"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerNoteRoute = registerNoteRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _notebooks2 = require("../../adaptors/notebooks");
var _saved_objects_notebooks_router = require("../../adaptors/notebooks/saved_objects_notebooks_router");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerNoteRoute(router) {
  // Fetch all the notebooks available
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    let notebooksData = [];
    try {
      notebooksData = await _notebooks2.BACKEND.viewNotes(opensearchNotebooksClient, _notebooks.wreckOptions);
      return response.ok({
        body: {
          data: notebooksData
        }
      });
    } catch (error) {
      console.log('Notebook:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Get all paragraphs of notebooks
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const notebookinfo = await _notebooks2.BACKEND.fetchNote(opensearchNotebooksClient, request.params.noteId, _notebooks.wreckOptions);
      return response.ok({
        body: notebookinfo
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Add a Notebook
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const addResponse = await _notebooks2.BACKEND.addNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: addResponse.message.objectId
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Rename a notebook
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/rename`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const renameResponse = await _notebooks2.BACKEND.renameNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: renameResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Clone a notebook
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/clone`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const cloneResponse = await _notebooks2.BACKEND.cloneNote(opensearchNotebooksClient, request.body, _notebooks.wreckOptions);
      return response.ok({
        body: cloneResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Delete notebooks
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/{noteList}`,
    validate: {
      params: _configSchema.schema.object({
        noteList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const delResponse = await _notebooks2.BACKEND.deleteNote(opensearchNotebooksClient, request.params.noteList, _notebooks.wreckOptions);
      return response.ok({
        body: delResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Add sample notebooks
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/addSampleNotebooks`,
    validate: {
      body: _configSchema.schema.object({
        visIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const addSampleNotesResponse = await _notebooks2.BACKEND.addSampleNotes(opensearchNotebooksClient, request.body.visIds, _notebooks.wreckOptions);
      return response.ok({
        body: addSampleNotesResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const notebooksData = await opensearchNotebooksClient.find({
        type: _observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT,
        perPage: 1000
      });
      const fetchedNotebooks = (0, _saved_objects_notebooks_router.fetchNotebooks)(notebooksData.saved_objects);
      return response.ok({
        body: {
          data: fetchedNotebooks
        }
      });
    } catch (error) {
      console.log('Notebook:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    let notebooksData;
    try {
      const newNotebookObject = (0, _saved_objects_notebooks_router.createNotebook)(request.body);
      notebooksData = await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, newNotebookObject);
      return response.ok({
        body: `${notebooksData.id}`
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const notebookinfo = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.noteId);
      return response.ok({
        body: notebookinfo.attributes.savedNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/clone`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const getNotebook = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId);
      const createCloneNotebook = (0, _saved_objects_notebooks_router.cloneNotebook)(getNotebook.attributes.savedNotebook, request.body.name);
      const createdNotebook = await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, createCloneNotebook);
      return response.ok({
        body: createdNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/rename`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const renamedNotebook = (0, _saved_objects_notebooks_router.renameNotebook)(request.body);
      const updatedNotebook = await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId, renamedNotebook);
      return response.ok({
        body: updatedNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const deletedNotebooks = await opensearchNotebooksClient.delete(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.noteId);
      return response.ok({
        body: deletedNotebooks
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/addSampleNotebooks`,
    validate: {
      body: _configSchema.schema.object({
        visIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const sampleNotebooks = await (0, _saved_objects_notebooks_router.addSampleNotes)(opensearchNotebooksClient, request.body.visIds);
      return response.ok({
        body: sampleNotebooks
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/migrate`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);
    try {
      const getNotebook = await _notebooks2.BACKEND.fetchNote(opensearchNotebooksClient, request.body.noteId, _notebooks.wreckOptions);
      const createCloneNotebook = (0, _saved_objects_notebooks_router.cloneNotebook)(getNotebook, request.body.name);
      const createdNotebook = await context.core.savedObjects.client.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, createCloneNotebook);
      return response.ok({
        body: createdNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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