"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processTaskError = exports.isIndexNotFoundError = exports.getTaskState = exports.getTaskInitProgress = exports.getLatestTaskForForecasterQuery = exports.getLatestForecasterTasksQuery = exports.getErrorMessage = exports.convertTaskAndJobFieldsToCamelCase = exports.convertStaticFieldsToCamelCase = exports.convertForecastKeysToSnakeCase = exports.convertForecastKeysToCamelCase = exports.combineTaskState = exports.buildEntityListQuery = void 0;
var _lodash = require("lodash");
var _constants = require("../../utils/constants");
var _helpers = require("../../utils/helpers");
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

const convertForecastKeysToSnakeCase = payload => {
  return {
    ...(0, _helpers.mapKeysDeep)({
      ...(0, _lodash.omit)(payload, ['filterQuery', 'featureAttributes']) // Exclude the filterQuery,
    }, _helpers.toSnake),
    filter_query: (0, _lodash.get)(payload, 'filterQuery', {}),
    ui_metadata: (0, _lodash.get)(payload, 'uiMetadata', {}),
    feature_attributes: (0, _lodash.get)(payload, 'featureAttributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregationQuery'])
      }, _helpers.toSnake),
      aggregation_query: feature.aggregationQuery
    }))
  };
};
exports.convertForecastKeysToSnakeCase = convertForecastKeysToSnakeCase;
const processTaskError = error => {
  const errorWithPrefixRemoved = error.replace(_constants.OPENSEARCH_EXCEPTION_PREFIX, '');
  return (0, _lodash.isEmpty)(errorWithPrefixRemoved) || errorWithPrefixRemoved.endsWith('.') ? errorWithPrefixRemoved : errorWithPrefixRemoved + '.';
};

// Following checks/transformations need to be made here:
// - set to INACTIVE_NOT_STARTED if there is no existing task for this forecaster
// - set to UNEXPECTED_FAILURE if the task is in a FAILED state & the error message is unreadable / is a stack trace
// - set to INITIALIZING_FORECAST if the task is in a CREATED state
// - set to INACTIVE_STOPPED if the task is in a STOPPED state
// - override INACTIVE states (INACTIVE_STOPPED or INACTIVE_NOT_STARTED) with test-related states
//   (INITIALIZING_TEST, TEST_COMPLETE, or INIT_TEST_FAILURE) if runOnceTask exists with these states
exports.processTaskError = processTaskError;
const combineTaskState = (realTimeTask, runOnceTask) => {
  const realTimeState = (0, _lodash.get)(realTimeTask, 'state', 'INACTIVE_NOT_STARTED');
  const updatedStateString = realTimeState === 'CREATED' ? 'INITIALIZING_FORECAST' : realTimeState === 'STOPPED' ? 'INACTIVE_STOPPED' : realTimeState;
  //@ts-ignore
  let updatedState = _constants.FORECASTER_STATE[updatedStateString];

  // at the beginning, runOnceTask is inactive before going into initializing test state
  // if runOnceTask is not empty and the error is empty, set the state to INIT_TEST
  const runOnceState = (0, _lodash.get)(runOnceTask, 'state', 'INACTIVE_NOT_STARTED');
  const runOnceStateError = (0, _lodash.get)(runOnceTask, 'error', '');
  const updatedRunOnceStateString = runOnceState === 'INACTIVE' && runOnceStateError === '' ? 'INIT_TEST' : runOnceState;
  const realTimeLastUpdateTime = (0, _lodash.get)(realTimeTask, 'last_update_time', 0);
  const runOnceLastUpdateTime = (0, _lodash.get)(runOnceTask, 'last_update_time', 0);

  // Check if current state is INACTIVE and runOnceTask has a priority state
  // when realTimeTask is not updated for a while
  if ((updatedState === _constants.FORECASTER_STATE.INACTIVE_STOPPED || updatedState === _constants.FORECASTER_STATE.INACTIVE_NOT_STARTED) && runOnceTask && runOnceLastUpdateTime > realTimeLastUpdateTime) {
    // Convert runOnceState to enum value before comparison
    const runOnceStateEnum = _constants.FORECASTER_STATE[updatedRunOnceStateString];
    if (runOnceStateEnum === _constants.FORECASTER_STATE.INIT_TEST || runOnceStateEnum === _constants.FORECASTER_STATE.TEST_COMPLETE || runOnceStateEnum === _constants.FORECASTER_STATE.INIT_TEST_FAILED) {
      updatedState = runOnceStateEnum;
    }
  }
  return updatedState;
};
exports.combineTaskState = combineTaskState;
const getTaskState = realTimeTask => {
  const state = (0, _lodash.get)(realTimeTask, 'state', 'INACTIVE_NOT_STARTED');
  const updatedStateString = state === 'CREATED' ? 'INITIALIZING_FORECAST' : state === 'STOPPED' ? 'INACTIVE_STOPPED' : state;
  //@ts-ignore
  return _constants.FORECASTER_STATE[updatedStateString];
};
exports.getTaskState = getTaskState;
const convertForecastKeysToCamelCase = response => {
  let camelCaseResponse = {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'ui_metadata', 'feature_query', 'feature_attributes', 'forecasterJob', 'runOnceTask']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    enabled: (0, _lodash.get)(response, 'adJob.enabled', false),
    enabledTime: (0, _lodash.get)(response, 'adJob.enabled_time'),
    disabledTime: (0, _lodash.get)(response, 'adJob.disabled_time'),
    categoryField: (0, _lodash.get)(response, 'category_field')
  };
  if (!(0, _lodash.isEmpty)((0, _lodash.get)(response, 'runOnceTask', {}))) {
    camelCaseResponse = {
      ...camelCaseResponse,
      //@ts-ignore
      taskId: (0, _lodash.get)(response, 'runOnceTask.task_id'),
      taskState: getTaskState((0, _lodash.get)(response, 'runOnceTask', {})),
      taskProgress: (0, _lodash.get)(response, 'runOnceTask.task_progress'),
      taskError: processTaskError((0, _lodash.get)(response, 'runOnceTask.error', '')),
      detectionDateRange: {
        startTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.start_time'),
        endTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.end_time')
      }
    };
  }
  return camelCaseResponse;
};
exports.convertForecastKeysToCamelCase = convertForecastKeysToCamelCase;
const isIndexNotFoundError = err => {
  return err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception';
};
exports.isIndexNotFoundError = isIndexNotFoundError;
const getErrorMessage = err => {
  return !(0, _lodash.isEmpty)((0, _lodash.get)(err, 'body.error.reason')) ? (0, _lodash.get)(err, 'body.error.reason') : (0, _lodash.get)(err, 'message');
};

// Filtering by 'is_latest=true' is not enough here. It is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.getErrorMessage = getErrorMessage;
const getLatestForecasterTasksQuery = realtime => {
  const taskTypes = realtime ? _constants.FORECAST_REALTIME_TASK_TYPES : _constants.FORECAST_RUN_ONCE_TASK_TYPES;
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    },
    aggs: {
      forecasters: {
        terms: {
          field: 'forecaster_id',
          size: _constants.MAX_FORECASTER
        },
        aggs: {
          latest_tasks: {
            top_hits: {
              size: 1,
              sort: {
                execution_start_time: _constants.SORT_DIRECTION.DESC
              }
            }
          }
        }
      }
    }
  };
};

// Converts the static forecaster fields into camelcase. Ignores any job or task-related fields
exports.getLatestForecasterTasksQuery = getLatestForecasterTasksQuery;
const convertStaticFieldsToCamelCase = response => {
  return {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'feature_query', 'feature_attributes', 'ui_metadata', 'forecaster_job', 'realtime_task', 'run_once_task']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    lastUiBreakingChangeTime: (0, _lodash.get)(response, 'last_ui_breaking_change_time'),
    lastUpdateTime: (0, _lodash.get)(response, 'last_update_time')
  };
};

// Filtering by 'is_latest=true' is not enough here. It is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.convertStaticFieldsToCamelCase = convertStaticFieldsToCamelCase;
const getLatestTaskForForecasterQuery = (forecasterId, realtime) => {
  const taskTypes = realtime ? _constants.FORECAST_REALTIME_TASK_TYPES : _constants.FORECAST_RUN_ONCE_TASK_TYPES;
  return {
    size: 1,
    sort: {
      execution_start_time: _constants.SORT_DIRECTION.DESC
    },
    query: {
      bool: {
        filter: [{
          term: {
            forecaster_id: forecasterId
          }
        }, {
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    }
  };
};
exports.getLatestTaskForForecasterQuery = getLatestTaskForForecasterQuery;
const getTaskInitProgress = task => {
  if ((task === null || task === void 0 ? void 0 : task.init_progress) !== undefined) {
    return {
      percentageStr: `${((0, _lodash.get)(task, 'init_progress', 0) * 100).toFixed(0)}%`,
      estimatedMinutesLeft: task.estimated_minutes_left
    };
  }
  return undefined;
};

// Converts the task-related detector fields into camelcase
exports.getTaskInitProgress = getTaskInitProgress;
const convertTaskAndJobFieldsToCamelCase = (realtimeTask, runOnceTask, forecasterJob) => {
  // Populate Forecaster job fields
  return {
    enabled: (0, _lodash.get)(forecasterJob, 'enabled', false),
    enabledTime: (0, _lodash.get)(forecasterJob, 'enabled_time'),
    disabledTime: (0, _lodash.get)(forecasterJob, 'disabled_time'),
    curState: combineTaskState(realtimeTask, runOnceTask),
    stateError: (0, _lodash.get)(realtimeTask, 'error') || (0, _lodash.get)(runOnceTask, 'error'),
    initProgress: getTaskInitProgress(realtimeTask),
    realTimeLastUpdateTime: (0, _lodash.get)(realtimeTask, 'last_update_time'),
    runOnceLastUpdateTime: (0, _lodash.get)(runOnceTask, 'last_update_time'),
    taskId: (0, _lodash.get)(runOnceTask, 'task_id'),
    taskState: getTaskState(runOnceTask),
    taskProgress: (0, _lodash.get)(runOnceTask, 'task_progress'),
    taskError: processTaskError((0, _lodash.get)(runOnceTask, 'error', ''))
  };
};

/**
 * Builds an OpenSearch query for matching multiple ForecastEntity objects
 * Each entity's key-value pairs are grouped with "must" clauses
 * Different entities are combined with "should" clauses
 */
exports.convertTaskAndJobFieldsToCamelCase = convertTaskAndJobFieldsToCamelCase;
const buildEntityListQuery = entityList => {
  if (!entityList || entityList.length === 0) {
    return undefined;
  }

  // Create an array of bool queries - one for each entity
  const shouldClauses = entityList.map(entity => {
    // For each entity, create nested queries for each key-value pair
    const mustClauses = Object.entries(entity).map(([name, value]) => {
      return {
        nested: {
          path: "entity",
          query: {
            bool: {
              must: [{
                term: {
                  "entity.name": name
                }
              }, {
                term: {
                  "entity.value": value
                }
              }]
            }
          },
          ignore_unmapped: false,
          score_mode: "avg"
        }
      };
    });

    // All key-value pairs for this entity must match
    return {
      bool: {
        must: mustClauses
      }
    };
  });

  // At least one entity should match
  return {
    bool: {
      should: shouldClauses,
      minimum_should_match: 1
    }
  };
};
exports.buildEntityListQuery = buildEntityListQuery;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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