/*************************************************************************
 *
 *  $RCSfile: silitem.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: kso $ $Date: 2001/07/25 15:09:42 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _CHAOS_SILITEM_HXX
#define _CHAOS_SILITEM_HXX

#ifndef _COM_SUN_STAR_UCB_SENDINFO_HPP_
#include <com/sun/star/ucb/SendInfo.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_SENDMEDIATYPES_HPP_
#include <com/sun/star/ucb/SendMediaTypes.hpp>
#endif

#ifndef _INETTYPE_HXX
#include <svtools/inettype.hxx>
#endif
#ifndef _LIST_HXX
#include <tools/list.hxx>
#endif
#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif

#ifndef _CASTMACS_HXX
#include <castmacs.hxx>
#endif
#ifndef _RCPNITEM_HXX
#include <rcpnitem.hxx>
#endif

namespace chaos {

//============================================================================
/// A template for items that map objects of one type to objects of another
/// type.
///
/// @descr  Because of the use of nonstandard compilers, the three types
/// Entry::first_type (abbreviated Eft), Entry::second_type (abbreivated Est),
/// and Entry::rep_type (abbreviated Ert) have to be explicitly spelled out as
/// template parameters of CntMapItem.  With the use of only standard
/// compilers, this would not be necessary, and instead the two typedefs for
/// key_type and data_type would use the typename keyword with
/// Entry::first_type and Entry::second_type, respectively, and the methods of
/// CntMapItem would use the typename keyword and Entry::rep_type instead of
/// Ert.
///
/// @descr  Ideally, this class would be derived from
/// std::map<Entry::first_type, Entry::second_type> instead of List.  Then,
/// findEntry() would be replaced by a method find() returning a pair.
///
/// @descr  The template parameter Entry must be a class with the following
/// public properties:
/// - A type first_type, the type of the 'key' objects.
/// - A type second_type, the type of the 'data' objects.
/// - A type rep_type, the element type of the sequences for QueryValue() and
///   PutValue().
/// - A variable first holding an object of first_type.
/// - A variable second holding an object of second_type.
/// - A default constructor.
/// - A copy constructor.
/// - A constructor with an object of first_type and an object of second_type,
///   in that order.
/// - A destructor.
/// - A const method getRep() returning an object of rep_type.
/// - A method setRep() accepting an object of rep_type and returning success
///   (true) or failure (false).
/// - A const method write() accepting a stream.
/// - A method read() accepting a stream and returning success (true) or
///   failure (false).
template<class Entry, class Eft, class Est, class Ert>
class CntMapItem: public SfxPoolItem, private List
{
public:
	/// The type of the 'key' objects (those the mapping is from).
	typedef Eft key_type;

	/// The type of the 'data' objects (those the mapping is to).
	typedef Est data_type;

	/// An integral type large enough to hold all possible sizes (i.e., number
	/// of key/data pairs) of a mapping.
	typedef ULONG size_type;

	typedef CntMapItem<Entry, Eft, Est, Ert> self_type;

private:
	const Entry & getEntry(size_type nIndex) const
	{ return *INSECURE_DYNAMIC_CAST(Entry *, GetObject(nIndex)); }

	Entry & getEntry(size_type nIndex)
	{ return *INSECURE_DYNAMIC_CAST(Entry *, GetObject(nIndex)); }

	void eraseEntry(size_type nIndex)
	{ delete INSECURE_DYNAMIC_CAST(Entry *, Remove(nIndex)); }

public:
	/// The support for pseudo RTTI.
	///
	/// @descr  There must be a corresponding TYPEINIT...() definition for
	/// each template instantiation class.
	TYPEINFO();

	/// Create an item with an empty mapping.
	///
	/// @param nWhich  The Which-ID of the item.
	CntMapItem(USHORT nWhich = 0): SfxPoolItem(nWhich) {}

	/// The copy constructor.
	///
	/// @descr  Each item owns its mapping and all the entries of the mapping,
	/// so copies of those objects are create.
	///
	/// @param rItem  Some mapping item.
	inline CntMapItem(const self_type & rItem);

	/// The destructor.
	///
	/// @descr  Each item owns its mapping and all the entries of the mapping,
	/// so those objects are deleted.
	virtual ~CntMapItem() { clear(); }

	/// The assignment operator.
	///
	/// @descr  Each item owns its mapping and all the entries of the mapping,
	/// so copies of those objects are create.
	///
	/// @param rItem  Some mapping item.
	///
	/// @return  This item.
	inline self_type & operator =(const self_type & rItem);

	/// Test for equality of items.
	///
	/// @param rType  Some item.
	///
	/// @return  TRUE if rItem is a mapping item with the same key and data
	/// types, the same Which-ID, and the same mapping as this item.
	virtual inline int operator ==(const SfxPoolItem & rItem) const;

	virtual USHORT GetVersion(USHORT) const
	{ return 1; } // because it uses SfxPoolItem::read/writeUnicodeString()

	/// Get the mapping of this item.
	///
	/// @param rVal  The object to store the mapping in, in the form of a
	/// sequence of Entry::rep_type objects.
	///
	/// @return  TRUE.
	virtual inline BOOL QueryValue(com::sun::star::uno::Any & rVal, BYTE = 0) const;

	/// Set the mapping of this item.
	///
	/// @param rVal  Should be a sequence of Entry::rep_type objects.
	///
	/// @return  TRUE if rVal has the right reflection, and the mapping of
	/// this item has thus been changed.
	virtual inline BOOL PutValue(const com::sun::star::uno::Any & rVal, BYTE = 0);

	/// Read a mapping item from a stream.
	///
	/// @param rStream  A stream to read the presentation of a mapping item
	/// from.
	///
	/// @return  Either a new mapping item (if a valid presentation could be
	/// read from the stream), or 0.
	virtual inline SfxPoolItem * Create(SvStream & rStream, USHORT nVersion)
		const;

	/// Write this item into a stream.
	///
	/// @param rStream  A stream to write the presentation of this item into.
	///
	/// @return  rStream.
	virtual inline SvStream & Store(SvStream & rStream, USHORT) const;

	/// Create a copy of this item.
	///
	/// @descr  Each item owns its mapping and all the entries of the mapping,
	/// so copies of those objects are create.
	virtual inline SfxPoolItem * Clone(SfxItemPool * = 0) const
	{ return new self_type(*this); }


	/// Get the size (i.e., number of key/data pairs) of this mapping.
	///
	/// @return  The size of this mapping.
	size_type size() const { return Count(); }

	/// Get the data object associated with a given key object.
	///
	/// @param rKey  Some key object.
	///
	/// @return  The data object associated with rKey.  If no entry exists
	/// for rKey, one containing a default data object is created.
	inline data_type & operator [](const key_type & rKey);

	/// Erase the entry for a given key object.
	///
	/// @param rKey  Some key object.
	///
	/// @return  1 if an entry existed for rKey, 0 otherwise.
	inline size_type erase(const key_type & rKey);

	/// Clear the mapping (erase all entries).
	inline void clear();

	/// Find the data object associated with a given key object.
	///
	/// @param rKey  Some key object.
	///
	/// @return  The data object associated with rKey, or 0 if no entry exists
	/// for rKey.
	inline const data_type * findEntry(const key_type & rKey) const;
};

template<class Entry, class Eft, class Est, class Ert>
inline CntMapItem<Entry, Eft, Est, Ert>::CntMapItem(const self_type & rItem):
	SfxPoolItem(rItem),
	List(rItem.size())
{
	for (size_type i = 0; i < rItem.size();)
		Insert(new Entry(rItem.getEntry(i++)), LIST_APPEND);
}

template<class Entry, class Eft, class Est, class Ert>
inline
CntMapItem<Entry, Eft, Est, Ert>::self_type &
CntMapItem<Entry, Eft, Est, Ert>::operator =(const self_type & rItem)
{
	if (&rItem != this)
	{
		SetWhich(rItem.Which());
		clear();
		for (size_type i = 0; i < rItem.size();)
			Insert(new Entry(rItem.getEntry(i++)), LIST_APPEND);
	}
	return *this;
}

// virtual
template<class Entry, class Eft, class Est, class Ert>
inline
int CntMapItem<Entry, Eft, Est, Ert>::operator ==(const SfxPoolItem & rItem)
	const
{
	if (Which() != rItem.Which())
		return FALSE;
	const self_type * pMapItem = PTR_CAST(self_type, &rItem);
	if (!(pMapItem && size() == pMapItem->size()))
		return FALSE;
	for (size_type i = 0; i < size();)
	{
		const Entry & rEntry = getEntry(i++);
		const data_type * pData = pMapItem->findEntry(rEntry.first);
		if (!(pData && *pData == rEntry.second))
			return FALSE;
	}
	return TRUE;
}

// virtual
template<class Entry, class Eft, class Est, class Ert>
inline BOOL CntMapItem<Entry, Eft, Est, Ert>::QueryValue(
										com::sun::star::uno::Any & rVal, BYTE)
	const
{
	com::sun::star::uno::Sequence<Ert> aSeq(size());
	Ert * pSeqArray = aSeq.getArray();
	for (size_type i = 0; i < size();)
		*pSeqArray++ = getEntry(i++).getRep();
	rVal <<= aSeq;
	return TRUE;
}

// virtual
template<class Entry, class Eft, class Est, class Ert>
inline
BOOL CntMapItem<Entry, Eft, Est, Ert>::PutValue(
								const com::sun::star::uno::Any & rVal, BYTE)
{
	com::sun::star::uno::Sequence<Ert> aSeq;
	if (!(rVal >>= aSeq))
		return FALSE;
	clear();
	const Ert * pSeqArray = aSeq.getConstArray();
	for (sal_Int32 i = aSeq.getLength(); i--;)
	{
		Entry aEntry;
		if (aEntry.setRep(*pSeqArray++))
			(*this)[aEntry.first] = aEntry.second;
	}
	return TRUE;
}

// virtual
template<class Entry, class Eft, class Est, class Ert>
inline
SfxPoolItem * CntMapItem<Entry, Eft, Est, Ert>::Create(SvStream & rStream,
													   USHORT nVersion)
	const
{
	self_type * pItem = new self_type(Which());
	USHORT nSize = 0;
	rStream >> nSize;
	for (USHORT i = 0; i < nSize; ++i)
	{
		Entry aEntry;
		if (!aEntry.read(rStream, nVersion >= 1))
		{
			delete pItem;
			return 0;
		}
		(*pItem)[aEntry.first] = aEntry.second;
	}
	return pItem;
}

// virtual
template<class Entry, class Eft, class Est, class Ert>
inline
SvStream & CntMapItem<Entry, Eft, Est, Ert>::Store(SvStream & rStream, USHORT)
	const
{
	size_type nSize = size();
	if (nSize > USHRT_MAX)
		nSize = USHRT_MAX;
	rStream << USHORT(nSize);
	for (size_type i = 0; nSize--;)
		getEntry(i++).write(rStream);
	return rStream;
}

template<class Entry, class Eft, class Est, class Ert>
inline
CntMapItem<Entry, Eft, Est, Ert>::data_type &
CntMapItem<Entry, Eft, Est, Ert>::operator [](const key_type & rKey)
{
	for (size_type i = 0; i < size();)
	{
		Entry & rEntry = getEntry(i++);
		if (rEntry.first == rKey)
			return rEntry.second;
	}
	Entry * pEntry = new Entry(rKey, data_type());
	Insert(pEntry, LIST_APPEND);
	return pEntry->second;
}

template<class Entry, class Eft, class Est, class Ert>
inline
CntMapItem<Entry, Eft, Est, Ert>::size_type
CntMapItem<Entry, Eft, Est, Ert>::erase(const key_type & rKey)
{
	for (size_type i = 0; i < size(); ++i)
		if (getEntry(i).first == rKey)
		{
			eraseEntry(i);
			return 1;
		}
	return 0;
}

template<class Entry, class Eft, class Est, class Ert>
inline void CntMapItem<Entry, Eft, Est, Ert>::clear()
{
	while (size())
		eraseEntry(size() - 1);
}

template<class Entry, class Eft, class Est, class Ert>
inline
const CntMapItem<Entry, Eft, Est, Ert>::data_type *
CntMapItem<Entry, Eft, Est, Ert>::findEntry(const key_type & rKey) const
{
	for (size_type i = 0; i < size();)
	{
		const Entry & rEntry = getEntry(i++);
		if (rEntry.first == rKey)
			return &rEntry.second;
	}
	return 0;
}

//============================================================================
/// A pair of a CntOutMsgProtocolType and a String, used to define an item
/// mapping from CntOutMsgProtocolType to String (i.e., used as the template
/// parameter Entry of a CntMapItem).
///
/// @descr  Ideally, this class would be derived from
/// std::pair<CntOutMsgProtoclType, String>.
struct CntSendInfoListEntry
{
	typedef CntOutMsgProtocolType first_type;

	typedef String second_type;

	typedef com::sun::star::ucb::SendInfo rep_type;

	first_type first;
	second_type second;

	CntSendInfoListEntry(): first(CNT_OUTMSG_PROTOCOL_END) {}

	CntSendInfoListEntry(first_type eProtocol, const second_type & rValue):
		first(eProtocol), second(rValue) {}

	rep_type getRep() const;

	BOOL setRep(const rep_type & rInfo);

	void write(SvStream & rStream) const;

	BOOL read(SvStream & rStream, bool bUnicode);
};

//============================================================================
/// An item mapping from CntOutMsgProtocolType to String.
typedef CntMapItem<CntSendInfoListEntry,
	               CntSendInfoListEntry::first_type,
	               CntSendInfoListEntry::second_type,
	               CntSendInfoListEntry::rep_type>
CntSendInfoListItem;

//============================================================================
/// An Internet media type (like "text/plain" and "application/octet-stream").
class CntMediaType
{
	enum { NOT_CANONIC = -2, NOT_DETERMINED = -1 };

	MUTABLE String m_aCanonic;
	MUTABLE INetContentType m_eID;

public:
	/// Construct an instance from a string representation of the media type.
	///
	/// @param rType  A valid Internet media type (RFCs 2045, 2046).  It can
	/// be in any combination of upper and lower case letters, but it may not
	/// include any whitespace at the front or end, or around the '/'.
	CntMediaType(const String & rType):
		m_aCanonic(rType), m_eID(INetContentType(NOT_CANONIC)) {}

	/// Construct an instance from an identifier representing a media type.
	///
	/// @param eTheID  An identifier representing a media type.  This should
	/// not be CONTENT_TYPE_UNKNOWN.
	CntMediaType(INetContentType eTheID): m_eID(eTheID) {}

	/// Test for equality of media types.
	///
	/// @param rType  Some media type.
	///
	/// @return  TRUE if this media type and rType are equal, i.e., if they
	/// have the same canonic string representation.
	inline BOOL operator ==(const CntMediaType & rType) const;

	/// Test for inequality of media types.
	///
	/// @param rType  Some media type.
	///
	/// @return  TRUE if this media type and rType are inequal, i.e., if they
	/// have different canonic string representations.
	BOOL operator !=(const CntMediaType & rType) const
	{ return !(*this == rType); }

	/// Get the canonic string representation.
	///
	/// @return  The canonic (i.e., all lower case) string representation of
	/// this media type.
	inline const String & getCanonic() const;

	/// Get the identifier representing this media type.
	///
	/// @return  The identifier representing this media type, or
	/// CONTENT_TYPE_UNKNOWN if none is defined.
	inline INetContentType getID() const;
};

inline BOOL CntMediaType::operator ==(const CntMediaType & rType) const
{
	return getID() == rType.getID()
	       && (m_eID != CONTENT_TYPE_UNKNOWN
			   || getCanonic() == rType.getCanonic());
}

inline const String & CntMediaType::getCanonic() const
{
	if (m_eID == NOT_CANONIC)
	{
		MUTATE(CntMediaType, m_aCanonic).ToLowerAscii();
		MUTATE(CntMediaType, m_eID) = INetContentType(NOT_DETERMINED);
	}
	else if (m_aCanonic.Len() == 0)
	{
		MUTATE(CntMediaType, m_aCanonic)
			= INetContentTypes::GetContentType(m_eID);
		MUTATE(CntMediaType, m_aCanonic).ToLowerAscii();
	}
	return m_aCanonic;
}

inline INetContentType CntMediaType::getID() const
{
	switch (m_eID)
	{
		case NOT_CANONIC:
			MUTATE(CntMediaType, m_aCanonic).ToLowerAscii();
		case NOT_DETERMINED:
			MUTATE(CntMediaType, m_eID)
			 = INetContentTypes::GetContentType(m_aCanonic);
			break;
	}
	return m_eID;
}

//============================================================================
/// A set of Internet media types.
///
/// @descr  Ideally, this would be
///   typedef std::set<String> CntMediaTypeSet;
/// Then, getEntry() would be replaced by iteration, and findEntry() would be
/// replaced by a method find() returning a pair.
class CntMediaTypeSet: private List
{
public:
	typedef CntMediaType key_type;

	typedef CntMediaType value_type;

	typedef ULONG size_type;

	CntMediaTypeSet() {}

	CntMediaTypeSet(const CntMediaTypeSet & rSet);

	~CntMediaTypeSet() { clear(); }

	CntMediaTypeSet & operator =(const CntMediaTypeSet & rSet);

	BOOL operator ==(const CntMediaTypeSet & rSet) const;

	size_type size() const { return Count(); }

	void insert(const value_type & rEntry);

	void clear();

	const value_type & getEntry(size_type nIndex) const
	{ return *INSECURE_DYNAMIC_CAST(value_type *, GetObject(nIndex)); }

	BOOL findEntry(const key_type & rKey) const;
};

//============================================================================
/// A pair of a CntOutMsgProtocolType and a CntMediaTypeSet, used to define
/// an item mapping from CntOutMsgProtocolType to CntMediaTypeSet (i.e., used
/// as the template parameter Entry of a CntMapItem).
///
/// @descr  Ideally, this class would be derived from
/// std::pair<CntOutMsgProtoclType, CntMediaTypeSet>.
struct CntSendMediaTypesEntry
{
	typedef CntOutMsgProtocolType first_type;

	typedef CntMediaTypeSet second_type;

	typedef com::sun::star::ucb::SendMediaTypes rep_type;

	first_type first;
	second_type second;

	CntSendMediaTypesEntry(): first(CNT_OUTMSG_PROTOCOL_END) {}

	CntSendMediaTypesEntry(first_type eProtocol, const second_type & rValue):
		first(eProtocol), second(rValue) {}

	rep_type getRep() const;

	BOOL setRep(const rep_type & rMediaTypes);

	void write(SvStream & rStream) const;

	BOOL read(SvStream & rStream, bool bUnicode);
};

//============================================================================
/// An item mapping from CntOutMsgProtocolType to CntMediaTypeSet.
typedef CntMapItem<CntSendMediaTypesEntry,
	               CntSendMediaTypesEntry::first_type,
	               CntSendMediaTypesEntry::second_type,
	               CntSendMediaTypesEntry::rep_type>
CntSendMediaTypesItem;

}

#endif // _CHAOS_SILITEM_HXX

