/*************************************************************************
 *
 *  $RCSfile: cntrules.cxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: hr $ $Date: 2001/10/12 16:38:56 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <algorithm>
#include <limits>

#ifndef _COM_SUN_STAR_UCB_RULESET_HPP_
#include <com/sun/star/ucb/RuleSet.hpp>
#endif
#ifndef _COM_SUN_STAR_UNO_SEQUENCE_HXX_
#include <com/sun/star/uno/Sequence.hxx>
#endif
#ifndef _COM_SUN_STAR_UTIL_DATE_HPP_
#include <com/sun/star/util/Date.hpp>
#endif
#ifndef _SVTOOLS_CENUMITM_HXX
#include <svtools/cenumitm.hxx>
#endif
#ifndef _SVTOOLS_CINTITEM_HXX
#include <svtools/cintitem.hxx>
#endif
#ifndef _SVTOOLS_CTYPEITM_HXX
#include <svtools/ctypeitm.hxx>
#endif
#ifndef _SVTOOLS_CUSTRITM_HXX
#include <svtools/custritm.hxx>
#endif
#ifndef _DATETIMEITEM_HXX
#include <svtools/dateitem.hxx>
#endif
#ifndef _SFXITEMSET_HXX
#include <svtools/itemset.hxx>
#endif
#ifndef _TOOLS_INTN_HXX
#include <tools/intn.hxx>
#endif
#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif
#ifndef _UNOTOOLS_TEXTSEARCH_HXX
#include <unotools/textsearch.hxx>
#endif
#ifndef _UNOTOOLS_INTLWRAPPER_HXX
#include <unotools/intlwrapper.hxx>
#endif

#ifndef _CNTPOOL_HXX
#include <cntpool.hxx>
#endif
#ifndef _CHAOS_CNTRULES_HXX
#include <cntrules.hxx>
#endif

using namespace chaos;
using namespace com::sun::star;

//============================================================================
namespace chaos {

class OutputGuard_Impl
{
	SvStream & m_rStream;
	sal_uInt32 m_nBegin;

public:
	inline OutputGuard_Impl(SvStream & rTheStream);

	inline ~OutputGuard_Impl();
};

inline OutputGuard_Impl::OutputGuard_Impl(SvStream & rTheStream):
	m_rStream(rTheStream),
	m_nBegin(rTheStream.Tell())
{
	m_rStream << m_nBegin; // write place holder for end
}

inline OutputGuard_Impl::~OutputGuard_Impl()
{
	sal_uInt32 nEnd = m_rStream.Tell();
	m_rStream.Seek(m_nBegin);
	m_rStream << nEnd;
	m_rStream.Seek(nEnd);
}

//============================================================================
class InputGuard_Impl
{
	SvStream & m_rStream;
	sal_uInt32 m_nEnd;

public:
	InputGuard_Impl(SvStream & rTheStream): m_rStream(rTheStream), m_nEnd(0)
	{ m_rStream >> m_nEnd; }

	inline ~InputGuard_Impl();
};

inline InputGuard_Impl::~InputGuard_Impl()
{
	DBG_ASSERT(m_nEnd >= m_rStream.Tell(),
			   "InputGuard_Impl::~InputGuard_Impl(): Read past end of data");
	m_rStream.Seek(m_nEnd);
}

}

//============================================================================
//
//  class CntNodeRuleTerm
//
//============================================================================

inline StringCompare CntNodeRuleTerm::compare(String const & rArgument,
                                              IntlWrapper const &
                                                  rIntlWrapper) const
{
    sal_Int32 nComp;
    if ( m_bCaseSensitive )
        nComp = rIntlWrapper.getCaseCollator()->compareString(
            rArgument, m_aStringOperand );
    else
        nComp = rIntlWrapper.getCollator()->compareString(
            rArgument, m_aStringOperand );
    if ( nComp < 0 )
        return COMPARE_LESS;
    if ( nComp > 0 )
        return COMPARE_GREATER;
    return COMPARE_EQUAL;
}

//============================================================================
inline
CntNodeRuleTerm::SearchResult CntNodeRuleTerm::search(String const &
													      rArgument,
                                                      IntlWrapper const &
                                                          rIntlWrapper)
	const
{
	xub_StrLen nStart = 0;
	xub_StrLen nEnd = rArgument.Len();
	return
		utl::TextSearch(utl::SearchParam(m_aStringOperand,
										 m_bRegexp ?
										     utl::SearchParam::SRCH_REGEXP :
										     utl::SearchParam::SRCH_NORMAL,
										 m_bCaseSensitive),
                        rIntlWrapper.getLanguage()).
		            SearchFrwrd(rArgument, &nStart, &nEnd)
		        == 0 ?
		    SEARCH_CONTAINS_NOT :
		nStart > 0 || nEnd < rArgument.Len() ?
		    SEARCH_CONTAINS :
		    SEARCH_MATCHES;
}

//============================================================================
bool CntNodeRuleTerm::matchString(String const & rArgument,
                                  IntlWrapper const & rIntlWrapper) const
{
	switch (m_eOperator)
	{
		case CNT_NODE_RULE_OPERATOR_CONTAINS:
            return search(rArgument, rIntlWrapper) != SEARCH_CONTAINS_NOT;

		case CNT_NODE_RULE_OPERATOR_CONTAINSNOT:
            return search(rArgument, rIntlWrapper) == SEARCH_CONTAINS_NOT;

		case CNT_NODE_RULE_OPERATOR_GREATEREQUAL:
			return !m_bRegexp
                   && compare(rArgument, rIntlWrapper) != COMPARE_LESS;

		case CNT_NODE_RULE_OPERATOR_LESSEQUAL:
			return !m_bRegexp
                   && compare(rArgument, rIntlWrapper) != COMPARE_GREATER;

		case CNT_NODE_RULE_OPERATOR_EQUAL:
			return m_bRegexp ?
                    search(rArgument, rIntlWrapper) == SEARCH_MATCHES :
                    compare(rArgument, rIntlWrapper) == COMPARE_EQUAL;

		case CNT_NODE_RULE_OPERATOR_NOTEQUAL:
			return m_bRegexp ?
                    search(rArgument, rIntlWrapper) != SEARCH_MATCHES :
                    compare(rArgument, rIntlWrapper) != COMPARE_EQUAL;

		default:
			return false;
	}
}

//============================================================================
bool CntNodeRuleTerm::matchDate(Date const & rArgument) const
{
	if ( ! rArgument.IsValid() )
		return FALSE;

	switch (m_eOperator)
	{
		case CNT_NODE_RULE_OPERATOR_GREATEREQUAL:
			return (rArgument >= GetDateOperand()) != false;

		case CNT_NODE_RULE_OPERATOR_LESSEQUAL:
			return (rArgument <= GetDateOperand()) != false;

		case CNT_NODE_RULE_OPERATOR_EQUAL:
			return (rArgument == GetDateOperand()) != false;

		case CNT_NODE_RULE_OPERATOR_NOTEQUAL:
			return (rArgument != GetDateOperand()) != false;

		default:
			return false;
	}
}

//============================================================================
bool CntNodeRuleTerm::matchBool(bool bArgument) const
{
	switch (m_eOperator)
	{
		case CNT_NODE_RULE_OPERATOR_TRUE:
			return bArgument;

		case CNT_NODE_RULE_OPERATOR_FALSE:
			return !bArgument;

		default:
			return false;
	}
}

//============================================================================
bool CntNodeRuleTerm::matchNumeric(sal_Int32 nArgument) const
{
	switch (m_eOperator)
	{
		case CNT_NODE_RULE_OPERATOR_GREATEREQUAL:
			return nArgument >= GetNumericOperand();

		case CNT_NODE_RULE_OPERATOR_LESSEQUAL:
			return nArgument <= GetNumericOperand();

		case CNT_NODE_RULE_OPERATOR_EQUAL:
			return nArgument == GetNumericOperand();

		case CNT_NODE_RULE_OPERATOR_NOTEQUAL:
			return nArgument != GetNumericOperand();

		default:
			return false;
	}
}

//============================================================================
bool CntNodeRuleTerm::matchDateNumeric(Date const & rArgument,
									   Date const * pReferenceDate) const
{
	if ( ! rArgument.IsValid() )
		return FALSE;

	Date * pDate = pReferenceDate ? new Date(*pReferenceDate) : new Date;
	*pDate -= GetNumericOperand();
	bool bResult;
	switch (m_eOperator)
	{
		case CNT_NODE_RULE_OPERATOR_GREATEREQUAL:
			bResult = (rArgument >= *pDate) != false;
			break;

		case CNT_NODE_RULE_OPERATOR_LESSEQUAL:
			bResult = (rArgument <= *pDate) != false;
			break;

		default:
			bResult = false;
	}
	delete pDate;
	return bResult;
}

//============================================================================
CntNodeRuleTerm::CntNodeRuleTerm(CntNodeRule * pRule, USHORT nThePropertyWID,
								 CntNodeRuleOperator eTheOperator,
								 String const & rTheStringOperand,
								 bool bTheCaseSensitive, bool bTheRegexp):
	m_eOperator(eTheOperator),
	m_nPropertyWID(nThePropertyWID),
	m_eOperandType(CNT_NODE_RULE_OPERANDTYPE_STRING),
	m_aStringOperand(rTheStringOperand),
	m_bCaseSensitive(bTheCaseSensitive),
	m_bRegexp(bTheRegexp),
	m_aDateOperand(0)
{
	if (pRule) pRule->InsertTerm(this);
}

//============================================================================
CntNodeRuleTerm::CntNodeRuleTerm(CntNodeRule * pRule, USHORT nThePropertyWID,
								 CntNodeRuleOperator eTheOperator,
								 Date const & rTheDateOperand):
	m_eOperator(eTheOperator),
	m_nPropertyWID(nThePropertyWID),
	m_eOperandType(CNT_NODE_RULE_OPERANDTYPE_DATE),
	m_aDateOperand(rTheDateOperand)
{
	if (pRule) pRule->InsertTerm(this);
}

//============================================================================
CntNodeRuleTerm::CntNodeRuleTerm(CntNodeRule * pRule, USHORT nThePropertyWID,
								 CntNodeRuleOperator eTheOperator):
	m_eOperator(eTheOperator),
	m_nPropertyWID(nThePropertyWID),
	m_eOperandType(CNT_NODE_RULE_OPERANDTYPE_BOOL),
	m_aDateOperand(0)
{
	if (pRule) pRule->InsertTerm(this);
}

//============================================================================
CntNodeRuleTerm::CntNodeRuleTerm(CntNodeRule * pRule, USHORT nThePropertyWID,
								 CntNodeRuleOperator eTheOperator,
								 sal_Int32 nTheNumericOperand):
	m_eOperator(eTheOperator),
	m_nPropertyWID(nThePropertyWID),
	m_eOperandType(CNT_NODE_RULE_OPERANDTYPE_NUMERIC),
	m_aDateOperand(0),
	m_nNumericOperand(nTheNumericOperand)
{
	if (pRule) pRule->InsertTerm(this);
}

//============================================================================
bool CntNodeRuleTerm::operator ==(CntNodeRuleTerm const & rTerm) const
{
	if (m_eOperator != rTerm.m_eOperator
		|| m_nPropertyWID != rTerm.m_nPropertyWID
		|| m_eOperandType != rTerm.m_eOperandType)
		return false;
	switch (m_eOperandType)
	{
		case CNT_NODE_RULE_OPERANDTYPE_STRING:
			return m_bCaseSensitive == rTerm.m_bCaseSensitive
		           && m_bRegexp == rTerm.m_bRegexp
		           && m_aStringOperand == rTerm.m_aStringOperand;

		case CNT_NODE_RULE_OPERANDTYPE_DATE:
			return (m_aDateOperand == rTerm.m_aDateOperand) != false;

		case CNT_NODE_RULE_OPERANDTYPE_BOOL:
			return true;

		default: // CNT_NODE_RULE_OPERANDTYPE_NUMERIC
			return m_nNumericOperand == rTerm.m_nNumericOperand;
	}
}

//============================================================================
bool CntNodeRuleTerm::matches(SfxPoolItem const & rArgument,
                              IntlWrapper const & rIntlWrapper,
							  Date const * pReferenceDate) const
{
	switch (m_eOperandType)
	{
		case CNT_NODE_RULE_OPERANDTYPE_STRING:
			if (rArgument.ISA(CntUnencodedStringItem)
				&& !rArgument.ISA(CntContentTypeItem))
				return
					matchString(static_cast< CntUnencodedStringItem const * >(
						                &rArgument)->
								    GetValue(),
                                rIntlWrapper);
			else
			{
				String aStringArgument;
				rArgument.GetPresentation(SFX_ITEM_PRESENTATION_NAMEONLY,
										  SFX_MAPUNIT_APPFONT,
										  SFX_MAPUNIT_APPFONT,
                                          aStringArgument, &rIntlWrapper);
                return matchString(aStringArgument, rIntlWrapper);
			}

		case CNT_NODE_RULE_OPERANDTYPE_DATE:
			return rArgument.ISA(SfxDateTimeItem)
				   && matchDate(static_cast< SfxDateTimeItem const * >(
					                    &rArgument)->
								    GetDateTime());

		case CNT_NODE_RULE_OPERANDTYPE_BOOL:
			return
				rArgument.ISA(CntBoolItem)
				&& matchBool(static_cast< CntBoolItem const * >(&rArgument)->
							         GetValue()
							     != false);

		default: // CNT_NODE_RULE_OPERANDTYPE_NUMERIC
			if (rArgument.ISA(CntUInt32Item))
			{
				sal_Int32 nNumericArgument
					= sal_Int32(static_cast< CntUInt32Item const * >(
						                &rArgument)->
								    GetValue());
				return
					matchNumeric(nNumericArgument >= 0 ?
								     nNumericArgument :
								     std::numeric_limits< sal_Int32 >::max());
			}
			else if (rArgument.ISA(CntUInt16Item))
				return matchNumeric(static_cast< CntUInt16Item const * >(
					                        &rArgument)->
									    GetValue());
			else if (rArgument.ISA(CntEnumItem))
				return matchNumeric(static_cast< CntEnumItem const * >(
					                        &rArgument)->
									    GetValue());
			else return rArgument.ISA(SfxDateTimeItem)
				        && matchDateNumeric(
							  static_cast< SfxDateTimeItem const * >(
								      &rArgument)->
							      GetDateTime(),
							  pReferenceDate);
	}
}

//============================================================================
void CntNodeRuleTerm::store(SvStream & rStream) const
{
	OutputGuard_Impl aGuard(rStream);
	rStream << sal_uInt16(2); // version
	switch (m_eOperandType)
	{
		case CNT_NODE_RULE_OPERANDTYPE_STRING:
			SfxPoolItem::writeUnicodeString(rStream, m_aStringOperand);
			rStream << sal_uInt16(m_eOperator) << m_nPropertyWID
					<< sal_uInt32(0) // date operand
					<< m_bCaseSensitive << m_bRegexp
					<< sal_uInt16(CNT_NODE_RULE_OPERANDTYPE_STRING);

		case CNT_NODE_RULE_OPERANDTYPE_DATE:
			SfxPoolItem::writeUnicodeString(rStream, String());
				// string operand
			rStream << sal_uInt16(m_eOperator) << m_nPropertyWID
					<< m_aDateOperand.GetDate()
					<< sal_uInt8(false) // string operand case sensitive
					<< sal_uInt8(false) // string operand regexp
					<< sal_uInt16(CNT_NODE_RULE_OPERANDTYPE_DATE);

		case CNT_NODE_RULE_OPERANDTYPE_BOOL:
			SfxPoolItem::writeUnicodeString(rStream, String());
				// string operand
			rStream << sal_uInt16(m_eOperator) << m_nPropertyWID
					<< sal_uInt32(0) // date operand
					<< sal_uInt8(false) // string operand case sensitive
					<< sal_uInt8(false) // string operand regexp
					<< sal_uInt16(CNT_NODE_RULE_OPERANDTYPE_BOOL);

		default: // CNT_NODE_RULE_OPERANDTYPE_NUMERIC:
			SfxPoolItem::writeUnicodeString(rStream, String());
				// string operand
			rStream << sal_uInt16(m_eOperator) << m_nPropertyWID
					<< sal_uInt32(0) // date operand
					<< sal_uInt8(false) // string operand case sensitive
					<< sal_uInt8(false) // string operand regexp
					<< sal_uInt16(CNT_NODE_RULE_OPERANDTYPE_NUMERIC)
					<< m_nNumericOperand;
	}
}

//============================================================================
// static
CntNodeRuleTerm * CntNodeRuleTerm::load(SvStream & rStream)
{
	InputGuard_Impl aGuard(rStream);
	sal_uInt16 nVersion = 0;
	rStream >> nVersion;
	if (nVersion <= 2)
	{
		String aTheStringOperand;
		sal_uInt16 nTheOperator = 0;
		sal_uInt16 nThePropertyWID = 0;
		sal_uInt32 nTheDateOperand = 0;
		sal_uInt8 nTheCaseSensitive = false;
		sal_uInt8 nTheRegexp = false;
		sal_uInt16 nTheOperandType = 0;
		SfxPoolItem::readUnicodeString(rStream, aTheStringOperand,
									   nVersion >= 1);
		rStream >> nTheOperator >> nThePropertyWID >> nTheDateOperand
				>> nTheCaseSensitive >> nTheRegexp >> nTheOperandType;
		if (nTheOperator >= CNT_NODE_RULE_OPERATOR_CONTAINS
			&& nTheOperator <= CNT_NODE_RULE_OPERATOR_FALSE
			&& nThePropertyWID != 0)
			switch (nTheOperandType)
			{
				case CNT_NODE_RULE_OPERANDTYPE_STRING:
					return
					 new CntNodeRuleTerm(0, nThePropertyWID,
										 CntNodeRuleOperator(nTheOperator),
										 aTheStringOperand,
										 nTheCaseSensitive != false,
										 nTheRegexp != false);

				case CNT_NODE_RULE_OPERANDTYPE_DATE:
					return
					 new CntNodeRuleTerm(0, nThePropertyWID,
										 CntNodeRuleOperator(nTheOperator),
										 Date(nTheDateOperand));

				case CNT_NODE_RULE_OPERANDTYPE_BOOL:
					return
					 new CntNodeRuleTerm(0, nThePropertyWID,
										 CntNodeRuleOperator(nTheOperator));

				case CNT_NODE_RULE_OPERANDTYPE_NUMERIC:
				{
					sal_Int32 nTheNumericOperand = 0;
					if (nVersion >= 1)
						rStream >> nTheNumericOperand;
					return
					 new CntNodeRuleTerm(0, nThePropertyWID,
										 CntNodeRuleOperator(nTheOperator),
										 nTheNumericOperand);
				}
			}
	}
	return 0;
}

//============================================================================
// static
CntNodeRuleTerm * CntNodeRuleTerm::translate(ucb::RuleTerm const & rTerm,
											 CntNodeRule * pRule)
{
	CntItemMapEntry const * pEntry
		= CntItemPool::GetItemMap()->Prop2Which(rTerm.Property);
	if (!pEntry)
		return 0;

	uno::Any aOperand = rTerm.Operand;
	uno::Type aType = aOperand.getValueType();

	if (aType == getCppuType(static_cast< rtl::OUString const * >(0)))
	{
		rtl::OUString aValue;
		return (aOperand >>= aValue) ?
			       new CntNodeRuleTerm(pRule, pEntry->nWhich,
									   CntNodeRuleOperator(rTerm.Operator),
									   aValue, rTerm.CaseSensitive != false,
									   rTerm.RegularExpression != false) :
			           0;
	}
	else if (aType == getCppuType(static_cast< util::Date const * >(0)))
	{
		util::Date aValue;
		return (aOperand >>= aValue) ?
			       new CntNodeRuleTerm(pRule, pEntry->nWhich,
									   CntNodeRuleOperator(rTerm.Operator),
									   Date(aValue.Day, aValue.Month,
											aValue.Year)) :
			       0;
	}
	else if (aType == getCppuType(static_cast< sal_Int32 const * >(0)))
	{
		sal_Int32 nValue;
		return (aOperand >>= nValue) ?
			       new CntNodeRuleTerm(pRule, pEntry->nWhich,
									   CntNodeRuleOperator(rTerm.Operator),
									   nValue) :
			           0;
	}
	else if (aType == getCppuType(static_cast< sal_Bool const * >(0)))
		return new CntNodeRuleTerm(pRule, pEntry->nWhich,
								   CntNodeRuleOperator(rTerm.Operator));
	return 0;
}

//============================================================================
//
//  class CntNodeRule
//
//============================================================================

void CntNodeRule::clearTerms()
{
	while (TermCount() != 0)
		delete static_cast< CntNodeRuleTerm * >(m_aTerms.Remove(TermCount()
																    - 1));
}

//============================================================================
void CntNodeRule::copyTerms(CntNodeRule const & rRule)
{
	for (sal_uInt32 i = 0; i < rRule.TermCount(); ++i)
		InsertTerm(new CntNodeRuleTerm(*rRule.GetTerm(i)));
}

//============================================================================
bool CntNodeRule::operator ==(CntNodeRule const & rRule) const
{
	if (TermCount() != rRule.TermCount() || m_eAction != rRule.m_eAction
		|| m_aActionParameter != rRule.m_aActionParameter)
		return false;
	for (sal_uInt32 i = 0; i < TermCount(); ++i)
		if (*GetTerm(i) != *rRule.GetTerm(i))
			return false;
	return true;
}

//============================================================================
CntNodeAction CntNodeRule::queryAction(SfxItemSet const & rItems,
                                       IntlWrapper const & rIntlWrapper,
									   Date const * pReferenceDate) const
{
	for (sal_uInt32 i = 0; i < TermCount(); ++i)
	{
		CntNodeRuleTerm const * pTerm = GetTerm(i);
		if (!pTerm->matches(rItems.Get(pTerm->GetPropertySID()),
                            rIntlWrapper, pReferenceDate))
			return CNT_NODE_ACTION_NONE;
	}
	return m_eAction;
}

//============================================================================
void CntNodeRule::store(SvStream & rStream) const
{
	OutputGuard_Impl aGuard(rStream);
	rStream << sal_uInt16(3); // version
	sal_uInt32 nCount
		= std::min(TermCount(),
				   sal_uInt32(std::numeric_limits< sal_uInt16 >::max()));
	rStream << sal_uInt16(nCount);
	for (sal_uInt32 i = 0; i < nCount; ++i)
		GetTerm(i)->store(rStream);
	rStream << sal_uInt16(0) << sal_uInt16(m_eAction);
	SfxPoolItem::writeUnicodeString(rStream, m_aActionParameter);
}

//============================================================================
void CntNodeRule::load(SvStream & rStream)
{
	InputGuard_Impl aGuard(rStream);
	sal_uInt16 nVersion = 0;
	rStream >> nVersion;
	if (nVersion <= 3)
	{
		clearTerms();
		sal_uInt16 nCount = 0;
		rStream >> nCount;
		while (nCount--)
		{
			CntNodeRuleTerm * pTerm = CntNodeRuleTerm::load(rStream);
			if (pTerm)
				InsertTerm(pTerm);
		}
		sal_uInt16 nDummy = 0;
		sal_uInt16 nTheAction = 0;
		rStream >> nDummy >> nTheAction;
		if (nTheAction <= CNT_NODE_LAST_ACTION)
		{
			m_eAction = CntNodeAction(nTheAction);
			if (nVersion >= 2)
			{
				m_aActionParameter.Erase();
				SfxPoolItem::readUnicodeString(rStream, m_aActionParameter,
											   nVersion >= 3);
			}
		}
	}
}

//============================================================================
bool CntNodeRule::getTerms(uno::Sequence< ucb::RuleTerm > & rTerms) const
{
	if (TermCount() > std::numeric_limits< sal_Int32 >::max())
		return false;
	CntItemMap const * pItemMap = CntItemPool::GetItemMap();
	rTerms.realloc(sal_Int32(TermCount()));
	ucb::RuleTerm * pList = rTerms.getArray();
	for (sal_uInt32 i = 0; i < TermCount(); ++i)
	{
		CntNodeRuleTerm const & rTerm = *GetTerm(i);
		CntItemMapEntry const * pEntry
			= pItemMap ? pItemMap->Which2Prop(rTerm.GetPropertySID()) : 0;
		if (!pEntry)
			return false;
		pList->Property = rtl::OUString::createFromAscii(pEntry->pName);
		pList->Operator = rTerm.GetOperator();
		switch (rTerm.GetOperandType())
		{
			case CNT_NODE_RULE_OPERANDTYPE_STRING:
				pList->Operand <<= rtl::OUString(rTerm.GetStringOperand());
				pList->CaseSensitive = rTerm.IsCaseSensitive();
				pList->RegularExpression = rTerm.IsRegularExpression();
				break;

			case CNT_NODE_RULE_OPERANDTYPE_DATE:
				pList->Operand
					<<= util::Date(rTerm.GetDateOperand().GetDay(),
								   rTerm.GetDateOperand().GetMonth(),
								   rTerm.GetDateOperand().GetYear());
				break;

			case CNT_NODE_RULE_OPERANDTYPE_BOOL:
				pList->Operand
					<<= sal_Bool(rTerm.GetOperator()
								     == CNT_NODE_RULE_OPERATOR_TRUE);
				break;

			default: // CNT_NODE_RULE_OPERANDTYPE_NUMERIC:
				pList->Operand <<= sal_Int32(rTerm.GetNumericOperand());
				break;
		}
		++pList;
	}
	return true;
}

//============================================================================
bool CntNodeRule::addTerms(uno::Sequence< ucb::RuleTerm > const & rTerms)
{
	sal_Int32 nCount = rTerms.getLength();
	for (ucb::RuleTerm const * pList = rTerms.getConstArray(); nCount-- != 0;
		 ++pList)
		if (CntNodeRuleTerm::translate(*pList, this) == 0)
			return false;
	return true;
}

//============================================================================
//
//  class CntNodeActionSet
//
//============================================================================

CntNodeActionSet::~CntNodeActionSet()
{
	while (Count() != 0)
		delete static_cast< CntNodeActionRec * >(m_aRecs.Remove(Count() - 1));
}

//============================================================================
// Diese Funktion fuegt eine neue Action in das ActionSet ein. Es werden keine
// Actions doppelt eingefuegt. Wenn zu einer schon eingefuegten Action eine
// gegenteilige Action kommt, wird die weniger gefaehrliche Action behalten.
void CntNodeActionSet::Insert(CntNodeActionRec * pRec, bool bClone)
{
	CntNodeAction eAction = pRec->GetAction();
	if (bClone)
		pRec = new CntNodeActionRec(pRec->GetParameter(), eAction);
	for (sal_uInt32 i = 0; i < Count(); ++i)
	{
		CntNodeActionRec const * pAction = Get(i);
		if (*pRec == *pAction)
		{
			delete pRec;
			return;
		}
		switch (pAction->GetAction())
		{
			case CNT_NODE_ACTION_SHOW:
				if (eAction == CNT_NODE_ACTION_HIDE)
				{
					// Wenn es schon ein SHOW gibt, dann wird HIDE ignoriert.
					// Da die neue Action nicht eingefuegt wurde, muessen wir
					// sie jetzt loeschen.
					delete pRec;
					return;
				}
				break;

			case CNT_NODE_ACTION_HIDE:
				if (eAction == CNT_NODE_ACTION_SHOW)
				{
					// Ein existierendes HIDE wird durch ein SHOW ersetzt. Die
					// alte action durch die neue Action ersetzen und
					// anschliessend entsorgen.
					m_aRecs.Replace(pRec, i);
					delete const_cast< CntNodeActionRec * >(pAction);
					return;
				}
				break;

			case CNT_NODE_ACTION_MARK:
				if (eAction == CNT_NODE_ACTION_UNMARK)
				{
					// MARK wird NICHT durch UNMARK ersetzt.
					delete pRec;
					return;
				}
				break;

			case CNT_NODE_ACTION_UNMARK:
				if (eAction == CNT_NODE_ACTION_MARK)
				{
					// UNMARK wird durch MARK ersetzt.
					m_aRecs.Replace(pRec, i);
					delete const_cast< CntNodeActionRec * >(pAction);
					return;
				}
				break;

			case CNT_NODE_ACTION_MARKREAD:
				if (eAction == CNT_NODE_ACTION_MARKUNREAD)
				{
					m_aRecs.Replace(pRec, i);
					delete const_cast< CntNodeActionRec * >(pAction);
					return;
				}
				break;

			case CNT_NODE_ACTION_MARKUNREAD:
				if (eAction == CNT_NODE_ACTION_MARKREAD)
				{
					delete pRec;
					return;
				}
				break;

			case CNT_NODE_ACTION_MOVE:
				if (eAction == CNT_NODE_ACTION_MOVE)
				{
					// Das erste MOVE gewinnt.
					delete pRec;
					return;
				}
				else if (eAction == CNT_NODE_ACTION_COPY)
				{
					// Ein COPY wird vor einem MOVE eingefuegt.
					m_aRecs.Insert(pRec, i);
					return;
				}
				else if (eAction == CNT_NODE_ACTION_LINK)
				{
					// Ein CREATE_LINK wird vor einem MOVE eingefuegt.
					m_aRecs.Insert(pRec, i);
					return;
				}
				else if (eAction == CNT_NODE_ACTION_DELETE)
				{
					// Nach einem MOVE brauchen wir kein DELETE mehr.
					delete pRec;
					return;
				}
				break;

			case CNT_NODE_ACTION_DELETE:
				if (eAction == CNT_NODE_ACTION_DELETE)
				{
					// Ein DELETE reicht.
					delete pRec;
					return;
				}
				else if (eAction == CNT_NODE_ACTION_MOVE)
				{
					// Ein DELETE wird durch ein MOVE ersetzt.
					m_aRecs.Replace(pRec, i);
					delete const_cast< CntNodeActionRec * >(pAction);
					return;
				}
				else if (eAction == CNT_NODE_ACTION_COPY)
				{
					// Ein COPY wird vor einem DELETE eingefuegt.
					m_aRecs.Insert(pRec, i);
					return;
				}
				else if (eAction == CNT_NODE_ACTION_LINK)
				{
					// Ein CREATE_LINK wird vor einem DELETE eingefuegt.
					m_aRecs.Insert(pRec, i);
					return;
				}
				break;
		}
	}
	m_aRecs.Insert(pRec, LIST_APPEND);
}

//============================================================================
//
//  class CntNodeRuleSet
//
//============================================================================

void CntNodeRuleSet::clearRules()
{
	while (RuleCount() != 0)
		delete static_cast< CntNodeRule * >(m_aRules.Remove(RuleCount() - 1));
}

//============================================================================
void CntNodeRuleSet::copyRules(CntNodeRuleSet const & rSet)
{
	for (sal_uInt32 i = 0; i < rSet.RuleCount(); ++i)
		InsertRule(new CntNodeRule(*rSet.GetRule(i)));
}

//============================================================================
bool CntNodeRuleSet::operator ==(CntNodeRuleSet const & rSet) const
{
	if (RuleCount() != rSet.RuleCount()
		|| m_bHandleFolder != rSet.m_bHandleFolder)
		return false;
	for (sal_uInt32 i = 0; i < RuleCount(); ++i)
		if (*GetRule(i) != *rSet.GetRule(i))
			return false;
	return true;
}

//============================================================================
CntNodeAction CntNodeRuleSet::GetDefaultAction(bool & rNeitherNor) const
{
	sal_uInt16 i = 0;
	CntNodeAction nDefault = CNT_NODE_ACTION_SHOW;
	rNeitherNor = true;
	while (rNeitherNor && i < RuleCount())
		switch (GetRule(i)->GetAction())
		{
			case CNT_NODE_ACTION_SHOW:
				nDefault = CNT_NODE_ACTION_HIDE;
				rNeitherNor = false;
				break;

			case CNT_NODE_ACTION_HIDE:
				rNeitherNor = false;
				break;

			default:
				++i;
				break;
		}
	return nDefault;
}

//============================================================================
void CntNodeRuleSet::execute(SfxItemSet const & rItems,
							 CntNodeActionSet & rActions,
                             IntlWrapper const & rIntlWrapper,
							 Date const * pReferenceDate) const
{
	for (sal_uInt32 i = 0; i < RuleCount(); ++i)
	{
		CntNodeRule const * pRule = GetRule(i);
        CntNodeAction eAction = pRule->queryAction(rItems, rIntlWrapper,
												   pReferenceDate);
		if (eAction != CNT_NODE_ACTION_NONE)
			rActions.Insert(new CntNodeActionRec(pRule->GetParameter(),
												 eAction),
							false);
	}
}

//============================================================================
void CntNodeRuleSet::store(SvStream & rStream) const
{
	OutputGuard_Impl aGuard(rStream);
	rStream << sal_uInt16(1); // version
	sal_uInt32 nCount
		= std::min(RuleCount(),
				   sal_uInt32(std::numeric_limits< sal_uInt16 >::max()));
	rStream << sal_uInt16(nCount);
	for (sal_uInt32 i = 0; i < nCount; ++i)
		GetRule(i)->store(rStream);
}

//============================================================================
void CntNodeRuleSet::load(SvStream & rStream)
{
	InputGuard_Impl aGuard(rStream);
	sal_uInt16 nVersion = 0;
	rStream >> nVersion;
	if (nVersion <= 1)
	{
		sal_uInt16 nCount = 0;
		rStream >> nCount;
		while (nCount--)
		{
			CntNodeRule * pRule = new CntNodeRule(this, CNT_NODE_ACTION_SHOW);
			pRule->load(rStream);
			InsertRule(pRule);
		}
		m_bHandleFolder = false;
	}
}

//============================================================================
//
//  class CntRuleSetItem
//
//============================================================================

TYPEINIT1_AUTOFACTORY(CntRuleSetItem, SfxPoolItem);

//============================================================================
// virtual
int CntRuleSetItem::operator ==(SfxPoolItem const & rItem) const
{
	return Which() == rItem.Which() && rItem.ISA(CntRuleSetItem)
	       && m_aSet == static_cast< CntRuleSetItem const * >(&rItem)->m_aSet;
}

//============================================================================
// virtual
SfxItemPresentation CntRuleSetItem::GetPresentation(SfxItemPresentation,
													SfxMapUnit, SfxMapUnit,
													String & rText,
                                                    IntlWrapper const *)
	const
{
	rText.Erase();
	return SFX_ITEM_PRESENTATION_NONE;
}

//============================================================================
// virtual
BOOL CntRuleSetItem::QueryValue(uno::Any & rVal, BYTE) const
{
	sal_uInt32 nCount = m_aSet.RuleCount();
	if (nCount > std::numeric_limits< sal_Int32 >::max())
		return false;
	ucb::RuleSet aSet;
	aSet.Rules.realloc(sal_Int32(nCount));
	ucb::Rule * pRules = aSet.Rules.getArray();
	for (sal_uInt32 i = 0; i < nCount; ++i)
	{
		CntNodeRule const * pCurRule = m_aSet.GetRule(i);
		if (!pCurRule->getTerms(pRules->Terms))
			return false;
		pRules->Parameter = pCurRule->GetParameter();
		pRules->Action = sal_Int16(pCurRule->GetAction());
		++pRules;
	}
	aSet.HandleFolder = m_aSet.GetHandleFolder();
	rVal <<= aSet;
	return true;
}

//============================================================================
// virtual
BOOL CntRuleSetItem::PutValue(uno::Any const & rVal, BYTE)
{
	ucb::RuleSet aValue;
	if (rVal >>= aValue)
	{
		CntNodeRuleSet aRuleSet;
		aRuleSet.SetHandleFolder(aValue.HandleFolder != false);
		ucb::Rule * pRules = aValue.Rules.getArray();
		sal_Int32 nRuleCount = aValue.Rules.getLength();
		for (sal_Int32 n = 0; n < nRuleCount; ++n)
		{
			CntNodeRule * pRule
				= new CntNodeRule(&aRuleSet, CntNodeAction(pRules[n].Action),
								  pRules[n].Parameter);
			if (pRule->addTerms(pRules[n].Terms))
				aRuleSet.InsertRule(pRule);
			else
			{
				delete pRule;
				return false;
			}
		}
		m_aSet = aRuleSet;
		return true;
	}
	return false;
}

//============================================================================
// virtual
SfxPoolItem * CntRuleSetItem::Create(SvStream & rStream, USHORT) const
{
	CntRuleSetItem * pItem = new CntRuleSetItem(Which());
	pItem->m_aSet.load(rStream);
	return pItem;
}

//============================================================================
// virtual
SvStream & CntRuleSetItem::Store(SvStream & rStream, USHORT) const
{
	m_aSet.store(rStream);
	return rStream;
}
