/*************************************************************************
 *
 *  $RCSfile: pkgchk_packages.cxx,v $
 *
 *  $Revision: 1.5.4.2 $
 *
 *  last change: $Author: khendricks $ $Date: 2002/12/21 10:26:33 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2002 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "pkgchk_env.h"

#include <rtl/uri.hxx>


using namespace ::std;
using namespace ::rtl;
using namespace ::osl;
using namespace ::com::sun::star::uno;
using namespace ::com::sun::star;

namespace pkgchk
{

enum ctype { ZIP, SHARED_LIBRARY, JAR, BASIC_LIBRARY, RDB, OTHER };

//==================================================================================================
static inline ctype path_get_ctype( OUString const & path ) SAL_THROW( () )
{
    sal_Int32 dot = path.lastIndexOf( '.' );
    if (dot >= 0)
    {
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".zip" ))
        {
            return ZIP;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, SAL_DLLEXTENSION ))
        {
            return SHARED_LIBRARY;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".jar" ))
        {
            return JAR;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".rdb" ))
        {
            return RDB;
        }
        if (0 == rtl_ustr_ascii_compareIgnoreAsciiCase(
                path.pData->buffer + dot, ".xlb" ))
        {
            return BASIC_LIBRARY;
        }
    }
    return OTHER;
}

#if defined (WIN32)
static char const s_platform_path [] = "/windows.plt";
#elif defined (SOLARIS) && defined (SPARC)
static char const s_platform_path [] = "/solaris_sparc.plt";
#elif defined (SOLARIS) && defined (INTEL)
static char const s_platform_path [] = "/solaris_x86.plt";
#elif defined (LINUX) && defined (INTEL)
static char const s_platform_path [] = "/linux_x86.plt";
#elif defined (LINUX) && defined (POWERPC)
static char const s_platform_path [] = "/linux_powerpc.plt";
#elif defined (MACOSX) && defined (POWERPC)
static char const s_platform_path [] = "/macosx_powerpc.plt";
#elif defined (NETBSD) && defined (SPARC)
static char const s_platform_path [] = "/netbsd_sparc.plt";
#else
#error "unkonwn platform"
insert your platform identifier above; inserted for the case the preprocessor ignores error
#endif

//==================================================================================================
static bool package_ignore_path(
    OUString const & path )
    SAL_THROW( () )
{
    // check if platform path
    if (! ends_with_ignore_ascii_case( path, CSTR_LEN(".plt") ))
        return false;
    // check if this platform path
    if (ends_with_ignore_ascii_case( path, CSTR_LEN(s_platform_path) ))
        return false;
    return true;
}

//==================================================================================================
static OUString make_reg_url( OUString const & reg_base_path, OUString const & package )
    SAL_THROW( () )
{
    if (reg_base_path.getLength())
    {
#if SUPD > 641
        if (0 == reg_base_path.compareToAscii( RTL_CONSTASCII_STRINGPARAM(EXPAND_PROTOCOL ":") ))
#else
        if (true)
#endif
        {
            // make bootstrap macro url
            OUStringBuffer buf( 128 );
            buf.append( reg_base_path );
            buf.append( (sal_Unicode)'/' );
            // escape package url $, ... (rtl bootstrap macro)
            sal_Int32 nPos = 0;
            sal_Int32 nLen = package.getLength();
            while (nPos < nLen)
            {
                sal_Unicode c = package[ nPos ];
                switch (c)
                {
                case '$':
                    // following are unlikely in package string (file-url), though escaped...
                case '\\':
                case '{':
                case '}':
                    buf.append( (sal_Unicode)'\\' );
                    break;
                }
                buf.append( c );
                ++nPos;
            }
            OUString macro( buf.makeStringAndClear() );
#if SUPD > 641
            static sal_Bool const * s_char_class_uric = 0;
            if (! s_char_class_uric)
            {
                s_char_class_uric = rtl_getUriCharClass( rtl_UriCharClassUric );
            }
            OUString uri( Uri::encode( macro, s_char_class_uric,
                                       rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 ) );
            // macro string starts with "vnd.sun.star.expand:"
            // uric char class allows characters '.' and ':'
            // so encoding the whole string (including protocol) is ok
            OSL_ASSERT(
                0 == uri.compareToAscii( RTL_CONSTASCII_STRINGPARAM(EXPAND_PROTOCOL ":") ) );
#ifdef __DIAG
            OString cstr_base_path( OUStringToOString( reg_base_path, RTL_TEXTENCODING_ASCII_US ) );
            OString cstr_package( OUStringToOString( package, RTL_TEXTENCODING_ASCII_US ) );
            OString cstr_macro( OUStringToOString( macro, RTL_TEXTENCODING_ASCII_US ) );
            OString cstr_uri( OUStringToOString( uri, RTL_TEXTENCODING_ASCII_US ) );
            printf(
                "[diag] make_reg_url(): reg_base_path=%s + package=%s => %s => %s\n",
                cstr_base_path.getStr(), cstr_package.getStr(),
                cstr_macro.getStr(), cstr_uri.getStr() );
#endif
            return uri;
#else /* SUPD <= 641 */
            return macro;
#endif
        }
        else // no expand path
        {
            return path_concat( reg_base_path, package );
        }
    }
    else
    {
        return package;
    }
}

//==================================================================================================
static void package_unlink(
    OUString const & package, OUString const & base_path, OUString const & reg_base_path,
    pkgchk_env & env, bool skip_registration = false )
    SAL_THROW( (RuntimeException) )
{
    OUString path( path_concat( base_path, package ) );
    FileStatus status( c_file_status_mask );
    path_get_status( &status, path );
    FileStatus::Type file_type = status.getFileType();
    
    if (FileStatus::Directory == file_type)
    {
        if (package_ignore_path( path ))
        {
#ifdef __DIAG
            OString cstr_path( OUStringToOString( path, RTL_TEXTENCODING_ASCII_US ) );
            printf( "[diag] package_unlink(): ignoring path %s\n", cstr_path.getStr() );
#endif
            return;
        }
        skip_registration |= ends_with_ignore_ascii_case( path, CSTR_LEN("/skip_registration") );
        
        Directory dir( path );
        dir_open( dir, path );
        while (true)
        {
            {
            DirectoryItem dirItem;
            Directory::RC rc = dir.getNextItem( dirItem );
            if (Directory::E_NOENT == rc)
                break;
            if (Directory::E_None != rc || !dirItem.is())
            {
                throw RuntimeException(
                    OUSTR("cannot get next packages dir item from ") + path,
                    Reference< XInterface >() );
            }
            diritem_get_status( &status, dirItem );
            } // release dir item
            // recurse
            package_unlink(
                path_concat( package, file_status_get_encoded_name( status ) ),
                base_path, reg_base_path, env, skip_registration );
        }
    }
    // do something
    else if (FileStatus::Regular == file_type || FileStatus::Link == file_type)
    {
        ctype ct = path_get_ctype( path );
        switch (ct)
        {
        case SHARED_LIBRARY:
        case JAR:
        {            
            if (JAR == ct) // remove from classpath file
            {
                env.classpath_remove( package );
            }
            if (skip_registration)
            {
#ifdef __DIAG
                OUString url( make_reg_url( reg_base_path, package ) );
                OString cstr_url( OUStringToOString( url, RTL_TEXTENCODING_ASCII_US ) );
                printf(
                    "[diag] package_unlink(): skipping registration for %s\n", cstr_url.getStr() );
#endif
            }
            else // revoke from services rdb
            {
                OUString url( make_reg_url( reg_base_path, package ) );
                Reference< registry::XSimpleRegistry > xServices_rdb( env.get_services_rdb() );
                
                OUStringBuffer buf( 128 );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("revoking ") );
                buf.append( url );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" (") );
                buf.append( path );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(") from registry ") );
                buf.append( xServices_rdb->getURL() );
                
                if (env.get_impreg()->revokeImplementation( url, xServices_rdb ))
                {
                    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ok.") );
                }
                else
                {
                    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": failed!") );
                }
                // just log errors when revoking
                env.log( buf.makeStringAndClear() );
            }
            break;
        }
        case BASIC_LIBRARY:
        {
            OUStringBuffer buf( 128 );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("revoking ") );
            buf.append( path );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" from basic library container") );
            if (env.basic_remove( path /* take absolute path */ ))
            {
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ok.") );
            }
            else
            {
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": failed.  Library not known.") );
            }
            env.log( buf.makeStringAndClear() );
            break;
        }
        default:
            break;
        }
    }
    else
    {
        OUStringBuffer buf( 128 );
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("> warning: ") );
        buf.append( path );
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" has unexpected file type") );
        env.log( buf.makeStringAndClear() );
    }
}
//==================================================================================================
static void package_link(
    OUString const & package, OUString const & base_path, OUString const & reg_base_path,
    pkgchk_env & env, bool skip_registration = false )
    SAL_THROW( (RuntimeException) )
{
    OUString path( path_concat( base_path, package ) );
    FileStatus status( c_file_status_mask );
    path_get_status( &status, path );
    FileStatus::Type file_type = status.getFileType();
    
    if (FileStatus::Directory == file_type)
    {
        if (package_ignore_path( path ))
        {
#ifdef __DIAG
            OString cstr_path( OUStringToOString( path, RTL_TEXTENCODING_ASCII_US ) );
            printf( "[diag] package_link(): ignoring path %s\n", cstr_path.getStr() );
#endif
            return;
        }
        skip_registration |= ends_with_ignore_ascii_case( path, CSTR_LEN("/skip_registration") );
        
        Directory dir( path );
        dir_open( dir, path );
        while (true)
        {
            {
            DirectoryItem dirItem;
            Directory::RC rc = dir.getNextItem( dirItem );
            if (Directory::E_NOENT == rc)
                break;
            if (Directory::E_None != rc || !dirItem.is())
            {
                throw RuntimeException(
                    OUSTR("cannot get next packages dir item from ") + path,
                    Reference< XInterface >() );
            }
            diritem_get_status( &status, dirItem );
            } // release dir item
            // recurse
            package_link(
                path_concat( package, file_status_get_encoded_name( status ) ),
                base_path, reg_base_path, env, skip_registration );
        }
    }
    // do something
    else if (FileStatus::Regular == file_type || FileStatus::Link == file_type)
    {
        ctype ct = path_get_ctype( path );
        switch (ct)
        {
        case SHARED_LIBRARY:
        case JAR:
        {
            if (skip_registration)
            {
#ifdef __DIAG
                OUString url( make_reg_url( reg_base_path, package ) );
                OString cstr_url( OUStringToOString( url, RTL_TEXTENCODING_ASCII_US ) );
                printf(
                    "[diag] package_link(): skipping registration for %s\n", cstr_url.getStr() );
#endif
            }
            else // register into services rdb
            {
                OUString url( make_reg_url( reg_base_path, package ) );
                Reference< registry::XSimpleRegistry > xServices_rdb( env.get_services_rdb() );
                
                OUStringBuffer buf( 128 );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("registering ") );
                buf.append( url );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" (") );
                buf.append( path );
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(") into registry ") );
                buf.append( xServices_rdb->getURL() );
                
                try
                {
                    env.get_impreg()->registerImplementation(
                        SHARED_LIBRARY == ct
                        ? OUSTR("com.sun.star.loader.SharedLibrary")
                        : OUSTR("com.sun.star.loader.Java2"),
                        url, xServices_rdb );
                
                    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ok.") );
                    env.log( buf.makeStringAndClear() );
                }
                catch (registry::CannotRegisterImplementationException & exc)
                {
                    buf.appendAscii(
                        RTL_CONSTASCII_STRINGPARAM(": failed! (cannot register implementation") );
                    if (exc.Message.getLength())
                    {
                        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ") );
                        buf.append( exc.Message );
                    }
                    buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(")!") );
                    env.err( buf.makeStringAndClear() );
                }
            }            
            if (JAR == ct) // add jar to classpath file
            {
                env.classpath_insert( package );
            }
            break;
        }
        case BASIC_LIBRARY:
        {
            OUStringBuffer buf( 128 );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("inserting ") );
            buf.append( path );
            buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" into basic library container") );
            if (env.basic_insert( path /* take absolute path */ ))
            {
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": ok.") );
                env.log( buf.makeStringAndClear() );
            }
            else
            {
                buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(": failed!") );
                env.err( buf.makeStringAndClear() );
            }
            break;
        }
        case RDB:
        {
            try
            {
                // merge types in
                env.get_types_rdb()->mergeKey( OUSTR("/"), path );
            }
            catch (registry::InvalidRegistryException & exc)
            {
                throw RuntimeException(
                    exc.Message.getLength()
                    ? exc.Message : OUSTR("invalid registry exception occured!"),
                    Reference< XInterface >() );
            }
            catch (registry::MergeConflictException & exc)
            {
                throw RuntimeException(
                    exc.Message.getLength()
                    ? exc.Message : OUSTR("merge conflict exception occured!"),
                    Reference< XInterface >() );
            }
            break;
        }
        default:
            break;
        }
    }
    else
    {
        OUStringBuffer buf( 128 );
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM("> warning: ") );
        buf.append( path );
        buf.appendAscii( RTL_CONSTASCII_STRINGPARAM(" has unexpected file type!") );
        env.log( buf.makeStringAndClear() );
    }
}

//##################################################################################################
//##################################################################################################
//##################################################################################################

//__________________________________________________________________________________________________
bool pkgchk_env::packages_check()
    SAL_THROW( (RuntimeException) )
{
    // ensure packages path
    dir_ensure( m_packages_path );
    
    // new check
    m_packages_to_be_installed.clear();
    m_packages_to_be_removed.clear();
    
    log( OUSTR("scanning uno packages from ") + m_packages_path );
    
    FileStatus status( c_file_status_mask );
    Directory packages_dir( m_packages_path );
    dir_open( packages_dir, m_packages_path );
    while (true)
    {
        DirectoryItem dirItem;
        Directory::RC rc = packages_dir.getNextItem( dirItem );
        if (Directory::E_NOENT == rc)
            break;
        if (Directory::E_None != rc || !dirItem.is())
        {
            throw RuntimeException(
                OUSTR("cannot get next packages dir item from ") + m_packages_path,
                Reference< XInterface >() );
        }
        diritem_get_status( &status, dirItem );
        
        OUString package_path( status.getFileURL() );
        
        // ignore cache dir
        if (path_equals( package_path, m_cache_path ))
        {
            continue;
        }
        FileStatus::Type file_type = status.getFileType();
        if (FileStatus::Directory == file_type)
        {
            err( OUSTR("directories not allowed: ") + package_path );
            continue;
        }
        if (FileStatus::Regular != file_type &&
            FileStatus::Link != file_type)
        {
            err( package_path + OUSTR(" is of unsupported file type!") );
            continue;
        }
        
        m_packages_to_be_installed.insert( file_status_get_encoded_name( status ) );
    }
    
    OUString cache_packages_path( path_concat( m_cache_path, CSTR_LEN("uno_packages") ) );
    // output packages cache directory
    Directory cache_packages_dir( cache_packages_path );
    dir_open( cache_packages_dir, cache_packages_path, true /* create if not existing */ );
    
    // read installed entries from cache directory
    while (true)
    {
        {
        DirectoryItem dirItem;
        Directory::RC rc = cache_packages_dir.getNextItem( dirItem );
        if (Directory::E_NOENT == rc)
            break;
        if (Directory::E_None != rc || !dirItem.is())
        {
            throw RuntimeException(
                OUSTR("cannot get next cache dir item from ") + cache_packages_path,
                Reference< XInterface >() );
        }
        diritem_get_status( &status, dirItem );
        FileStatus::Type file_type = status.getFileType();
        if (FileStatus::Directory != file_type &&
            FileStatus::Regular != file_type &&
            FileStatus::Link != file_type)
        {
            err( status.getFileURL() + OUSTR(" is of unexpected file type!") );
            continue;
        }
        } // release dir item
        
        OUString package( file_status_get_encoded_name( status ) );
        
        if (FileStatus::Directory == status.getFileType()) // zip inflation dir
        {
            // decode time stamp of inflation dir
            sal_Int32 dot = package.lastIndexOf( '.' );
            OSL_ENSURE( 0 <= dot, "no zip inflation dir?!" );
            if (0 <= dot)
            {
                sal_uInt32 tCacheEntry = (sal_uInt32)package.copy( dot +1 ).toInt64();
                OSL_ENSURE( tCacheEntry, "zip inflation dir without encoded time stamp?!" );
                if (tCacheEntry)
                {
                    OUString package_name = package.copy( 0, dot );
                    OSL_ASSERT( ZIP == path_get_ctype( package_name ) );
                    // is in packages dir?
                    if (m_packages_to_be_installed.find( package_name ) !=
                        m_packages_to_be_installed.end())
                    {
                        path_get_status(
                            &status, path_concat( m_packages_path, package_name ) );
                        TimeValue time = status.getModifyTime();
                        if (tCacheEntry == time.Seconds)
                        {
                            // same version
                            m_packages_to_be_installed.erase( package_name );
                            continue;
                        }
                    }
                }
            }
        }
        else if (m_packages_to_be_installed.find( package ) !=
                 m_packages_to_be_installed.end()) // in packages dir?
        {
            TimeValue time = status.getModifyTime();
            sal_uInt32 tCacheEntry = time.Seconds;
            path_get_status( &status, path_concat( m_packages_path, package ) );
            time = status.getModifyTime();
            if (tCacheEntry == time.Seconds)
            {
                // same version
                m_packages_to_be_installed.erase( package );
                continue;
            }
        }
        
        // to be removed
        m_packages_to_be_removed.insert( package );
    }    
    
    log( OUSTR("finished scanning uno packages.") );
    
    return packages_to_be_balanced();
}
//__________________________________________________________________________________________________
void pkgchk_env::packages_balance()
    SAL_THROW( (RuntimeException) )
{
    OUString uno_packages( RTL_CONSTASCII_USTRINGPARAM("uno_packages") );
    OUString cache_packages_path( path_concat( m_cache_path, uno_packages ) );
    
    log( OUSTR("balancing cache dir ") + m_cache_path );
    
    // first install packages
    t_string_set::const_iterator iPos( m_packages_to_be_removed.begin() );
    t_string_set::const_iterator iEnd( m_packages_to_be_removed.end() );
    for ( ; iPos != iEnd; ++iPos )
    {
        OUString const & package = *iPos;
        // prefix with uno_packages/
        OUString rel_package_path( path_concat( uno_packages, package ) );
        // remove from services registry
        package_unlink(
            rel_package_path, m_cache_path, m_reg_cache_path, *this );
        // erase path
        path_erase(
            path_concat( m_cache_path, rel_package_path ), *this );
    }
    
    // then install packages
    FileStatus status( c_file_status_mask );
    iPos = m_packages_to_be_installed.begin();
    iEnd = m_packages_to_be_installed.end();
    for ( ; iPos != iEnd; ++iPos )
    {
        OUString const & package = *iPos;
        if (ZIP == path_get_ctype( package )) // inflate
        {
            // decorate name with time stamp
            OUStringBuffer buf( 128 );
            buf.append( package );
            buf.append( (sal_Unicode)'.' );
            path_get_status( &status, path_concat( m_packages_path, package ) );
            TimeValue time = status.getModifyTime();
            buf.append( OUString::valueOf( (sal_Int64)time.Seconds ) );
            // prefix with uno_packages/
            OUString reg_package_path( path_concat( uno_packages, buf.makeStringAndClear() ) );
            // inflate
            zip_inflate(
                path_concat( m_cache_path, reg_package_path ),
                path_concat( m_packages_path, package ), *this );
            // link into installation
            package_link(
                reg_package_path,
                m_cache_path, m_reg_cache_path, *this );
        }
        else
        {
            // copy into cache dir
            path_copy(
                cache_packages_path, path_concat( m_packages_path, package ), *this );
            // link into installation
            package_link(
                path_concat( uno_packages, package ), // prefix with uno_packages/
                m_cache_path, m_reg_cache_path, *this );
        }
    }
    
    log( OUSTR("finished balancing cache dir.") );
}
//__________________________________________________________________________________________________
void pkgchk_env::packages_remove_all()
    SAL_THROW( (RuntimeException) )
{
    // erase cache entries from script.xlc, dialog.xlc
    basic_remove_cache_entries();
    
    // log files resides in cache dir?
    if (0 == m_log_file_path.compareTo( m_cache_path, m_cache_path.getLength() ))
    {
        if (m_log_file) // if Initialized
        {
            // close old log file
            OSL_VERIFY( osl_File_E_None == osl_closeFile( m_log_file ) );
            m_log_file = 0;
        }
        // de-install log file
        OUString log_file_path( m_log_file_path );
        m_log_file_path = OUString();
        
        // erase cache dir
        path_erase( m_cache_path, *this );
        // ensure cache path
        dir_ensure( m_cache_path );
        
        // new log
        m_log_file_path = log_file_path;
    }
    else
    {
        // erase cache dir
        path_erase( m_cache_path, *this );
        // ensure cache path
        dir_create( m_cache_path );
    }
}

}
