/*************************************************************************
 *
 *  $RCSfile: hierarchy.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: jl $ $Date: 2001/03/23 11:49:03 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
#ifndef _HIERARCHY_HXX
#include "hierarchy.hxx"
#endif

#ifndef _HIERARCHY_TASKWRAPPER_HXX_
#include "taskwrapper.hxx"
#endif

#ifndef _COM_SUN_STAR_UCB_CONTENTACTION_HPP_
#include <com/sun/star/ucb/ContentAction.hpp>
#endif

#ifndef _COM_SUN_STAR_BEANS_XPROPERTYCONTAINER_HPP_
#include <com/sun/star/beans/XPropertyContainer.hpp>
#endif

#ifndef _COM_SUN_STAR_BEANS_XMULTIPROPERTYSET_HPP_
#include <com/sun/star/beans/XMultiPropertySet.hpp>
#endif

#ifndef _COM_SUN_STAR_FRAME_XCONFIGMANAGER_HPP_
#include <com/sun/star/frame/XConfigManager.hpp>
#endif

#ifndef _COM_SUN_STAR_REGISTRY_XSIMPLEREGISTRY_HPP_
#include <com/sun/star/registry/XSimpleRegistry.hpp>
#endif

#ifndef _USR_SMARTSERVICES_HXX_
#include <usr/smartservices.hxx>
#endif

#ifndef _SEQUENCE_HELPER_HXX_
#include "sequence_helper.hxx"
#endif

#include "encode.hxx"

#include <osl/diagnose.h>


using namespace com::sun::star::frame;
using namespace com::sun::star::registry;

/**************************************************************************
 *
 * helper functions
 *
 **************************************************************************
 */

inline Reference< XConfigManager > getSpecialConfigManager(const Reference< XMultiServiceFactory >& xServiceMgr)
{
	Reference< XInterface > xIfc = xServiceMgr->createInstance(L"com.sun.star.config.SpecialConfigManager");

	if(xIfc.is())
	{
		return Reference< XConfigManager >::query(xIfc);
	}
	else
		return Reference< XConfigManager >();
}

//------------------------------------------------------------------------

inline OWString getUserConfigURL(const Reference< XConfigManager >& xSpecialConfigMgr)
{
	Reference< XSimpleRegistry > xSimReg=Reference< XSimpleRegistry >::query(xSpecialConfigMgr);

    if(xSimReg.is())
	{
		Reference< XRegistryKey > xRootKey = xSimReg->getRootKey();

		if(xRootKey.is())
		{
			OWString aURL;

            Reference< XRegistryKey > xRegKey = xRootKey->
				openKey(L"Directories/UserConfig-Path");

            if(xRegKey.is())
			{
				if(xRegKey->getValueType() == RegistryValueType_STRING)
				{
					aURL = xRegKey->getStringValue();
				}

                xRegKey->closeKey();
			}

            return xSpecialConfigMgr->substituteVariables(aURL);
		}
	}

    return OWString();
}


//------------------------------------------------------------------------



/////////////////////////////////////////////////////////////////////////////
//
//	OContentHolder
//
/////////////////////////////////////////////////////////////////////////////

OContentHolder::OContentHolder( const Reference< XContent > & xContent ) :
OContentHelper(),
m_xPhysicalContent( xContent )
{
	if ( m_xPhysicalContent.is() )
		m_xPhysicalContent->addContentEventListener( this );
}

//-----------------------------------------------------------------------------

OContentHolder::~OContentHolder()
{
	if ( m_xPhysicalContent.is() )
		m_xPhysicalContent->removeContentEventListener( this );
}

//-----------------------------------------------------------------------------
//	XEventListener implementation
//-----------------------------------------------------------------------------

void SAL_CALL OContentHolder::disposing( const EventObject& Source ) throw(RuntimeException)
{
	if ( Source.Source == m_xPhysicalContent )
		m_xPhysicalContent = Reference< XContent >();
}

//-----------------------------------------------------------------------------
//	XContentEventListener implementation
//-----------------------------------------------------------------------------

Reference< XContent > OContentHolder::packContent( const Reference< XContent > & xContent )
{
	return xContent;
}

//-----------------------------------------------------------------------------

void SAL_CALL OContentHolder::contentEvent( const ContentEvent & evt ) throw(RuntimeException)
{
	if ( evt.Content != Reference< XContent >(this) )
	{
		ContentEvent	aEvent = evt;

		aEvent.Content = packContent( aEvent.Content );
		aEvent.Source = SAL_STATIC_CAST( XContent *, this );
		broadcastContentEvent( aEvent );
	}
	else
		broadcastContentEvent( evt );
}

//-----------------------------------------------------------------------------
//	XInterface implementation
//-----------------------------------------------------------------------------

#if defined(TF_TYPE) || (SUPD >= 564)
Any SAL_CALL OContentHolder::queryInterface( const Type& type ) throw( RuntimeException )
{
	Any	aRet = ::cppu::queryInterface(
		type,
		SAL_STATIC_CAST( XCommandTaskProcessor*, this ),
		SAL_STATIC_CAST( XPropertyTaskProcessor*, this ),
		SAL_STATIC_CAST( XContentHierarchy*, this ),
		SAL_STATIC_CAST( XContentHierarchy*, this ),
		SAL_STATIC_CAST( XContentCreator*, this ),
		SAL_STATIC_CAST( XContentEventListener*, this )
		);

	return ( aRet.hasValue() ? aRet : OContentHelper::queryInterface( type ) );
}
#else
sal_Bool SAL_CALL OContentHolder::queryInterface( const Uik& type, Any& ifc ) throw( RuntimeException )
{
	return ::com::sun::star::uno::queryInterface(
		type, ifc,
		SAL_STATIC_CAST( XCommandTaskProcessor*, this ),
		SAL_STATIC_CAST( XPropertyTaskProcessor*, this ),
		SAL_STATIC_CAST( XContentHierarchy*, this ),
		SAL_STATIC_CAST( XContentHierarchy*, this ),
		SAL_STATIC_CAST( XContentCreator*, this ),
		SAL_STATIC_CAST( XContentEventListener*, this )
		)
		|| OContentHelper::queryInterface( type, ifc );
}
#endif

//-----------------------------------------------------------------------------
//	XContentCreator implementation
//-----------------------------------------------------------------------------

Sequence< ContentInfo > SAL_CALL OContentHolder::queryCreatableContentsInfo(  ) throw(RuntimeException)
{
	Sequence< ContentInfo >	aSeq;
	Reference< XContentCreator >	xCreator = Reference< XContentCreator >::query( m_xPhysicalContent );

	if ( xCreator.is() )
		aSeq = xCreator->queryCreatableContentsInfo();

	return aSeq;
}

//-----------------------------------------------------------------------------

Reference< XContent > SAL_CALL OContentHolder::createNewContent( const ContentInfo& Info ) throw(RuntimeException)
{
	Reference< XContent >	xContent;
	Reference< XContentCreator >	xCreator = Reference< XContentCreator >::query( m_xPhysicalContent );

	if ( xCreator.is() )
		xContent = xCreator->createNewContent( Info );

	return xContent;
}

//-----------------------------------------------------------------------------

#if ( SUPD < 565 )
sal_Bool SAL_CALL OContentHolder::insertContent( const Reference< XContent >& Content ) throw(RuntimeException)
{
	Reference< XContentCreator >	xCreator = Reference< XContentCreator >::query( m_xPhysicalContent );

	if ( xCreator.is() )
		return xCreator->insertContent( Content );
	else
		return sal_False;
}
#endif

//-----------------------------------------------------------------------------
//	XContentHierarchy implementation
//-----------------------------------------------------------------------------

Reference< XContent > OContentHolder::getContentFromHierarchyIdentifier( const Sequence< Any > &aIdentifier ) throw( IllegalArgumentException )
{
	Reference< XContent >	xContent;

	if ( aIdentifier.getLength() == 0 )
		xContent = m_xPhysicalContent;
	else if ( m_xPhysicalContent.is() )
	{
		Reference< XContentHierarchy >	xParser = Reference< XContentHierarchy >::query( m_xPhysicalContent );
		if ( xParser.is() )
		{
			try
			{
				xContent = xParser->getContentFromHierarchyIdentifier( aIdentifier );
			}
			catch ( IllegalArgumentException )
			{
				throw;
			}
		}
	}

	return xContent;
}

//---------------------------------------------------------------------------

Sequence< Any > OContentHolder::getHierarchyIdentifierFromContent( const Reference< XContent > &xContent ) throw( IllegalArgumentException )
{
	Sequence< Any >	aIdentifier;

	if ( xContent.is() && m_xPhysicalContent.is() && xContent != m_xPhysicalContent )
	{
		Reference< XContentHierarchy >	xParser = Reference< XContentHierarchy >::query( m_xPhysicalContent );

		if ( xParser.is() )
			aIdentifier = xParser->getHierarchyIdentifierFromContent( xContent );
	}

	return aIdentifier;
}

//-----------------------------------------------------------------------------

sal_Bool SAL_CALL OContentHolder::compareHierarchyIdentifiers( const Sequence< Any > &id1, const Sequence< Any > &id2 ) throw()
{
	if ( m_xPhysicalContent.is() )
	{
		Reference< XContentHierarchy >	xParser = Reference< XContentHierarchy >::query( m_xPhysicalContent );

		if ( xParser.is() )
			return xParser->compareHierarchyIdentifiers( id1, id2 );
	}

	return sal_False;
}

//-----------------------------------------------------------------------------
//	XContent implementation
//-----------------------------------------------------------------------------

Reference< XContentIdentifier > SAL_CALL OContentHolder::getIdentifier(  ) throw(RuntimeException)
{
	Reference< XContentIdentifier >	xIdentifier;

	if ( m_xPhysicalContent.is() )
		xIdentifier = m_xPhysicalContent->getIdentifier();

	return xIdentifier;
}

//-----------------------------------------------------------------------------

OWString SAL_CALL OContentHolder::getContentType(  ) throw(RuntimeException)
{
	if ( m_xPhysicalContent.is() )
		return m_xPhysicalContent->getContentType();

	return OWString();
}


//-----------------------------------------------------------------------------
//	XCommandTaskProcessor implementation
//-----------------------------------------------------------------------------

#if (SUPD >= 566 )
Reference< XCommandTask > SAL_CALL OContentHolder::createCommandTask( const Command& TheCommand, const Reference< XResultAcceptor > &ResultAcceptor, const Reference< XContentTaskEnvironment >& Environment ) throw(RuntimeException)
#else
Reference< XCommandTask > SAL_CALL OContentHolder::createCommandTask( const Command& TheCommand, const Reference< XContentTaskEnvironment >& Environment ) throw(RuntimeException)
#endif
{
	Reference< XCommandTaskProcessor >		xProcessor = Reference< XCommandTaskProcessor >::query( m_xPhysicalContent );
	Reference< XContentTaskEnvironment >	xEnv = new OCommandTaskWrapper( TheCommand, Environment );
	Reference< XCommandTask >				xTask, xWrappedTask = Reference< XCommandTask >::query( xEnv );

	if ( xProcessor.is() )
#if (SUPD >= 566 )
		xTask = xProcessor->createCommandTask( TheCommand, ResultAcceptor, xEnv );
#else
		xTask = xProcessor->createCommandTask( TheCommand, xEnv );
#endif

	if ( xTask.is() && xEnv.is() )
		xEnv->getClient()->contentTaskEvent( ContentTaskEvent(xTask, ContentTaskStatus_CREATED, ContentTaskStatus_CREATED) );

	return xWrappedTask.is() ? xWrappedTask : xTask;
}

//-----------------------------------------------------------------------------

Reference< XCommandInfo > SAL_CALL OContentHolder::getCommandsInfo(  ) throw(RuntimeException)
{
	Reference< XCommandInfo >	xInfo;
	Reference< XCommandTaskProcessor >	xProcessor = Reference<XCommandTaskProcessor>::query( m_xPhysicalContent );

	if ( xProcessor.is() )
		xInfo = xProcessor->getCommandsInfo();

	return xInfo;
}

//-----------------------------------------------------------------------------
//	XPropertyTaskProcessor implementation
//-----------------------------------------------------------------------------

Reference< XPropertyTask > SAL_CALL OContentHolder::createPropertyTask( PropertyTaskType Type, const Sequence< PropertyValue >& Properties, const Reference< XContentTaskEnvironment >& Environment ) throw()
{
	Reference< XPropertyTaskProcessor >		xProcessor = Reference< XPropertyTaskProcessor >::query( m_xPhysicalContent );
	Reference< XContentTaskEnvironment >	xEnv = new OPropertyTaskWrapper( Type, Environment );
	Reference< XPropertyTask >				xTask, xWrappedTask = Reference< XPropertyTask >::query( xEnv );

	if ( xProcessor.is() && Properties.getLength() )
		xTask = xProcessor->createPropertyTask( Type, Properties, xEnv );

	if ( xTask.is() && xEnv.is() )
		xEnv->getClient()->contentTaskEvent( ContentTaskEvent(xTask, ContentTaskStatus_CREATED, ContentTaskStatus_CREATED) );

	return xWrappedTask.is() ? xWrappedTask : xTask;
}

//-----------------------------------------------------------------------------

Reference< XPropertySetInfo > SAL_CALL OContentHolder::getPropertySetInfo() throw()
{
	Reference< XPropertySetInfo >	xInfo;

	if ( m_xPhysicalContent.is() )
	{
		Reference< XPropertyTaskProcessor >	xProcessor = Reference<XPropertyTaskProcessor>::query( m_xPhysicalContent );

		if ( xProcessor.is() )
			xInfo = xProcessor->getPropertySetInfo();
	}

	return xInfo;
}

//-----------------------------------------------------------------------------

void SAL_CALL OContentHolder::addPropertiesChangeListener( const Sequence< OWString >& PropertyNames, const Reference< XPropertiesChangeListener >& Listener ) throw(RuntimeException)
{
	if ( m_xPhysicalContent.is() )
	{
		Reference< XPropertyTaskProcessor >	xProcessor = Reference<XPropertyTaskProcessor>::query( m_xPhysicalContent );

		if ( xProcessor.is() )
			xProcessor->addPropertiesChangeListener( PropertyNames, Listener );
	}
}

//-----------------------------------------------------------------------------

void SAL_CALL OContentHolder::removePropertiesChangeListener( const Sequence< OWString >& PropertyNames, const Reference< XPropertiesChangeListener >& Listener ) throw(RuntimeException)
{
	if ( m_xPhysicalContent.is() )
	{
		Reference< XPropertyTaskProcessor >	xProcessor = Reference<XPropertyTaskProcessor>::query( m_xPhysicalContent );

		if ( xProcessor.is() )
			xProcessor->removePropertiesChangeListener( PropertyNames, Listener );
	}
}

//-----------------------------------------------------------------------------
//	XCommandInfo implementation
//-----------------------------------------------------------------------------

Sequence< CommandInfo > SAL_CALL OContentHolder::getCommands(  ) throw(RuntimeException)
{
	Sequence< CommandInfo >	aCommands;

	if ( m_xPhysicalContent.is() )
	{
		Reference< XCommandInfo >	xCommandInfo = Reference< XCommandInfo >::query( m_xPhysicalContent );

		if ( xCommandInfo.is() )
			aCommands = getCommands();
	}

	return aCommands;
}

///////////////////////////////////////////////////////////////////////////////
//
// ODynamicHierarchy
//
///////////////////////////////////////////////////////////////////////////////

#define VFS_SCHEME  L"vfs"
#define FILE_SCHEME	L"file"
#define FILE_ROOT	L"file:///"
#define GROUPCONFIG	L"GroupViewConfig"

Reference< XContentProviderManager >		ODynamicHierarchy::s_xProviderManager;
Reference< XContentProvider >				ODynamicHierarchy::s_xUCBProvider;
Reference< XMultiServiceFactory >			ODynamicHierarchy::s_xServiceFactory;
Reference< XHierarchyRegistry >				ODynamicHierarchy::s_xDesktopRegistry;
Reference< XContentIdentifierFactory >		ODynamicHierarchy::s_xIdentifierFactory;
Reference< XPropertySetRegistryFactory >	ODynamicHierarchy::s_xPropertyRegistryFactory;
Reference< XPropertySetRegistry >			ODynamicHierarchy::s_xPropertyRegistry;


//-----------------------------------------------------------------------------

OWString ODynamicHierarchy::getKeyName()
{
	Reference< XContentIdentifier >	xIdentifier = getIdentifier();

	if ( xIdentifier.is() )
		return xIdentifier->getContentIdentifier();
	else
		return L"<Root>";
}


//-----------------------------------------------------------------------------

static void SetTitle( Reference< XContent > & xContent, const OWString & crTitle )
{
	Reference< XPropertyTaskProcessor >	xTaskProcessor = Reference< XPropertyTaskProcessor >::query( xContent );

	if ( !xTaskProcessor.is() )
		return;

	Any	aTitle;
	aTitle <<= crTitle;

#if ( SUPD >= 565 )
	PropertyValue	aProperty( L"Title", -1, Type(), 0, aTitle, PropertyState_DIRECT_VALUE );
#else
	PropertyValue	aProperty( L"Title", -1, Reference< ::com::sun::star::reflection::XIdlClass >(), 0, aTitle, PropertyState_DIRECT_VALUE );
#endif
	Reference< XPropertyTask >	xTask = xTaskProcessor->createPropertyTask( PropertyTaskType_SET, Sequence< PropertyValue >( &aProperty,1 ), Reference< XContentTaskEnvironment >() );

	if ( xTask.is() )
		xTask->execute();
}

//-----------------------------------------------------------------------------

ODynamicHierarchy::ODynamicHierarchy( const Reference< XContent > & xContent, const Sequence< Any > & root ) :
OContentHolder( xContent ),
m_bIsOpen( sal_False ),
m_aOwnIdentifier( root )
{
	if ( xContent.is() )
	{
		Reference< XContentIdentifier >	xIdentifier = xContent->getIdentifier();

		if ( xIdentifier.is() )
			m_aOwnURL = xIdentifier->getContentIdentifier();

		OSL_ASSERT( m_aOwnURL.getLength() );
	}

	Sequence< Any >	aChildList;

	if ( s_xDesktopRegistry.is() )
		aChildList = s_xDesktopRegistry->listRegisteredChilds( getKeyName() );

	for ( int i = 0; i < aChildList.getLength(); i++ )
	{
		OWString	aURL;

		aChildList[i] >>= aURL;

		Reference< XContentIdentifier >	xIdentifier	= s_xIdentifierFactory->createContentIdentifier( aURL );

		if ( xIdentifier.is() && aURL.getLength() )
		{
			Reference< XContent > xContent = s_xUCBProvider->queryContent( xIdentifier );
			if ( xContent.is() )
				addChildContent( xContent );
		}
	}

#if 0
	if ( aChildList.getLength() )
	{
		OWString aURL = L"pop3://hro@sdmail/";

		Reference< XContentIdentifier >	xIdentifier	= s_xIdentifierFactory->createContentIdentifier( aURL );

		if ( xIdentifier.is() && aURL.getLength() )
		{
			Reference< XContent > xContent = s_xUCBProvider->queryContent( xIdentifier );
			if ( xContent.is() )
			{
				addChildContent( xContent );

				SetTitle( xContent, L"Mail Account" );
			}
		}

		aURL = L"ftp://anonymous:huhu@rabbit/";

		xIdentifier	= s_xIdentifierFactory->createContentIdentifier( aURL );

		if ( xIdentifier.is() && aURL.getLength() )
		{
			Reference< XContent > xContent = s_xUCBProvider->queryContent( xIdentifier );
			if ( xContent.is() )
				addChildContent( xContent );

			SetTitle( xContent, L"FTP Account" );
		}


		aURL = L"vfs://file:///c|/test.stg?/";

		xIdentifier	= s_xIdentifierFactory->createContentIdentifier( aURL );

		if ( xIdentifier.is() && aURL.getLength() )
		{
			Reference< XContent > xContent = s_xUCBProvider->queryContent( xIdentifier );
			if ( xContent.is() )
				addChildContent( xContent );

			SetTitle( xContent, L"FTP Account" );
		}
	}
#endif
}

//-----------------------------------------------------------------------------

ODynamicHierarchy::ODynamicHierarchy( const Reference< XMultiServiceFactory > & xServiceFactory ) :
OContentHolder( Reference< XContent >() ),
m_bIsOpen( sal_False ),
m_aOwnURL( L"<Root>" )
{
	Reference< XContentProvider >	xVfsProvider;
	// Make the initial service factory accessible
	s_xServiceFactory = xServiceFactory;

	if ( s_xServiceFactory.is() )
	{
		// Get the desktop registry service
		s_xDesktopRegistry = Reference< XHierarchyRegistry >::query( s_xServiceFactory->createInstance( L"com.sun.star.hierarchy.DesktopRegistry" ) );

		// Get the view property service
		s_xPropertyRegistryFactory = Reference< XPropertySetRegistryFactory >::query( s_xServiceFactory->createInstance( L"com.sun.star.ucb.Store" ) );

		if ( s_xPropertyRegistryFactory.is() )
		{
			OWString	aURL = L"\\hierarchy.vdt";
			OWString    aConfigURL;
			Reference< XConfigManager >	xConfigManager = getSpecialConfigManager( xServiceFactory );

			aConfigURL = getUserConfigURL( xConfigManager );

			aConfigURL += aURL;
			aConfigURL = OWString( L"file:///" ) + aConfigURL;

			s_xPropertyRegistry = s_xPropertyRegistryFactory->createPropertySetRegistry( aConfigURL );
		}

		// Get the UCB Provider manager
		s_xProviderManager = Reference< XContentProviderManager >::query(	s_xServiceFactory->createInstance(	L"com.sun.star.ucb.UniversalContentBroker" ) );

		if ( s_xProviderManager.is() )
		{
			xVfsProvider = Reference< XContentProvider >::query( s_xServiceFactory->createInstance( L"com.sun.star.hierarchy.VirtualFileSystem" ) );

			if ( xVfsProvider.is() )
				s_xProviderManager->registerContentProvider( xVfsProvider, VFS_SCHEME, sal_True );



			s_xUCBProvider = Reference< XContentProvider >( s_xProviderManager, UNO_QUERY );;

			if ( !s_xUCBProvider.is() )
				s_xUCBProvider = Reference< XContentProvider >( s_xServiceFactory->createInstance(	L"com.sun.star.ucb.UniversalContentBroker" ), UNO_QUERY );

			s_xIdentifierFactory = Reference< XContentIdentifierFactory >( s_xProviderManager, UNO_QUERY );
		}

		if ( s_xUCBProvider.is() && s_xIdentifierFactory.is() )
		{
			Reference< XContentIdentifier >	xIdentifier	= s_xIdentifierFactory->createContentIdentifier( L"vfs://file:///c|/test.stg?/" );
			
			if ( xIdentifier.is() )
			{
				try
				{
					Reference< XContent > xRoot = xVfsProvider->queryContent( xIdentifier );
					if ( xRoot.is() )
						addChildContent( xRoot );

				} catch ( IllegalIdentifierException )
				{
				}
			}

			xIdentifier	= s_xIdentifierFactory->createContentIdentifier( FILE_ROOT );
			
			if ( xIdentifier.is() )
			{
				try
				{
					Reference< XContent > xFileRoot = s_xUCBProvider->queryContent( xIdentifier );
					if ( xFileRoot.is() )
						addChildContent( xFileRoot );

				} catch ( IllegalIdentifierException )
				{
				}
			}

		}
	}
}

//-----------------------------------------------------------------------------

Reference< XContent > ODynamicHierarchy::packContent( const Reference< XContent > & xContent )
{
	Reference< XDynamicHierarchy >	xHierarchy = Reference< XDynamicHierarchy >::query( xContent );

	if ( xHierarchy.is() )
		return xContent;
	else
	{
		try 
		{
			Sequence< Any >	aIdentifier = getHierarchyIdentifierFromContent( xContent );
			Reference< XContent >	xWrappedContent = getContentFromHierarchyIdentifier( aIdentifier );

			return xWrappedContent;
		} catch( IllegalArgumentException )
		{
			return xContent;
		}
	}
}

//-----------------------------------------------------------------------------
//	OContentHelper implementation
//-----------------------------------------------------------------------------

void ODynamicHierarchy::doCommand( const Command & rCommand )
{
	Guard< MutexWrap > aGuard2( m_aContentList );

	if ( rCommand.Name == L"open" /*&& !m_bIsOpen*/ )
	{
		int len = m_aContentList.size();
		for ( safe_list< ContentEntry >::iterator it = m_aContentList.begin(); it != m_aContentList.end(); it++ )
		{
			ContentEvent	aEvent;

			aEvent.Action = ContentAction::INSERTED;
			aEvent.Source = (XContent *)this;
			aEvent.Content = it->xContent;

			broadcastContentEvent( aEvent );
		}

		m_bIsOpen = sal_True;
	}
	else if ( rCommand.Name == L"update" && m_bIsOpen )
	{
		// Nothing to do
	}
	else if ( rCommand.Name == L"synchronize" && m_bIsOpen )
	{
		// Nothing to do
	}
	else if ( rCommand.Name == L"close" && m_bIsOpen )
	{
		m_bIsOpen = sal_False;
	}
}

//-----------------------------------------------------------------------------


Sequence< PropertyValueInfo > ODynamicHierarchy::getProperties( const Sequence< PropertyValue > & Properties )
{
	Sequence< PropertyValueInfo >	aResult;
	int nProperties = Properties.getLength();

	if ( !m_xPropertySet.is() )
	{
		if ( s_xPropertyRegistry.is() )
			m_xPropertySet = s_xPropertyRegistry->openPropertySet( getKeyName(), sal_False );
	}

	for ( int i = 0; i < nProperties; i++ )
	{
		if ( m_xPropertySet.is() )
		{
			try
			{
				Any aValue = m_xPropertySet->getPropertyValue( Properties[i].Name );

				aResult.realloc( aResult.getLength() + 1 );
				aResult[aResult.getLength() - 1].Value = aValue;
				aResult[aResult.getLength() - 1].Attributes = Properties[i].Attributes;
				aResult[aResult.getLength() - 1].Type = Properties[i].Type;
				aResult[aResult.getLength() - 1].Handle = Properties[i].Handle;
				aResult[aResult.getLength() - 1].Name = Properties[i].Name;
				aResult[aResult.getLength() - 1].State = Properties[i].State;
				aResult[aResult.getLength() - 1].ValueState = PropertyValueState_PROCESSED;
			}
			catch ( UnknownPropertyException )
			{
//				OWString	aMessage( L"Unknown Property: " );
//				aMessage += Properties[i].Name;
//				OSL_ENSURE( 0, OWStringToOString( aMessage, RTL_TEXTENCODING_ASCII_US ) );
			}
			catch ( WrappedTargetException ) 
			{
			}
		}
	}

	return aResult;
}

//-----------------------------------------------------------------------------

Sequence< PropertyValueInfo > ODynamicHierarchy::setProperties( const Sequence< PropertyValue > & Properties )
{
	Sequence< PropertyValueInfo >	aResult;
	int nProperties = Properties.getLength();

	if ( !m_xPropertySet.is() )
	{
		if ( s_xPropertyRegistry.is() )
			m_xPropertySet = s_xPropertyRegistry->openPropertySet( getKeyName(), sal_True );
	}

	if ( m_xPropertySet.is() )
	{
		try
		{
				m_xPropertySet->addPropertyChangeListener( L"", this );
		}
		catch( UnknownPropertyException )
		{
			OSL_ENSURE( 0, "UnknownPropertyException with addPropertyChangeListener" );
		}
	}

	for ( int i = 0; i < nProperties; i++ )
	{
		if ( m_xPropertySet.is() )
		{
			try
			{

				try
				{
					m_xPropertySet->setPropertyValue( Properties[i].Name, Properties[i].Value );
				}
				catch( UnknownPropertyException )
				{
					Reference< XPropertyContainer >	xContainer = Reference< XPropertyContainer >::query( m_xPropertySet );

					if ( xContainer.is() )
					{
						try
						{
							xContainer->addProperty( Properties[i].Name, Properties[i].Attributes, Properties[i].Value );
						}
						catch ( PropertyExistException )
						{
							OSL_ASSERT( 0 );
						}
					}
					m_xPropertySet->setPropertyValue( Properties[i].Name, Properties[i].Value );
				}



			aResult.realloc( aResult.getLength() + 1 );
			aResult[aResult.getLength() - 1].Attributes = Properties[i].Attributes;
			aResult[aResult.getLength() - 1].Type = Properties[i].Type;
			aResult[aResult.getLength() - 1].Handle = Properties[i].Handle;
			aResult[aResult.getLength() - 1].Name = Properties[i].Name;
			aResult[aResult.getLength() - 1].State = Properties[i].State;
			aResult[aResult.getLength() - 1].Value = Properties[i].Value;
			aResult[aResult.getLength() - 1].ValueState = PropertyValueState_PROCESSED;

			}
			catch ( WrappedTargetException ) 
			{
			}
			catch ( IllegalArgumentException ) 
			{
			}
			catch ( PropertyVetoException ) 
			{
			}


		}
	}

	return aResult;
}

//-----------------------------------------------------------------------------
//	XInterface implementation
//-----------------------------------------------------------------------------

#if defined(TF_TYPE) || (SUPD >= 564)
Any SAL_CALL ODynamicHierarchy::queryInterface( const Type& type ) throw( RuntimeException )
{
	Any	aRet = ::cppu::queryInterface(
		type,
		SAL_STATIC_CAST( XDynamicHierarchy*, this ),
		SAL_STATIC_CAST( XHierarchyProvider*, this ),
		SAL_STATIC_CAST( XPropertiesChangeListener*, this ),
		SAL_STATIC_CAST( XPropertyChangeListener*, this )
		);

	return ( aRet.hasValue() ? aRet : OContentHolder::queryInterface( type ) );
}
#else
sal_Bool SAL_CALL ODynamicHierarchy::queryInterface( const Uik& type, Any& ifc ) throw( RuntimeException )
{
	return ::com::sun::star::uno::queryInterface(
		type, ifc,
		SAL_STATIC_CAST( XDynamicHierarchy*, this ),
		SAL_STATIC_CAST( XHierarchyProvider*, this ),
		SAL_STATIC_CAST( XPropertiesChangeListener*, this ),
		SAL_STATIC_CAST( XPropertyChangeListener*, this )
		)
		|| OContentHolder::queryInterface( type, ifc );
}
#endif

//-----------------------------------------------------------------------------
//	XEventListener implementation
//-----------------------------------------------------------------------------

void SAL_CALL ODynamicHierarchy::disposing( const EventObject& Source ) throw(RuntimeException)
{
	OContentHolder::disposing( Source );

	if ( Source.Source == m_xPropertySet )
		m_xPropertySet = Reference< XPersistentPropertySet >();
}

//-----------------------------------------------------------------------------
//	XHierarchyProvider implementation
//-----------------------------------------------------------------------------

void SAL_CALL ODynamicHierarchy::setProviderRoot( const Sequence< Any > &aIdentifier ) throw(RuntimeException)
{
	m_aOwnIdentifier = aIdentifier;
}

//-----------------------------------------------------------------------------

Sequence< Any > SAL_CALL ODynamicHierarchy::getProviderRoot() throw(RuntimeException)
{
	return m_aOwnIdentifier;
}

//-----------------------------------------------------------------------------
//	XDynamicHierarchy implementation
//-----------------------------------------------------------------------------


Reference< XDynamicHierarchy > SAL_CALL ODynamicHierarchy::getSubHierarchy( const Sequence< Any > &aIdentifier ) throw(RuntimeException)
{
	Guard< MutexWrap >					aGuard( m_aContentList );
	Reference< XDynamicHierarchy >	xHierarchy;

	Reference< XContent >	xContent = getContentFromHierarchyIdentifier( aIdentifier );

	xHierarchy = Reference< XDynamicHierarchy >::query( xContent );

	if ( !xHierarchy.is() )
		xHierarchy = new ODynamicHierarchy( xContent, concat( m_aOwnIdentifier, aIdentifier ) );

	return xHierarchy;
}

//-----------------------------------------------------------------------------

Sequence< Any > ODynamicHierarchy::addChildContent( const Reference< XContent > & xContent )
{
	Guard< MutexWrap >	aGuard( m_aContentList );
	
	if ( xContent.is() )
	{
		Reference< XContentIdentifier >	xContentIdentifier = xContent->getIdentifier();
		if ( xContentIdentifier.is() )
		{
			OWString		aURL = xContentIdentifier->getContentIdentifier();

			ContentEntry		aEntry;
			aEntry.xContent = xContent;
			aEntry.aToken <<= aURL;

			if ( m_aContentList.find( aEntry ) == m_aContentList.end() )
				m_aContentList.push_back( aEntry );

			if ( m_bIsOpen )
			{
				ContentEvent	aEvent;

				aEvent.Source = SAL_STATIC_CAST(XContent *, this);
				aEvent.Content = xContent;
				aEvent.Action = ContentAction::INSERTED;

				broadcastContentEvent( aEvent );
			}

			return Sequence< Any >( &aEntry.aToken, 1 );
		}
	}
	
	return Sequence< Any >();
}

//---------------------------------------------------------------------------

Sequence< Any > SAL_CALL ODynamicHierarchy::registerChildContent( const Reference< XContent >& xContent ) throw(RuntimeException)
{
	Sequence< Any >	aSeq = addChildContent( xContent );

	if ( aSeq.getLength() && s_xDesktopRegistry.is() )
			s_xDesktopRegistry->registerChild( SAL_STATIC_CAST( XContentHierarchy *, this), getKeyName(), aSeq[0] );

	return aSeq;
}

//---------------------------------------------------------------------------

void SAL_CALL ODynamicHierarchy::deregisterChildHierarchy( const Sequence< Any > &aIdentifier ) throw(RuntimeException)
{
	Guard< MutexWrap >	aGuard( m_aContentList );

	if ( aIdentifier.getLength() == 1 )
	{
		Reference< XContent > xContent = getContentFromHierarchyIdentifier( aIdentifier );

		if ( xContent.is() )
		{
			ContentEntry	aEntry;
			aEntry.xContent = xContent;
			aEntry.aToken = aIdentifier[0];

			m_aContentList.remove( aEntry );

			if ( m_bIsOpen )
			{
				ContentEvent	aEvent;
				aEvent.Source = (XContent *)this;
				aEvent.Content = xContent;
				aEvent.Action = ContentAction::REMOVED;

				broadcastContentEvent( aEvent );
			}
		}
	}
}

//-----------------------------------------------------------------------------
//	XContentHierarchy
//-----------------------------------------------------------------------------

Reference< XContent > ODynamicHierarchy::getContentFromHierarchyIdentifier( const Sequence< Any > &aIdentifier )
{
	Guard< MutexWrap > aGuard( m_aContentList );

	// If length is 0 return ourself

	if ( aIdentifier.getLength() == 0 )
		return Reference< XContent >( this );

	Reference< XContent > xContent;
	Any	aToken	= aIdentifier[0];

	for ( safe_list< ContentEntry >::iterator it = m_aContentList.begin(); it != m_aContentList.end(); it++ )
	{
		if (it->aToken == aToken)
		{
			Sequence< Any >	aChildIdentifier = child( aIdentifier );

			if ( aChildIdentifier.getLength() )
			{
				Reference< XContentHierarchy >	xParser = Reference< XContentHierarchy >::query( it->xContent );
				if ( xParser.is() )
					xContent = xParser->getContentFromHierarchyIdentifier( aChildIdentifier );
			}
			else
				xContent = it->xContent;

			break;
		}
	}

	if ( !xContent.is() )
		xContent = OContentHolder::getContentFromHierarchyIdentifier( aIdentifier );

	if ( xContent.is() && !Reference< XDynamicHierarchy >::query( xContent ).is() )
			xContent = new ODynamicHierarchy( xContent, concat( m_aOwnIdentifier, aIdentifier ) );

	return xContent;
}


//---------------------------------------------------------------------------

Sequence< Any > ODynamicHierarchy::getHierarchyIdentifierFromContent( const Reference< XContent > &xContent )
{
	Guard< MutexWrap > aGuard( m_aContentList );

	Reference< XContentIdentifier >	xIdentifier;
	
	if ( xContent.is() )
		xIdentifier = xContent->getIdentifier();

	OWString	aScheme = xIdentifier->getContentProviderScheme();
	Reference< XContentProvider >	xProvider = s_xProviderManager->queryContentProvider( aScheme );

	for ( safe_list< ContentEntry >::iterator it = m_aContentList.begin(); it != m_aContentList.end(); it++ )
	{
		if ( it->xContent.is() )
		{
			Reference< XContentIdentifier >	xTempIdentifier = it->xContent->getIdentifier();
			OWString	aTempScheme = xTempIdentifier->getContentProviderScheme();

			if ( aTempScheme == aScheme )
			{
				if ( 0 == xProvider->compareContentIds( xTempIdentifier, xIdentifier ) )
					return Sequence< Any >( &it->aToken, 1 );
			}
		}

	}

	Sequence< Any > aIdentifier = OContentHolder::getHierarchyIdentifierFromContent( xContent );

	if ( aIdentifier.getLength() )
		return aIdentifier;


	for ( it = m_aContentList.begin(); it != m_aContentList.end() && !aIdentifier.getLength(); it++ )
	{
		Reference< XContentHierarchy >	xContentHierarchy( it->xContent, UNO_QUERY );

		if ( xContentHierarchy.is() )
		{
			try
			{
				aIdentifier = xContentHierarchy->getHierarchyIdentifierFromContent( xContent );
				if ( aIdentifier.getLength() )
					aIdentifier = concat( Sequence< Any >( &it->aToken, 1 ), aIdentifier );
			} catch ( IllegalArgumentException )
			{
			}
		}
	}

	return aIdentifier;
}

//-----------------------------------------------------------------------------

sal_Bool SAL_CALL ODynamicHierarchy::compareHierarchyIdentifiers( const Sequence< Any > &id1, const Sequence< Any > &id2 ) throw()
{
	Guard< MutexWrap > aGuard( m_aContentList );

	if ( id1.getLength() != id2.getLength() )
		return sal_False;

	return OContentHolder::compareHierarchyIdentifiers( id1, id2 );
}

//-----------------------------------------------------------------------------
//	XPropertyTaskProcessor implementation
//-----------------------------------------------------------------------------

Reference< XPropertyTask > SAL_CALL ODynamicHierarchy::createPropertyTask( PropertyTaskType Type, const Sequence< PropertyValue >& Properties, const Reference< XContentTaskEnvironment >& Environment ) throw()
{
	Sequence< PropertyValue >			aPhysicalProperties( Properties.getLength() );
	Sequence< PropertyValue >			aViewProperties( Properties.getLength() );

	int	nPhysical = 0;
	int nView = 0;

	for ( int i = 0; i < Properties.getLength(); i++ )
	{
		OWString	aName = Properties[i].Name;

		if ( m_xPhysicalPropertySetInfo.is() && m_xPhysicalPropertySetInfo->hasPropertyByName( aName ) )
			aPhysicalProperties[nPhysical++] = Properties[i];
		else
			aViewProperties[nView++] = Properties[i];
	}

	aViewProperties.realloc( nView );
	aPhysicalProperties.realloc( nPhysical );

	// HACK: Alles umsonst weil das CHAOS Scheie erz"ahlt

//	aViewProperties = Properties;
//	aPhysicalProperties = Properties;

	Reference< XPropertyTask >				xWrappedTask = OContentHolder::createPropertyTask( Type, aPhysicalProperties, Environment );
	Reference< XContentTaskEnvironment >	xEnv = Reference< XContentTaskEnvironment >::query( xWrappedTask );

	Reference< XPropertyTask >				xTask = new ODynamicHierarchyPropertyTask( xEnv.is() ? xEnv : Environment, Type, aViewProperties, this );

	if ( xTask.is() && xEnv.is() )
		xEnv->getClient()->contentTaskEvent( ContentTaskEvent(xTask, ContentTaskStatus_CREATED, ContentTaskStatus_CREATED) );

	return xWrappedTask.is() ? xWrappedTask : xTask;
}

//-----------------------------------------------------------------------------

Reference< XPropertySetInfo > SAL_CALL ODynamicHierarchy::getPropertySetInfo() throw()
{
	if ( !m_xPhysicalPropertySetInfo.is() )
		m_xPhysicalPropertySetInfo = OContentHolder::getPropertySetInfo();
	
	Sequence< Property >	aProperties;

	if ( m_xPhysicalPropertySetInfo.is() )
		aProperties = m_xPhysicalPropertySetInfo->getProperties();

	// Add the view properties here
	// TODO: ...

	Sequence< Property >	aViewProperties( 1 );

	aViewProperties[0].Name = GROUPCONFIG;
	aViewProperties[0].Handle = -1;

	Reference< XPropertySetInfo >	xNewInfo = new OPropertySetInfoHelper( concat( aProperties, aViewProperties ) );


	return xNewInfo.is() ? xNewInfo : m_xPhysicalPropertySetInfo;
}


//-----------------------------------------------------------------------------

void SAL_CALL ODynamicHierarchy::addPropertiesChangeListener( const Sequence< OWString >& PropertyNames, const Reference< XPropertiesChangeListener >& Listener ) throw(RuntimeException)
{
	if ( !m_xPropertySet.is() )
	{
		if ( s_xPropertyRegistry.is() )
			m_xPropertySet = s_xPropertyRegistry->openPropertySet( getKeyName(), sal_True );
	}

	if ( m_xPropertySet.is() )
	{
		if ( PropertyNames.getLength() )
		{
			try
			{
				m_xPropertySet->addPropertyChangeListener( L"", this );
			}
			catch( UnknownPropertyException )
			{
				OSL_ENSURE( 0, "UnknownPropertyException with addPropertyChangeListener" );
			}
		}
		else for ( int i = 0; i < PropertyNames.getLength(); i++ )
		{
			try
			{
				m_xPropertySet->addPropertyChangeListener( PropertyNames[i], this );
			}
			catch( UnknownPropertyException )
			{
				OSL_ENSURE( 0, "UnknownPropertyException with addPropertyChangeListener" );
			}
		}
	}

	if ( !m_xPhysicalPropertySetInfo.is() )
		m_xPhysicalPropertySetInfo = OContentHolder::getPropertySetInfo();

	Sequence< OWString >			aPhysicalProperties;
	Sequence< OWString >			aViewProperties;

	int	nPhysical = aPhysicalProperties.getLength();
	int nView = aViewProperties.getLength();

	for ( int i = 0; i < PropertyNames.getLength(); i++ )
	{
		if ( m_xPhysicalPropertySetInfo.is() && m_xPhysicalPropertySetInfo->hasPropertyByName( PropertyNames[i] ) )
		{
			aPhysicalProperties.realloc( nPhysical + 1 );
			aPhysicalProperties[nPhysical++] = PropertyNames[i];
		}
		else
		{
			aViewProperties.realloc( nView + 1 );
			aViewProperties[nView++] = PropertyNames[i];
		}
	}

	if ( nPhysical != 0 || PropertyNames.getLength() == 0 )
		OContentHolder::addPropertiesChangeListener( aPhysicalProperties, Listener );

	if ( nView != 0 || PropertyNames.getLength() == 0 )
		OPropertyHelper::addPropertiesChangeListener( aViewProperties, Listener );

}

//-----------------------------------------------------------------------------

void SAL_CALL ODynamicHierarchy::removePropertiesChangeListener( const Sequence< OWString >& PropertyNames, const Reference< XPropertiesChangeListener >& Listener ) throw(RuntimeException)
{
	if ( !m_xPhysicalPropertySetInfo.is() )
		m_xPhysicalPropertySetInfo = OContentHolder::getPropertySetInfo();

	Sequence< OWString >			aPhysicalProperties;
	Sequence< OWString >			aViewProperties;

	int	nPhysical = aPhysicalProperties.getLength();
	int nView = aViewProperties.getLength();

	for ( int i = 0; i < PropertyNames.getLength(); i++ )
	{
		if ( m_xPhysicalPropertySetInfo.is() && m_xPhysicalPropertySetInfo->hasPropertyByName( PropertyNames[i] ) )
		{
			aPhysicalProperties.realloc( nPhysical + 1 );
			aPhysicalProperties[nPhysical++] = PropertyNames[i];
		}
		else
		{
			aViewProperties.realloc( nView + 1 );
			aViewProperties[nView++] = PropertyNames[i];
		}
	}

	if ( nPhysical != 0 || PropertyNames.getLength() == 0 )
		OContentHolder::removePropertiesChangeListener( aPhysicalProperties, Listener );

	if ( nView != 0 || PropertyNames.getLength() == 0 )
		OPropertyHelper::removePropertiesChangeListener( aViewProperties, Listener );
}

//-----------------------------------------------------------------------------
// XPropertiesChangeListener implementation
//-----------------------------------------------------------------------------

void SAL_CALL ODynamicHierarchy::propertiesChange( const Sequence< PropertyChangeEvent >& Events ) throw(RuntimeException)
{
	Sequence< PropertyChangeEvent >	aEvents( Events );

	for ( int i = 0; i < aEvents.getLength(); i++ )
		aEvents[i].Source = SAL_STATIC_CAST(XContent *,this);

	broadcastPropertiesChangeEvent( aEvents );
}

//-----------------------------------------------------------------------------
//	XCommandTaskProcessor implementation
//-----------------------------------------------------------------------------

#if (SUPD >= 566 )
Reference< XCommandTask > SAL_CALL ODynamicHierarchy::createCommandTask( const Command& TheCommand, const Reference< XResultAcceptor > & ResultAcceptor, const Reference< XContentTaskEnvironment >& Environment ) throw(RuntimeException)
#else
Reference< XCommandTask > SAL_CALL ODynamicHierarchy::createCommandTask( const Command& TheCommand, const Reference< XContentTaskEnvironment >& Environment ) throw(RuntimeException)
#endif
{
#if (SUPD >= 566 )
	Reference< XCommandTask >				xWrappedTask = OContentHolder::createCommandTask( TheCommand, ResultAcceptor, Environment );
#else
	Reference< XCommandTask >				xWrappedTask = OContentHolder::createCommandTask( TheCommand, Environment );
#endif
	Reference< XContentTaskEnvironment >	xEnv = Reference< XContentTaskEnvironment >::query( xWrappedTask );

	Reference< XCommandTask >				xTask = new ODynamicHierarchyCommandTask( xEnv.is() ? xEnv : Environment, TheCommand, this );

	if ( xTask.is() && xEnv.is() )
		xEnv->getClient()->contentTaskEvent( ContentTaskEvent(xTask, ContentTaskStatus_CREATED, ContentTaskStatus_CREATED) );

	return xWrappedTask.is() ? xWrappedTask : xTask;
}

//-----------------------------------------------------------------------------

Reference< XCommandInfo > SAL_CALL ODynamicHierarchy::getCommandsInfo(  ) throw(RuntimeException)
{
	return this;
}

//-----------------------------------------------------------------------------
//	XCommandInfo implementation
//-----------------------------------------------------------------------------

Sequence< CommandInfo > SAL_CALL ODynamicHierarchy::getCommands(  ) throw(RuntimeException)
{
	Sequence< CommandInfo >	aInfo =	OContentHolder::getCommands();
	int						nLen = aInfo.getLength();

	//if ( !OContentHolder::hasCommandByName( L"open" ) )
	{
		aInfo.realloc( nLen + 1 );

		aInfo[nLen].Name = L"open";
		aInfo[nLen].Handle = -1;
//		aInfo[nLen].ArgType = (void *);

		nLen++;
	}

	//if ( !OContentHolder::hasCommandByName( L"update" ) )
	{
		aInfo.realloc( nLen + 1 );

		aInfo[nLen].Name = L"update";
		aInfo[nLen].Handle = -1;
//		aInfo[nLen].ArgType = (void *);

		nLen++;
	}

	//if ( !OContentHolder::hasCommandByName( L"synchronize" ) )
	{
		aInfo.realloc( nLen + 1 );

		aInfo[nLen].Name = L"synchronize";
		aInfo[nLen].Handle = -1;
//		aInfo[nLen].ArgType = (void *);

		nLen++;
	}

	//if ( !OContentHolder::hasCommandByName( L"close" ) )
	{
		aInfo.realloc( nLen + 1 );

		aInfo[nLen].Name = L"close";
		aInfo[nLen].Handle = -1;
//		aInfo[nLen].ArgType = (void *);

		nLen++;
	}

	return aInfo;
}



///////////////////////////////////////////////////////////////////////////////
//
// ODynamicHierarchyCommandTask
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// OContentTaskHelper implementation
//-----------------------------------------------------------------------------

void ODynamicHierarchyCommandTask::doExecute()
{
	if ( m_pHierarchy )
		m_pHierarchy->doCommand( getCommand() );
}

///////////////////////////////////////////////////////////////////////////////
//
// ODynamicHierarchyPropertyTask
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// OContentTaskHelper implementation
//-----------------------------------------------------------------------------

void ODynamicHierarchyPropertyTask::doExecute()
{
	if ( m_pHierarchy )
	{
		switch ( getType() )
		{
		case PropertyTaskType_GET:
			m_aProperties = m_pHierarchy->getProperties( m_aPropertyValues );
			break;
		case PropertyTaskType_SET:
			m_aProperties = m_pHierarchy->setProperties( m_aPropertyValues );
			break;
		default:
			break;
		}
	}
}

