/*************************************************************************
 *
 *  $RCSfile: argumentanalyzer.cxx,v $
 *
 *  $Revision: 1.17.2.1 $
 *
 *  last change: $Author: mh $ $Date: 2002/10/31 20:57:03 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

//_________________________________________________________________________________________________________________
//	my own includes
//_________________________________________________________________________________________________________________

#ifndef __FRAMEWORK_CLASSES_ARGUMENTANALYZER_HXX_
#include <classes/argumentanalyzer.hxx>
#endif

//_________________________________________________________________________________________________________________
//	interface includes
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	other includes
//_________________________________________________________________________________________________________________

#ifndef _URLOBJ_HXX
#include <tools/urlobj.hxx>
#endif

//_________________________________________________________________________________________________________________
//	const
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	namespace
//_________________________________________________________________________________________________________________

namespace framework{

//_________________________________________________________________________________________________________________
//	non exported const
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	non exported declarations
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	definitions
//_________________________________________________________________________________________________________________

/*-************************************************************************************************************//**
    @short      set argument list for analyzing
    @descr      Normaly we analyze all arguments of given list ... but if you call us without our default mask
                we analyze masked values only!
                Use EArgument values - combined by "|" - to build these mask.
                e.g. "nMask = E_URL | E_JUMPMARK | E_FILTERNAME"

    @seealso    methods setArguments()
    @seealso    enum EArgument

    @param      "pArgumentList" , sequence of arguments specified by name and value
    @param      "nMask"         , mask subset of arguments, which should be analyzed
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
ArgumentAnalyzer::ArgumentAnalyzer( css::uno::Sequence< css::beans::PropertyValue >& rArguments ,
                                    sal_uInt32                                       nMask      )
{
	m_pReadOnly  = NULL;
	m_pArguments = NULL;
    setArguments( rArguments, nMask );
}

//*****************************************************************************************************************
ArgumentAnalyzer::ArgumentAnalyzer( const css::uno::Sequence< css::beans::PropertyValue >& rArguments ,
                                          sal_uInt32                                       nMask      )
{
	m_pReadOnly  = NULL;
	m_pArguments = NULL;
    setArguments( rArguments, nMask );
}

//*****************************************************************************************************************
ArgumentAnalyzer::~ArgumentAnalyzer()
{
    // Forget all informations. These pointers should never be used again.
    m_aAnalyzed.forgetIt();
    m_pArguments = NULL;
    if (m_pReadOnly!=NULL)
        delete m_pReadOnly;
    m_pReadOnly = NULL;
}

/*-************************************************************************************************************//**
    @short      correct the internal saved MediaDescriptor
    @descr      There exist some inconsitent properties inside this structure.
                Sometimes we must handle two names for the same meaning ...
                Then we prefer the new notation. We delete the old (mostly deprecated
                item) and replace it by the new one.

    @attention  This method should be called before setArguments() analyze the given descriptor!
                Or they have to be called again to actualize the internal saved informations.
                The reason: we change the descriptor!

    @param      pDescriptor
                    pointer to the descriptor, we should reorganize
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::implst_correctOldFormat( css::uno::Sequence< css::beans::PropertyValue >* pDescriptor )
{
    if (pDescriptor==NULL)
        return;

    // analyze it
    // We safe indices to the positions of searched members
    // Such informations can be used to reorganize the descriptor later!
    sal_Int32  nURL          = -1;
    sal_Int32  nFileName     = -1;
    sal_Int32  nMediaType    = -1;
    sal_Int32  nContentType  = -1;
    sal_uInt32 nCount        = pDescriptor->getLength();
    sal_Int32  nPackedLength = nCount;

    for( sal_uInt32 nItem=0; nItem<nCount; ++nItem )
    {
        if( (*pDescriptor)[nItem].Name == DECLARE_ASCII("URL") )
            nURL = nItem;
        else
        if( (*pDescriptor)[nItem].Name == DECLARE_ASCII("FileName") )
            nFileName = nItem;
        else
        if( (*pDescriptor)[nItem].Name == DECLARE_ASCII("MediaType") )
            nMediaType = nItem;
        else
        if( (*pDescriptor)[nItem].Name == DECLARE_ASCII("ContentType") )
            nContentType = nItem;
    }

    // reorganize it
    // a) if "FileName" but not "URL" exist -> rename it to "URL"
    if (nURL==-1 && nFileName!=-1)
        (*pDescriptor)[nFileName].Name = DECLARE_ASCII("URL");

    // b) if "ContentType" but not "MediaType" exist -> rename it to "MediaType"
    if (nMediaType==-1 && nContentType!=-1)
        (*pDescriptor)[nContentType].Name = DECLARE_ASCII("MediaType");

    // c) if "FileName" and "URL" exist -> prefer "URL" and delete "FileName"
    //    Deletion is done by copying the last item of the descriptor to this position.
    //    But you have to look, if it's neccessary or possible to do so!
    if (nURL!=-1 && nFileName!=-1)
    {
        --nPackedLength;
        if (nFileName!=nPackedLength)
            (*pDescriptor)[nFileName] = (*pDescriptor)[nPackedLength];
    }

    // c1) check first, if copy action (for URL and FileName before) has copied
    //     one of the required items for the next action! Correct it ...
    if (nPackedLength==nMediaType)
        nMediaType = nFileName;
    else
    if (nPackedLength==nContentType)
        nContentType = nFileName;

    // d) if "ContentType" and "MediaType" exist -> prefer "MediaType" and delete "ContentType"
    if (nMediaType!=-1 && nContentType!=-1)
    {
        --nPackedLength;
        if (nContentType!=nPackedLength)
            (*pDescriptor)[nFileName] = (*pDescriptor)[nPackedLength];
    }

    // now it's time to pack the descriptor
    // reallocate it to the new size
    // Last (two!?) items should be empty and can be deleted.
    pDescriptor->realloc(nPackedLength);
}

/*-************************************************************************************************************//**
    @short      set argument list to analyze
    @descr      Normaly you give me a argument list at creation by using my ctor ...
                but sometimes it's neccessary to set a new list during runtime.

    @seealso    constructor

    @param      "lArgumentList" , sequence of arguments specified by name and value
    @param      "nMask"         , mask subset of arguments, which should be analyzed
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::setArguments( const css::uno::Sequence< css::beans::PropertyValue >& rArguments ,
                                           sal_uInt32                                       nMask      )
{
    // Trick to make it realy readonly!
    // We copy the whole descriptor to an internal member and work further on it.
    // Then our member m_pArguments doesnt point to rArguments - it points
    // to our other member m_aReadOnly.
    if (m_pReadOnly!=NULL)
        delete m_pReadOnly;
    m_pReadOnly = NULL;

    m_pReadOnly = new css::uno::Sequence< css::beans::PropertyValue >(rArguments);
    setArguments( *m_pReadOnly, nMask );
}

void ArgumentAnalyzer::setArguments( css::uno::Sequence< css::beans::PropertyValue >& rArguments ,
                                     sal_uInt32                                       nMask      )
{
    // Method isn't defined for invalid parameters!
    LOG_ASSERT2( &rArguments ==NULL, "ArgumentAnalyzer::setArguments()", "Invalid parameter detected!" )

    // Save pointer to list internal ...
    // ... step over given list and try to find supported arguments.
    // If someone found set it on internal info structure.
    // Use mask to analyze specified items only. (default=all!)
    m_pArguments = &rArguments;
    m_aAnalyzed.forgetIt();

    // Before we analyze the given MediaDescriptor - we have to correct
    // some inconsistent (deprecated) arguments! It works on our internal saved
    // list (available on member m_pArguments) and change it. So it must be valid here ...
    ArgumentAnalyzer::implst_correctOldFormat(m_pArguments);

    sal_uInt32                  nCount = m_pArguments->getLength();
    css::beans::PropertyValue*  pItem  = m_pArguments->getArray() ;
    for( sal_uInt32 nItem=0; nItem<nCount; ++nItem, ++pItem )
	{
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_CHARACTERSET            )   &&
            ( pItem->Name == ARGUMENTNAME_CHARACTERSET )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"CharacterSet\"?" )
            m_aAnalyzed.psCharacterSet = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_MEDIATYPE            )   &&
            ( pItem->Name == ARGUMENTNAME_MEDIATYPE )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"MediaType\"?" )
            m_aAnalyzed.psMediaType = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_DETECTSERVICE            )   &&
            ( pItem->Name == ARGUMENTNAME_DETECTSERVICE )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"DetectService\"?" )
            m_aAnalyzed.psDetectService = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_EXTENSION            )   &&
            ( pItem->Name == ARGUMENTNAME_EXTENSION )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Extension\"?" )
            m_aAnalyzed.psExtension = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_URL            )   &&
            ( pItem->Name == ARGUMENTNAME_URL )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"URL\"?" )
            m_aAnalyzed.psURL = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_FILTERNAME            )   &&
            ( pItem->Name == ARGUMENTNAME_FILTERNAME )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"FilterName\"?" )
            m_aAnalyzed.psFilterName = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_TYPENAME            )   &&
            ( pItem->Name == ARGUMENTNAME_TYPENAME )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"TypeName\"?" )
            m_aAnalyzed.psTypeName = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_FORMAT            )   &&
            ( pItem->Name == ARGUMENTNAME_FORMAT )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Format\"?" )
            m_aAnalyzed.psFormat = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_FRAMENAME            )   &&
            ( pItem->Name == ARGUMENTNAME_FRAMENAME )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"FrameName\"?" )
            m_aAnalyzed.psFrameName = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_PATTERN            )   &&
            ( pItem->Name == ARGUMENTNAME_PATTERN )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Pattern\"?" )
            m_aAnalyzed.psPattern = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_POSTDATA            )   &&
            ( pItem->Name == ARGUMENTNAME_POSTDATA )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"PostData\"?" )
            m_aAnalyzed.psPostData = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_POSTSTRING            )   &&
            ( pItem->Name == ARGUMENTNAME_POSTSTRING )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"PostString\"?" )
            m_aAnalyzed.psPostString = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_REFERRER            )   &&
            ( pItem->Name == ARGUMENTNAME_REFERRER )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Referrer\"?" )
            m_aAnalyzed.psReferrer = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_TEMPLATENAME            )   &&
            ( pItem->Name == ARGUMENTNAME_TEMPLATENAME )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"TemplateName\"?" )
            m_aAnalyzed.psTemplateName = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_TEMPLATEREGIONNAME            )   &&
            ( pItem->Name == ARGUMENTNAME_TEMPLATEREGIONNAME )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"TemplateRegionName\"?" )
            m_aAnalyzed.psTemplateRegionName = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_VERSION            )   &&
            ( pItem->Name == ARGUMENTNAME_VERSION )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_SHORT, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Version\"?" )
            m_aAnalyzed.pnVersion = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_VIEWID            )   &&
            ( pItem->Name == ARGUMENTNAME_VIEWID )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_SHORT, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"ViewId\"?" )
            m_aAnalyzed.pnViewId = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_FLAGS            )   &&
            ( pItem->Name == ARGUMENTNAME_FLAGS )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_LONG, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Flags\"?" )
            m_aAnalyzed.pnFlags = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_ASTEMPLATE            )   &&
            ( pItem->Name == ARGUMENTNAME_ASTEMPLATE )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_BOOLEAN, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"AsTemplate\"?" )
            m_aAnalyzed.pbAsTemplate = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_HIDDEN            )   &&
            ( pItem->Name == ARGUMENTNAME_HIDDEN )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_BOOLEAN, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Hidden\"?" )
            m_aAnalyzed.pbHidden = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_OPENNEWVIEW            )   &&
            ( pItem->Name == ARGUMENTNAME_OPENNEWVIEW )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_BOOLEAN, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"OpenNewView\"?" )
            m_aAnalyzed.pbOpenNewView = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_READONLY            )   &&
            ( pItem->Name == ARGUMENTNAME_READONLY )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_BOOLEAN, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"ReadOnly\"?" )
            m_aAnalyzed.pbReadOnly = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_PREVIEW            )   &&
            ( pItem->Name == ARGUMENTNAME_PREVIEW )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_BOOLEAN, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Preview\"?" )
            m_aAnalyzed.pbPreview = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_SILENT            )   &&
            ( pItem->Name == ARGUMENTNAME_SILENT )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_BOOLEAN, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"Silent\"?" )
            m_aAnalyzed.pbSilent = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_DEEPDETECTION            )   &&
            ( pItem->Name == ARGUMENTNAME_DEEPDETECTION )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_BOOLEAN, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"DeepDetection\"?" )
            m_aAnalyzed.pbDeepDetection = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_POSSIZE            )   &&
            ( pItem->Name == ARGUMENTNAME_POSSIZE )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRUCT, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"PosSize\"?" )
            m_aAnalyzed.paPosSize = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_INPUTSTREAM            )   &&
            ( pItem->Name == ARGUMENTNAME_INPUTSTREAM )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_INTERFACE, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"InputStream\"?" )
            m_aAnalyzed.pxInputStream = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_OUTPUTSTREAM            )   &&
            ( pItem->Name == ARGUMENTNAME_OUTPUTSTREAM )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_INTERFACE, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"OutputStream\"?" )
            m_aAnalyzed.pxOutputStream = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_STATUSINDICATOR            )   &&
            ( pItem->Name == ARGUMENTNAME_STATUSINDICATOR )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_INTERFACE, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"StatusIndicator\"?" )
            m_aAnalyzed.pxStatusIndicator = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_INTERACTIONHANDLER            )   &&
            ( pItem->Name == ARGUMENTNAME_INTERACTIONHANDLER )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_INTERFACE, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"InteractionHandler\"?" )
            m_aAnalyzed.pxInteractionHandler = nItem;
		}
		else
		//---------------------------------------------------------------------------------------------------------
        if(
            ( nMask       &  E_JUMPMARK            )   &&
            ( pItem->Name == ARGUMENTNAME_JUMPMARK )
          )
		{
            LOG_ASSERT2( pItem->Value.getValueTypeClass()!=css::uno::TypeClass_STRING, "ArgumentAnalyzer::setArguments()", "Who has changed the type of argument \"JumpMark\"?" )
            m_aAnalyzed.psJumpMark = nItem;
		}
		//---------------------------------------------------------------------------------------------------------
/*
        #ifdef ENABLE_WARNINGS
        else LOG_ASSERT2( nMask&ANALYZE_ALL_ARGUMENTS, "ArgumentAnalyzer::setArguments()", "Unsupported argument found!" )
        #endif
*/
    }

	// We must analyze a possible set URL again. May the user forgot to
	// set the JumpMark property. Then we do it here ...
    // But supress it for readonly mode !
    if (m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST && m_pReadOnly == NULL)
	{
		::rtl::OUString sURL;
		if ((*m_pArguments)[m_aAnalyzed.psURL].Value >>= sURL)
			setArgument(E_URL,sURL);
	}
}

/*-************************************************************************************************************//**
    @short      get valid state and value of an argument
    @descr      After calling setArguments() you can call these methods to get information about the content
                of set list. You give us a reference to an valid memory position and we fill the value
                at these position if search item exist. We return the valid state too. If we return FALSE
                the value not exist in argument list and return value was not filled!

    @seealso    method setArguments()

    @param      "eArgument" , specify, which argument is searched
    @param      "..Value"   , reference to user memory, which should be filled with right value, if argument exist
    @return     True, if argument could be found, false otherwise.

    @onerror    We return false.
*//*-*************************************************************************************************************/
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, ::rtl::OUString& sValue ) const
{
    sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_CHARACTERSET         :   if( m_aAnalyzed.psCharacterSet != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psCharacterSet].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_MEDIATYPE            :   if( m_aAnalyzed.psMediaType != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psMediaType].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_DETECTSERVICE        :   if( m_aAnalyzed.psDetectService != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psDetectService].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_EXTENSION            :   if( m_aAnalyzed.psExtension != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psExtension].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_URL                  :   if( m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psURL].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_FILTERNAME           :   if( m_aAnalyzed.psFilterName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psFilterName].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_FORMAT               :   if( m_aAnalyzed.psFormat != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psFormat].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_FRAMENAME            :   if( m_aAnalyzed.psFrameName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psFrameName].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_PATTERN              :   if( m_aAnalyzed.psPattern != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psPattern].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_POSTDATA             :   if( m_aAnalyzed.psPostData != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psPostData].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_POSTSTRING           :   if( m_aAnalyzed.psPostString != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psPostString].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_REFERRER             :   if( m_aAnalyzed.psReferrer != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psReferrer].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATENAME         :   if( m_aAnalyzed.psTemplateName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psTemplateName].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATEREGIONNAME   :   if( m_aAnalyzed.psTemplateRegionName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psTemplateRegionName].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_JUMPMARK             :   if( m_aAnalyzed.psJumpMark != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psJumpMark].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_TYPENAME             :   if( m_aAnalyzed.psTypeName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( (*m_pArguments)[m_aAnalyzed.psTypeName].Value >>= sValue );
										}
										break;
		//_________________________________________________________________________________________________________
/*
        #ifdef  ENABLE_WARNINGS
        default                     :   LOG_WARNING( "ArgumentAnalyzer::getArgument( string )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::util::URL& aValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_URL  :   {
                            if( m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST )
                            {
                                ::rtl::OUString sURL     ;
                                ::rtl::OUString sJumpMark;
                                (*m_pArguments)[m_aAnalyzed.psURL].Value >>= sURL;

                                INetURLObject aAssembler( sURL );
                                if( m_aAnalyzed.psJumpMark != ARGUMENT_NOT_EXIST )
                                {
                                    (*m_pArguments)[m_aAnalyzed.psJumpMark].Value >>= sJumpMark;
                                    aAssembler.SetMark( sJumpMark, INetURLObject::ENCODE_ALL );
                                }

                                INetURLObject aParser( aAssembler.GetMainURL( INetURLObject::NO_DECODE ) );
                                aValue.Protocol   = INetURLObject::GetScheme( aParser.GetProtocol() );
                                aValue.User       = aParser.GetUser   ( INetURLObject::DECODE_WITH_CHARSET );
                                aValue.Password   = aParser.GetPass   ( INetURLObject::DECODE_WITH_CHARSET );
                                aValue.Server     = aParser.GetHost   ( INetURLObject::DECODE_WITH_CHARSET );
                                aValue.Port       = (sal_Int16)aParser.GetPort();
                                aValue.Path       = aParser.GetURLPath( INetURLObject::NO_DECODE           );
                                aValue.Arguments  = aParser.GetParam  ( INetURLObject::NO_DECODE           );
                                aValue.Mark       = aParser.GetMark   ( INetURLObject::DECODE_WITH_CHARSET );
                                aParser.SetMark ( ::rtl::OUString() );
                                aParser.SetParam( ::rtl::OUString() );
                                aValue.Main       = aParser.GetMainURL( INetURLObject::NO_DECODE           );
                                bValid = sal_True;
                            }
                        }
                        break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default     :   LOG_WARNING( "ArgumentAnalyzer::getArgument( URL )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, sal_Int16& nValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_VERSION  :   if( m_aAnalyzed.pnVersion != ARGUMENT_NOT_EXIST )
							{
                                bValid = ( (*m_pArguments)[m_aAnalyzed.pnVersion].Value >>= nValue );
							}
							break;
		//_________________________________________________________________________________________________________
        case E_VIEWID   :   if( m_aAnalyzed.pnViewId != ARGUMENT_NOT_EXIST )
							{
                                bValid = ( (*m_pArguments)[m_aAnalyzed.pnViewId].Value >>= nValue );
							}
							break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default         :   LOG_WARNING( "ArgumentAnalyzer::getArgument( int16 )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, sal_Int32& nValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_FLAGS    :   if( m_aAnalyzed.pnFlags != ARGUMENT_NOT_EXIST )
							{
                                bValid = ( (*m_pArguments)[m_aAnalyzed.pnFlags].Value >>= nValue );
							}
							break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default         :   LOG_WARNING( "ArgumentAnalyzer::getArgument( int32 )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, sal_Bool& bValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_ASTEMPLATE       :   if( m_aAnalyzed.pbAsTemplate != ARGUMENT_NOT_EXIST )
									{
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pbAsTemplate].Value >>= bValue );
									}
                                    break;
		//_________________________________________________________________________________________________________
        case E_HIDDEN           :   if( m_aAnalyzed.pbHidden != ARGUMENT_NOT_EXIST )
									{
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pbHidden].Value >>= bValue );
									}
                                    break;
		//_________________________________________________________________________________________________________
        case E_OPENNEWVIEW      :   if( m_aAnalyzed.pbOpenNewView != ARGUMENT_NOT_EXIST )
									{
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pbOpenNewView].Value >>= bValue );
									}
                                    break;
		//_________________________________________________________________________________________________________
        case E_READONLY         :   if( m_aAnalyzed.pbReadOnly != ARGUMENT_NOT_EXIST )
									{
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pbReadOnly].Value >>= bValue );
									}
                                    break;
		//_________________________________________________________________________________________________________
        case E_PREVIEW          :   if( m_aAnalyzed.pbPreview != ARGUMENT_NOT_EXIST )
									{
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pbPreview].Value >>= bValue );
									}
                                    break;
		//_________________________________________________________________________________________________________
        case E_SILENT           :   if( m_aAnalyzed.pbSilent != ARGUMENT_NOT_EXIST )
									{
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pbSilent].Value >>= bValue );
									}
                                    break;
		//_________________________________________________________________________________________________________
        case E_DEEPDETECTION    :   if( m_aAnalyzed.pbDeepDetection != ARGUMENT_NOT_EXIST )
									{
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pbDeepDetection].Value >>= bValue );
									}
									break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                 :   LOG_WARNING( "ArgumentAnalyzer::getArgument( boolean )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::awt::Rectangle& aValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_POSSIZE  :   if( m_aAnalyzed.paPosSize != ARGUMENT_NOT_EXIST )
							{
                                bValid = ( (*m_pArguments)[m_aAnalyzed.paPosSize].Value >>= aValue );
							}
							break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default         :   LOG_WARNING( "ArgumentAnalyzer::getArgument( com.sun.star.awt.Rectangle )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
//	interface
//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::uno::Reference< css::io::XInputStream >& xValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_INPUTSTREAM  :   if( m_aAnalyzed.pxInputStream != ARGUMENT_NOT_EXIST )
								{
                                    bValid = ( (*m_pArguments)[m_aAnalyzed.pxInputStream].Value >>= xValue );
								}
								break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default             :   LOG_WARNING( "ArgumentAnalyzer::getArgument( com.sun.star.io.XInputStream )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::uno::Reference< css::io::XOutputStream >& xValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_OUTPUTSTREAM  :  if( m_aAnalyzed.pxOutputStream != ARGUMENT_NOT_EXIST )
								{
                                    bValid = ( (*m_pArguments)[m_aAnalyzed.pxOutputStream].Value >>= xValue );
								}
								break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default              :   LOG_WARNING( "ArgumentAnalyzer::getArgument( com.sun.star.io.XOutputStream )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::uno::Reference< css::task::XStatusIndicator >& xValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_STATUSINDICATOR  :   if( m_aAnalyzed.pxStatusIndicator != ARGUMENT_NOT_EXIST )
                                    {
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pxStatusIndicator].Value >>= xValue );
                                    }
                                    break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                 :   LOG_WARNING( "ArgumentAnalyzer::getArgument( com.sun.star.task.XStatusIndicator )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

//*****************************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::uno::Reference< css::task::XInteractionHandler >& xValue ) const
{
	sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_INTERACTIONHANDLER:  if( m_aAnalyzed.pxInteractionHandler != ARGUMENT_NOT_EXIST )
                                    {
                                        bValid = ( (*m_pArguments)[m_aAnalyzed.pxInteractionHandler].Value >>= xValue );
                                    }
                                    break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                 :   LOG_WARNING( "ArgumentAnalyzer::getArgument( com.sun.star.task.XInteractionHandler )", "Unknown argument detected!" )
        #endif
*/
	}

	return bValid;
}

/*-************************************************************************************************************//**
    @short      change or add a argument to current list
    @descr      Use these methods to add a new argument to current list or set a new value for it.

    @seealso    method getArgument()

    @param      "eArgument" , specify, which argument should be changed or added
    @param      "...Value"  , new value for this argument
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::setArgument( EArgument eArgument, const ::rtl::OUString& sValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(string)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_CHARACTERSET         :   {
                                            ppArgument = &(m_aAnalyzed.psCharacterSet);
                                            sName      = ARGUMENTNAME_CHARACTERSET ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_MEDIATYPE            :   {
                                            ppArgument = &(m_aAnalyzed.psMediaType);
                                            sName      = ARGUMENTNAME_MEDIATYPE ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_DETECTSERVICE        :   {
                                            ppArgument = &(m_aAnalyzed.psDetectService);
                                            sName      = ARGUMENTNAME_DETECTSERVICE ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_EXTENSION            :   {
                                            ppArgument = &(m_aAnalyzed.psExtension);
                                            sName      = ARGUMENTNAME_EXTENSION ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_URL                  :   {
                                            ppArgument = &(m_aAnalyzed.psURL);
                                            sName      = ARGUMENTNAME_URL ;

                                            INetURLObject aParser( sValue );
                                            if (aParser.GetProtocol()!=INET_PROT_NOT_VALID && aParser.HasMark())
                                            {
                                                sal_Int32*    ppJumpMark     = &(m_aAnalyzed.psJumpMark);
                                                css::uno::Any aPackedJumpMark;
                                                aPackedJumpMark<<=::rtl::OUString(aParser.GetMark( INetURLObject::DECODE_WITH_CHARSET ));
                                                impl_addOrChangeArgument( m_pArguments, ppJumpMark, ARGUMENTNAME_JUMPMARK, aPackedJumpMark );
                                            }
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_FILTERNAME           :   {
                                            ppArgument = &(m_aAnalyzed.psFilterName);
                                            sName      = ARGUMENTNAME_FILTERNAME ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_FORMAT               :   {
                                            ppArgument = &(m_aAnalyzed.psFormat);
                                            sName      = ARGUMENTNAME_FORMAT ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_FRAMENAME            :   {
                                            ppArgument = &(m_aAnalyzed.psFrameName);
                                            sName      = ARGUMENTNAME_FRAMENAME ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_PATTERN              :   {
                                            ppArgument = &(m_aAnalyzed.psPattern);
                                            sName      = ARGUMENTNAME_PATTERN ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_POSTDATA             :   {
                                            ppArgument = &(m_aAnalyzed.psPostData);
                                            sName      = ARGUMENTNAME_POSTDATA ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_POSTSTRING           :   {
                                            ppArgument = &(m_aAnalyzed.psPostString);
                                            sName      = ARGUMENTNAME_POSTSTRING ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_REFERRER             :   {
                                            ppArgument = &(m_aAnalyzed.psReferrer);
                                            sName      = ARGUMENTNAME_REFERRER ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATENAME         :   {
                                            ppArgument = &(m_aAnalyzed.psTemplateName);
                                            sName      = ARGUMENTNAME_TEMPLATENAME ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATEREGIONNAME   :   {
                                            ppArgument = &(m_aAnalyzed.psTemplateRegionName);
                                            sName      = ARGUMENTNAME_TEMPLATEREGIONNAME ;
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_JUMPMARK             :   {
                                            ppArgument = &(m_aAnalyzed.psJumpMark);
                                            sName      = ARGUMENTNAME_JUMPMARK ;
                                            // if we change the jumpmark - we have to change the complete URL too!
                                            // If no URL item currently exist inside this descriptor - set jumpmark
                                            // as the only partz of it.
                                            sal_Int32* ppURL = &(m_aAnalyzed.psURL);
                                            if (*ppURL==ARGUMENT_NOT_EXIST)
                                            {
                                                ::rtl::OUStringBuffer sBuilder(256);
                                                sBuilder.appendAscii("#"   );
                                                sBuilder.append     (sValue);
                                                css::uno::Any aPackedURL;
                                                aPackedURL<<=sBuilder.makeStringAndClear();
                                                impl_addOrChangeArgument( m_pArguments, ppURL, ARGUMENTNAME_URL, aPackedURL );
                                            }
                                            else
                                            {
                                                ::rtl::OUString sOldURL;
                                                ::rtl::OUString sNewURL;
                                                if ( (*m_pArguments)[m_aAnalyzed.psURL].Value >>= sOldURL )
                                                {
                                                    INetURLObject aAssembler(sOldURL);
                                                    if (aAssembler.GetProtocol()!=INET_PROT_NOT_VALID)
                                                    {
                                                        aAssembler.SetMark(sValue);
                                                        sNewURL = aAssembler.GetMainURL( INetURLObject::NO_DECODE );
                                                    }
                                                }

                                                if (sNewURL.getLength()<1)
                                                {
                                                    sal_Int32* ppLastArgument = m_aAnalyzed.getLastReference();
                                                    if(
                                                        ( ppURL          != NULL )  &&
                                                        ( ppLastArgument != NULL )
                                                      )
                                                    {
                                                        impl_deleteArgument( m_pArguments, ppURL, ppLastArgument );
                                                    }
                                                }
                                                else
                                                {
                                                    css::uno::Any aPackedURL;
                                                    aPackedURL<<=sNewURL;
                                                    impl_addOrChangeArgument( m_pArguments, ppURL, ARGUMENTNAME_URL, aPackedURL );
                                                }
                                            }
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_TYPENAME             :   {
                                            ppArgument = &(m_aAnalyzed.psTypeName);
                                            sName      = ARGUMENTNAME_TYPENAME ;
                                        }
										break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                     :   LOG_WARNING( "ArgumentAnalyzer::setArgument( string )", "Unknown argument detected!" )
        #endif
*/
	}

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=sValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::util::URL& aValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(util::URL)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    /*Attention
        This method is a special one! We don't hold an util::URL directly ... This one isn't supported as direct argument!
        But we can build it on different parts of our argument list ...{URL and JUMPMARK}.
        That's why we use our own "setArgument()" methods to set it.
     */

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
		case E_URL  		:	{
                                    // We can use setArgument( E_URL, string ) .. because
                                    // he analyze complete URLs too and split it into URL and JumpMark part!
                                    setArgument( E_URL, aValue.Complete );
								}
								break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default             :   LOG_WARNING( "ArgumentAnalyzer::setArgument( URL )", "Unknown argument detected!" )
        #endif
*/
	}
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, sal_Int16 nValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(short)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_VERSION  :   {
                                ppArgument = &(m_aAnalyzed.pnVersion);
                                sName      = ARGUMENTNAME_VERSION ;
                            }
                            break;
		//_________________________________________________________________________________________________________
        case E_VIEWID   :   {
                                ppArgument = &(m_aAnalyzed.pnViewId);
                                sName      = ARGUMENTNAME_VIEWID ;
                            }
                            break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default             :   LOG_WARNING( "ArgumentAnalyzer::setArgument( int16 )", "Unknown argument detected!" )
        #endif
*/
    }

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=nValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, sal_Int32 nValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(long)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_FLAGS  :   {
                              ppArgument = &(m_aAnalyzed.pnFlags);
                              sName      = ARGUMENTNAME_FLAGS ;
                          }
                          break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default             :   LOG_WARNING( "ArgumentAnalyzer::setArgument( int32 )", "Unknown argument detected!" )
        #endif
*/
    }

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=nValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, sal_Bool bValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(boolean)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
		case E_ASTEMPLATE		:	{
                                        ppArgument = &(m_aAnalyzed.pbAsTemplate);
                                        sName      = ARGUMENTNAME_ASTEMPLATE ;
									}
									break;
		//_________________________________________________________________________________________________________
		case E_HIDDEN			:	{
                                        ppArgument = &(m_aAnalyzed.pbHidden);
                                        sName      = ARGUMENTNAME_HIDDEN ;
									}
									break;
		//_________________________________________________________________________________________________________
		case E_OPENNEWVIEW		:	{
                                        ppArgument = &(m_aAnalyzed.pbOpenNewView);
                                        sName      = ARGUMENTNAME_OPENNEWVIEW ;
									}
									break;
		//_________________________________________________________________________________________________________
		case E_READONLY			:	{
                                        ppArgument = &(m_aAnalyzed.pbReadOnly);
                                        sName      = ARGUMENTNAME_READONLY ;
									}
									break;
		//_________________________________________________________________________________________________________
		case E_PREVIEW			:	{
                                        ppArgument = &(m_aAnalyzed.pbPreview);
                                        sName      = ARGUMENTNAME_PREVIEW ;
									}
									break;
		//_________________________________________________________________________________________________________
		case E_SILENT			:	{
                                        ppArgument = &(m_aAnalyzed.pbSilent);
                                        sName      = ARGUMENTNAME_SILENT ;
									}
									break;
		//_________________________________________________________________________________________________________
		case E_DEEPDETECTION	:	{
                                        ppArgument = &(m_aAnalyzed.pbDeepDetection);
                                        sName      = ARGUMENTNAME_DEEPDETECTION ;
									}
									break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                 :   LOG_WARNING( "ArgumentAnalyzer::setArgument( boolean )", "Unknown argument detected!" )
        #endif
*/
	}

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=bValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::awt::Rectangle& aValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(rectangle)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
		case E_POSSIZE	:	{
                                ppArgument = &(m_aAnalyzed.paPosSize);
                                sName      = ARGUMENTNAME_POSSIZE ;
							}
							break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default         :   LOG_WARNING( "ArgumentAnalyzer::setArgument( com.sun.star.awt.Rectangle )", "Unknown argument detected!" )
        #endif
*/
	}

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=aValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::uno::Reference< css::io::XInputStream >& xValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(inputstream)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
		case E_INPUTSTREAM	:	{
                                    ppArgument = &(m_aAnalyzed.pxInputStream);
                                    sName      = ARGUMENTNAME_INPUTSTREAM ;
								}
								break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default             :   LOG_WARNING( "ArgumentAnalyzer::setArgument( com.sun.star.io.XInputStream )", "Unknown argument detected!" )
        #endif
*/
	}

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=xValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::uno::Reference< css::io::XOutputStream >& xValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(outputstream)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_OUTPUTSTREAM  :   {
                                     ppArgument = &(m_aAnalyzed.pxOutputStream);
                                     sName      = ARGUMENTNAME_OUTPUTSTREAM ;
                                 }
                                 break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default             :   LOG_WARNING( "ArgumentAnalyzer::setArgument( com.sun.star.io.XOutputStream )", "Unknown argument detected!" )
        #endif
*/
	}

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=xValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::uno::Reference< css::task::XStatusIndicator >& xValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(statusindicator)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_STATUSINDICATOR  :   {
                                        ppArgument = &(m_aAnalyzed.pxStatusIndicator);
                                        sName      = ARGUMENTNAME_STATUSINDICATOR    ;
                                    }
                                    break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                 :   LOG_WARNING( "ArgumentAnalyzer::setArgument( com.sun.star.task.XStatusIndicator )", "Unknown argument detected!" )
        #endif
*/
	}

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=xValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::uno::Reference< css::task::XInteractionHandler >& xValue )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(interactionhandler)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32*      ppArgument = NULL;
    ::rtl::OUString sName            ;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_INTERACTIONHANDLER:  {
                                        ppArgument = &(m_aAnalyzed.pxInteractionHandler);
                                        sName      = ARGUMENTNAME_INTERACTIONHANDLER    ;
                                    }
                                    break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                 :   LOG_WARNING( "ArgumentAnalyzer::setArgument( com.sun.star.task.XInteractionHandler )", "Unknown argument detected!" )
        #endif
*/
	}

    //  ppArgument points into m_aAnalyzed structure to a css::beans::PropertyValue*
    // *ppArgument points to right item in list m_pArguments from type css::beans::PropertyValue

    // Try tp pack new/changed value into an any.
    // If it failed, we have no chance to finish follow operations successfully!
    css::uno::Any aPackedValue;
    aPackedValue<<=xValue;
    impl_addOrChangeArgument( m_pArguments, ppArgument, sName, aPackedValue );
}

//*****************************************************************************************************************
//	interface
//*****************************************************************************************************************
void ArgumentAnalyzer::deleteArgument( EArgument eArgument )
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::deleteArgument()", "You try to work on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32* ppArgument     = NULL;
    sal_Int32* ppLastArgument = m_aAnalyzed.getLastReference();

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_CHARACTERSET         :   ppArgument = &(m_aAnalyzed.psCharacterSet);
										break;
		//_________________________________________________________________________________________________________
        case E_MEDIATYPE            :   ppArgument = &(m_aAnalyzed.psMediaType);
										break;
		//_________________________________________________________________________________________________________
        case E_DETECTSERVICE        :   ppArgument = &(m_aAnalyzed.psDetectService);
										break;
		//_________________________________________________________________________________________________________
        case E_EXTENSION            :   ppArgument = &(m_aAnalyzed.psExtension);
										break;
		//_________________________________________________________________________________________________________
        case E_URL                  :   ppArgument = &(m_aAnalyzed.psURL);
										break;
		//_________________________________________________________________________________________________________
        case E_FILTERNAME           :   ppArgument = &(m_aAnalyzed.psFilterName);
										break;
		//_________________________________________________________________________________________________________
        case E_FORMAT               :   ppArgument = &(m_aAnalyzed.psFormat);
										break;
		//_________________________________________________________________________________________________________
        case E_FRAMENAME            :   ppArgument = &(m_aAnalyzed.psFrameName);
										break;
		//_________________________________________________________________________________________________________
        case E_PATTERN              :   ppArgument = &(m_aAnalyzed.psPattern);
										break;
		//_________________________________________________________________________________________________________
        case E_POSTDATA             :   ppArgument = &(m_aAnalyzed.psPostData);
										break;
		//_________________________________________________________________________________________________________
        case E_POSTSTRING           :   ppArgument = &(m_aAnalyzed.psPostString);
										break;
		//_________________________________________________________________________________________________________
        case E_REFERRER             :   ppArgument = &(m_aAnalyzed.psReferrer);
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATENAME         :   ppArgument = &(m_aAnalyzed.psTemplateName);
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATEREGIONNAME   :   ppArgument = &(m_aAnalyzed.psTemplateRegionName);
										break;
		//_________________________________________________________________________________________________________
        case E_JUMPMARK             :   ppArgument = &(m_aAnalyzed.psJumpMark);
										break;
		//_________________________________________________________________________________________________________
        case E_VERSION              :   ppArgument = &(m_aAnalyzed.pnVersion);
										break;
		//_________________________________________________________________________________________________________
        case E_VIEWID               :   ppArgument = &(m_aAnalyzed.pnViewId);
										break;
		//_________________________________________________________________________________________________________
        case E_FLAGS                :   ppArgument = &(m_aAnalyzed.pnFlags);
										break;
		//_________________________________________________________________________________________________________
        case E_ASTEMPLATE           :   ppArgument = &(m_aAnalyzed.pbAsTemplate);
										break;
		//_________________________________________________________________________________________________________
        case E_HIDDEN               :   ppArgument = &(m_aAnalyzed.pbHidden);
										break;
		//_________________________________________________________________________________________________________
        case E_OPENNEWVIEW          :   ppArgument = &(m_aAnalyzed.pbOpenNewView);
										break;
		//_________________________________________________________________________________________________________
        case E_READONLY             :   ppArgument = &(m_aAnalyzed.pbReadOnly);
										break;
		//_________________________________________________________________________________________________________
        case E_PREVIEW              :   ppArgument = &(m_aAnalyzed.pbPreview);
										break;
		//_________________________________________________________________________________________________________
        case E_SILENT               :   ppArgument = &(m_aAnalyzed.pbSilent);
										break;
		//_________________________________________________________________________________________________________
        case E_DEEPDETECTION        :   ppArgument = &(m_aAnalyzed.pbDeepDetection);
										break;
		//_________________________________________________________________________________________________________
        case E_POSSIZE              :   ppArgument = &(m_aAnalyzed.paPosSize);
										break;
		//_________________________________________________________________________________________________________
        case E_INPUTSTREAM          :   ppArgument = &(m_aAnalyzed.pxInputStream);
                                        break;
		//_________________________________________________________________________________________________________
        case E_OUTPUTSTREAM         :   ppArgument = &(m_aAnalyzed.pxOutputStream);
                                        break;
		//_________________________________________________________________________________________________________
        case E_TYPENAME             :   ppArgument = &(m_aAnalyzed.psTypeName);
                                        break;
		//_________________________________________________________________________________________________________
        case E_STATUSINDICATOR      :   ppArgument = &(m_aAnalyzed.pxStatusIndicator);
                                        break;
		//_________________________________________________________________________________________________________
        case E_INTERACTIONHANDLER   :   ppArgument = &(m_aAnalyzed.pxInteractionHandler);
                                        break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                     :   LOG_WARNING( "ArgumentAnalyzer::deleteArgument()", "Unknown argument detected!" )
        #endif
*/
	}

    if(
        ( ppArgument     != NULL               )   &&
        ( ppLastArgument != NULL               )   &&
        ( *ppArgument    != ARGUMENT_NOT_EXIST )
      )
    {
        impl_deleteArgument( m_pArguments, ppArgument, ppLastArgument );
    }
}

/*-************************************************************************************************************//**
    @short      clear the whole internaly saved sequence and reset this instance of the ArgumentAnalyzer
    @descr      -
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::deleteAll()
{
    if (m_pReadOnly!=NULL)
    {
        LOG_WARNING( "ArgumentAnalyzer::deleteAll()", "You try to work on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    if (m_pArguments!=NULL)
		m_pArguments->realloc(0);
	m_aAnalyzed.forgetIt();
}

/*-************************************************************************************************************//**
    @short      check for already existing arguments in sequence
    @descr      -

    @seealso    methods getArgument()

    @param      "eArgument", specify queried argument
    @return     true, if it exist ... false otherwise

    @onerror    We return false.
*//*-*************************************************************************************************************/
sal_Bool ArgumentAnalyzer::existArgument( EArgument eArgument ) const
{
    sal_Bool bExist = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_CHARACTERSET         :   bExist = ( m_aAnalyzed.psCharacterSet != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_MEDIATYPE            :   bExist = ( m_aAnalyzed.psMediaType != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_DETECTSERVICE        :   bExist = ( m_aAnalyzed.psDetectService != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_EXTENSION            :   bExist = ( m_aAnalyzed.psExtension != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_URL                  :   bExist = ( m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_FILTERNAME           :   bExist = ( m_aAnalyzed.psFilterName != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_FORMAT               :   bExist = ( m_aAnalyzed.psFormat != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_FRAMENAME            :   bExist = ( m_aAnalyzed.psFrameName != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_PATTERN              :   bExist = ( m_aAnalyzed.psPattern != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_POSTDATA             :   bExist = ( m_aAnalyzed.psPostData != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_POSTSTRING           :   bExist = ( m_aAnalyzed.psPostString != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_REFERRER             :   bExist = ( m_aAnalyzed.psReferrer != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATENAME         :   bExist = ( m_aAnalyzed.psTemplateName != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATEREGIONNAME   :   bExist = ( m_aAnalyzed.psTemplateRegionName != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_JUMPMARK             :   bExist = ( m_aAnalyzed.psJumpMark != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_VERSION              :   bExist = ( m_aAnalyzed.pnVersion != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_VIEWID               :   bExist = ( m_aAnalyzed.pnViewId != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_FLAGS                :   bExist = ( m_aAnalyzed.pnFlags != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_ASTEMPLATE           :   bExist = ( m_aAnalyzed.pbAsTemplate != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_HIDDEN               :   bExist = ( m_aAnalyzed.pbHidden != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_OPENNEWVIEW          :   bExist = ( m_aAnalyzed.pbOpenNewView != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_READONLY             :   bExist = ( m_aAnalyzed.pbReadOnly != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_PREVIEW              :   bExist = ( m_aAnalyzed.pbPreview != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_SILENT               :   bExist = ( m_aAnalyzed.pbSilent != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_DEEPDETECTION        :   bExist = ( m_aAnalyzed.pbDeepDetection != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_POSSIZE              :   bExist = ( m_aAnalyzed.paPosSize != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_INPUTSTREAM          :   bExist = ( m_aAnalyzed.pxInputStream != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_OUTPUTSTREAM         :   bExist = ( m_aAnalyzed.pxOutputStream != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_TYPENAME             :   bExist = ( m_aAnalyzed.psTypeName != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_STATUSINDICATOR      :   bExist = ( m_aAnalyzed.pxStatusIndicator != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_INTERACTIONHANDLER   :   bExist = ( m_aAnalyzed.pxInteractionHandler != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
/*
        #ifdef ENABLE_WARNINGS
        default                     :   LOG_WARNING( "ArgumentAnalyzer::existArgument()", "Unknown argument detected!" )
        #endif
*/
	}

    return bExist;
}

/*-************************************************************************************************************//**
    @short      add/delete a property or change a already existing one
    @descr      If a user set or delete a value on these class we must add/delete these property to/from our internal
                sequences(!) - because if user get these lists; all setted properties must exist with right values!

    @seealso    methods set...()
    @seealso    methods delete...()

    @param      -
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::impl_addOrChangeArgument(       css::uno::Sequence< css::beans::PropertyValue >* pList       ,
                                                       sal_Int32*                                       pPosition   ,
                                                 const ::rtl::OUString&                                 sName       ,
                                                 const css::uno::Any&                                   aValue      )
{
    // Element doesnt already exist.
    // Resize sequence and add new item at the end of it.
    // Don't forget to set *ppArgument! m_aAnalyzed.pxxx must point to right item in m_pArguments!
    if( *pPosition == ARGUMENT_NOT_EXIST )
    {
        *pPosition = pList->getLength();
        pList->realloc( (*pPosition)+1 );
    }

    // It doesn't matter, if element should be changed or added here!
    // *ppArguments must point to right place in m_pArguments always ...
    // and name and value are well known. So we can set it here directly for
    // both cases.
    (*pList)[*pPosition].Name  = sName ;
    (*pList)[*pPosition].Value = aValue;
}

//*****************************************************************************************************************
void ArgumentAnalyzer::impl_deleteArgument( css::uno::Sequence< css::beans::PropertyValue >* pList     ,
                                            sal_Int32*                                       pPosition ,
                                            sal_Int32*                                       pLast     )
{
	// Algorithm:
    //  a)  copy last item of list to position of item, which should be deleted
    //      If last item should be deleted, do nothing! It will be automticly done by "realloc()" call of b) !
    //      Otherwise dont forget to actualize index of copied last item ...
    //  b)  reallocate list of arguments. Last Item will be destroyed then ...
    //  c)  forget pointer to deleted argument!!!

    // a)
    if( *pPosition != *pLast )
    {
        (*pList)[*pPosition] = (*pList)[*pLast];
        *pLast               = *pPosition;
    }

    // b)
    pList->realloc( pList->getLength()-1 );

    // c)
    *pPosition = ARGUMENT_NOT_EXIST;
}

//___________________________________________________
/**
    write the whole content of the current MediaDescriptor to a log file
    Can be used to get an overwview about the current state of such descriptor
    before and after an operation using this ArgumentAnalyzer.
 */
#ifdef ENABLE_COMPONENT_SELF_CHECK

    // by the way - define a log file for following debug operations
    #define ARGUMENTANALYZER_LOGFILE "dump_mediadescriptor.log"

void ArgumentAnalyzer::impldbg_dumpArguments(const char* pMessage, ::rtl::OUStringBuffer* pBuffer)
{
    pBuffer->appendAscii("\"");
    pBuffer->appendAscii(pMessage);
    pBuffer->appendAscii("\"\n\n");

    if (m_pArguments==NULL || m_pArguments->getLength()<1)
        pBuffer->appendAscii("is empty\n");
    else
    {
        sal_Int32 nCount = m_pArguments->getLength();
        for (sal_Int32 nItem = 0; nItem < nCount ; ++nItem )
        {
            pBuffer->appendAscii("\"");
            pBuffer->append     ((*m_pArguments)[nItem].Name);
            pBuffer->appendAscii("\"\t=\t");

            if ((*m_pArguments)[nItem].Value.getValueTypeClass()==css::uno::TypeClass_STRING)
            {
                ::rtl::OUString sValue;
                (*m_pArguments)[nItem].Value >>= sValue;
                pBuffer->appendAscii("\"");
                pBuffer->append     (sValue);
                pBuffer->appendAscii("\"");
            }
            else
            if ((*m_pArguments)[nItem].Value.getValueTypeClass()==css::uno::TypeClass_SHORT)
            {
                sal_Int16 nValue;
                (*m_pArguments)[nItem].Value >>= nValue;
                pBuffer->append((sal_Int32)nValue);
            }
            else
            if ((*m_pArguments)[nItem].Value.getValueTypeClass()==css::uno::TypeClass_LONG)
            {
                sal_Int32 nValue;
                (*m_pArguments)[nItem].Value >>= nValue;
                pBuffer->append((sal_Int32)nValue);
            }
            else
            if ((*m_pArguments)[nItem].Value.getValueTypeClass()==css::uno::TypeClass_BOOLEAN)
            {
                sal_Bool bValue;
                (*m_pArguments)[nItem].Value >>= bValue;
                if (bValue)
                    pBuffer->appendAscii("true");
                else
                    pBuffer->appendAscii("false");
            }
            else
            if ((*m_pArguments)[nItem].Value.getValueTypeClass()==css::uno::TypeClass_INTERFACE)
            {
                css::uno::Reference< css::uno::XInterface > xValue;
                (*m_pArguments)[nItem].Value >>= xValue;
                if (xValue.is())
                    pBuffer->appendAscii("valid");
                else
                    pBuffer->appendAscii("null");
            }

            pBuffer->appendAscii("\n");
        }
    }
}
#endif // ENABLE_COMPONENT_SELF_CHECK

//___________________________________________________
/**
    helper method for special debug mode.
    It searches for an item inside the given sequence and
    returns his value. Of course - we must implement it twice.
    We can't use original function of the normal ArgumentAnalyzer!
    We will check her functionality ...
 */
#ifdef ENABLE_COMPONENT_SELF_CHECK
sal_Bool ArgumentAnalyzer::impldbg_getArgument(const char*                                        pName      ,
                                                     ::rtl::OUString*                             pString    ,
                                                     sal_Int16*                                   pShort     ,
                                                     sal_Int32*                                   pLong      ,
                                                     sal_Bool*                                    pBool      ,
                                                     css::uno::Reference< css::uno::XInterface >* pInterface )
{
    if (m_pArguments==NULL || m_pArguments->getLength()<1)
        return sal_False;

    ::rtl::OUString sSearched = ::rtl::OUString::createFromAscii(pName);
    css::uno::Any   aValue    ;
    sal_Bool        bState    = sal_False;

    sal_Int32 nCount = m_pArguments->getLength();
    for (sal_Int32 i=0; i<nCount; ++i )
    {
        if ((*m_pArguments)[i].Name == sSearched)
        {
            aValue = (*m_pArguments)[i].Value;
            bState = sal_True;
            break;
        }
    }

    if (aValue.getValueTypeClass()==css::uno::TypeClass_STRING)
        aValue >>= *pString;
    else
    if (aValue.getValueTypeClass()==css::uno::TypeClass_SHORT)
        aValue >>= *pShort;
    else
    if (aValue.getValueTypeClass()==css::uno::TypeClass_LONG)
        aValue >>= *pLong;
    else
    if (aValue.getValueTypeClass()==css::uno::TypeClass_BOOLEAN)
        aValue >>= *pBool;
    else
    if (aValue.getValueTypeClass()==css::uno::TypeClass_INTERFACE)
        aValue >>= *pInterface;

    return bState;
}
#endif // ENABLE_COMPONENT_SELF_CHECK

//___________________________________________________
/**
    implements a test scenario for this class ArgumentAnalyzer
    It's a static function, which is builded for debug only.
    It create a MediaDescriptor and work with an instance of an ArgumentAnalyzer
    on top of it.
 */
#ifdef ENABLE_COMPONENT_SELF_CHECK
void ArgumentAnalyzer::impldbg_checkIt()
{
    ::rtl::OUStringBuffer sOut(10000);

    css::uno::Sequence< css::beans::PropertyValue > lDescriptor;
    ArgumentAnalyzer aAnalyzer(lDescriptor);

    ::rtl::OUString sOrgURLMain    = DECLARE_ASCII("http://so-berlin/test/test.sxw");
    ::rtl::OUString sOrgURLJump    = DECLARE_ASCII("jump1");
    ::rtl::OUString sOrgURLArg     = DECLARE_ASCII("arg1=1");
    ::rtl::OUString sOrgURL        = sOrgURLMain + DECLARE_ASCII("?") + sOrgURLArg + DECLARE_ASCII("#") + sOrgURLJump;
    ::rtl::OUString sOrgMediaType  = DECLARE_ASCII("text/plain");
    ::rtl::OUString sOrgContentType= DECLARE_ASCII("bravo/superhits");
    ::rtl::OUString sOrgFileName   = DECLARE_ASCII("c:\\test\\irgendwas.txt");
    ::rtl::OUString sOrgURLJump2   = DECLARE_ASCII("jump2");
    ::rtl::OUString sOrgURL2       = sOrgURLMain + DECLARE_ASCII("?") + sOrgURLArg + DECLARE_ASCII("#") + sOrgURLJump2;
	::rtl::OUString sOrgURLPrivate = DECLARE_ASCII("private:factory/swriter");

    ::rtl::OUString sCheckURL         ;
    ::rtl::OUString sCheckURLJump     ;
    ::rtl::OUString sCheckMediaType   ;
    ::rtl::OUString sCheckContentType ;
    ::rtl::OUString sCheckFileName    ;
    ::rtl::OUString sCheckURLPrivate  ;

    //------------------------------------------------------------
    sOut.appendAscii("set \"URL\" on empty descriptor ... ");
    aAnalyzer.setArgument(E_URL, sOrgURL );

    if (
        ! aAnalyzer.impldbg_getArgument("FileName",&sCheckURL,NULL,NULL,NULL,NULL    )  &&
          aAnalyzer.impldbg_getArgument("URL"     ,&sCheckURL,NULL,NULL,NULL,NULL    )  &&
          aAnalyzer.impldbg_getArgument("JumpMark",&sCheckURLJump,NULL,NULL,NULL,NULL)  &&
          sCheckURL     == sOrgURL                                                      &&
          sCheckURLJump == sOrgURLJump
       )
    {
        sOut.appendAscii("OK\n");
    }
    else
    {
        sOut.appendAscii("failed\n");
        aAnalyzer.impldbg_dumpArguments("Please check manually, why URL/JumpMark are wrong or FileName exist? It shouldn't.", &sOut);
    }

    //------------------------------------------------------------
    sOut.appendAscii("set \"ContentType\" in the descriptor but outside the analyzer ... ");
    lDescriptor.realloc(lDescriptor.getLength()+1);
    lDescriptor[lDescriptor.getLength()-1].Name    = DECLARE_ASCII("ContentType");
    lDescriptor[lDescriptor.getLength()-1].Value <<= sOrgMediaType;
	aAnalyzer.setArguments(lDescriptor);

    if (
        ! aAnalyzer.impldbg_getArgument("ContentType",&sCheckMediaType,NULL,NULL,NULL,NULL )    &&
          aAnalyzer.impldbg_getArgument("MediaType"  ,&sCheckMediaType,NULL,NULL,NULL,NULL )    &&
          sCheckMediaType == sOrgMediaType
       )
    {
        sOut.appendAscii("OK\n");
    }
    else
    {
        sOut.appendAscii("failed\n");
        aAnalyzer.impldbg_dumpArguments("Please check manually, why ContentType instead of MediaType exist?", &sOut);
    }

    //------------------------------------------------------------
    sOut.appendAscii("set \"FileName\" in the descriptor but outside the analyzer ... ");
    lDescriptor.realloc(lDescriptor.getLength()+1);
    lDescriptor[lDescriptor.getLength()-1].Name    = DECLARE_ASCII("FileName");
    lDescriptor[lDescriptor.getLength()-1].Value <<= sOrgFileName;
	aAnalyzer.setArguments(lDescriptor);

    if (
        ! aAnalyzer.impldbg_getArgument("FileName",&sCheckFileName,NULL,NULL,NULL,NULL )  &&
          aAnalyzer.impldbg_getArgument("URL"     ,&sCheckURL,NULL,NULL,NULL,NULL      )  &&
          sOrgURL == sCheckURL
       )
    {
        sOut.appendAscii("OK\n");
    }
    else
    {
        sOut.appendAscii("failed\n");
        aAnalyzer.impldbg_dumpArguments("Please check manually, why FileName and URL are different? They shouldn't.", &sOut);
    }

    //------------------------------------------------------------
    sOut.appendAscii("set \"JumpMark\" using the analyzer ... ");
    aAnalyzer.setArgument(E_JUMPMARK,sOrgURLJump2);

    if (
        aAnalyzer.impldbg_getArgument("URL"     ,&sCheckURL,NULL,NULL,NULL,NULL      )  &&
        aAnalyzer.impldbg_getArgument("JumpMark",&sCheckURLJump,NULL,NULL,NULL,NULL  )  &&
        sCheckURL     == sOrgURL2                                                       &&
        sCheckURLJump == sOrgURLJump2
       )
    {
        sOut.appendAscii("OK\n");
    }
    else
    {
        sOut.appendAscii("failed\n");
        aAnalyzer.impldbg_dumpArguments("Please check manually, why the new jumpmark wasn't realy used.", &sOut);
    }

    //------------------------------------------------------------
    sOut.appendAscii("set private URL ... ");
	aAnalyzer.deleteArgument(E_URL     );
	aAnalyzer.deleteArgument(E_JUMPMARK);
    aAnalyzer.setArgument(E_URL,sOrgURLPrivate);

    if (
          aAnalyzer.impldbg_getArgument("URL"     ,&sCheckURLPrivate,NULL,NULL,NULL,NULL )  &&
		! aAnalyzer.impldbg_getArgument("JumpMark",&sCheckURLJump,NULL,NULL,NULL,NULL    )  &&
          sCheckURLPrivate == sOrgURLPrivate
       )
    {
        sOut.appendAscii("OK\n");
    }
    else
    {
        sOut.appendAscii("failed\n");
        aAnalyzer.impldbg_dumpArguments("Please check manually, why private URLs are handled wrong.", &sOut);
    }

    //------------------------------------------------------------
    sOut.appendAscii("check handling of two superflous items \"FileName\" and \"ContentType\" ... ");
	aAnalyzer.deleteAll();
    lDescriptor.realloc(4);
    lDescriptor[0].Name    = DECLARE_ASCII("URL");
    lDescriptor[0].Value <<= sOrgURL;
    lDescriptor[1].Name    = DECLARE_ASCII("MediaType");
    lDescriptor[1].Value <<= sOrgMediaType;
    lDescriptor[2].Name    = DECLARE_ASCII("FileName");
    lDescriptor[2].Value <<= sOrgFileName;
    lDescriptor[3].Name    = DECLARE_ASCII("ContentType");
    lDescriptor[3].Value <<= sOrgContentType;
    aAnalyzer.setArguments(lDescriptor);

    if (
          aAnalyzer.impldbg_getArgument("URL"        ,&sCheckURL        ,NULL,NULL,NULL,NULL )  &&
		  aAnalyzer.impldbg_getArgument("JumpMark"   ,&sCheckURLJump    ,NULL,NULL,NULL,NULL )  &&
          aAnalyzer.impldbg_getArgument("MediaType"  ,&sCheckMediaType  ,NULL,NULL,NULL,NULL )  &&
        ! aAnalyzer.impldbg_getArgument("FileName"   ,&sCheckFileName   ,NULL,NULL,NULL,NULL )  &&
        ! aAnalyzer.impldbg_getArgument("ContentType",&sCheckContentType,NULL,NULL,NULL,NULL )  &&
          sCheckURL       == sOrgURL															&&
          sCheckURLJump   == sOrgURLJump														&&
          sCheckMediaType == sOrgMediaType
       )
    {
        sOut.appendAscii("OK\n");
    }
    else
    {
        sOut.appendAscii("failed\n");
        aAnalyzer.impldbg_dumpArguments("Please check manually, why there still exist \"FileName\" or \"ContentType\".", &sOut);
    }

    WRITE_LOGFILE(ARGUMENTANALYZER_LOGFILE, U2B(sOut.makeStringAndClear()) )

	exit(-1);
}
#endif // ENABLE_COMPONENT_SELF_CHECK

}		//	namespace framework
