/*************************************************************************
 *
 *  $RCSfile: inetdns.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: th $ $Date: 2001/05/11 11:52:51 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INETDNS_CXX "$Revision: 1.5 $"

#ifndef __STL_USE_NEWALLOC
#define __STL_USE_NEWALLOC 1
#endif
#ifndef __UTILITY__
#include <utility>
#endif
#ifndef __HASH_MAP__
#include <hash_map>
#endif
#ifndef __MAP__
#include <map>
#endif

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif
#ifndef _OSL_INTERLCK_H_
#include <osl/interlck.h>
#endif
#ifndef _OSL_SOCKET_H_
#include <osl/socket.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif
#ifndef _OSL_ENDIAN_H_
#include <osl/endian.h>
#endif
#ifndef _VOS_MUTEX_HXX_
#include <vos/mutex.hxx>
#endif
#ifndef _VOS_OBJECT_HXX_
#include <vos/object.hxx>
#endif
#ifndef _VOS_REFERNCE_HXX_
#include <vos/refernce.hxx>
#endif
#ifndef _VOS_THREAD_HXX_
#include <vos/thread.hxx>
#endif
#ifndef _VOS_TIMER_HXX_
#include <vos/timer.hxx>
#endif

#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif

#ifndef _INET_MACROS_HXX
#include <inet/macros.hxx>
#endif
#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif

#ifndef _INET_CONFIG_HXX
#include <inetcfg.hxx>
#endif
#ifndef _INETDNS_HXX
#include <inetdns.hxx>
#endif

#include <time.h>

using namespace inet;
using rtl::OUString;

/*=====================================================================
 *
 * Debug trace environment.
 *
 *===================================================================*/
#ifdef DBG_UTIL_PRIVATE
#include <stdio.h>

#if (defined(OS2) || defined(WIN) || defined(WNT))
#define _TRACELOGNAME "c:\\inetdns.log"
#elif (defined(UNX))
#define _TRACELOGNAME "/tmp/inetdns.log"
#else
#define _TRACELOGNAME "inetdns.log"
#endif

#ifdef WIN
#define _TRACELOGFORMAT "%ls\n"
#else
#define _TRACELOGFORMAT "%s\n"
#endif

#define _TRACE(a) \
{ \
    FILE *fp = fopen (_TRACELOGNAME, "a+"); \
    if (fp) \
    { \
        fprintf (fp, _TRACELOGFORMAT, (const char *)((a)) ); \
        fclose (fp); \
    } \
}

#else
#define _TRACE(a)
#endif /* DBG_UTIL_PRIVATE */

/*=====================================================================
 *
 * INetCoreDNSResolver internals.
 *
 *===================================================================*/
typedef NAMESPACE_INET(INetSocket) socket_type;

#if (!defined(OSL_BIGENDIAN) && !defined(OSL_LITENDIAN))
#error  Unknown Byte Order
#endif

#define INETCOREDNS_NAMESERVER_PORT    53     /* UDP */
#define INETCOREDNS_SEND_MSGSIZE      512     /* UDP */
#define INETCOREDNS_RECV_MSGSIZE     2048     /* DNS 16K */

#define INETCOREDNS_TYPE_A              1     /* Address */
#define INETCOREDNS_TYPE_NS             2     /* Name server */
#define INETCOREDNS_TYPE_CNAME          5     /* Canonical name */
#define INETCOREDNS_TYPE_PTR           12     /* Pointer */
#define INETCOREDNS_TYPE_MX            15     /* Mail exchanger */

#define INETCOREDNS_CLASS_IN            1     /* AF_INET */

#define INETCOREDNS_INADDR_ANY         0x00000000 /* INADDR_ANY  */
#define INETCOREDNS_INADDR_NONE        0xffffffff /* INADDR_NONE */

#define INETCOREDNS_CONFIG_SOCKET      0x0001
#define INETCOREDNS_CONFIG_SERVER      0x0002
#define INETCOREDNS_CONFIG_LOCALHOST   0x0004
#define INETCOREDNS_CONFIG_LOCALDOMAIN 0x0008
#define INETCOREDNS_CONFIG_HOSTADDR    0x0010

#define INETCOREDNS_CONFIG_BASE \
    (INETCOREDNS_CONFIG_SOCKET  | INETCOREDNS_CONFIG_SERVER)
#define INETCOREDNS_CONFIG_INITIAL \
    (INETCOREDNS_CONFIG_SOCKET  | INETCOREDNS_CONFIG_LOCALHOST)
#define INETCOREDNS_CONFIG_QUERY \
    (INETCOREDNS_CONFIG_BASE    | INETCOREDNS_CONFIG_LOCALDOMAIN)
#define INETCOREDNS_CONFIG_FULL \
    (INETCOREDNS_CONFIG_INITIAL | INETCOREDNS_CONFIG_LOCALDOMAIN)

#define INETCOREDNS_SOCKET_MSGSIZE    (-osl_Socket_E_MsgSize)
#define INETCOREDNS_SOCKET_WOULDBLOCK (-osl_Socket_E_WouldBlock)

#define INETCOREDNS_CACHE_TIMEOUT    172800L /*  48h */
#define INETCOREDNS_RESOLVER_TIMEOUT 120000L /* 120s */

#ifndef COPYCTOR_API
#define COPYCTOR_API(C) C (const C&); C& operator= (const C&)
#endif

/*
 * dn_htons.
 */
inline sal_uInt16 dn_htons (sal_uInt16 h)
{
#ifdef OSL_LITENDIAN
    return VOS_SWAPWORD(h);
#else
    return h;
#endif
}

/*
 * dn_ntohs.
 */
inline sal_uInt16 dn_ntohs (sal_uInt16 n)
{
#ifdef OSL_LITENDIAN
    return VOS_SWAPWORD(n);
#else
    return n;
#endif
}

/*
 * dn_isNumericAscii.
 */
static sal_Bool dn_isNumericAscii (const sal_Unicode *p, sal_Int32 n)
{
    for (const sal_Unicode *q = p + n; p < q; p++)
    {
        if ((*p < 0x30) || (*p > 0x39))
            return sal_False;
    }
    return sal_True;
}

/*
 * dn_inet_addr.
 */
static sal_uInt32 dn_inet_addr (const OUString &rDottedAddr)
{
    UniString aDottedAddr (rDottedAddr);
    if (!aDottedAddr.Len())
        return INETCOREDNS_INADDR_ANY;

    xub_StrLen i, n = aDottedAddr.GetTokenCount('.');
    if (!(n == 4))
        return INETCOREDNS_INADDR_NONE;

    sal_uInt32 nAddr = 0;
    for (i = n; i > 0; i--)
    {
        UniString aToken (aDottedAddr.GetToken (i - 1, '.'));
        if (dn_isNumericAscii (aToken.GetBuffer(), aToken.Len()))
            nAddr |= ((aToken.ToInt32()) << ((n - i) * 8));
        else
            return INETCOREDNS_INADDR_NONE;
    }

#ifdef OSL_LITENDIAN
    nAddr = VOS_SWAPDWORD(nAddr);
#endif /* OSL_LITENDIAN */
    return nAddr;
}

/*
 * dn_copyString.
 */
inline void dn_copyString (sal_Char *&to, const void *from, sal_Int32 size)
{
    to = (sal_Char*)rtl_reallocateMemory (to, size);
    rtl_copyMemory (to, from, size);
}

/*====================================================================
 *
 * INetDNSHeader_Impl.
 * (Declaration taken from 4BSD nameser.h).
 *
 * ===> Bit fields in network byte order have MSB first <===
 *
 *==================================================================*/
#pragma pack(1) /* bit fields require byte packing */

struct INetDNSHeader_Impl
{
    /* 1st and 2nd byte */
    sal_uInt16 id;       /* Query identification number. */

#if (defined(OSL_BIGENDIAN))
    /* fields in 3rd byte */
    sal_uInt16 qr:1;     /* query/response flag */
    sal_uInt16 opcode:4; /* query code */
    sal_uInt16 aa:1;     /* authoritative answer */
    sal_uInt16 tc:1;     /* truncated message */
    sal_uInt16 rd:1;     /* recursion desired */

    /* fields in 4th byte */
    sal_uInt16 ra:1;     /* recursion available */
    sal_uInt16 unused:3; /* unused (MBZ) */
    sal_uInt16 rcode:4;  /* response code */
#endif /* OSL_BIGENDIAN */

#if (defined(OSL_LITENDIAN))
    /* fields in 3rd byte */
    sal_uInt16 rd:1;     /* recursion desired */
    sal_uInt16 tc:1;     /* truncated message */
    sal_uInt16 aa:1;     /* authoritative answer */
    sal_uInt16 opcode:4; /* query code */
    sal_uInt16 qr:1;     /* query/response flag */

    /* fields in 4th byte */
    sal_uInt16 rcode:4;  /* response code */
    sal_uInt16 unused:3; /* unused (MBZ) */
    sal_uInt16 ra:1;     /* recursion available */
#endif /* OSL_LITENDIAN */

    /* remaining bytes */
    sal_uInt16 qdcount;  /* number of question entries */
    sal_uInt16 ancount;  /* number of answer entries */
    sal_uInt16 nscount;  /* number of authority entries */
    sal_uInt16 arcount;  /* number of resource entries */
};

#ifdef SOLARIS
#pragma pack(4)
#else
#pragma pack() /* back to default packing */
#endif

/*====================================================================
 *
 * INetDNSResolver_Impl interface.
 *
 *==================================================================*/
#ifdef _USE_NAMESPACE
namespace inet {
#endif

/*
 * INetDNSEntry_Impl.
 */
struct INetDNSEntry_Impl
{
    sal_uInt16  m_nType;
    sal_uInt16  m_nClass;
    time_t      m_nExpires;

    OUString    m_aName;
    OUString    m_aData;

    INetDNSEntry_Impl (const OUString &rName)
        : m_aName (rName)
    {}

    ~INetDNSEntry_Impl (void)
    {}

    void setData (const OUString &rData)
    {
        m_aData = rData;
    }
};

typedef OUString dn_key;
typedef void*    dn_val;

struct dn_hash : public std::unary_function<dn_key, sal_Int32>
{
    sal_Int32 operator() (const dn_key& k) const
    {
        return k.hashCode();
    }
};

struct dn_cmp : public std::binary_function<dn_key, dn_key, sal_Bool>
{
    sal_Bool operator() (const dn_key& k1, const dn_key& k2) const
    {
        return k1.equalsIgnoreAsciiCase (k2);
    }
};

typedef std::hash_map<dn_key, dn_val, dn_hash, dn_cmp> dn_map;
typedef std::pair<const dn_key, dn_val> dn_entry;

/*
 * INetDNSCache_Impl.
 */
class INetDNSCache_Impl : public dn_map
{
public:
    INetDNSCache_Impl (void);
    virtual ~INetDNSCache_Impl (void);

    void insertHostAddr (
        const OUString    &dn,
        const oslHostAddr  hAddr);
    void insertResponse (
        const OUString  &dn,
        const sal_uInt8 *pBuffer);

    sal_Int32 lookup (
        const OUString       &dn,
        INetCoreDNSHostEntry *he);

private:
    /** Disassemble a Resource Record into a cache entry.
     */
    const sal_uInt8* create (
        INetDNSEntry_Impl *&rpEntry,
        const sal_uInt8    *pBuffer,
        const sal_uInt8    *pOffset);

    /** Disassemble sequence of labels/pointers into domain name.
     */
    const sal_uInt8* expand (
        const sal_uInt8 *pBuffer,
        const sal_uInt8 *pOffset,
        void            *pData,
        sal_uInt16       nSize);

    /** Not implemented.
     */
    COPYCTOR_API(INetDNSCache_Impl);
};

/*
 * INetDNSRequest_Impl.
 */
class INetDNSRequest_Impl : public NAMESPACE_VOS(OTimer)
{
    VOS_DECLARE_CLASSINFO (VOS_NAMESPACE (INetDNSRequest_Impl, inet));

public:
    INetCoreDNSHostEntry *m_pHostent;
    INetCoreDNSCallback  *m_pfnCB;
    void                 *m_pDataCB;

    INetDNSRequest_Impl (sal_uInt16 nID = 0);

    /** RequestCallback.
     */
    typedef sal_Bool callback_type (
        sal_uInt16         nID,
        const oslHostAddr  hAddr,
        void              *pData);

    /** Start timer with given timeout [ms].
     */
    void start (
        callback_type *pfnCB,
        void          *pDataCB,
        sal_uInt32     nTimeout = INETCOREDNS_RESOLVER_TIMEOUT);

    /** Assemble a Query Message into buffer.
     */
    sal_Bool generateQuery (
        sal_uInt16      nID,
        sal_uInt16      nQueryType,
        sal_Bool        bRecursive,
        const OUString &rDomainName,
        sal_uInt8      *pBuffer,
        sal_uInt16      nBufSiz,
        sal_uInt16     &rBufLen);

    /** Execute TYPE_A query.
     */
    sal_Bool getHostByName (
        const OUString &rDomainName);

protected:
    /** Destruction.
     */
    virtual ~INetDNSRequest_Impl (void);

private:
    /** Executable.
     */
    class executor : public NAMESPACE_VOS(OThread)
    {
        typedef NAMESPACE_VOS(IReference) reference_type;

    public:
        executor (reference_type *pParent, sal_uInt16 nID);
        virtual ~executor (void);

        void done (const oslHostAddr hAddr)
        {
            if (m_pfnCB)
                m_pfnCB (m_nID, hAddr, m_pDataCB);
        }

        void setCallback (callback_type *pfnCB, void *pDataCB)
        {
            m_pfnCB   = pfnCB;
            m_pDataCB = pDataCB;
        }

        sal_Bool getHostByName (const OUString &rName);

        virtual void SAL_CALL terminate (void);

    private:
        /** Representation.
         */
        reference_type *m_pParent;
        sal_uInt16      m_nID;
        OUString        m_aName;
        callback_type  *m_pfnCB;
        void           *m_pDataCB;

        /** OThread.
         */
        virtual void SAL_CALL run (void);
        virtual void SAL_CALL onTerminated (void);

        /** Not implemented.
         */
        COPYCTOR_API(executor);
    };

    /** Representation.
     */
    executor m_aExecutor;

    /** OTimer.
     */
    virtual void SAL_CALL onShot (void);

    /** Not implemented.
     */
    COPYCTOR_API(INetDNSRequest_Impl);
};

typedef NAMESPACE_INET(INetDNSRequest_Impl)     Request;
typedef std::less<sal_uInt16>                   RequestCmp;
typedef std::map<sal_uInt16, void*, RequestCmp> RequestMap;

/*
 * INetDNSResolver_Impl.
 */
class INetDNSResolver_Impl :
    public NAMESPACE_INET(INetDNSCache_Impl),
    public NAMESPACE_VOS(OReference),
    public NAMESPACE_VOS(OObject)
{
    VOS_DECLARE_CLASSINFO (VOS_NAMESPACE (INetDNSResolver_Impl, inet));

public:
    /** getOrCreate the single instance.
     */
    static INetDNSResolver_Impl* getOrCreate (void);

    /** configure (incremental initialization).
     */
    sal_Int32 configure (sal_uInt32 nOptions);
    sal_Bool  hasOption (sal_uInt32 nOptions)
    {
        return ((m_nOptions & nOptions) == nOptions);
    }

    /** getLocalHostname.
     */
    sal_Bool getLocalHostname (
        INetCoreDNSHostEntry *pHostEntry);

    /** getHostByName.
     */
    sal_Bool getHostByName (
        INetCoreDNSHostEntry *pHostEntry,
        INetCoreDNSCallback  *pfnCallback,
        void                 *pData);

    /** getHostByAddr.
     */
    sal_Bool getHostByAddr (
        INetCoreDNSHostEntry *pHostEntry,
        INetCoreDNSCallback  *pfnCallback,
        void                 *pData);

protected:
    INetDNSResolver_Impl (void);
    virtual ~INetDNSResolver_Impl (void);

private:
    /** The single instance.
     */
    static INetDNSResolver_Impl        *m_pThis;

    /** Representation.
     */
    NAMESPACE_VOS(OMutex)               m_aMutex;
    NAMESPACE_VOS(ORef)<INetUDPSocket>  m_xSocket;

    NAMESPACE_VOS(OInetSocketAddr)      m_aToAddr;
    INetCoreDNSHostEntry                m_aHostAddr;

    OUString                            m_aNameServer;
    OUString                            m_aLocalHost;
    OUString                            m_aLocalDomain;
    sal_uInt32                          m_nOptions;

    RequestMap                          m_aRequestMap;
    oslInterlockedCount                 m_nRequest;

    /** Config(Callback|Handler).
     */
    static sal_Bool ConfigCallback (
        sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry, void *pData);
    void ConfigHandler (
        sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry);

    /** setNameServer.
     */
    void setNameServer (
        const OUString &rDottedName);

    /** setLocalHostname.
     */
    void setLocalHostname (
        const OUString &rDomainName);

    /** ExecuteCallback.
     */
    static sal_Bool ExecuteCallback (
        sal_uInt16         nRequestID,
        const oslHostAddr  hAddr,
        void              *pData);

    /** Request(Callback|Handler).
     */
    static sal_Bool RequestCallback (
        const NAMESPACE_VOS(ORef)<INetSocket> &rxSocket,
        sal_Int32         nEvent,
        void             *pData);
    void RequestHandler (
        const NAMESPACE_VOS(ORef)<INetSocket> &rxSocket,
        sal_Int32         nEvent);

    sal_Bool enqueue (
        sal_uInt16            nRequestID,
        INetDNSRequest_Impl *&rpRequest);
    sal_Bool dequeue (
        sal_uInt16            nRequestID,
        INetDNSRequest_Impl *&rpRequest);

    sal_Int32 query (
        sal_uInt16            nQueryType,
        const OUString       &rDomainName,
        INetCoreDNSHostEntry *pHostEntry,
        INetCoreDNSCallback  *pfnCallback,
        void                 *pData);

    void onResponse (
        sal_uInt16         nRequestID,
        const oslHostAddr  hAddr     = NULL,
        const sal_uInt8   *pResponse = NULL);

    /** Not implemented.
     */
    COPYCTOR_API(INetDNSResolver_Impl);
};

#ifdef _USE_NAMESPACE
}
#endif

/*
 * __getGlobalMutex_Impl (protect the single instance creation).
 */
static NAMESPACE_VOS(IMutex)& __getGlobalMutex_Impl (void)
{
    static NAMESPACE_VOS(IMutex) *pMutex = NULL;
    if (!pMutex)
    {
        NAMESPACE_VOS(OGuard) aGuard (NAMESPACE_VOS(OMutex)::getGlobalMutex());
        if (!pMutex)
        {
            static NAMESPACE_VOS(OMutex) aGlobalMutex;
            pMutex = &aGlobalMutex;
        }
    }
    return *pMutex;
}

/*====================================================================
 *
 * INetDNSResolver_Impl implementation.
 *
 *==================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetDNSResolver_Impl, inet),
    VOS_NAMESPACE (INetDNSResolver_Impl, inet),
    VOS_NAMESPACE (OObject, vos),
    0);

/*
 * The single instance.
 */
NAMESPACE_INET(INetDNSResolver_Impl)*
NAMESPACE_INET(INetDNSResolver_Impl)::m_pThis = NULL;

/*
 * INetDNSResolver_Impl.
 */
INetDNSResolver_Impl::INetDNSResolver_Impl (void)
    : m_xSocket      (NULL),
      m_aHostAddr    (OUString(), 0),
      m_nOptions     (0),
      m_nRequest     (0)
{
    // Assign single instance.
    m_pThis = this;
}

/*
 * ~INetDNSResolver_Impl.
 */
INetDNSResolver_Impl::~INetDNSResolver_Impl (void)
{
    NAMESPACE_VOS(OClearableGuard) aGuard (__getGlobalMutex_Impl());
    m_pThis = NULL;
    aGuard.clear();

    // Check request queue.
    if (!m_aRequestMap.empty())
    {
        // Cleanup request queue.
        RequestMap::const_iterator first = m_aRequestMap.begin();
        RequestMap::const_iterator last  = m_aRequestMap.end();

        for (RequestMap::const_iterator it = first; it != last; ++it)
        {
            Request *pRequest = (Request*)((*it).second);
            if (pRequest)
            {
                pRequest->stop();
                pRequest->release();
            }
        }
        m_aRequestMap.clear();
    }

    // Cleanup request socket.
    m_xSocket.unbind();
}

/*
 * getOrCreate.
 */
INetDNSResolver_Impl* INetDNSResolver_Impl::getOrCreate (void)
{
    NAMESPACE_VOS(OGuard) aGuard (__getGlobalMutex_Impl());
    if (!m_pThis)
        new INetDNSResolver_Impl();
    return m_pThis;
}

/*
 * configure (incremental initialization).
 */
sal_Int32 INetDNSResolver_Impl::configure (sal_uInt32 nOptions)
{
    // Acquire exclusive access.
    NAMESPACE_VOS(OGuard) aGuard (m_aMutex);

    if (!hasOption (INETCOREDNS_CONFIG_SOCKET))
    {
        // Create socket and reset configuration.
        m_xSocket  = new INetUDPSocket();
        m_nOptions = INETCOREDNS_CONFIG_SOCKET;
    }

    if (nOptions & INETCOREDNS_CONFIG_SERVER)
    {
        NAMESPACE_VOS(ORef)<INetConfig> xConfig;
        if (INetConfig::getOrCreate (xConfig))
        {
            // Set NameServer and SocketAddr.
            setNameServer (xConfig->getDomainNameServer());
        }
    }

    if (nOptions & INETCOREDNS_CONFIG_LOCALHOST)
    {
        m_nOptions &= ~INETCOREDNS_CONFIG_LOCALHOST;
        m_nOptions &= ~INETCOREDNS_CONFIG_LOCALDOMAIN;

        OUString aName;
        if (osl_getLocalHostname (&(aName.pData)) == osl_Socket_Ok)
        {
            // Set LocalHost and LocalDomain.
            setLocalHostname (aName);
        }
        else
        {
            // Only possible reason is osl_Socket_E_NetDown.
            oslSocketError eErrCode = osl_getLastSocketError (NULL);
            m_nOptions &= ~INETCOREDNS_CONFIG_BASE;
        }
    }

    if (nOptions & INETCOREDNS_CONFIG_LOCALDOMAIN)
    {
        if (!hasOption (INETCOREDNS_CONFIG_LOCALDOMAIN) &&
            !hasOption (INETCOREDNS_CONFIG_HOSTADDR   )    )
        {
            if (hasOption (INETCOREDNS_CONFIG_INITIAL))
            {
                // Start resolving own domain name.
                m_aHostAddr = INetCoreDNSHostEntry (m_aLocalHost);
                m_nOptions |= INETCOREDNS_CONFIG_HOSTADDR;

                sal_Int32 result = query (
                    INETCOREDNS_TYPE_A,
                    m_aLocalHost, &m_aHostAddr,
                    ConfigCallback, this);
                if (!(result == INETCOREDNS_RESOLVER_START))
                {
                    // Resolution finished. Analyze result.
                    ConfigCallback (result, &m_aHostAddr, this);
                }
            }
        }
    }

    // Done.
    if (hasOption (nOptions))
        return INETCOREDNS_RESOLVER_SUCCESS;
    else if (hasOption (INETCOREDNS_CONFIG_SOCKET))
        return INETCOREDNS_RESOLVER_START;
    else
        return INETCOREDNS_RESOLVER_ERROR;
}

/*
 * ConfigCallback.
 */
sal_Bool INetDNSResolver_Impl::ConfigCallback (
    sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry, void *pData)
{
    // Check context.
    INetDNSResolver_Impl *pThis = (INetDNSResolver_Impl*)pData;
    if (!(pThis && (pThis == m_pThis)))
        return sal_False;

    // Handle callback reason.
    m_pThis->acquire();
    m_pThis->ConfigHandler (nStatus, pHostEntry);
    m_pThis->release();

    // Done.
    return sal_True;
}

/*
 * ConfigHandler.
 */
void INetDNSResolver_Impl::ConfigHandler (
    sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry)
{
    if (!(nStatus == INETCOREDNS_RESOLVER_START))
    {
        // Acquire exclusive access.
        NAMESPACE_VOS(OGuard) aGuard (m_aMutex);

        // Check reply.
        if (nStatus == INETCOREDNS_RESOLVER_SUCCESS)
        {
            // Set (fully qualified) hostname.
            if (pHostEntry->m_aCName.getLength())
                setLocalHostname (pHostEntry->m_aCName);
            else
                setLocalHostname (pHostEntry->m_aName);
        }

        // Disable further lookups.
        m_nOptions |= INETCOREDNS_CONFIG_HOSTADDR;
    }
}

/*
 * setNameServer.
 */
void INetDNSResolver_Impl::setNameServer (const OUString &rDottedName)
{
    if (hasOption (INETCOREDNS_CONFIG_SERVER))
    {
        if (!m_aNameServer.equalsIgnoreAsciiCase (rDottedName))
            m_nOptions &= ~INETCOREDNS_CONFIG_SERVER;
        else
            return;
    }

    if (hasOption (INETCOREDNS_CONFIG_SOCKET))
    {
        m_xSocket->deregisterEventHandler (RequestCallback);
        if (rDottedName.getLength() <= 0)
            return;

        oslSocketAddr hAddr = osl_createInetSocketAddr (
            rDottedName.pData, INETCOREDNS_NAMESERVER_PORT);
        if (!hAddr)
            return;
        m_aToAddr = hAddr;

        if (m_xSocket->registerEventHandler (RequestCallback, this))
        {
            m_aNameServer = rDottedName;
            m_nOptions |= INETCOREDNS_CONFIG_SERVER;
        }
    }

#ifdef DBG_UTIL_PRIVATE
    if (hasOption (INETCOREDNS_CONFIG_SERVER))
    {
        String aTrace ("INetDNS::setNameServer(): NameServer: ");
        aTrace += m_pNameServer;
        _TRACE(aTrace);
    }
#endif /* DBG_UTIL_PRIVATE */
}

/*
 * setLocalHostname.
 */
void INetDNSResolver_Impl::setLocalHostname (const OUString &rDomainName)
{
    m_nOptions &= ~INETCOREDNS_CONFIG_LOCALHOST;
    m_nOptions &= ~INETCOREDNS_CONFIG_LOCALDOMAIN;

    sal_Int32 n = rDomainName.getLength();
    if (n > 0)
    {
        m_aLocalHost = rDomainName;
        m_nOptions |= INETCOREDNS_CONFIG_LOCALHOST;

        sal_Int32 k = m_aLocalHost.indexOf ('.');
        if (k > 0)
        {
            m_aLocalDomain = m_aLocalHost.copy (k, n - k);
            m_nOptions |= INETCOREDNS_CONFIG_LOCALDOMAIN;
        }
    }

#ifdef DBG_UTIL_PRIVATE
    if (hasOption (INETCOREDNS_CONFIG_LOCALHOST))
    {
        String aTrace ("INetDNS::setLocalHostname(): LocalHost: ");
        aTrace += m_pLocalHost;
        _TRACE(aTrace);
    }
    if (hasOption (INETCOREDNS_CONFIG_LOCALDOMAIN))
    {
        String aTrace ("INetDNS::setLocalHostname(): LocalDomain: ");
        aTrace += m_pLocalDomain;
        _TRACE(aTrace);
    }
#endif /* DBG_UTIL_PRIVATE */
}

/*
 * getLocalHostname.
 */
sal_Bool INetDNSResolver_Impl::getLocalHostname (INetCoreDNSHostEntry *he)
{
    // Acquire exclusive access.
    NAMESPACE_VOS(OGuard) aGuard (m_aMutex);

    // Check configuration.
    if (!hasOption (INETCOREDNS_CONFIG_INITIAL))
    {
        // Perform initial configuration.
        configure (INETCOREDNS_CONFIG_INITIAL);
    }

    // Check Hostname.
    if (hasOption (INETCOREDNS_CONFIG_LOCALHOST))
    {
        // DomainName is "localhost".
        he->m_aName = OUString::createFromAscii ("localhost");

        // CanonicalName is local hostname.
        he->m_aCName = m_aLocalHost;

        // DottedDecimalName is "127.0.0.1".
        he->m_aDotDecName = OUString::createFromAscii ("127.0.0.1");
    }

    // Done.
    return hasOption (INETCOREDNS_CONFIG_LOCALHOST);
}

/*
 * getHostByName.
 */
sal_Bool INetDNSResolver_Impl::getHostByName (
    INetCoreDNSHostEntry *pHostEntry,
    INetCoreDNSCallback  *pfnCallback,
    void                 *pData)
{
    // Check arguments.
    if (!(pHostEntry && pfnCallback))
        return sal_False;

    // Start resolution.
    int result = INETCOREDNS_RESOLVER_START;
    (pfnCallback) (result, pHostEntry, pData);

    // Check for DottedDecimal name.
    sal_uInt32 nAddr = dn_inet_addr (pHostEntry->m_aName);
    if (nAddr == INETCOREDNS_INADDR_NONE)
    {
        // Acquire exclusive access.
        NAMESPACE_VOS(OGuard) aGuard (m_aMutex);

        // Check configuration.
        if (!hasOption (INETCOREDNS_CONFIG_FULL))
        {
            // Initiate full configuration now.
            result = configure (INETCOREDNS_CONFIG_FULL);
        }
        if (result != INETCOREDNS_RESOLVER_ERROR)
        {
            // Perform cache lookup.
            result = lookup (pHostEntry->m_aName, pHostEntry);
            if (result == INETCOREDNS_RESOLVER_NOTFOUND)
            {
                // Check for fully qualified domain name.
                sal_Int32 k = pHostEntry->m_aName.indexOf ('.');
                if ((k < 0) && hasOption (INETCOREDNS_CONFIG_LOCALDOMAIN))
                {
                    // Concatenate local domain.
                    OUString aName (pHostEntry->m_aName);
                    pHostEntry->m_aName = aName.concat (m_aLocalDomain);

                    // Perform cache lookup.
                    result = lookup (pHostEntry->m_aName, pHostEntry);
                }
            }
        }
    }
    else
    {
        // No DomainName resolution necessary.
        pHostEntry->m_aDotDecName = pHostEntry->m_aName;
        result = INETCOREDNS_RESOLVER_SUCCESS;
    }

    // Check result.
    if (result == INETCOREDNS_RESOLVER_NOTFOUND)
    {
        // Perform nameserver query.
        result = query (
            INETCOREDNS_TYPE_A, pHostEntry->m_aName,
            pHostEntry, pfnCallback, pData);
    }
    if (result != INETCOREDNS_RESOLVER_START)
    {
        // Notify caller.
        (pfnCallback) (result, pHostEntry, pData);
    }
    return (result != INETCOREDNS_RESOLVER_ERROR);
}

/*
 * getHostByAddr.
 */
sal_Bool INetDNSResolver_Impl::getHostByAddr (
    INetCoreDNSHostEntry *pHostEntry,
    INetCoreDNSCallback  *pfnCallback,
    void                 *pData)
{
    // Check arguments.
    if (!(pHostEntry && pfnCallback))
        return sal_False;

    // Start resolution.
    int result = INETCOREDNS_RESOLVER_START;
    (pfnCallback) (result, pHostEntry, pData);

    // Check for DottedDecimal name.
    sal_uInt32 nAddr = dn_inet_addr (pHostEntry->m_aName);
    if (nAddr == INETCOREDNS_INADDR_NONE)
    {
        // Failure.
        result = INETCOREDNS_RESOLVER_ERROR;
    }
    else
    {
        // Acquire exclusive access.
        NAMESPACE_VOS(OGuard) aGuard (m_aMutex);

        // Check configuration.
        if (!hasOption (INETCOREDNS_CONFIG_FULL))
        {
            // Initiate full configuration now.
            result = configure (INETCOREDNS_CONFIG_FULL);
        }
        if (result != INETCOREDNS_RESOLVER_ERROR)
        {
            // Store DottedDecimal name.
            pHostEntry->m_aDotDecName = pHostEntry->m_aName;

            // Create pointer type domain name.
            rtl::OUStringBuffer aBuffer (32);

            sal_uInt8 *c = (sal_uInt8*)&nAddr;
            aBuffer.append (sal_Int32(c[3]));
            aBuffer.append (sal_Unicode('.'));
            aBuffer.append (sal_Int32(c[2]));
            aBuffer.append (sal_Unicode('.'));
            aBuffer.append (sal_Int32(c[1]));
            aBuffer.append (sal_Unicode('.'));
            aBuffer.append (sal_Int32(c[0]));

            aBuffer.appendAscii (".IN-ADDR.ARPA", 13);
            pHostEntry->m_aName = aBuffer.makeStringAndClear();

            // Perform cache lookup.
            result = lookup (pHostEntry->m_aName, pHostEntry);
        }
    }

    // Check result.
    if (result == INETCOREDNS_RESOLVER_NOTFOUND)
    {
        // Perform nameserver query.
        result = query (
            INETCOREDNS_TYPE_PTR, pHostEntry->m_aName,
            pHostEntry, pfnCallback, pData);
    }
    if (result != INETCOREDNS_RESOLVER_START)
    {
        // Notify caller.
        (pfnCallback) (result, pHostEntry, pData);
    }
    return (result != INETCOREDNS_RESOLVER_ERROR);
}

/*
 * query.
 */
sal_Int32 INetDNSResolver_Impl::query (
    sal_uInt16            nQueryType,
    const OUString       &rDomainName,
    INetCoreDNSHostEntry *pHostEntry,
    INetCoreDNSCallback  *pfnCallback,
    void                 *pData)
{
    oslInterlockedCount n = osl_incrementInterlockedCount (&m_nRequest);
    sal_uInt16 nRequestID = (sal_uInt16)(n & 0xffff);
    Request   *pRequest   = NULL;

    sal_Int32 result = INETCOREDNS_RESOLVER_ERROR;
    if (enqueue (nRequestID, pRequest))
    {
        pRequest->m_pHostent = pHostEntry;
        pRequest->m_pfnCB    = pfnCallback;
        pRequest->m_pDataCB  = pData;

        pRequest->start (ExecuteCallback, this);
        if (hasOption (INETCOREDNS_CONFIG_QUERY))
        {
            // Asynchronous lookup via name server query.
            sal_uInt8  pBuffer[INETCOREDNS_SEND_MSGSIZE];
            sal_uInt16 nBufLen = 0;

            if (pRequest->generateQuery (
                nRequestID, nQueryType, TRUE, rDomainName,
                pBuffer, sizeof (pBuffer), nBufLen))
            {
                sal_Int32 nWrite = m_xSocket->sendTo (
                    m_aToAddr, pBuffer, nBufLen);
                if (nWrite > 0)
                {
                    // Try reading any response message.
                    RequestCallback (
                        NAMESPACE_VOS(ORef)<INetSocket>(&*m_xSocket),
                        socket_type::EVENT_READ, this);
                    result = INETCOREDNS_RESOLVER_START;
                }
                else
                {
                    // Failure.
                    result = INETCOREDNS_RESOLVER_ERROR;
                }
            }
        }
        else
        {
            // Asynchronous lookup via "getXbyY" routines.
            if (nQueryType == INETCOREDNS_TYPE_A)
            {
                if (pRequest->getHostByName (rDomainName))
                    result = INETCOREDNS_RESOLVER_START;
                else
                    result = INETCOREDNS_RESOLVER_ERROR;
            }
            else if (nQueryType == INETCOREDNS_TYPE_PTR)
            {
                // (NYI).
                result = INETCOREDNS_RESOLVER_ERROR;
            }
            else
            {
                // Failure.
                result = INETCOREDNS_RESOLVER_ERROR;
            }
        }

        if (!(result == INETCOREDNS_RESOLVER_START))
        {
            // Failure.
            if (dequeue (nRequestID, pRequest))
                pRequest->release();
        }
    }
    return result;
}

/*
 * enqueue.
 */
sal_Bool INetDNSResolver_Impl::enqueue (
    sal_uInt16 nRequestID, Request *&rpRequest)
{
    NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
    rpRequest = new Request (nRequestID);
    rpRequest->acquire();

    typedef std::pair<const sal_uInt16, void*> entry_t;
    entry_t entry (nRequestID, rpRequest);

    typedef std::pair<RequestMap::iterator, bool> result_t;
    result_t result = m_aRequestMap.insert (entry);
    if (!result.second)
    {
        rpRequest->release();
        rpRequest = NULL;
    }
    return (!!rpRequest);
}

/*
 * dequeue.
 */
sal_Bool INetDNSResolver_Impl::dequeue (
    sal_uInt16 nRequestID, Request *&rpRequest)
{
    NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
    rpRequest = NULL;

    RequestMap::iterator it = m_aRequestMap.find (nRequestID);
    if (it != m_aRequestMap.end())
    {
        rpRequest = (Request *)((*it).second);
        m_aRequestMap.erase (it);
    }
    return (!!rpRequest);
}

/*
 * ExecuteCallback.
 */
sal_Bool INetDNSResolver_Impl::ExecuteCallback (
    sal_uInt16         nRequestID,
    const oslHostAddr  hAddr,
    void              *pData)
{
    // Check context.
    INetDNSResolver_Impl *pThis = (INetDNSResolver_Impl*)pData;
    if (!(pThis && (pThis == m_pThis)))
        return 0;

    // Handle callback reason.
    m_pThis->acquire();
    m_pThis->onResponse (nRequestID, hAddr, NULL);
    m_pThis->release();

    // Wait for next callback.
    return 1;
}

/*
 * RequestCallback.
 */
sal_Bool INetDNSResolver_Impl::RequestCallback (
    const NAMESPACE_VOS(ORef)<INetSocket> &rxSocket,
    sal_Int32         nEvent,
    void             *pData)
{
    // Check context.
    INetDNSResolver_Impl *pThis = (INetDNSResolver_Impl*)pData;
    if (!(pThis && (pThis == m_pThis)))
        return 0;

    // Handle callback reason.
    m_pThis->acquire();
    m_pThis->RequestHandler (rxSocket, nEvent);
    m_pThis->release();

    // Wait for next callback.
    return 1;
}

/*
 * RequestHandler.
 */
void INetDNSResolver_Impl::RequestHandler (
    const NAMESPACE_VOS(ORef)<INetSocket> &rxSocket, sal_Int32 nEvent)
{
    // Jump into state machine.
    while (1)
    {
        if (nEvent & socket_type::EVENT_READ)
        {
            // Read response message.
            NAMESPACE_VOS(OInetSocketAddr) aFromAddr;
            sal_uInt8 pBuffer[INETCOREDNS_RECV_MSGSIZE];

            sal_Int32 nRead = m_xSocket->recvFrom (
                aFromAddr, pBuffer, sizeof (pBuffer));
            if (nRead > 0)
            {
                // Process response message.
                INetDNSHeader_Impl *pHeader = (INetDNSHeader_Impl*)pBuffer;
                onResponse (pHeader->id, NULL, pBuffer);
            }
            else if (nRead == INETCOREDNS_SOCKET_MSGSIZE)
            {
                // Disable lookup via nameserver query.
                NAMESPACE_VOS(OClearableGuard) aGuard (m_aMutex);
                m_nOptions &= ~INETCOREDNS_CONFIG_SERVER;
                aGuard.clear();

                // Response message too large.
                INetDNSHeader_Impl *pHeader = (INetDNSHeader_Impl*)pBuffer;
                onResponse (pHeader->id, NULL, NULL);
            }
            else if (nRead == INETCOREDNS_SOCKET_WOULDBLOCK)
            {
                // Ok, wait for next callback.
                break;
            }
            else
            {
                // Failure. Wait for close event.
                m_xSocket->close();
                break;
            }
        }
        else if (nEvent & socket_type::EVENT_WRITE)
        {
            // Ok, wait for next callback.
            break;
        }
        else if (nEvent & socket_type::EVENT_CLOSE)
        {
            // Socket closed (due to failure).
            NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
            m_nOptions &= ~INETCOREDNS_CONFIG_SOCKET;
            break;
        }
        else
        {
            // Ignore event.
            break;
        }
    }
}

/*
 * onResponse.
 */
void INetDNSResolver_Impl::onResponse (
    sal_uInt16         nRequestID,
    const oslHostAddr  hAddr,
    const sal_uInt8   *pResponse)
{
    // Lookup request.
#ifdef DBG_UTIL_PRIVATE
    _TRACE("INetDNSResolver_Impl::OnResponse(): Enter.");
#endif /* DBG_UTIL_PRIVATE */

    Request *pRequest;
    if (dequeue (nRequestID, pRequest))
    {
        // Request entry found.
        INetCoreDNSHostEntry *pHostEntry = pRequest->m_pHostent;
        INetCoreDNSCallback  *pfnCB = pRequest->m_pfnCB;
        void                 *pData = pRequest->m_pDataCB;

        // Process response.
        int result = INETCOREDNS_RESOLVER_NOTFOUND;
        if (hAddr || pResponse)
        {
            NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
            if (hAddr)
                insertHostAddr (pHostEntry->m_aName, hAddr);
            if (pResponse)
                insertResponse (pHostEntry->m_aName, pResponse);
            result = lookup (pHostEntry->m_aName, pHostEntry);
        }

        // Notify caller.
#ifdef DBG_UTIL_PRIVATE
        _TRACE("INetDNSResolver_Impl::OnResponse(): Notify.");
#endif /* DBG_UTIL_PRIVATE */
        if (pfnCB) (pfnCB) (result, pHostEntry, pData);

        // Cleanup request.
        pRequest->release();
    }

    // Done.
#ifdef DBG_UTIL_PRIVATE
    _TRACE("INetDNSResolver_Impl::OnResponse(): Leave.");
#endif /* DBG_UTIL_PRIVATE */
}

/*====================================================================
 *
 * INetDNSCache_Impl Implementation.
 *
 *==================================================================*/
/*
 * INetDNSCache_Impl.
 */
INetDNSCache_Impl::INetDNSCache_Impl (void)
{
    // Create permanent entry for "localhost/127.0.0.1".
    INetDNSEntry_Impl *ce =
        new INetDNSEntry_Impl (OUString::createFromAscii ("localhost"));

    ce->m_nType = INETCOREDNS_TYPE_A;
    ce->m_nClass = INETCOREDNS_CLASS_IN;
    ce->m_nExpires = (time_t)(sal_uInt32)(~0);

    ce->setData (OUString::createFromAscii ("127.0.0.1"));
    insert (dn_entry (ce->m_aName, ce));
}

/*
 * ~INetDNSCache_Impl.
 */
INetDNSCache_Impl::~INetDNSCache_Impl (void)
{
    while (!empty())
    {
        dn_map::iterator it = begin();

        INetDNSEntry_Impl *e = (INetDNSEntry_Impl*)((*it).second);
        erase (it);

        delete e;
    }
}

/*
 * insertHostAddr.
 */
void INetDNSCache_Impl::insertHostAddr (
    const OUString &dn, const oslHostAddr hAddr)
{
    // Check arguments.
    if ((dn.getLength() <= 0) || (hAddr == NULL))
        return;

    // Create cache entry for Domain Name.
    INetDNSEntry_Impl *ce = new INetDNSEntry_Impl (dn);

    // Type, Class, TTL.
    OUString aName;
    osl_getHostnameOfHostAddr (hAddr, &(aName.pData));
    if (aName.equalsIgnoreAsciiCase (ce->m_aName))
        ce->m_nType = INETCOREDNS_TYPE_A;
    else
        ce->m_nType = INETCOREDNS_TYPE_CNAME;

    ce->m_nClass = INETCOREDNS_CLASS_IN;
    ce->m_nExpires = time (NULL) + INETCOREDNS_CACHE_TIMEOUT;

    // Resource data.
    if (ce->m_nType == INETCOREDNS_TYPE_CNAME)
    {
        ce->setData (aName);
        insert (dn_entry (ce->m_aName, ce));

        // Recurse for CNAME address.
        insertHostAddr (aName, hAddr);
    }
    else
    {
        if (osl_Socket_Ok == osl_getDottedInetAddrOfSocketAddr (
            osl_getSocketAddrOfHostAddr (hAddr), &(aName.pData)))
        {
            ce->setData (aName);
            insert (dn_entry (ce->m_aName, ce));
        }
    }
}

/*
 * insertResponse.
 */
void INetDNSCache_Impl::insertResponse (
    const OUString &dn, const sal_uInt8 *pBuffer)
{
    // Check arguments.
    if ((dn.getLength() <= 0) || (pBuffer == NULL))
        return;

    // Message header.
    INetDNSHeader_Impl *pHeader = (INetDNSHeader_Impl *)pBuffer;

    pHeader->qdcount = dn_ntohs (pHeader->qdcount);
    pHeader->ancount = dn_ntohs (pHeader->ancount);
    pHeader->nscount = dn_ntohs (pHeader->nscount);
    pHeader->arcount = dn_ntohs (pHeader->arcount);

    // Message body.
    const sal_uInt8 *pOffset = pBuffer + sizeof (INetDNSHeader_Impl);

    // Question section.
    sal_uInt16 i;
    for (i = 0; i < pHeader->qdcount; i++)
    {
        sal_Char name[256];
        pOffset = expand (pBuffer, pOffset, name, sizeof(name));

        sal_uInt16 qtype  = (sal_uInt16)((pOffset[0] << 8) | pOffset[1]);
        pOffset += sizeof (sal_uInt16);
        sal_uInt16 qclass = (sal_uInt16)((pOffset[0] << 8) | pOffset[1]);
        pOffset += sizeof (sal_uInt16);

        if ((pHeader->rcode == 3) && (pHeader->aa))
        {
            // Cache authoritative name error. (status _NOTEXIST).
        }
    }

    // Answer section RR(s).
    for (i = 0; i < pHeader->ancount; i++)
    {
        INetDNSEntry_Impl *ce = NULL;
        pOffset = create (ce, pBuffer, pOffset);
        if (ce->m_nType == INETCOREDNS_TYPE_NS)
            delete ce;
        else
            insert (dn_entry (ce->m_aName, ce));
    }

    // Authority section RR(s).
    for (i = 0; i < pHeader->nscount; i++)
    {
        INetDNSEntry_Impl *ce = NULL;
        pOffset = create (ce, pBuffer, pOffset);
        if (ce->m_nType == INETCOREDNS_TYPE_NS)
            delete ce;
        else
            insert (dn_entry (ce->m_aName, ce));
    }

    // Additional section RR(s).
    for (i = 0; i < pHeader->arcount; i++)
    {
        INetDNSEntry_Impl *ce = NULL;
        pOffset = create (ce, pBuffer, pOffset);
        if (ce->m_nType == INETCOREDNS_TYPE_NS)
            delete ce;
        else
            insert (dn_entry (ce->m_aName, ce));
    }
}

/*
 * lookup.
 */
sal_Int32 INetDNSCache_Impl::lookup (
    const OUString &dn, INetCoreDNSHostEntry *he)
{
    // Check arguments.
    sal_Int32 result = INETCOREDNS_RESOLVER_ERROR;
    if ((dn.getLength() <= 0) || (he == NULL))
        return result;

    result = INETCOREDNS_RESOLVER_NOTFOUND;
    dn_map::iterator it = find (dn);
    if (it != end())
    {
        INetDNSEntry_Impl *ce = (INetDNSEntry_Impl*)((*it).second);
        sal_Int32 nDataLen = ce->m_aData.getLength();
        switch (ce->m_nType)
        {
            case INETCOREDNS_TYPE_CNAME:
            case INETCOREDNS_TYPE_NS:
                if (nDataLen)
                {
                    // Canonical Name (i.e. the Domain Name is an alias).
                    he->m_aCName = ce->m_aData;

                    // Recursive lookup for address of Canonical Name.
                    result = lookup (he->m_aCName, he);
                }
                break;

            case INETCOREDNS_TYPE_PTR:
                if (nDataLen)
                {
                    // Canonical Name (of pointer record).
                    he->m_aCName = ce->m_aData;
                    result = INETCOREDNS_RESOLVER_SUCCESS;
                }
                break;

            case INETCOREDNS_TYPE_A:
                if (nDataLen)
                {
                    // Address.
                    he->m_aDotDecName = ce->m_aData;
                }
                if ((0 < ce->m_nExpires) && (ce->m_nExpires < time(NULL)))
                    result = INETCOREDNS_RESOLVER_EXPIRED;
                else
                    result = INETCOREDNS_RESOLVER_SUCCESS;
                break;

            default:
                // Name exists but no data.
                result = INETCOREDNS_RESOLVER_NODATA;
                break;
        }

        if (!(result == INETCOREDNS_RESOLVER_SUCCESS))
        {
            erase (it);
            delete ce;
        }
    }
    return result;
}

/*
 * create (disassemble a Resource Record into a cache entry).
 */
#define INET_NTOHS(c, h) \
    ((h)  = ((sal_uInt16)(*((c)++))) << 8L, \
     (h) |= ((sal_uInt16)(*((c)++))))

#define INET_NTOHL(c, h) \
    ((h)  = ((sal_uInt32)(*((c)++))) << 24L, \
     (h) |= ((sal_uInt32)(*((c)++))) << 16L, \
     (h) |= ((sal_uInt32)(*((c)++))) <<  8L, \
     (h) |= ((sal_uInt32)(*((c)++))))

const sal_uInt8* INetDNSCache_Impl::create (
    INetDNSEntry_Impl *&rpEntry,
    const sal_uInt8    *pBuffer,
    const sal_uInt8    *pOffset)
{
    // Domain Name.
    sal_Char pName[256];
    pOffset = expand (pBuffer, pOffset, pName, sizeof(pName));
    rpEntry = new INetDNSEntry_Impl (OUString (
        pName, rtl_str_getLength (pName), RTL_TEXTENCODING_UTF8));

    // Type and Class.
    INET_NTOHS (pOffset, rpEntry->m_nType);
    INET_NTOHS (pOffset, rpEntry->m_nClass);

    // TTL (don't cache when zero).
    INET_NTOHL (pOffset, rpEntry->m_nExpires);
    rpEntry->m_nExpires += time(NULL);

    // Resource data.
    sal_uInt16 nDataLen = 0;
    INET_NTOHS (pOffset, nDataLen);
    switch (rpEntry->m_nType)
    {
        case INETCOREDNS_TYPE_CNAME:
        case INETCOREDNS_TYPE_NS:
        case INETCOREDNS_TYPE_PTR:
            expand (pBuffer, pOffset, pName, sizeof (pName));
            rpEntry->setData (OUString (
                pName, rtl_str_getLength (pName), RTL_TEXTENCODING_UTF8));
            break;

        case INETCOREDNS_TYPE_A:
            if (nDataLen == 4)
            {
                // Address.
                rtl::OUStringBuffer aBuffer (16);
                aBuffer.append (sal_Int32(pOffset[0]));
                aBuffer.append (sal_Unicode('.'));
                aBuffer.append (sal_Int32(pOffset[1]));
                aBuffer.append (sal_Unicode('.'));
                aBuffer.append (sal_Int32(pOffset[2]));
                aBuffer.append (sal_Unicode('.'));
                aBuffer.append (sal_Int32(pOffset[3]));
                rpEntry->setData (aBuffer.makeStringAndClear());
            }
            break;

        case INETCOREDNS_TYPE_MX:
        default: // unspecific, yet.
#if 0   /* OLD */
            rpEntry->setData (pOffset, nDataLen);
#endif  /* OLD */
            break;
    }

    pOffset += nDataLen;
    return pOffset;
}

/*
 * expand (disassemble sequence of labels/pointers into domain name).
 */
#define INETCOREDNS_DNAME_ISPTR(a) ((a) & 0xc0)
#define INETCOREDNS_DNAME_OFFSETMASK 0x3fff

const sal_uInt8* INetDNSCache_Impl::expand (
    const sal_uInt8 *pBuffer,
    const sal_uInt8 *pOffset,
    void            *pData,
    sal_uInt16       nSize)
{
    sal_uInt8 *pLabel = (sal_uInt8*)pData;

    while (*pOffset && !INETCOREDNS_DNAME_ISPTR(*pOffset))
    {
        // Copy label.
        sal_uInt8 n = *pOffset++;
        for (sal_uInt8 i = 0; i < n; i++)
            *pLabel++ = *pOffset++;

        // Insert trailing dot, except for root.
        if (*pOffset)
            *pLabel++ = '.';
    }

    if (INETCOREDNS_DNAME_ISPTR(*pOffset))
    {
        sal_uInt16 nOffset = 0;
        INET_NTOHS (pOffset, nOffset);
        nOffset &= INETCOREDNS_DNAME_OFFSETMASK;

        sal_uInt16 nLabel = nSize - (pLabel - (sal_uInt8*)pData);
        expand (pBuffer, pBuffer + nOffset, pLabel, nLabel);
    }
    else
    {
        // Terminate with trailing zero octet.
        *pLabel++ = *pOffset++;
    }

    return pOffset;
}

/*=====================================================================
 *
 * INetDNSRequest_Impl implementation.
 *
 *===================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetDNSRequest_Impl, inet),
    VOS_NAMESPACE (INetDNSRequest_Impl, inet),
    VOS_NAMESPACE (OTimer, vos),
    0);

/*
 * INetDNSRequest_Impl.
 */
INetDNSRequest_Impl::INetDNSRequest_Impl (sal_uInt16 nID)
    : m_aExecutor (this, nID)
{
}

/*
 * ~INetDNSRequest_Impl.
 */
INetDNSRequest_Impl::~INetDNSRequest_Impl (void)
{
    m_aExecutor.setCallback (NULL, NULL);
    OTimer::stop();
}

/*
 * generateQuery (assemble a Query Message).
 */
sal_Bool INetDNSRequest_Impl::generateQuery (
    sal_uInt16      nID,
    sal_uInt16      nQueryType,
    sal_Bool        bRecursive,
    const OUString &rDomainName,
    sal_uInt8      *pBuffer,
    sal_uInt16      nBufSiz,
    sal_uInt16     &rBufLen)
{
    // Determine TextEncoding.
    rtl_TextEncoding eEncoding;
    if (nQueryType == INETCOREDNS_TYPE_PTR)
        eEncoding = RTL_TEXTENCODING_ASCII_US;
    else
        eEncoding = RTL_TEXTENCODING_UTF8;

    // Encode domain name.
    rtl::OString aDomainName (
        rDomainName.pData->buffer,
        rDomainName.pData->length,
        eEncoding);

    // Evaluate query message length.
    sal_Int32 k = sizeof (INetDNSHeader_Impl);
    sal_Int32 n = aDomainName.pData->length;
    rBufLen = k + 1 + n + 1 + 2 * sizeof (sal_uInt16);

    // Check buffer.
    if (!(pBuffer && (rBufLen <= nBufSiz)))
        return sal_False;

    // Setup the message header.
    INetDNSHeader_Impl *pHeader = (INetDNSHeader_Impl*)pBuffer;
    rtl_zeroMemory (pHeader, k);
    pBuffer += k;

    pHeader->id      = nID;         /* request id */
    pHeader->rd      = bRecursive;  /* recursion desired */
    pHeader->qdcount = dn_htons(1); /* number of question entries */

    // Store QNAME.
    const sal_Char *p = aDomainName.pData->buffer;
    while ((k = rtl_str_indexOfChar_WithLength (p, n, '.')) >= 0)
    {
        *pBuffer++ = (sal_uInt8)(k & 0x3f);
        rtl_copyMemory (pBuffer, p, k);
        pBuffer += k;

        p += k + 1;
        n -= k + 1;
    }

    *pBuffer++ = (sal_uInt8)(n & 0x3f);
    rtl_copyMemory (pBuffer, p, n);
    pBuffer += n;

    *pBuffer++ = '\0';

    // Store QTYPE and QCLASS.
    *pBuffer++ = (sal_uInt8)((nQueryType >> 8) & 0xff);
    *pBuffer++ = (sal_uInt8)((nQueryType     ) & 0xff);

    *pBuffer++ = (sal_uInt8)((INETCOREDNS_CLASS_IN >> 8) & 0xff);
    *pBuffer++ = (sal_uInt8)((INETCOREDNS_CLASS_IN     ) & 0xff);

    // Done.
    return sal_True;
}

/*
 * getHostByName (execute TYPE_A query).
 */
sal_Bool INetDNSRequest_Impl::getHostByName (const OUString &rDomainName)
{
    return m_aExecutor.getHostByName (rDomainName);
}

/*
 * start.
 */
void INetDNSRequest_Impl::start (
    callback_type *pfnCB, void *pDataCB, sal_uInt32 nTimeout)
{
    m_aExecutor.setCallback (pfnCB, pDataCB);

    OTimer::setRemainingTime (NAMESPACE_VOS(TTimeValue)(nTimeout));
    OTimer::start();
}

/*
 * onShot.
 */
void SAL_CALL INetDNSRequest_Impl::onShot (void)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetDNSRequest_Impl> xThis (this);

    // Request timed out.
    m_aExecutor.terminate();
    m_aExecutor.done (NULL);
}

/*=====================================================================
 *
 * INetDNSRequest_Impl::executor implementation.
 *
 *===================================================================*/
/*
 * executor.
 */
INetDNSRequest_Impl::executor::executor (
    reference_type *pParent, sal_uInt16 nID)
    : m_pParent (pParent),
      m_nID     (nID),
      m_pfnCB   (NULL),
      m_pDataCB (NULL)
{
}

/*
 * ~executor.
 */
INetDNSRequest_Impl::executor::~executor (void)
{
}

/*
 * getHostByName.
 */
sal_Bool INetDNSRequest_Impl::executor::getHostByName (const OUString &rName)
{
    // Save domain name.
    m_aName = rName;

    // Start resolution.
    sal_Bool result = createSuspended();
    if (result)
    {
        // Ensure clean destruction.
        if (m_pParent)
            m_pParent->acquire();
        resume();
    }
    return (result);
}

/*
 * run.
 */
void SAL_CALL INetDNSRequest_Impl::executor::run (void)
{
    // Resolve domain name.
    oslHostAddr hAddr = osl_createHostAddrByName (m_aName.pData);
    if (schedule())
    {
        // Notify caller.
        done (hAddr);
    }
    osl_destroyHostAddr (hAddr);
}

/*
 * terminate.
 */
void SAL_CALL INetDNSRequest_Impl::executor::terminate (void)
{
    if (m_pParent && isRunning())
    {
        // Release execution reference.
        m_pParent->release();
        m_pParent = NULL;
    }
    OThread::terminate();
}

/*
 * onTerminated.
 */
void SAL_CALL INetDNSRequest_Impl::executor::onTerminated (void)
{
    if (m_pParent)
    {
        // Release execution reference.
        m_pParent->release();
        m_pParent = NULL;
    }
}

/*=====================================================================
 *
 * INetCoreDNSResolver Implementation.
 *
 *===================================================================*/
/*
 * INetCoreDNSResolver.
 */
INetCoreDNSResolver::INetCoreDNSResolver (void)
    : m_pImpl (INetDNSResolver_Impl::getOrCreate())
{
    if (m_pImpl)
    {
        m_pImpl->acquire();
        m_pImpl->configure (INETCOREDNS_CONFIG_BASE);
    }
}

/*
 * ~INetCoreDNSResolver.
 */
INetCoreDNSResolver::~INetCoreDNSResolver (void)
{
    if (m_pImpl)
        m_pImpl->release();
}

/*
 * GetHostName.
 */
sal_Bool INetCoreDNSResolver::GetHostName (
    INetCoreDNSHostEntry *pHostEntry)
{
    if (m_pImpl)
        return m_pImpl->getLocalHostname (pHostEntry);
    else
        return sal_False;
}

/*
 * GetHostByName.
 */
sal_Bool INetCoreDNSResolver::GetHostByName (
    INetCoreDNSHostEntry *pHostEntry,
    INetCoreDNSCallback  *pfnCallback, void *pData)
{
    if (m_pImpl)
        return m_pImpl->getHostByName (pHostEntry, pfnCallback, pData);
    else
        return sal_False;
}

/*
 * GetHostByAddress.
 */
sal_Bool INetCoreDNSResolver::GetHostByAddress (
    INetCoreDNSHostEntry *pHostEntry,
    INetCoreDNSCallback  *pfnCallback, void *pData)
{
    if (m_pImpl)
        return m_pImpl->getHostByAddr (pHostEntry, pfnCallback, pData);
    else
        return sal_False;
}

/*
 * GetMailExchangerByName.
 */
sal_Bool INetCoreDNSResolver::GetMailExchangerByName (
    INetCoreDNSHostEntry *pHostEntry,
    INetCoreDNSCallback  *pfnCallback, void *pData)
{
    // (NYI).
    return sal_False;
}

/*========================================================================
 *
 * INetCoreDNSHostEntry Implementation.
 *
 *======================================================================*/
/*
 * INetCoreDNSHostEntry.
 */
INetCoreDNSHostEntry::INetCoreDNSHostEntry (
    const OUString &rDomainName, sal_uInt16 nPort)
    : m_aName (rDomainName),
      m_nPort (nPort)
{
}

/*
 * INetCoreDNSHostEntry (Copy Ctor).
 */
INetCoreDNSHostEntry::INetCoreDNSHostEntry (
    const INetCoreDNSHostEntry& rHostEntry)
    : m_aDotDecName (rHostEntry.m_aDotDecName),
      m_aCName      (rHostEntry.m_aCName),
      m_aName       (rHostEntry.m_aName),
      m_nPort       (rHostEntry.m_nPort)
{
}

/*
 * INetCoreDNSHostEntry (Assignment).
 */
INetCoreDNSHostEntry&
INetCoreDNSHostEntry::operator= (const INetCoreDNSHostEntry& rHostEntry)
{
    if (this != &rHostEntry)
    {
        m_aDotDecName = rHostEntry.m_aDotDecName;
        m_aCName      = rHostEntry.m_aCName;
        m_aName       = rHostEntry.m_aName;
        m_nPort       = rHostEntry.m_nPort;
    }
    return *this;
}

/*
 * ~INetCoreDNSHostEntry.
 */
INetCoreDNSHostEntry::~INetCoreDNSHostEntry (void)
{
}

/*========================================================================
 *
 * INetCoreDNSHostEntry (obsolete) Implementation.
 *
 *======================================================================*/
#if 0  /* OBSOLETE */

/*
 * INetCoreDNSHostEntry.
 */
INetCoreDNSHostEntry::INetCoreDNSHostEntry (
    const sal_Char *pDomainName, sal_uInt16 nPort)
    : m_pCName (NULL),
      m_pName  (NULL),
      m_nPort  (nPort)
{
    if (pDomainName)
    {
        sal_Int32 nNameLen = rtl_str_getLength (pDomainName);
        m_pName = (sal_Char*)rtl_allocateMemory (nNameLen + 1);

        if (rtl_str_indexOfChar_WithLength (pDomainName, nNameLen, ' ') < 0)
        {
            // Well-formatted domain name.
            rtl_copyMemory (m_pName, pDomainName, nNameLen + 1);
        }
        else
        {
            // Ill-formatted domain name. Collapse around <SP> characters.
            const sal_Char *p = pDomainName;
            sal_Char       *q = m_pName;

            while (*p)
            {
                if (*p == ' ')
                    p++;
                else
                    *q++ = *p++;
            }
            *q = '\0';
        }
    }
    rtl_zeroMemory (m_pDotDecName, sizeof(m_pDotDecName));
}

/*
 * INetCoreDNSHostEntry (Copy Ctor).
 */
INetCoreDNSHostEntry::INetCoreDNSHostEntry (
    const INetCoreDNSHostEntry& rHostEntry)
    : m_pCName (NULL),
      m_pName  (NULL),
      m_nPort  (rHostEntry.m_nPort)
{
    sal_Int32 nNameLen = sizeof (rHostEntry.m_pDotDecName);
    rtl_copyMemory (m_pDotDecName, rHostEntry.m_pDotDecName, nNameLen);

    if (rHostEntry.m_pCName)
    {
        nNameLen = rtl_str_getLength (rHostEntry.m_pCName) + 1;
        dn_copyString (m_pCName, rHostEntry.m_pCName, nNameLen);
    }

    if (rHostEntry.m_pName)
    {
        nNameLen = rtl_str_getLength (rHostEntry.m_pName) + 1;
        dn_copyString (m_pName, rHostEntry.m_pName, nNameLen);
    }
}

/*
 * INetCoreDNSHostEntry (Assignment).
 */
INetCoreDNSHostEntry& INetCoreDNSHostEntry::operator = (
    const INetCoreDNSHostEntry& rHostEntry)
{
    if (this != &rHostEntry)
    {
        // Cleanup and Copy.
        sal_Int32 nNameLen = sizeof (rHostEntry.m_pDotDecName);
        rtl_copyMemory (m_pDotDecName, rHostEntry.m_pDotDecName, nNameLen);

        if (rHostEntry.m_pCName)
        {
            nNameLen = rtl_str_getLength (rHostEntry.m_pCName) + 1;
            dn_copyString (m_pCName, rHostEntry.m_pCName, nNameLen);
        }
        else
        {
            rtl_freeMemory (m_pCName);
            m_pCName = NULL;
        }

        if (rHostEntry.m_pName)
        {
            nNameLen = rtl_str_getLength (rHostEntry.m_pName) + 1;
            dn_copyString (m_pName, rHostEntry.m_pName, nNameLen);
        }
        else
        {
            rtl_freeMemory (m_pName);
            m_pName = NULL;
        }

        m_nPort = rHostEntry.m_nPort;
    }
    return *this;
}

/*
 * ~INetCoreDNSHostEntry.
 */
INetCoreDNSHostEntry::~INetCoreDNSHostEntry (void)
{
    rtl_freeMemory (m_pCName);
    rtl_freeMemory (m_pName);
}

#endif /* OBSOLETE */

