/*************************************************************************
 *
 *  $RCSfile: inetnntp.cxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: th $ $Date: 2001/05/11 12:12:03 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INETNNTP_CXX "$Revision: 1.4 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_STRBUF_HXX_
#include <rtl/strbuf.hxx>
#endif
#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif

#ifndef _VOS_REF_HXX_
#include <vos/ref.hxx>
#endif

#ifndef _DATETIME_HXX
#include <tools/datetime.hxx>
#endif
#ifndef _LIST_HXX
#include <tools/list.hxx>
#endif

#ifndef _INET_MACROS_HXX
#include <inet/macros.hxx>
#endif
#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif

#ifndef _INET_CONFIG_HXX
#include <inetcfg.hxx>
#endif
#ifndef _INETDNS_HXX
#include <inetdns.hxx>
#endif
#ifndef _INETNNTP_HXX
#include <inetnntp.hxx>
#endif

#ifndef _INETCOREMSG_HXX
#include <inetmsg.hxx>
#endif
#ifndef _INETCORESTRM_HXX
#include <inetstrm.hxx>
#endif

#include <stdlib.h>
#include <stdio.h>

#ifdef _USE_NAMESPACE
using namespace inet;
#endif

using rtl::OString;
using rtl::OStringBuffer;

using rtl::OUString;
using rtl::OUStringBuffer;

/*=======================================================================
 *
 * INetCoreNNTPConnection Implementation.
 *
 * References:
 *   RFC  977 - Network News Transfer Protocol (Proposed Standard).
 *   RFC 1920 - Internet Official Protocol Standards (STD 1).
 *
 *=====================================================================*/
typedef NAMESPACE_INET(INetSocket) socket_type;

enum INetCoreNNTPStreamState
{
    INETCORENNTP_EOL_BEGIN = 0,
    INETCORENNTP_EOL_SCR   = 1,
    INETCORENNTP_EOL_FCR   = 2,
    INETCORENNTP_EOL_FLF   = 3
};

#define INETCORENNTP_SOCKET_DISPOSE(socket) \
if ((socket).isValid()) \
{ \
    (socket)->deregisterEventHandler(onSocketEvent); \
    (socket)->close(); \
    (socket).unbind(); \
}

#define INETCORENNTP_SOCKET_WOULDBLOCK (-osl_Socket_E_WouldBlock)

//=======================================================================

inline sal_Bool ascii_isDigit( sal_Unicode ch )
{
    return ((ch >= 0x0030) && (ch <= 0x0039));
}

inline sal_Bool ascii_isWhitespace( sal_Unicode ch )
{
    return ((ch <= 0x20) && ch);
}

inline sal_Unicode ascii_toLowerCase( sal_Unicode ch )
{
    if ( (ch >= 0x0041) && (ch <= 0x005A) )
        return ch + 0x20;
    else
        return ch;
}

/*========================================================================
 *
 * OutputStreamBuffer_Impl.
 *
 *======================================================================*/
namespace inet
{
namespace nntp
{

class OutputStreamBuffer_Impl
{
    sal_uInt32  m_nSize;
    sal_uInt32  m_nGrow;
    sal_Char   *m_pBuffer;
    sal_Char   *m_pOffset;

protected:
    void ensureCapacity (sal_uInt32 nBytes);

public:
    OutputStreamBuffer_Impl (
        sal_uInt32 nSize = 0,
        sal_uInt32 nGrow = 256);
    ~OutputStreamBuffer_Impl (void);

    const sal_Char* getBuffer (void) const
    {
        return m_pBuffer;
    }
    sal_uInt32 getLength (void) const
    {
        return (m_pOffset - m_pBuffer);
    }
    void rewind (void)
    {
        m_pOffset = m_pBuffer;
    }

    void operator<< (sal_Char c);
};

inline OutputStreamBuffer_Impl::OutputStreamBuffer_Impl (
    sal_uInt32 nSize, sal_uInt32 nGrow)
    : m_nSize (nSize), m_nGrow (nGrow), m_pBuffer (0), m_pOffset (0)
{
}

inline OutputStreamBuffer_Impl::~OutputStreamBuffer_Impl (void)
{
    rtl_freeMemory (m_pBuffer);
}

inline void OutputStreamBuffer_Impl::ensureCapacity (sal_uInt32 nBytes)
{
    sal_uInt32 nOffset = m_pOffset - m_pBuffer;
    if (m_nSize < (nOffset + nBytes))
    {
        m_nSize  += ((nBytes + m_nGrow) / m_nGrow) * m_nGrow;
        m_pBuffer = (sal_Char*)rtl_reallocateMemory (m_pBuffer, m_nSize);
        m_pOffset = m_pBuffer + nOffset;
    }
}

inline void OutputStreamBuffer_Impl::operator<< (sal_Char c)
{
    ensureCapacity(1);
    *m_pOffset++ = c;
}

} // nntp
} // inet

using namespace inet::nntp;

/*========================================================================
 *
 * INetCoreNNTPReplyStream Interface.
 *
 *======================================================================*/
class INetCoreNNTPReplyStream : public INetCoreOStream
{
    OutputStreamBuffer_Impl m_aBuffer;
    INetCoreNNTPStreamState m_eState;
    sal_Bool                m_bTransparent;

    sal_Int32               m_nReplyCode;
    OString                 m_aReplyText;

    int         ParseStatus (void *pCtx);

protected:
    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPReplyStream (void);
    virtual ~INetCoreNNTPReplyStream (void);

    sal_Int32       GetReplyCode (void) const;
    const sal_Char* GetReplyText (void);
};

inline sal_Int32 INetCoreNNTPReplyStream::GetReplyCode (void) const
{
    return m_nReplyCode;
}

inline const sal_Char* INetCoreNNTPReplyStream::GetReplyText (void)
{
    if (m_aReplyText.getLength())
        return m_aReplyText.getStr();
    else
        return NULL;
}

/*=======================================================================
 *
 * INetCoreNNTPConnectReplyStream Interface.
 *
 *=====================================================================*/
class INetCoreNNTPConnectReplyStream : public INetCoreNNTPReplyStream
{
    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPConnectReplyStream (void);
    virtual ~INetCoreNNTPConnectReplyStream (void);
};

/*========================================================================
 *
 * INetCoreNNTPSimpleReplyStream Interface.
 *
 *======================================================================*/
class INetCoreNNTPSimpleReplyStream : public INetCoreNNTPReplyStream
{
    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPSimpleReplyStream (void);
    virtual ~INetCoreNNTPSimpleReplyStream (void);
};

/*========================================================================
 *
 * INetCoreNNTPInputStream Interface.
 *
 *======================================================================*/
class INetCoreNNTPInputStream : public INetCoreIStream
{
    sal_uInt32               m_nBufSiz;
    sal_Char                *m_pBuffer;
    sal_Char                *m_pRead;
    sal_Char                *m_pWrite;

    INetCoreNNTPStreamState  m_eState;
    sal_Bool                 m_bEndOfMessage;

    virtual int GetData (
        sal_Char *pData, sal_uInt32 nSize, void *pCtx);

protected:
    virtual int GetLine (
        sal_Char *pData, sal_uInt32 nSize, void *pCtx) = 0;

public:
    INetCoreNNTPInputStream (sal_uInt32 nBufSiz = 1024);
    virtual ~INetCoreNNTPInputStream (void);
};

/*========================================================================
 *
 * INetCoreNNTPMsgDataInputStream Interface.
 *
 *======================================================================*/
class INetCoreNNTPMsgDataInputStream : public INetCoreNNTPInputStream
{
    INetCoreMessageIStream &m_rMsgStrm;

    virtual int GetLine (
        sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPMsgDataInputStream (INetCoreMessageIStream &rMsgStrm);
    virtual ~INetCoreNNTPMsgDataInputStream (void);
};

/*=======================================================================
 *
 * INetCoreNNTPOutputStream Interface.
 *
 *=====================================================================*/
class INetCoreNNTPOutputStream : public INetCoreOStream
{
    OutputStreamBuffer_Impl m_aBuffer;
    INetCoreNNTPStreamState m_eState;

    virtual int PutData (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

protected:
    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx) = 0;

public:
    INetCoreNNTPOutputStream (void);
    virtual ~INetCoreNNTPOutputStream (void);
};

/*=========================================================================
 *
 * INetCoreNNTPGroupListOutputStream Interface.
 *
 *=======================================================================*/
class INetCoreNNTPGroupListOutputStream : public INetCoreNNTPOutputStream
{
    List &m_rOutList;

    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPGroupListOutputStream (List& rOutList);
    virtual ~INetCoreNNTPGroupListOutputStream (void);
};

/*========================================================================
 *
 * INetCoreNNTPMsgListOutputStream Interface.
 *
 *======================================================================*/
class INetCoreNNTPMsgListOutputStream : public INetCoreNNTPOutputStream
{
    List &m_rOutList;

    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPMsgListOutputStream (List& rOutList);
    virtual ~INetCoreNNTPMsgListOutputStream (void);
};

/*========================================================================
 *
 * INetCoreNNTPOverFmtOutputStream Interface.
 *
 *======================================================================*/
class INetCoreNNTPOverFmtOutputStream : public INetCoreNNTPOutputStream
{
    List &m_rOutList;

    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPOverFmtOutputStream (List &rOutList);
    virtual ~INetCoreNNTPOverFmtOutputStream (void);
};

/*========================================================================
 *
 * INetCoreNNTPOverListOutputStream Interface.
 *
 *======================================================================*/
class INetCoreNNTPOverListOutputStream : public INetCoreNNTPOutputStream
{
    INetCoreNNTPConnection       *m_pConnection;
    INetCoreNNTPOverviewCallback *m_pfnCB;
    void                         *m_pDataCB;

    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPOverListOutputStream (
        INetCoreNNTPConnection       *pConnection,
        INetCoreNNTPOverviewCallback *pfnCallback,
        void                         *pData);
    virtual ~INetCoreNNTPOverListOutputStream (void);
};

/*=======================================================================
 *
 * INetCoreNNTPMsgDataOutputStream Interface.
 *
 *=====================================================================*/
class INetCoreNNTPMsgDataOutputStream : public INetCoreNNTPOutputStream
{
    INetCoreMessageOStream &m_rMsgStrm;

    virtual int PutLine (
        const sal_Char *pData, sal_uInt32 nSize, void *pCtx);

public:
    INetCoreNNTPMsgDataOutputStream (INetCoreMessageOStream &rMsgStrm);
    virtual ~INetCoreNNTPMsgDataOutputStream (void);
};

/*=======================================================================
 *
 * INetCoreNNTPCmdContext Interface.
 *
 *=====================================================================*/
enum INetCoreNNTPCmdState
{
    INETCORENNTP_CMD_SUCCESS = -2,
    INETCORENNTP_CMD_ERROR   = -1,
    INETCORENNTP_CMD_CONNECT =  0,
    INETCORENNTP_CMD_SEND    =  1,
    INETCORENNTP_CMD_RECV    =  2,
    INETCORENNTP_CMD_POST    =  3
};

#ifdef _USE_NAMESPACE
namespace inet {
#endif

struct INetCoreNNTPCmdContext
{
    INetCoreNNTPCmdContext (
        const OString            &rCommandLine,
        INetCoreNNTPReplyStream  *pReplyStream,
        INetCoreNNTPInputStream  *pSource,
        INetCoreNNTPOutputStream *pTarget,
        INetCoreNNTPCallback     *pfnCallback,
        void                     *pData);
    ~INetCoreNNTPCmdContext (void);

    INetCoreNNTPCmdState      m_eState;
    int                       m_nCode;
    INetCoreNNTPCmdState      m_eOkState;
    int                       m_nOkCode;

    OString                   m_aCmdLine;

    INetCoreNNTPReplyStream  *m_pReplyStream;
    INetCoreNNTPInputStream  *m_pSource;
    INetCoreNNTPOutputStream *m_pTarget;

    INetCoreNNTPCallback     *m_pfnCB;
    void                     *m_pDataCB;
};

#ifdef _USE_NAMESPACE
}
#endif

/*========================================================================
 *
 * INetCoreNNTPConnectionContext Interface.
 *
 *======================================================================*/
#ifdef _USE_NAMESPACE
namespace inet {
#endif

struct INetCoreNNTPConnectionContext
{
    INetCoreNNTPConnectionContext (sal_uInt32 nBufferSize = 4096);
    ~INetCoreNNTPConnectionContext (void);

    INetCoreNNTPCmdContext  *m_pCmdCtx;

    sal_Bool                 m_bIsOpen;
    sal_Bool                 m_bPostingAllowed;
    sal_Bool                 m_bAborting;

    INetCoreDNSResolver     *m_pResolver;
    INetCoreDNSHostEntry     m_aDestAddr;
    NAMESPACE_VOS(ORef)<INetActiveTCPSocket> m_xSocket;
    sal_uInt32               m_nXferCount;

    sal_uInt32               m_nBufSiz;
    sal_Char                *m_pBuffer;
    sal_Char                *m_pRead;
    sal_Char                *m_pWrite;

    INetCoreNNTPCallback    *m_pfnXferCB;
    void                    *m_pXferData;
    INetCoreNNTPCallback    *m_pfnTermCB;
    void                    *m_pTermData;

    /** create.
     */
    void create (INetCoreDNSHostEntry &rDstAddr);
};

#ifdef _USE_NAMESPACE
}
#endif

/*=======================================================================
 *
 * INetCoreNNTPConnection Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreNNTPConnection.
 */
INetCoreNNTPConnection::INetCoreNNTPConnection (void)
    : m_pConCtx (new INetCoreNNTPConnectionContext())
{
    VOS_POSTCOND(m_pConCtx, "INetCoreNNTPConnection::ctor(): no context");
}

/*
 * ~INetCoreNNTPConnection.
 */
INetCoreNNTPConnection::~INetCoreNNTPConnection (void)
{
    VOS_PRECOND(m_pConCtx, "INetCoreNNTPConnection::dtor(): no context");
    if (m_pConCtx)
    {
        INETCORENNTP_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
        delete m_pConCtx;
    }
}

/*
 * handleResolverEvent.
 */
sal_Bool INetCoreNNTPConnection::handleResolverEvent (
    sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry)
{
    // Check connection context.
    if (m_pConCtx == NULL)
        return 0;

    // Check for idle state (No command context).
    INetCoreNNTPCmdContext *pCtx = m_pConCtx->m_pCmdCtx;
    if (pCtx == NULL)
        return 1;

    // Check for abort during name resolution.
    if (m_pConCtx->m_bAborting)
        nStatus = INETCOREDNS_RESOLVER_ERROR;

    switch (nStatus)
    {
        case INETCOREDNS_RESOLVER_START:
            // Notify caller.
            if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                this, INETCORENNTP_REPLY_RESOLVER_WAIT,
                NULL, pCtx->m_pDataCB);

            // Wait for next event.
            return 1;

        case INETCOREDNS_RESOLVER_SUCCESS:
        case INETCOREDNS_RESOLVER_EXPIRED:
            // Initialize active socket to destination.
            m_pConCtx->create (*pHostEntry);
            m_pConCtx->m_xSocket->registerEventHandler (onSocketEvent, this);

            // Initiate connect.
            if (m_pConCtx->m_xSocket->connect (
                NAMESPACE_VOS(OInetSocketAddr (
                    pHostEntry->GetDottedDecimalName(),
                    pHostEntry->GetPort()))))
            {
                // Notify caller.
                if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                    this, INETCORENNTP_REPLY_CONNECT_WAIT,
                    NULL, pCtx->m_pDataCB);

                // Wait for connect event.
                return 1;
            }
            else
            {
                // Failure. Cleanup and notify caller.
                m_pConCtx->m_xSocket.unbind();

                m_pConCtx->m_pCmdCtx = NULL;
                if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                    this, INETCORENNTP_REPLY_CONNECT_ERROR,
                    NULL, pCtx->m_pDataCB);
                delete pCtx;
            }
            break;

        default:
            // Failure. Cleanup and notify caller.
            m_pConCtx->m_pCmdCtx = NULL;

            if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                this, INETCORENNTP_REPLY_RESOLVER_ERROR,
                NULL, pCtx->m_pDataCB);
            delete pCtx;
            break;
    }

    // Done.
    return 0;
}

/*
 * handleSocketEvent.
 * Implements NNTP as Finite State Machine.
 */
sal_Bool INetCoreNNTPConnection::handleSocketEvent (
    const NAMESPACE_VOS(ORef)<INetSocket> &rxSocket, sal_Int32 nEvent)
{
    // Check connection context.
    if (m_pConCtx == NULL)
        return 0;

    // Check command context.
    INetCoreNNTPCmdContext *pCtx = m_pConCtx->m_pCmdCtx;
    if (pCtx == NULL)
    {
        // No command context (STATE_IDLE).
        if (nEvent & socket_type::EVENT_READ)
        {
            // Jump into idle handler.
            while (1)
            {
                sal_Int32 nRead = rxSocket->recv (
                    m_pConCtx->m_pBuffer, m_pConCtx->m_nBufSiz);
                if (nRead > 0)
                {
                    // Absorbing whatever comes in.
                    continue;
                }
                else if (nRead == INETCORENNTP_SOCKET_WOULDBLOCK)
                {
                    // Wait for next event.
                    return 1;
                }
                else
                {
                    // Connection closed or Network failure.
                    rxSocket->close();
                    return 1;
                }
            }
        }

        if (nEvent & socket_type::EVENT_CLOSE)
        {
            // Connection closed.
            m_pConCtx->m_bIsOpen = sal_False;
            m_pConCtx->m_xSocket.unbind();

            // Notify caller (terminate callback).
            if (m_pConCtx->m_pfnTermCB) (m_pConCtx->m_pfnTermCB) (
                this, INETCORENNTP_REPLY_NETWORK_ERROR, NULL,
                m_pConCtx->m_pTermData);
        }

        // Leave.
        return 1;
    }

    // Check event.
    if (nEvent & socket_type::EVENT_CLOSE)
    {
        // Connection closed.
        m_pConCtx->m_bIsOpen = sal_False;
        m_pConCtx->m_xSocket.unbind();

        // Finish with network error.
        m_pConCtx->m_pCmdCtx->m_eState = INETCORENNTP_CMD_ERROR;
        m_pConCtx->m_pCmdCtx->m_nCode  = INETCORENNTP_REPLY_NETWORK_ERROR;
    }

    // Jump into state machine.
    while (1)
    {
        switch (pCtx->m_eState)
        {
            case INETCORENNTP_CMD_CONNECT:
                if (nEvent & socket_type::EVENT_CONNECT)
                {
                    if (nEvent & socket_type::EVENT_OOB)
                    {
                        // Finish with connect error.
                        pCtx->m_eState = INETCORENNTP_CMD_ERROR;
                        pCtx->m_nCode  = INETCORENNTP_REPLY_CONNECT_ERROR;

                        // Cleanup.
                        m_pConCtx->m_xSocket.unbind();
                    }
                    else
                    {
                        // Read greeting reply.
                        pCtx->m_eState = INETCORENNTP_CMD_RECV;
                        nEvent = socket_type::EVENT_READ;
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            case INETCORENNTP_CMD_SEND:
                if (nEvent & socket_type::EVENT_WRITE)
                {
                    // Write out command buffer.
                    OString aCommand (pCtx->m_aCmdLine);

                    sal_Int32 nWrite = rxSocket->send (
                        aCommand.pData->buffer, aCommand.pData->length);
                    if (nWrite > 0)
                    {
                        // Read reply.
                        pCtx->m_eState = INETCORENNTP_CMD_RECV;
                        nEvent = socket_type::EVENT_READ;
                    }
                    else if (nWrite == INETCORENNTP_SOCKET_WOULDBLOCK)
                    {
                        // Wait for next event.
                        return 1;
                    }
                    else
                    {
                        // Socket write error.
                        pCtx->m_eState = INETCORENNTP_CMD_ERROR;
                        pCtx->m_nCode = INETCORENNTP_REPLY_NETWORK_ERROR;
                        rxSocket->close();
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            case INETCORENNTP_CMD_RECV:
                if (nEvent & socket_type::EVENT_READ)
                {
                    sal_Int32 nRead = rxSocket->recv (
                        m_pConCtx->m_pBuffer, m_pConCtx->m_nBufSiz);
                    if (nRead > 0)
                    {
                        // Ok, increment transfer count.
                        m_pConCtx->m_nXferCount += nRead;

                        // Analyze reply.
                        int status = pCtx->m_pReplyStream->Write (
                            m_pConCtx->m_pBuffer, nRead, m_pConCtx);
                        if (status == INETCORESTREAM_STATUS_LOADED)
                        {
                            pCtx->m_nCode =
                                pCtx->m_pReplyStream->GetReplyCode();
                            if (pCtx->m_nCode/100 == pCtx->m_nOkCode/100)
                            {
                                pCtx->m_eState =
                                    pCtx->m_eOkState;
                                pCtx->m_eOkState =
                                    INETCORENNTP_CMD_SUCCESS;
                                pCtx->m_nOkCode =
                                    INETCORENNTP_REPLY_ARTICLE_OK;

                                if (pCtx->m_eState == INETCORENNTP_CMD_POST)
                                {
                                    nEvent = socket_type::EVENT_WRITE;
                                    m_pConCtx->m_nXferCount = 0;
                                }
                            }
                            else
                            {
                                pCtx->m_eState = INETCORENNTP_CMD_ERROR;
                            }
                        }
                        else
                        {
                            if (status != INETCORESTREAM_STATUS_OK)
                            {
                                // Syntax error, i.e. no reply code.
                                pCtx->m_eState =
                                    INETCORENNTP_CMD_ERROR;
                                pCtx->m_nCode =
                                    INETCORENNTP_REPLY_COMMAND_SYNTAX_ERROR;
                            }
                        }
                    }
                    else if (nRead == INETCORENNTP_SOCKET_WOULDBLOCK)
                    {
                        if (pCtx->m_pSource || pCtx->m_pTarget)
                        {
                            // Notify caller.
                            if (m_pConCtx->m_pfnXferCB)
                                (m_pConCtx->m_pfnXferCB) (
                                    this, INETCORENNTP_REPLY_TRANSFER_WAIT,
                                    NULL, m_pConCtx->m_pXferData);
                        }

                        // Wait for next read event.
                        return 1;
                    }
                    else
                    {
                        // Socket read error.
                        pCtx->m_eState = INETCORENNTP_CMD_ERROR;
                        pCtx->m_nCode  = INETCORENNTP_REPLY_NETWORK_ERROR;
                        rxSocket->close();
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            case INETCORENNTP_CMD_POST:
                if (nEvent & socket_type::EVENT_WRITE)
                {
                    // Send article to be posted.
                    if ((m_pConCtx->m_pRead - m_pConCtx->m_pWrite) > 0)
                    {
                        // Bytes not yet written. Write out to socket.
                        sal_Int32 nWrite = rxSocket->send (
                            m_pConCtx->m_pWrite,
                            (m_pConCtx->m_pRead - m_pConCtx->m_pWrite));
                        if (nWrite > 0)
                        {
                            // Have written some bytes.
                            m_pConCtx->m_nXferCount += nWrite;
                            m_pConCtx->m_pWrite     += nWrite;
                        }
                        else if (nWrite == INETCORENNTP_SOCKET_WOULDBLOCK)
                        {
                            // Notify caller.
                            if (m_pConCtx->m_pfnXferCB)
                                (m_pConCtx->m_pfnXferCB) (
                                    this, INETCORENNTP_REPLY_TRANSFER_WAIT,
                                    NULL, m_pConCtx->m_pXferData);

                            // Wait for next write event.
                            return 1;
                        }
                        else
                        {
                            // Socket write error.
                            pCtx->m_eState = INETCORENNTP_CMD_ERROR;
                            pCtx->m_nCode  = INETCORENNTP_REPLY_NETWORK_ERROR;
                            rxSocket->close();
                        }
                    }
                    else
                    {
                        // Buffer empty. Reset to begin.
                        m_pConCtx->m_pRead = m_pConCtx->m_pWrite =
                            m_pConCtx->m_pBuffer;

                        // Read source stream.
                        int nRead = pCtx->m_pSource->Read (
                            m_pConCtx->m_pBuffer, m_pConCtx->m_nBufSiz,
                            m_pConCtx);
                        if (nRead > 0)
                        {
                            // Have read some bytes.
                            m_pConCtx->m_pRead += nRead;
                        }
                        else if (nRead == 0)
                        {
                            // Source loaded. Read reply.
                            pCtx->m_eState = INETCORENNTP_CMD_RECV;
                            nEvent = socket_type::EVENT_READ;
                        }
                        else
                        {
                            // Source read error.
                            pCtx->m_eState = INETCORENNTP_CMD_ERROR;
                            pCtx->m_nCode  = 0;
                        }
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            default: // (_SUCCESS || _ERROR)
                if (pCtx)
                {
                    // Restore idle state.
                    m_pConCtx->m_pCmdCtx = NULL;

                    // Notify caller.
                    if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                        this, pCtx->m_nCode,
                        pCtx->m_pReplyStream->GetReplyText(),
                        pCtx->m_pDataCB);

                    // Cleanup and leave.
                    delete pCtx;
                    return 1;
                }
                break;
        } // switch (pCtx->m_eState)
    } // while (1)
}

/*
 * StartCommand.
 */
sal_Bool INetCoreNNTPConnection::StartCommand (INetCoreNNTPCmdContext *pCtx)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

    // Check connection context.
    if (m_pConCtx && m_pConCtx->m_bIsOpen && !m_pConCtx->m_bAborting)
    {
        // Check command context.
        if (!m_pConCtx->m_pCmdCtx && pCtx)
        {
            // Set connection context.
            m_pConCtx->m_pCmdCtx = pCtx;
            m_pConCtx->m_nXferCount = 0;

            // Start command.
            if (m_pConCtx->m_xSocket->postEvent (socket_type::EVENT_WRITE))
            {
                // Command started.
                return sal_True;
            }
            m_pConCtx->m_pCmdCtx = NULL;
        }
    }

    // Command failed.
    delete pCtx;
    return sal_False;
}

/*
 * Open.
 * Initiate connection to NNTPD.
 */
sal_Bool INetCoreNNTPConnection::Open (
    const OUString &rHost, sal_uInt16 nPort,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

    // Check connection context.
    if ((m_pConCtx == NULL) || (m_pConCtx->m_bIsOpen))
        return sal_False;

    // Check arguments.
    if ((rHost.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;
    if (nPort == 0)
        nPort = INETCORENNTP_DEF_PORT;

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        OString(),
        new INetCoreNNTPConnectReplyStream(),
        NULL, NULL,
        pfnCallback, pData);

    pCtx->m_eState   = INETCORENNTP_CMD_CONNECT;
    pCtx->m_nCode    = 0;
    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_SERVICE_READY;

    // Set connection context.
    m_pConCtx->m_pCmdCtx = pCtx;
    m_pConCtx->m_bIsOpen         = sal_False;
    m_pConCtx->m_bPostingAllowed = sal_False;

    // Start domain name resolution.
    m_pConCtx->m_aDestAddr = INetCoreDNSHostEntry (rHost, nPort);

    if (m_pConCtx->m_pResolver->GetHostByName (
        &(m_pConCtx->m_aDestAddr), onResolverEvent, this))
    {
        // Ok. Wait for things to come.
        return sal_True;
    }
    else
    {
        // Cleanup and fail.
        m_pConCtx->m_pCmdCtx = NULL;
        delete pCtx;
        return sal_False;
    }
}

/*
 * IsOpen.
 */
sal_Bool INetCoreNNTPConnection::IsOpen (void)
{
    return (m_pConCtx ? m_pConCtx->m_bIsOpen : sal_False);
}

/*
 * Close.
 */
sal_Bool INetCoreNNTPConnection::Close (
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        "QUIT\015\012",
        new INetCoreNNTPConnectReplyStream(),
        NULL, NULL,
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_SERVICE_CLOSING;

    // Start QUIT command.
    return StartCommand (pCtx);
}

/*
 * Destroy.
 */
void INetCoreNNTPConnection::Destroy (void)
{
    if (m_pConCtx)
    {
        m_pConCtx->m_bIsOpen         = sal_False;
        m_pConCtx->m_bPostingAllowed = sal_False;
        m_pConCtx->m_bAborting       = sal_True;

        if (m_pConCtx->m_pCmdCtx)
        {
            m_pConCtx->m_pCmdCtx->m_pSource = NULL; /* @@@ */
            m_pConCtx->m_pCmdCtx->m_pTarget = NULL; /* @@@ */

            m_pConCtx->m_pCmdCtx->m_pfnCB   = NULL;
            m_pConCtx->m_pCmdCtx->m_pDataCB = NULL;
        }

        m_pConCtx->m_pfnXferCB = NULL;
        m_pConCtx->m_pXferData = NULL;

        m_pConCtx->m_pfnTermCB = NULL;
        m_pConCtx->m_pTermData = NULL;

        INETCORENNTP_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
    }
}

/*
 * SetModeReader.
 */
sal_Bool INetCoreNNTPConnection::SetModeReader (
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        "MODE READER\015\012",
        new INetCoreNNTPSimpleReplyStream(),
        NULL, NULL,
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_SERVICE_READY;

    // Start MODE READER command.
    return StartCommand (pCtx);
}

/*
 * AuthinfoUser.
 */
sal_Bool INetCoreNNTPConnection::AuthinfoUser (
    const OUString &rUsername,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((rUsername.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("AUTHINFO USER ");
    aBuffer.append (OString (
        rUsername.pData->buffer,
        rUsername.pData->length,
        RTL_TEXTENCODING_ASCII_US));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPSimpleReplyStream(),
        NULL, NULL, pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_AUTHINFO_ACCEPTED;

    // Start AUTHINFO USER command.
    return StartCommand (pCtx);
}

/*
 * AuthinfoPass.
 */
sal_Bool INetCoreNNTPConnection::AuthinfoPass (
    const OUString &rPassword,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((rPassword.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("AUTHINFO PASS ");
    aBuffer.append (OString (
        rPassword.pData->buffer,
        rPassword.pData->length,
        RTL_TEXTENCODING_ASCII_US));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPSimpleReplyStream(),
        NULL, NULL,
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_AUTHINFO_ACCEPTED;

    // Start AUTHINFO PASS command.
    return StartCommand (pCtx);
}

/*
 * GetGroupList.
 * Starts LIST command.
 */
sal_Bool INetCoreNNTPConnection::GetGroupList (
    List& rGroupList, INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        "LIST\015\012",
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPGroupListOutputStream (rGroupList),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_GROUPS_LIST_OK;

    // Start LIST command.
    return StartCommand (pCtx);
}

/*
 * ConvertDateTime_Impl.
 */
static void ConvertDateTime_Impl (const DateTime &rDate, sal_Char Buffer[18])
{
    DateTime aDate (rDate);
    aDate.ConvertToUTC();

#if 0  /* Y2K */
    int year = aDate.GetYear() - 1900;
    if (year > 99) year -= 100;
#endif /* Y2K */
    int year = aDate.GetYear() % 100;

    sprintf (Buffer, "%02d%02d%02d %02d%02d%02d GMT",
             year           , aDate.GetMonth(), aDate.GetDay(),
             aDate.GetHour(), aDate.GetMin()  , aDate.GetSec());
}

/*
 * GetNewGroupsList.
 * Starts NEWGROUPS command.
 */
sal_Bool INetCoreNNTPConnection::GetNewGroupsList (
    const DateTime& rSinceDateTime,
    List& rNewGroupsList,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Build command line.
    sal_Char pBuffer[18];
    ConvertDateTime_Impl (rSinceDateTime, pBuffer);

    OStringBuffer aBuffer ("NEWGROUPS ");
    aBuffer.append (pBuffer);
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPGroupListOutputStream (rNewGroupsList),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_NEWGROUPS_LIST_OK;

    // Start NEWGROUPS command.
    return StartCommand (pCtx);
}

/*
 * GetNewNewsList.
 * Starts NEWNEWS command.
 */
sal_Bool INetCoreNNTPConnection::GetNewNewsList (
    const OUString &rGroups,
    const DateTime &rSinceDateTime,
    List           &rNewNewsList,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Build command line.
    sal_Char pBuffer[18];
    ConvertDateTime_Impl (rSinceDateTime, pBuffer);

    OStringBuffer aBuffer ("NEWNEWS ");
    if (rGroups.getLength())
        aBuffer.append (OString (
            rGroups.pData->buffer,
            rGroups.pData->length,
            RTL_TEXTENCODING_UTF8));
    else
        aBuffer.append (sal_Char ('*'));
    aBuffer.append (sal_Char (' '));
    aBuffer.append (pBuffer);
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPMsgListOutputStream (rNewNewsList),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_NEWARTICLE_LIST_OK;

    // Start NEWNEWS command.
    return StartCommand (pCtx);
}

/*
 * SelectGroup.
 * Starts GROUP command.
 */
sal_Bool INetCoreNNTPConnection::SelectGroup (
    const OUString &rGroupName,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((rGroupName.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("GROUP ");
    aBuffer.append (OString (
        rGroupName.pData->buffer,
        rGroupName.pData->length,
        RTL_TEXTENCODING_UTF8));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPSimpleReplyStream(),
        NULL, NULL,
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_GROUP_SELECTED;

    // Start GROUP command.
    return StartCommand (pCtx);
}

/*
 * GetArticleHeaderOverviewFormat.
 * Starts LIST OVERVIEW.FMT (extension) command.
 */
sal_Bool INetCoreNNTPConnection::GetArticleHeaderOverviewFormat (
    List& rOverviewHeaderList,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        "LIST OVERVIEW.FMT\015\012",
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPOverFmtOutputStream (rOverviewHeaderList),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_GROUPS_LIST_OK; // code re-used.

    // Start LIST OVERVIEW.FMT command.
    return StartCommand (pCtx);
}

/*
 * GetArticleHeaderOverview.
 * Starts XOVER (extension) command.
 */
sal_Bool INetCoreNNTPConnection::GetArticleHeaderOverview (
    sal_uInt32 nFirstArtNum, sal_uInt32 nLastArtNum,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((nFirstArtNum == 0) || (pfnCallback == NULL))
        return sal_False;

    if (nLastArtNum < nFirstArtNum)
        nLastArtNum = nFirstArtNum;

    // Build command line.
    OStringBuffer aBuffer ("XOVER ");
    aBuffer.append (sal_Int32 (nFirstArtNum));
    if (nLastArtNum > nFirstArtNum)
    {
        aBuffer.append (sal_Char ('-'));
        aBuffer.append (sal_Int32 (nLastArtNum));
    }
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPOverListOutputStream (
            this, (INetCoreNNTPOverviewCallback *)pfnCallback, pData),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_OVERVIEW_OK;

    // Start XOVER command.
    return StartCommand (pCtx);
}

/*
 * GetArticleHeader.
 * Starts HEAD (by ArticleNumber) command.
 */
sal_Bool INetCoreNNTPConnection::GetArticleHeader (
    sal_uInt32 nArticleNumber,
    INetCoreMessageOStream& rMessageStream,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((nArticleNumber == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("HEAD ");
    aBuffer.append (sal_Int32 (nArticleNumber));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPMsgDataOutputStream (rMessageStream),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_ARTICLE_HEAD_OK;

    // Start HEAD command.
    return StartCommand (pCtx);
}

/*
 * GetArticleHeader.
 * Starts HEAD (by MessageID) command.
 */
sal_Bool INetCoreNNTPConnection::GetArticleHeader (
    const OUString &rMessageID,
    INetCoreMessageOStream& rMessageStream,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    sal_Int32 nLength = rMessageID.getLength();
    if ((nLength == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("HEAD ");
    if (!(rMessageID.indexOf ('<') == 0))
        aBuffer.append (sal_Char ('<'));
    aBuffer.append (OString (
        rMessageID.pData->buffer,
        rMessageID.pData->length,
        RTL_TEXTENCODING_ASCII_US));
    if (!(rMessageID.lastIndexOf ('>') == (nLength - 1)))
        aBuffer.append (sal_Char ('>'));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPMsgDataOutputStream (rMessageStream),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_ARTICLE_HEAD_OK;

    // Start HEAD command.
    return StartCommand (pCtx);
}

/*
 * RetrieveArticle.
 * Starts ARTICLE (by ArticleNumber) command.
 */
sal_Bool INetCoreNNTPConnection::RetrieveArticle (
    sal_uInt32 nArticleNumber,
    INetCoreMessageOStream& rMessageStream,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((nArticleNumber == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("ARTICLE ");
    aBuffer.append (sal_Int32 (nArticleNumber));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPMsgDataOutputStream (rMessageStream),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_ARTICLE_OK;

    // Start ARTICLE command.
    return StartCommand (pCtx);
}

/*
 * RetrieveArticle.
 * Starts ARTICLE (by MessageID) command.
 */
sal_Bool INetCoreNNTPConnection::RetrieveArticle (
    const OUString &rMessageID,
    INetCoreMessageOStream& rMessageStream,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    sal_Int32 nLength = rMessageID.getLength();
    if ((nLength == 0) || (pfnCallback == NULL))
        return sal_False;

    // Build command line.
    OStringBuffer aBuffer ("ARTICLE ");
    if (!(rMessageID.indexOf ('<') == 0))
        aBuffer.append (sal_Char ('<'));
    aBuffer.append (OString (
        rMessageID.pData->buffer,
        rMessageID.pData->length,
        RTL_TEXTENCODING_ASCII_US));
    if (!(rMessageID.lastIndexOf ('>') == (nLength - 1)))
        aBuffer.append (sal_Char ('>'));
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        aBuffer.makeStringAndClear(),
        new INetCoreNNTPReplyStream(),
        NULL, new INetCoreNNTPMsgDataOutputStream (rMessageStream),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_ARTICLE_OK;

    // Start ARTICLE command.
    return StartCommand (pCtx);
}

/*
 * IsPostingAllowed.
 */
sal_Bool INetCoreNNTPConnection::IsPostingAllowed (void)
{
    return (m_pConCtx ? m_pConCtx->m_bPostingAllowed : sal_False);
}

/*
 * PostArticle.
 * Starts POST command.
 */
sal_Bool INetCoreNNTPConnection::PostArticle (
    INetCoreMessageIStream& rMessageStream,
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Initialize command context.
    INetCoreNNTPCmdContext *pCtx = new INetCoreNNTPCmdContext (
        "POST\015\012",
        new INetCoreNNTPSimpleReplyStream(),
        new INetCoreNNTPMsgDataInputStream (rMessageStream), NULL,
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORENNTP_CMD_POST;
    pCtx->m_nOkCode  = INETCORENNTP_REPLY_START_ARTICLE_POSTING;

    // Start POST command.
    return StartCommand (pCtx);
}

/*
 * GetTransferCount.
 */
sal_uInt32 INetCoreNNTPConnection::GetTransferCount (void)
{
    return (m_pConCtx ? m_pConCtx->m_nXferCount : 0);
}

/*
 * SetTransferCallback.
 */
sal_Bool INetCoreNNTPConnection::SetTransferCallback (
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    if (m_pConCtx && !m_pConCtx->m_bAborting)
    {
        m_pConCtx->m_pfnXferCB = pfnCallback;
        m_pConCtx->m_pXferData = pData;
        return sal_True;
    }
    return sal_False;
}

/*
 * SetTerminateCallback.
 */
sal_Bool INetCoreNNTPConnection::SetTerminateCallback (
    INetCoreNNTPCallback *pfnCallback, void *pData)
{
    if (m_pConCtx && !m_pConCtx->m_bAborting)
    {
        m_pConCtx->m_pfnTermCB = pfnCallback;
        m_pConCtx->m_pTermData = pData;
        return sal_True;
    }
    return sal_False;
}

/*=======================================================================
 *
 * INetCoreNNTPConnectionContext Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreNNTPConnectionContext.
 */
INetCoreNNTPConnectionContext::INetCoreNNTPConnectionContext (
    sal_uInt32 nBufSiz)
    : m_pCmdCtx (NULL),
      m_bIsOpen (sal_False),
      m_bPostingAllowed (sal_False),
      m_bAborting       (sal_False),
      m_pResolver       (new INetCoreDNSResolver()),
      m_aDestAddr       (OUString(), 0),
      m_xSocket         (NULL),
      m_nXferCount      (0),
      m_nBufSiz         (nBufSiz),
      m_pBuffer         ((sal_Char*)(rtl_allocateMemory (m_nBufSiz))),
      m_pRead           (m_pBuffer),
      m_pWrite          (m_pBuffer),
      m_pfnXferCB       (NULL),
      m_pXferData       (NULL),
      m_pfnTermCB       (NULL),
      m_pTermData       (NULL)
{
}

/*
 * ~INetCoreNNTPConnectionContext.
 */
INetCoreNNTPConnectionContext::~INetCoreNNTPConnectionContext (void)
{
    rtl_freeMemory (m_pBuffer);
    if (m_xSocket.isValid())
    {
        m_xSocket->close();
        m_xSocket.unbind();
    }
    delete m_pResolver;
    delete m_pCmdCtx;
}

/*
 * create.
 */
void INetCoreNNTPConnectionContext::create (INetCoreDNSHostEntry &rDstAddr)
{
    // Initialize active socket.
    m_xSocket = new INetActiveTCPSocket();

    // Check for SocksGateway.
    NAMESPACE_VOS(ORef)<INetConfig> xConfig;
    if (INetConfig::getOrCreate (xConfig))
    {
        NAMESPACE_VOS(ORef)<INetProxyPolicy> xProxyPolicy (
            xConfig->getProxyPolicy());
        if (xProxyPolicy.isValid())
        {
            OUStringBuffer aBuffer (OUString::createFromAscii ("news://"));
            if (rDstAddr.GetCanonicalName())
                aBuffer.append (rDstAddr.GetCanonicalName());
            else
                aBuffer.append (rDstAddr.GetDomainName());
            aBuffer.append (sal_Unicode (':'));
            aBuffer.append (rDstAddr.GetPort());
            aBuffer.append (sal_Unicode ('/'));

            OUString        aUrl (aBuffer.makeStringAndClear());
            INetProxyConfig aProxyConfig;

            if (xProxyPolicy->shouldUseProxy (aUrl, aProxyConfig))
            {
                if (aProxyConfig.hasSocksProxy())
                {
                    // Use SocksGateway.
                    m_xSocket->setSocksGateway (
                        NAMESPACE_VOS(OInetSocketAddr)(
                            aProxyConfig.getSocksProxyName(),
                            aProxyConfig.getSocksProxyPort()));
                }
            }
        }
    }
}

/*=======================================================================
 *
 * INetCoreNNTPCmdContext Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreNNTPCmdContext.
 */
INetCoreNNTPCmdContext::INetCoreNNTPCmdContext (
    const OString            &rCommandLine,
    INetCoreNNTPReplyStream  *pReplyStream,
    INetCoreNNTPInputStream  *pSource,
    INetCoreNNTPOutputStream *pTarget,
    INetCoreNNTPCallback     *pfnCallback,
    void                     *pData)
    : m_eState       (INETCORENNTP_CMD_SEND),
      m_nCode        (0),
      m_eOkState     (INETCORENNTP_CMD_SUCCESS),
      m_nOkCode      (0),
      m_aCmdLine     (rCommandLine),
      m_pReplyStream (pReplyStream),
      m_pSource      (pSource),
      m_pTarget      (pTarget),
      m_pfnCB        (pfnCallback),
      m_pDataCB      (pData)
{
}

/*
 * ~INetCoreNNTPCmdContext.
 */
INetCoreNNTPCmdContext::~INetCoreNNTPCmdContext (void)
{
    delete m_pReplyStream;
    delete m_pSource;
    delete m_pTarget;
}

/*========================================================================
 *
 * INetCoreNNTPReplyStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreNNTPReplyStream.
 */
INetCoreNNTPReplyStream::INetCoreNNTPReplyStream (void)
    : m_eState       (INETCORENNTP_EOL_SCR),
      m_bTransparent (sal_False),
      m_nReplyCode   (0)
{
}

/*
 * ~INetCoreNNTPReplyStream.
 */
INetCoreNNTPReplyStream::~INetCoreNNTPReplyStream (void)
{
}

/*
 * PutData.
 */
int INetCoreNNTPReplyStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    const sal_Char *pStop = pData + nSize;
    while (!m_bTransparent && (pData < pStop))
    {
        /*
         * Non-transparent. Extract status line.
         */
        if (m_eState == INETCORENNTP_EOL_FCR)
        {
            if (*pData == '\012')
            {
                // Found <CR><LF>.
                m_aBuffer << sal_Char('\0');
                int status = ParseStatus (pCtx);
                if (status != INETCORESTREAM_STATUS_OK) return status;

                m_aBuffer.rewind();
                m_bTransparent = sal_True;
                m_eState = INETCORENNTP_EOL_SCR;
            }
            else
            {
                // Found <CR> only.
                m_aBuffer << sal_Char('\0');
                int status = ParseStatus (pCtx);
                if (status != INETCORESTREAM_STATUS_OK) return status;

                m_aBuffer.rewind();
                m_bTransparent = sal_True;
                m_eState = INETCORENNTP_EOL_SCR;
            }
        }
        else if (*pData == '\015')
        {
            // Found <CR>.
            m_eState = INETCORENNTP_EOL_FCR;
        }
        else if (*pData == '\012')
        {
            // Found <LF> only.
            m_aBuffer << sal_Char('\0');
            int status = ParseStatus (pCtx);
            if (status != INETCORESTREAM_STATUS_OK) return status;

            m_aBuffer.rewind();
            m_bTransparent = sal_True;
            m_eState = INETCORENNTP_EOL_SCR;
        }
        else
        {
            m_aBuffer << *pData;
        }
        pData++;
    }

    INetCoreNNTPConnectionContext *ctx =
        (INetCoreNNTPConnectionContext *)pCtx;
    if (ctx && ctx->m_pCmdCtx && ctx->m_pCmdCtx->m_pTarget)
    {
        nSize = pStop - pData;
        if (nSize > 0)
        {
            // Put multi-line reply down to output stream.
            return ctx->m_pCmdCtx->m_pTarget->Write (pData, nSize, pCtx);
        }
        else
        {
            // Wait for multi-line reply.
            return INETCORESTREAM_STATUS_OK;
        }
    }
    else
    {
        nSize = pStop - pData;
        if (nSize > 0)
        {
            // Should not happen.
            return INETCORESTREAM_STATUS_WOULDBLOCK;
        }
        else
        {
            // Reset and wait for next reply.
            m_bTransparent = sal_False;
            m_eState = INETCORENNTP_EOL_SCR;
            return INETCORESTREAM_STATUS_OK;
        }
    }
}

/*
 * ParseStatus.
 */
int INetCoreNNTPReplyStream::ParseStatus (void *pCtx)
{
    int status = INETCORESTREAM_STATUS_OK;
    const sal_Char *pBuffer = m_aBuffer.getBuffer();
    if (ascii_isDigit (*pBuffer))
    {
        // Reply code.
        m_nReplyCode = atoi (pBuffer);

        if ((m_nReplyCode/100 == 4) || (m_nReplyCode/100 == 5))
            status = INETCORESTREAM_STATUS_LOADED;

        // Reply text.
        m_aReplyText = (const sal_Char*)(pBuffer + 3);
    }
    else
    {
        // Status line w/o status code -> something went wrong.
        status = INETCORESTREAM_STATUS_ERROR;
    }

    // Done.
    return status;
}

/*=======================================================================
 *
 * INetCoreNNTPConnectReplyStream Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreNNTPConnectReplyStream.
 */
INetCoreNNTPConnectReplyStream::INetCoreNNTPConnectReplyStream (void)
    : INetCoreNNTPReplyStream()
{
}

/*
 * ~INetCoreNNTPConnectReplyStream.
 */
INetCoreNNTPConnectReplyStream::~INetCoreNNTPConnectReplyStream (void)
{
}

/*
 * PutData.
 */
int INetCoreNNTPConnectReplyStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    int status = INetCoreNNTPReplyStream::PutData (pData, nSize, pCtx);
    if (status != INETCORESTREAM_STATUS_ERROR)
    {
        if (pCtx && GetReplyCode())
        {
            INetCoreNNTPConnectionContext *ctx =
                (INetCoreNNTPConnectionContext *)pCtx;
            int nCode = GetReplyCode();

            ctx->m_bIsOpen = (
                (nCode == INETCORENNTP_REPLY_SERVICE_READY_POSTING) ||
                (nCode == INETCORENNTP_REPLY_SERVICE_READY        )    );
            ctx->m_bPostingAllowed =
                (nCode == INETCORENNTP_REPLY_SERVICE_READY_POSTING);

            status = INETCORESTREAM_STATUS_LOADED;
        }
    }
    return status;
}

/*========================================================================
 *
 * INetCoreNNTPSimpleReplyStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreNNTPSimpleReplyStream.
 */
INetCoreNNTPSimpleReplyStream::INetCoreNNTPSimpleReplyStream (void)
    : INetCoreNNTPReplyStream()
{
}

/*
 * ~INetCoreNNTPSimpleReplyStream.
 */
INetCoreNNTPSimpleReplyStream::~INetCoreNNTPSimpleReplyStream (void)
{
}

/*
 * PutData.
 */
int INetCoreNNTPSimpleReplyStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    int status = INetCoreNNTPReplyStream::PutData (pData, nSize, pCtx);
    if ((status == INETCORESTREAM_STATUS_OK) && GetReplyCode())
        status = INETCORESTREAM_STATUS_LOADED;
    return status;
}

/*========================================================================
 *
 * INetCoreNNTPInputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreNNTPInputStream.
 */
INetCoreNNTPInputStream::INetCoreNNTPInputStream (sal_uInt32 nBufSiz)
    : INetCoreIStream()
{
    m_nBufSiz = nBufSiz;
    m_pBuffer = (sal_Char*)(rtl_allocateMemory (m_nBufSiz));
    m_pRead = m_pWrite = m_pBuffer;

    m_eState = INETCORENNTP_EOL_SCR;
    m_bEndOfMessage = sal_False;
}

/*
 * ~INetCoreNNTPInputStream.
 */
INetCoreNNTPInputStream::~INetCoreNNTPInputStream (void)
{
    rtl_freeMemory (m_pBuffer);
}

/*
 * GetData.
 */
int INetCoreNNTPInputStream::GetData (
    sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    sal_Char *pWrite = pData;
    while (pWrite < (pData + nSize))
    {
        // Caller's buffer not yet filled.
        if ((m_pRead - m_pWrite) > 0)
        {
            // Bytes still in buffer.
            if (m_eState == INETCORENNTP_EOL_BEGIN)
            {
                m_eState = INETCORENNTP_EOL_SCR;
                if ((*m_pWrite == '.') && !m_bEndOfMessage)
                {
                    /*
                     * A period as the first character must be doubled,
                     * except at end of message, of course.
                     */
                    *pWrite++ = '.';
                }
            }
            else if (m_eState == INETCORENNTP_EOL_FCR)
            {
                m_eState = INETCORENNTP_EOL_BEGIN;
                if (*m_pWrite != '\012')
                {
                    // Found <CR> only. Insert missing <LF>.
                    *pWrite++ = '\012';
                }
                else
                {
                    // Found standard <CR><LF> sequence.
                    *pWrite++ = *m_pWrite++;
                }
            }
            else if (*m_pWrite == '\015')
            {
                // Found <CR>.
                m_eState = INETCORENNTP_EOL_FCR;
                *pWrite++ = *m_pWrite++;
            }
            else if (*m_pWrite == '\012')
            {
                // Found <LF> only. Insert missing <CR>.
                m_eState = INETCORENNTP_EOL_FCR;
                *pWrite++ = '\015';
            }
            else
            {
                *pWrite++ = *m_pWrite++;
            }
        }
        else
        {
            // Buffer empty. Reset to <Begin-of-Buffer>.
            m_pRead = m_pWrite = m_pBuffer;

            // Read next message line.
            int nRead = GetLine (m_pBuffer, m_nBufSiz, pCtx);
            if (nRead > 0)
            {
                // Set read pointer.
                m_pRead = m_pBuffer + nRead;
            }
            else
            {
                if (!m_bEndOfMessage)
                {
                    // Generate <End-of-Message> sequence.
                    if (!(m_eState == INETCORENNTP_EOL_BEGIN))
                    {
                        *m_pRead++ = '\015';
                        *m_pRead++ = '\012';
                    }
                    *m_pRead++ = '.';
                    *m_pRead++ = '\015';
                    *m_pRead++ = '\012';

                    // Mark we're done.
                    m_bEndOfMessage = sal_True;
                }
                else
                {
                    // Done.
                    return (pWrite - pData);
                }
            }
        }
    }
    return (pWrite - pData);
}

/*========================================================================
 *
 * INetCoreNNTPMsgDataInputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreNNTPMsgDataInputStream.
 */
INetCoreNNTPMsgDataInputStream::INetCoreNNTPMsgDataInputStream (
    INetCoreMessageIStream &rMsgStrm)
    : INetCoreNNTPInputStream(),
      m_rMsgStrm (rMsgStrm)
{
}

/*
 * ~INetCoreNNTPMsgDataInputStream.
 */
INetCoreNNTPMsgDataInputStream::~INetCoreNNTPMsgDataInputStream (void)
{
}

/*
 * GetLine.
 */
int INetCoreNNTPMsgDataInputStream::GetLine (
    sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    INetCoreNNTPConnectionContext *ctx =
        (INetCoreNNTPConnectionContext *)pCtx;
    if ((ctx == NULL) || (ctx->m_bAborting))
        return INETCORESTREAM_STATUS_ERROR;

    return m_rMsgStrm.Read (pData, nSize, NULL);
}

/*=======================================================================
 *
 * INetCoreNNTPOutputStream Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreNNTPOutputStream.
 */
INetCoreNNTPOutputStream::INetCoreNNTPOutputStream (void)
    : m_eState (INETCORENNTP_EOL_SCR)
{
}

/*
 * ~INetCoreNNTPOutputStream.
 */
INetCoreNNTPOutputStream::~INetCoreNNTPOutputStream (void)
{
}

/*
 * PutData.
 */
int INetCoreNNTPOutputStream::PutData (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    INetCoreNNTPConnectionContext *ctx =
        (INetCoreNNTPConnectionContext *)pCtx;
    if ((ctx == NULL) || (ctx->m_bAborting))
        return INETCORESTREAM_STATUS_ERROR;

    while (nSize--)
    {
        if (m_eState == INETCORENNTP_EOL_FCR)
        {
            if (*pData == '\012')
                m_eState = INETCORENNTP_EOL_BEGIN;
            else
                m_eState = INETCORENNTP_EOL_SCR;
        }
        else if (*pData == '\015')
        {
            m_eState = INETCORENNTP_EOL_FCR;
        }
        else
        {
            if (*pData == '\012')
            {
                m_aBuffer << sal_Char('\015');
                m_eState = INETCORENNTP_EOL_BEGIN;
            }
        }

        m_aBuffer << *pData++;

        if (m_eState == INETCORENNTP_EOL_BEGIN)
        {
            if (*m_aBuffer.getBuffer() == '.')
            {
                if (*(m_aBuffer.getBuffer() + 1) == '\015')
                {
                    return INETCORESTREAM_STATUS_LOADED;
                }
                else
                {
                    int status = PutLine (
                        (m_aBuffer.getBuffer() + 1),
                        (m_aBuffer.getLength() - 1), pCtx);
                    if (status != INETCORESTREAM_STATUS_OK)
                        return status;
                }
            }
            else
            {
                int status = PutLine (
                    m_aBuffer.getBuffer(), m_aBuffer.getLength(), pCtx);
                if (status != INETCORESTREAM_STATUS_OK)
                    return status;
            }
            m_aBuffer.rewind();
            m_eState = INETCORENNTP_EOL_SCR;
        }
    }
    return INETCORESTREAM_STATUS_OK;
}

/*=========================================================================
 *
 * INetCoreNNTPGroupListOutputStream Implementation.
 *
 *=======================================================================*/
/*
 * INetCoreNNTPGroupListOutputStream.
 */
INetCoreNNTPGroupListOutputStream::INetCoreNNTPGroupListOutputStream (
    List& rOutList)
    : INetCoreNNTPOutputStream(),
      m_rOutList (rOutList)
{
}

/*
 * ~INetCoreNNTPGroupListOutputStream.
 */
INetCoreNNTPGroupListOutputStream::~INetCoreNNTPGroupListOutputStream (void)
{
}

/*
 * PutLine.
 */
int INetCoreNNTPGroupListOutputStream::PutLine (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    // Create ListEntry.
    INetCoreNNTPGroupListEntry *entry = new INetCoreNNTPGroupListEntry;

    // Parse line into "GROUP LAST FIRST FLAGS".
    sal_Char *p1, *p2;
    p1 = p2 = (sal_Char *)pData;

    while (*p2 && !ascii_isWhitespace (*p2)) p2++;
    *p2 = '\0';
    entry->m_aGroupName = OUString (p1, (p2 - p1), RTL_TEXTENCODING_UTF8);
    *p2 = ' ';

    p1 = p2;
    while (*p2 && ascii_isWhitespace (*p2)) p2++;
    p1 = p2;
    while (*p2 && ascii_isDigit (*p2)) p2++;
    *p2 = '\0';
    entry->m_nLastArticleNumber = atol (p1);
    *p2 = ' ';

    p1 = p2;
    while (*p2 && ascii_isWhitespace (*p2)) p2++;
    p1 = p2;
    while (*p2 && ascii_isDigit (*p2)) p2++;
    *p2 = '\0';
    entry->m_nFirstArticleNumber = atol (p1);
    *p2 = ' ';

    entry->m_nArticleCount =
        (entry->m_nLastArticleNumber + 1) - entry->m_nFirstArticleNumber;

    p1 = p2;
    while (*p2 && ascii_isWhitespace (*p2)) *p2++;
    p1 = p2;
    entry->m_nFlags = 0;
    switch (ascii_toLowerCase (*p1))
    {
        case 'm':
            entry->m_nFlags  = INETCORENNTP_GROUP_FLAG_MODERATED;
            // Fall trough.

        case 'y':
            entry->m_nFlags |= INETCORENNTP_GROUP_FLAG_POSTALLOWED;
            break;

        default:
            entry->m_nFlags = 0;
            break;
    }

    // Append to List.
    m_rOutList.Insert (entry, LIST_APPEND);

    // Done.
    return INETCORESTREAM_STATUS_OK;
}

/*========================================================================
 *
 * INetCoreNNTPMsgListOutputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreNNTPMsgListOutputStream.
 */
INetCoreNNTPMsgListOutputStream::INetCoreNNTPMsgListOutputStream (
    List& rOutList)
    : INetCoreNNTPOutputStream(),
      m_rOutList (rOutList)
{
}

/*
 * ~INetCoreNNTPMsgListOutputStream.
 */
INetCoreNNTPMsgListOutputStream::~INetCoreNNTPMsgListOutputStream (void)
{
}

/*
 * PutLine.
 */
int INetCoreNNTPMsgListOutputStream::PutLine (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    // Parse line into "<message-id>" string.
    sal_Char *p1, *p2;
    p1 = p2 = (sal_Char *)pData;

    while (*p2 && (*p2 != '<')) p2++;
    p1 = p2;
    while (*p2 && (*p2 != '>')) p2++;
    p2++;
    *p2 = '\0';

    // Append NewNewsListEntry to List.
    OUString *entry = new OUString (p1, (p2 - p1), RTL_TEXTENCODING_ASCII_US);
    m_rOutList.Insert (entry, LIST_APPEND);

    // Done.
    return INETCORESTREAM_STATUS_OK;
}

/*========================================================================
 *
 * INetCoreNNTPOverFmtOutputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreNNTPOverFmtOutputStream.
 */
INetCoreNNTPOverFmtOutputStream::INetCoreNNTPOverFmtOutputStream (
    List& rOutList)
    : INetCoreNNTPOutputStream(),
      m_rOutList (rOutList)
{
}

/*
 * ~INetCoreNNTPOverFmtOutputStream.
 */
INetCoreNNTPOverFmtOutputStream::~INetCoreNNTPOverFmtOutputStream (void)
{
}

/*
 * PutLine.
 */
int INetCoreNNTPOverFmtOutputStream::PutLine (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    // Check connection context.
    INetCoreNNTPConnectionContext *ctx =
        (INetCoreNNTPConnectionContext *)pCtx;
    if ((ctx == NULL) || (ctx->m_bAborting))
        return INETCORESTREAM_STATUS_ERROR;

    // Create ListEntry (strip off trailing <CRLF>).
    OString *entry = new OString (pData, sal_Int32 (nSize - 2));

    // Append to List.
    m_rOutList.Insert (entry, LIST_APPEND);

    // Done.
    return INETCORESTREAM_STATUS_OK;
}

/*========================================================================
 *
 * INetCoreNNTPOverListOutputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreNNTPOverListOutputStream.
 */
INetCoreNNTPOverListOutputStream::INetCoreNNTPOverListOutputStream (
    INetCoreNNTPConnection       *pConnection,
    INetCoreNNTPOverviewCallback *pfnCallback,
    void                         *pData)
    : INetCoreNNTPOutputStream()
{
    m_pConnection = pConnection;
    m_pfnCB   = pfnCallback;
    m_pDataCB = pData;
}

/*
 * ~INetCoreNNTPOverListOutputStream.
 */
INetCoreNNTPOverListOutputStream::~INetCoreNNTPOverListOutputStream (void)
{
}

/*
 * PutLine.
 */
int INetCoreNNTPOverListOutputStream::PutLine (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    // Check connection context.
    INetCoreNNTPConnectionContext *ctx =
        (INetCoreNNTPConnectionContext *)pCtx;
    if ((ctx == NULL) || (ctx->m_bAborting))
        return INETCORESTREAM_STATUS_ERROR;

    // Parse into "ARTICLENUMBER OVERVIEWHEADERS".
    sal_uInt32 nArticle;
    sal_Char  *p1, *p2;

    p1 = p2 = SAL_CONST_CAST(sal_Char*, pData);
    while (*p2 && ascii_isWhitespace (*p2)) p2++;
    p1 = p2;
    while (*p2 && ascii_isDigit (*p2)) p2++;
    *p2 = '\0';
    nArticle = atol (p1);
    p1 = ++p2;

    // Setup overview entry (strip off trailing <CRLF>).
    INetCoreNNTPOverviewEntry entry (
        nArticle, p1, sal_Int32 (nSize - 2 - (p1 - pData)));

    // Notify caller.
    if (m_pfnCB) (m_pfnCB) (
        m_pConnection, INETCORENNTP_REPLY_TRANSFER_OVERVIEW,
        &entry, m_pDataCB);

    // Done.
    return INETCORESTREAM_STATUS_OK;
}

/*=======================================================================
 *
 * INetCoreNNTPMsgDataOutputStream Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreNNTPMsgDataOutputStream.
 */
INetCoreNNTPMsgDataOutputStream::INetCoreNNTPMsgDataOutputStream (
    INetCoreMessageOStream &rMsgStrm)
    : INetCoreNNTPOutputStream(),
      m_rMsgStrm (rMsgStrm)
{
}

/*
 * ~INetCoreNNTPMsgDataOutputStream.
 */
INetCoreNNTPMsgDataOutputStream::~INetCoreNNTPMsgDataOutputStream (void)
{
}

/*
 * PutLine.
 */
int INetCoreNNTPMsgDataOutputStream::PutLine (
    const sal_Char *pData, sal_uInt32 nSize, void *pCtx)
{
    INetCoreNNTPConnectionContext *ctx =
        (INetCoreNNTPConnectionContext *)pCtx;
    if ((ctx == NULL) || (ctx->m_bAborting))
        return INETCORESTREAM_STATUS_ERROR;

    return m_rMsgStrm.Write (pData, nSize, NULL);
}

