/*************************************************************************
 *
 *  $RCSfile: inettcp.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: mhu $ $Date: 2001/03/19 11:27:12 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INETTCP_CXX "$Revision: 1.2 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_USTRING_HXX_
#include <rtl/ustring.hxx>
#endif
#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif

#ifndef _VOS_REF_HXX_
#include <vos/ref.hxx>
#endif

#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif

#ifndef _INET_MACROS_HXX
#include <inet/macros.hxx>
#endif
#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif

#ifndef _INET_CONFIG_HXX
#include <inetcfg.hxx>
#endif
#ifndef _INETDNS_HXX
#include <inetdns.hxx>
#endif
#ifndef _INETTCP_HXX
#include <inettcp.hxx>
#endif

#ifdef _USE_NAMESPACE
using namespace inet;
#endif

using rtl::OUString;

#define INETCORETCP_SOCKET_DISPOSE(socket) \
if ((socket).isValid()) \
{ \
	(socket)->deregisterEventHandler(onSocketEvent); \
	(socket)->close(); \
	(socket).unbind(); \
}

#define INETCORETCP_SOCKET_WOULDBLOCK (-osl_Socket_E_WouldBlock)

/*=======================================================================
 *
 * INetCoreTCPRecvContext Interface.
 *
 *=====================================================================*/
struct INetCoreTCPRecvContext
{
	INetCoreTCPRecvContext (
		INetCoreTCPConnectionContext &rMasterCtx,
		SvLockBytes                  &rResLB);
	~INetCoreTCPRecvContext (void);

	SvLockBytesRef       m_xLB;
	sal_uInt32           m_nBufSiz;
	sal_Char            *m_pBuffer;

	INetCoreTCPCallback *m_pfnCB;
	void                *m_pDataCB;
};

/*=======================================================================
 *
 * INetCoreTCPSendContext Interface.
 *
 *=====================================================================*/
struct INetCoreTCPSendContext
{
	INetCoreTCPSendContext (
		INetCoreTCPConnectionContext &rMasterCtx,
		SvLockBytes                  &rReqLB);
	~INetCoreTCPSendContext (void);

	SvLockBytesRef       m_xLB;
	SvStream            *m_pStrm;

	sal_uInt32           m_nBufSiz;
	sal_Char            *m_pBuffer;
	sal_Char            *m_pRead;
	sal_Char            *m_pWrite;

	INetCoreTCPCallback *m_pfnCB;
	void                *m_pDataCB;
};

/*=======================================================================
 *
 * INetCoreTCPConnectionContext Interface.
 *
 *=====================================================================*/
struct INetCoreTCPConnectionContext
{
	sal_Bool                 m_bIsOpen;
	sal_Bool                 m_bAborting;

	INetCoreDNSResolver     *m_pResolver;
	INetCoreDNSHostEntry     m_aDestAddr;
	NAMESPACE_VOS(ORef)<INetActiveTCPSocket> m_xSocket;
	OUString                 m_aScheme;

	INetCoreTCPRecvContext  *m_pRecvCtx;
	INetCoreTCPSendContext  *m_pSendCtx;
	sal_uInt32               m_nRecvCount;
	sal_uInt32               m_nSendCount;

	sal_uInt32               m_nBufSiz;
	sal_Char                *m_pRecvBuf;
	sal_Char                *m_pSendBuf;

	INetCoreTCPCallback     *m_pfnConnCB;
	void                    *m_pDataConnCB;
	INetCoreTCPCallback     *m_pfnTermCB;
	void                    *m_pDataTermCB;

	INetCoreTCPConnectionContext (sal_uInt32 nBufferSize = 4096);
	~INetCoreTCPConnectionContext (void);

	void create (INetCoreDNSHostEntry &rDstAddr);
};

/*=======================================================================
 *
 * INetCoreTCPConnection Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreTCPConnection.
 */
INetCoreTCPConnection::INetCoreTCPConnection (sal_uInt32 nBufferSize)
	: m_pConCtx (new INetCoreTCPConnectionContext (nBufferSize))
{
}

/*
 * ~INetCoreTCPConnection.
 */
INetCoreTCPConnection::~INetCoreTCPConnection (void)
{
	Abort();
	delete m_pConCtx;
}

/*
 * handleResolverEvent.
 */
sal_Bool INetCoreTCPConnection::handleResolverEvent (
	sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry)
{
	// Check context.
	if (!m_pConCtx)
		return 0;

	// Check for abort during name resolution.
	if (m_pConCtx->m_bAborting)
		nStatus = INETCOREDNS_RESOLVER_ERROR;

	// Obtain connect callback.
	INetCoreTCPCallback *pfnCB = m_pConCtx->m_pfnConnCB;
	void                *pData = m_pConCtx->m_pDataConnCB;

	switch (nStatus)
	{
		case INETCOREDNS_RESOLVER_START:
			if (pfnCB) (pfnCB) (
				this, INETCORETCP_STATUS_NAMERES_WAIT, pData);
			break;

		case INETCOREDNS_RESOLVER_SUCCESS:
		case INETCOREDNS_RESOLVER_EXPIRED:
			// Notify caller.
			if (pfnCB) (pfnCB) (
				this, INETCORETCP_STATUS_NAMERES_DONE, pData);

			// Initialize active socket.
			m_pConCtx->create (m_pConCtx->m_aDestAddr);
			m_pConCtx->m_xSocket->registerEventHandler (onSocketEvent, this);

			// Initiate connect.
			if (m_pConCtx->m_xSocket->connect (
				NAMESPACE_VOS(OInetSocketAddr)(
					m_pConCtx->m_aDestAddr.GetDottedDecimalName(),
					m_pConCtx->m_aDestAddr.GetPort())))
			{
				// Ok. Wait for connect event.
				if (pfnCB) (pfnCB) (
					this, INETCORETCP_STATUS_CONNECT_WAIT, pData);
			}
			else
			{
				// Failure. Cleanup and notify caller.
				m_pConCtx->m_xSocket.unbind();

				m_pConCtx->m_pfnConnCB = NULL;
				m_pConCtx->m_pDataConnCB = NULL;

				if (pfnCB) (pfnCB) (
					this, INETCORETCP_STATUS_CONNECT_ERROR, pData);
			}
			break;

		default:
			// Failure. Cleanup and notify caller.
			m_pConCtx->m_pfnConnCB = NULL;
			m_pConCtx->m_pDataConnCB = NULL;

			if (pfnCB) (pfnCB) (
				this, INETCORETCP_STATUS_NAMERES_ERROR, pData);
			break;
	}
	return 1;
}

/*
 * handleSocketEvent.
 * (Implements TCP transaction as Finite State Machine).
 */
sal_Bool INetCoreTCPConnection::handleSocketEvent (
	const NAMESPACE_VOS(ORef)<socket_type> &rxSocket, sal_Int32 nEvent)
{
	// Check context.
	if (!m_pConCtx)
		return 0;

	// Check for close event.
	if (nEvent & socket_type::EVENT_CLOSE)
	{
		// Cleanup and notify caller.
		m_pConCtx->m_bIsOpen = sal_False;
		m_pConCtx->m_xSocket.unbind();

		INetCoreTCPCallback *pfnCB   = NULL;
		void                *pDataCB = NULL;

		if (m_pConCtx->m_pRecvCtx)
		{
			// Recv in progress.
			pfnCB   = m_pConCtx->m_pRecvCtx->m_pfnCB;
			pDataCB = m_pConCtx->m_pRecvCtx->m_pDataCB;

			DELETEZ (m_pConCtx->m_pRecvCtx);

			if (pfnCB) (pfnCB) (
				this, INETCORETCP_STATUS_NETWORK_ERROR, pDataCB);
		}

		if (m_pConCtx->m_pSendCtx)
		{
			// Send in progress.
			pfnCB   = m_pConCtx->m_pSendCtx->m_pfnCB;
			pDataCB = m_pConCtx->m_pSendCtx->m_pDataCB;

			DELETEZ (m_pConCtx->m_pSendCtx);

			if (pfnCB) (pfnCB) (
				this, INETCORETCP_STATUS_NETWORK_ERROR, pDataCB);
		}

		if (m_pConCtx->m_pfnTermCB) (m_pConCtx->m_pfnTermCB) (
			this, INETCORETCP_STATUS_NETWORK_ERROR, m_pConCtx->m_pDataTermCB);

		// Done.
		return 0;
	}

	// Process event.
	while (1)
	{
		if ((nEvent & socket_type::EVENT_CONNECT) && (m_pConCtx->m_pfnConnCB))
		{
			// Set connected status.
			if (nEvent & socket_type::EVENT_OOB)
			{
				// Connect error. Cleanup.
				m_pConCtx->m_xSocket.unbind();
			}
			m_pConCtx->m_bIsOpen = m_pConCtx->m_xSocket.isValid();

			// Notify caller.
			INetCoreTCPCallback *pfnCB   = m_pConCtx->m_pfnConnCB;
			void                *pDataCB = m_pConCtx->m_pDataConnCB;
			m_pConCtx->m_pfnConnCB   = NULL;
			m_pConCtx->m_pDataConnCB = NULL;

			if (m_pConCtx->m_bIsOpen)
				(pfnCB) (this, INETCORETCP_STATUS_CONNECT_DONE, pDataCB);
			else
				(pfnCB) (this, INETCORETCP_STATUS_CONNECT_ERROR, pDataCB);

			return 1;
		}
		else if ((nEvent & socket_type::EVENT_READ) && (m_pConCtx->m_pRecvCtx))
		{
			INetCoreTCPRecvContext *pCtx = m_pConCtx->m_pRecvCtx;

			INetCoreTCPCallback *pfnCB   = pCtx->m_pfnCB;
			void                *pDataCB = pCtx->m_pDataCB;

			sal_Int32 nRead = rxSocket->recv (
				pCtx->m_pBuffer, pCtx->m_nBufSiz);
			if (nRead > 0)
			{
				// Have read some bytes. Write out to LockBytes.
				ULONG nCount = m_pConCtx->m_nRecvCount;
				ULONG nWrite = 0;

				m_pConCtx->m_nRecvCount += nRead;

				pCtx->m_xLB->WriteAt (nCount, pCtx->m_pBuffer, nRead, &nWrite);
				if (nWrite < (ULONG)nRead)
				{
					// LockBytes write error. Cleanup and fail.
					pCtx->m_xLB->Flush();

					delete pCtx;
					m_pConCtx->m_pRecvCtx = NULL;

					if (pfnCB) (pfnCB) (
						this, INETCORETCP_STATUS_RECV_ERROR, pDataCB);
					return 1;
				}
			}
			else if (nRead == INETCORETCP_SOCKET_WOULDBLOCK)
			{
				// Socket would block.
				pCtx->m_xLB->Flush();
				if (pfnCB) (pfnCB) (
					this, INETCORETCP_STATUS_RECV_WAIT, pDataCB);
				return 1;
			}
			else
			{
				// Socket read error. Shutdown and wait for close event.
				pCtx->m_xLB->Flush();
				rxSocket->close();
				return 1;
			}
		}
		else if ((nEvent & socket_type::EVENT_WRITE) && (m_pConCtx->m_pSendCtx))
		{
			INetCoreTCPSendContext *pCtx = m_pConCtx->m_pSendCtx;

			INetCoreTCPCallback *pfnCB   = pCtx->m_pfnCB;
			void                *pDataCB = pCtx->m_pDataCB;

			if ((pCtx->m_pRead - pCtx->m_pWrite) > 0)
			{
				// Bytes still in buffer. Write out to socket.
				sal_Int32 nWrite = rxSocket->send (
					pCtx->m_pWrite, (pCtx->m_pRead - pCtx->m_pWrite));
				if (nWrite > 0)
				{
					// Have written some bytes.
					m_pConCtx->m_nSendCount += nWrite;

					pCtx->m_pWrite += nWrite;
					if (pCtx->m_pWrite == pCtx->m_pRead)
						pCtx->m_pRead = pCtx->m_pWrite = pCtx->m_pBuffer;
				}
				else if (nWrite == INETCORETCP_SOCKET_WOULDBLOCK)
				{
					// Socket would block.
					if (pfnCB) (pfnCB) (
						this, INETCORETCP_STATUS_SEND_WAIT, pDataCB);
					return 1;
				}
				else
				{
					// Socket write error. Shutdown and wait for close event.
					rxSocket->close();
					return 1;
				}
			}
			else
			{
				// Buffer empty. Read source stream.
				int nRead = pCtx->m_pStrm->Read (
					pCtx->m_pBuffer, pCtx->m_nBufSiz);
				if (nRead > 0)
				{
					// Have read some bytes.
					pCtx->m_pRead = pCtx->m_pBuffer + nRead;
				}
				else
				{
					// Source stream exhausted. Cleanup and finish.
					delete pCtx;
					m_pConCtx->m_pSendCtx = NULL;

					if (pfnCB) (pfnCB) (
						this, INETCORETCP_STATUS_SEND_DONE, pDataCB);
					return 1;
				}
			}
		}
		else
		{
			// Idle. Check for read event.
			if (nEvent & socket_type::EVENT_READ)
			{
				// Idle handler.
				sal_Int32 nRead = rxSocket->recv (
					m_pConCtx->m_pRecvBuf, m_pConCtx->m_nBufSiz);
				if (nRead > 0)
				{
					// Absorbing whatever comes in.
					continue;
				}
				else if (nRead == INETCORETCP_SOCKET_WOULDBLOCK)
				{
					// Wait for next event.
					return 1;
				}
				else
				{
					/*
					 * Connection closed or Network failure.
					 * Shutdown and wait for close event.
					 */
					rxSocket->close();
					return 1;
				}
			}
			else
			{
				// Ignore event.
				return 1;
			}
		}
	} // while (1)
}

/*
 * Open.
 */
sal_Bool INetCoreTCPConnection::Open (
	const OUString &rScheme, const OUString &rHost, sal_uInt16 nPort,
	INetCoreTCPCallback *pfnCallback, void *pData)
{
	// Ensure clean destruction.
	NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

	// Check connection context.
	if (!m_pConCtx || m_pConCtx->m_bIsOpen)
		return sal_False;

	// Check arguments.
	if (rScheme.getLength() == 0)
		return sal_False;
	if ((rHost.getLength() == 0) || (nPort == 0))
		return sal_False;
	if (pfnCallback == NULL)
		return sal_False;

	// Setup callback.
	m_pConCtx->m_aScheme     = rScheme;
	m_pConCtx->m_pfnConnCB   = pfnCallback;
	m_pConCtx->m_pDataConnCB = pData;

	// Start domain name resolution.
	m_pConCtx->m_aDestAddr = INetCoreDNSHostEntry (rHost, nPort);

	if (m_pConCtx->m_pResolver->GetHostByName (
		&(m_pConCtx->m_aDestAddr), onResolverEvent, this))
	{
		// Ok. Wait for things to come.
		return sal_True;
	}
	else
	{
		// Cleanup and fail.
		m_pConCtx->m_pfnConnCB = NULL;
		m_pConCtx->m_pDataConnCB = NULL;

		return sal_False;
	}
}

/*
 * IsOpen.
 */
sal_Bool INetCoreTCPConnection::IsOpen (void)
{
	return (m_pConCtx ? m_pConCtx->m_bIsOpen : sal_False);
}

/*
 * Abort.
 */
void INetCoreTCPConnection::Abort (void)
{
	if (m_pConCtx)
	{
		m_pConCtx->m_bIsOpen   = sal_False;
		m_pConCtx->m_bAborting = sal_True;

		m_pConCtx->m_pfnConnCB = NULL;
		m_pConCtx->m_pfnTermCB = NULL;

		if (m_pConCtx->m_pRecvCtx)
			m_pConCtx->m_pRecvCtx->m_pfnCB = NULL;
		if (m_pConCtx->m_pSendCtx)
			m_pConCtx->m_pSendCtx->m_pfnCB = NULL;

		INETCORETCP_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
	}
}

/*
 * Recv.
 */
sal_Bool INetCoreTCPConnection::Recv (
	SvLockBytes         &rRecvLB,
	INetCoreTCPCallback *pfnCallback, void *pData)
{
	// Ensure clean destruction.
	NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

	// Check connection context.
	if (!(m_pConCtx && m_pConCtx->m_xSocket.isValid()))
		return sal_False;
	if (!m_pConCtx->m_bIsOpen || m_pConCtx->m_bAborting)
		return sal_False;

	// Check arguments.
	if (pfnCallback == NULL)
		return sal_False;

	// Cleanup previous receive context.
	DELETEZ (m_pConCtx->m_pRecvCtx);
	m_pConCtx->m_nRecvCount = 0;
	
	// Create new receive context.
	m_pConCtx->m_pRecvCtx = new INetCoreTCPRecvContext (*m_pConCtx, rRecvLB);
	m_pConCtx->m_pRecvCtx->m_pfnCB   = pfnCallback;
	m_pConCtx->m_pRecvCtx->m_pDataCB = pData;

	// Try Receive.
	return handleSocketEvent (
		&*(m_pConCtx->m_xSocket), socket_type::EVENT_READ);
}

/*
 * Send.
 */
sal_Bool INetCoreTCPConnection::Send (
	SvLockBytes         &rSendLB,
	INetCoreTCPCallback *pfnCallback, void *pData)
{
	// Ensure clean destruction.
	NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

	// Check connection context.
	if (!(m_pConCtx && m_pConCtx->m_xSocket.isValid()))
		return sal_False;
	if (!m_pConCtx->m_bIsOpen || m_pConCtx->m_bAborting)
		return sal_False;

	// Check arguments.
	if (pfnCallback == NULL)
		return sal_False;

	// Cleanup previous send context.
	DELETEZ (m_pConCtx->m_pSendCtx);
	m_pConCtx->m_nSendCount = 0;
	
	// Create new send context.
	m_pConCtx->m_pSendCtx = new INetCoreTCPSendContext (*m_pConCtx, rSendLB);
	m_pConCtx->m_pSendCtx->m_pfnCB   = pfnCallback;
	m_pConCtx->m_pSendCtx->m_pDataCB = pData;

	// Try Send.
	return handleSocketEvent (
		&*(m_pConCtx->m_xSocket), socket_type::EVENT_WRITE);
}

/*
 * GetRecvCount.
 */
sal_uInt32 INetCoreTCPConnection::GetRecvCount (void)
{
	return (m_pConCtx ? m_pConCtx->m_nRecvCount : 0);
}

/*
 * GetSendCount.
 */
sal_uInt32 INetCoreTCPConnection::GetSendCount (void)
{
	return (m_pConCtx ? m_pConCtx->m_nSendCount : 0);
}

/*
 * SetTerminateCallback.
 */
void INetCoreTCPConnection::SetTerminateCallback (
	INetCoreTCPCallback *pfnCallback, void *pData)
{
	if (m_pConCtx && !m_pConCtx->m_bAborting)
	{
		m_pConCtx->m_pfnTermCB   = pfnCallback;
		m_pConCtx->m_pDataTermCB = pData;
	}
}

/*=======================================================================
 *
 * INetCoreTCPConnectionContext Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreTCPConnectionContext.
 */
INetCoreTCPConnectionContext::INetCoreTCPConnectionContext (sal_uInt32 nBufSiz)
	: m_bIsOpen     (sal_False),
	  m_bAborting   (sal_False),
	  m_pResolver   (new INetCoreDNSResolver()),
	  m_aDestAddr   (OUString(), 0),
	  m_xSocket     (NULL),
	  m_pRecvCtx    (NULL),
	  m_pSendCtx    (NULL),
	  m_nRecvCount  (0),
	  m_nSendCount  (0),
	  m_nBufSiz     (nBufSiz),
	  m_pRecvBuf    ((sal_Char*)(rtl_allocateMemory (m_nBufSiz))),
	  m_pSendBuf    ((sal_Char*)(rtl_allocateMemory (m_nBufSiz))),
	  m_pfnConnCB   (NULL),
	  m_pDataConnCB (NULL),
	  m_pfnTermCB   (NULL),
	  m_pDataTermCB (NULL)
{
}

/*
 * ~INetCoreTCPConnectionContext.
 */
INetCoreTCPConnectionContext::~INetCoreTCPConnectionContext (void)
{
	if (m_xSocket.isValid())
	{
		m_xSocket->close();
		m_xSocket.unbind();
	}
	delete m_pResolver;

	delete m_pRecvCtx;
	delete m_pSendCtx;

	rtl_freeMemory (m_pRecvBuf);
	rtl_freeMemory (m_pSendBuf);
}

/*
 * create.
 */
void INetCoreTCPConnectionContext::create (INetCoreDNSHostEntry &rDstAddr)
{
	// Initialize active socket.
	m_xSocket = new INetActiveTCPSocket();

	// Check for SocksGateway.
	NAMESPACE_VOS(ORef)<INetConfig> xConfig;
	if (INetConfig::getOrCreate (xConfig))
	{
		NAMESPACE_VOS(ORef)<INetProxyPolicy> xProxyPolicy (
			xConfig->getProxyPolicy());
		if (xProxyPolicy.isValid())
		{
			rtl::OUStringBuffer aBuffer (m_aScheme);
			if (rDstAddr.GetCanonicalName())
				aBuffer.append (rDstAddr.GetCanonicalName());
			else
				aBuffer.append (rDstAddr.GetDomainName());
			aBuffer.append (sal_Unicode(':'));
			aBuffer.append (sal_Int32(rDstAddr.GetPort()));
			aBuffer.append (sal_Unicode('/'));

			OUString aUrl (aBuffer.makeStringAndClear());

			INetProxyConfig aProxyConfig;
			if (xProxyPolicy->shouldUseProxy (aUrl, aProxyConfig))
			{
				if (aProxyConfig.hasSocksProxy())
				{
					// Use SocksGateway.
					m_xSocket->setSocksGateway (
						NAMESPACE_VOS(OInetSocketAddr)(
							aProxyConfig.getSocksProxyName(),
							aProxyConfig.getSocksProxyPort()));
				}
			}
		}
	}
}

/*=======================================================================
 *
 * INetCoreTCPRecvContext Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreTCPRecvContext.
 */
INetCoreTCPRecvContext::INetCoreTCPRecvContext (
	INetCoreTCPConnectionContext& rMasterCtx, SvLockBytes& rResLB)
{
	m_nBufSiz = rMasterCtx.m_nBufSiz;
	m_pBuffer = rMasterCtx.m_pRecvBuf;

	m_xLB = &rResLB;

	m_pfnCB   = NULL;
	m_pDataCB = NULL;
}

/*
 * ~INetCoreTCPRecvContext.
 */
INetCoreTCPRecvContext::~INetCoreTCPRecvContext (void)
{
	m_xLB.Clear();
}

/*=======================================================================
 *
 * INetCoreTCPSendContext Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreTCPSendContext.
 */
INetCoreTCPSendContext::INetCoreTCPSendContext (
	INetCoreTCPConnectionContext& rMasterCtx, SvLockBytes& rReqLB)
{
	m_nBufSiz = rMasterCtx.m_nBufSiz;
	m_pBuffer = rMasterCtx.m_pSendBuf;
	m_pRead = m_pWrite = m_pBuffer;

	m_xLB   = &rReqLB;
	m_pStrm = new SvStream (&m_xLB);

	m_pfnCB   = NULL;
	m_pDataCB = NULL;
}

/*
 * INetCoreTCPSendContext.
 */
INetCoreTCPSendContext::~INetCoreTCPSendContext (void)
{
	delete m_pStrm;
}

