/*************************************************************************
 *
 *  $RCSfile: socks4.c,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: mhu $ $Date: 2001/05/04 09:42:04 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INET_SOCKS4_C_ "$Revision: 1.3 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif
#ifndef _OSL_SOCKET_H_
#include <osl/socket.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _OSL_ENDIAN_H_
#include <osl/endian.h>
#endif

#ifndef _INET_SOCKS4_H_
#include <inet/socks4.h>
#endif

/*
 * __OSL_NETWORD.
 */
#ifdef OSL_BIGENDIAN
#define __OSL_NETWORD(x) (x)
#else
#define __OSL_NETWORD(x) ((((x) >> 8) && 0x00ff) | (((x) & 0x00ff) << 8))
#endif

/*
 * in_addr_type.
 */
typedef struct in_addr_impl
{
	sal_uInt32 s_addr;
} in_addr_type;

/*
 * sock_addr_type.
 */
typedef struct sockaddr_in_impl
{
	sal_Int16    sin_family;
	sal_uInt16   sin_port;
	in_addr_type sin_addr;
	sal_Char     sin_zero[8];
} sock_addr_type;

/*
 * message_type.
 */
typedef struct socks_message_impl
{
	sal_uInt8  m_nVersion;
	sal_uInt8  m_nCode;
	sal_uInt16 m_nPort;
	sal_uInt32 m_nHost;
} message_type;

/*
 * state_type.
 */
typedef enum socks_state_impl
{
	STATE_INIT,
	STATE_CONNECT,
	STATE_REQUEST,
	STATE_RESPONSE,
	STATE_DONE,
	STATE_FORCE_EQUAL_SIZE = SAL_MAX_ENUM
} state_type;

/*
 * context_type.
 */
typedef struct socks_context_impl
{
	oslSocket     m_hSocket;
	oslSocketAddr m_hGwAddr;

	state_type    m_eState;
	sal_Char     *m_pBuffer;
	union
	{
		message_type m_aMessage;
		sal_Char     m_pBuffer[sizeof(message_type)];
	} bind;
	union
	{
		message_type m_aMessage;
		sal_Char     m_pBuffer[sizeof(message_type) + 6];
	} dest;
} context_type;

/*
 * __osl_getInetAddrOfSocketAddr.
 */
static sal_uInt32 __osl_getInetAddrOfSocketAddr (oslSocketAddr hAddr)
{
	if (osl_getFamilyOfSocketAddr(hAddr) == osl_Socket_FamilyInet)
	{
		sock_addr_type *pAddr = (sock_addr_type*)hAddr;
		return (pAddr->sin_addr.s_addr);
	}
	return OSL_INADDR_NONE;
}

/*
 * __osl_socks_getSocketAddr.
 */
static oslSocketAddr __osl_socks_getSocketAddr (message_type *pImpl)
{
	sock_addr_type *pAddr;
	pAddr = (sock_addr_type*)(rtl_allocateMemory (sizeof (sock_addr_type)));
	if (pAddr)
	{
		pAddr->sin_family      = 2; /* AF_INET */
		pAddr->sin_port        = pImpl->m_nPort;
		pAddr->sin_addr.s_addr = pImpl->m_nHost;
	}
	return ((oslSocketAddr)pAddr);
}

/*
 * __osl_socks_setSocketAddr.
 */
static sal_Bool __osl_socks_setSocketAddr (
	message_type *pImpl, oslSocketAddr hAddr)
{
	sal_Int32 nPort = osl_getInetPortOfSocketAddr (hAddr);
	if (!(nPort == OSL_INVALID_PORT))
	{
		pImpl->m_nHost = __osl_getInetAddrOfSocketAddr (hAddr);
		pImpl->m_nPort = __OSL_NETWORD((sal_uInt16)(nPort));

		return (!(pImpl->m_nHost == OSL_INADDR_NONE));
	}
	return sal_False;
}

/*
 * __osl_socks_connectSocket.
 */
static oslSocketResult __osl_socks_connectSocket (context_type *pImpl)
{
	oslSocketResult result;

	result = osl_connectSocketTo (pImpl->m_hSocket, pImpl->m_hGwAddr, NULL);
	if (!(result == osl_Socket_Ok))
	{
		oslSocketError error = osl_getLastSocketError (pImpl->m_hSocket);
		if (error == osl_Socket_E_IsConnected)
			result = osl_Socket_Ok;
		else if (error == osl_Socket_E_Already)
			result = osl_Socket_InProgress;
		else if (error == osl_Socket_E_InProgress)
			result = osl_Socket_InProgress;
		else if (error == osl_Socket_E_WouldBlock)
			result = osl_Socket_InProgress;
		else
			result = osl_Socket_Error;
	}
	return (result);
}

/*
 * __osl_socks_sendSocket.
 */
static oslSocketResult __osl_socks_sendSocket (context_type *pImpl)
{
	sal_Int32 n, m = sizeof (pImpl->dest.m_pBuffer);
	while ((n = pImpl->m_pBuffer - pImpl->dest.m_pBuffer) < m)
	{
		sal_Int32 k = osl_sendSocket (
			pImpl->m_hSocket, pImpl->m_pBuffer, m - n, osl_Socket_MsgNormal);
		if (k <= 0)
		{
			oslSocketError error = osl_getLastSocketError (pImpl->m_hSocket);
			if (error == osl_Socket_E_WouldBlock)
				return osl_Socket_InProgress;
			else
				return osl_Socket_Error;
		}
		pImpl->m_pBuffer += k;
	}
	return osl_Socket_Ok;
}

/*
 * __osl_socks_recvSocket.
 */
static oslSocketResult __osl_socks_recvSocket (context_type *pImpl)
{
	sal_Int32 n, m = sizeof (pImpl->bind.m_pBuffer);
	while ((n = pImpl->m_pBuffer - pImpl->bind.m_pBuffer) < m)
	{
		sal_Int32 k = osl_receiveSocket (
			pImpl->m_hSocket, pImpl->m_pBuffer, m - n, osl_Socket_MsgNormal);
		if (k <= 0)
		{
			oslSocketError error = osl_getLastSocketError (pImpl->m_hSocket);
			if (error == osl_Socket_E_WouldBlock)
				return osl_Socket_InProgress;
			else
				return osl_Socket_Error;
		}
		pImpl->m_pBuffer += k;
	}
	return osl_Socket_Ok;
}

/*
 * __osl_socks_createContext.
 */
oslSocksContext SAL_CALL __osl_socks_createContext (
	oslSocket hSocket, oslSocketAddr hGwAddr)
{
	context_type *pImpl = (context_type*)NULL;

	if (!(hSocket && hGwAddr))
		return ((oslSocksContext)NULL);

	pImpl = (context_type*)(rtl_allocateMemory (sizeof (context_type)));
	if (!pImpl)
		return ((oslSocksContext)NULL);

	rtl_zeroMemory (pImpl, sizeof (context_type));

	osl_acquireSocket (hSocket);
	pImpl->m_hSocket = hSocket;
	pImpl->m_hGwAddr = osl_copySocketAddr (hGwAddr);

	pImpl->m_eState  = STATE_INIT;

	return ((oslSocksContext)pImpl);
}

/*
 * __osl_socks_initContext.
 */
oslSocketResult SAL_CALL __osl_socks_initContext (
	oslSocksContext hContext,
	oslSocksCommand eCommand,
	oslSocketAddr   hDestAddr)
{
	context_type *pImpl = (context_type*)hContext;
	if (!pImpl)
		return osl_Socket_Error;

	if (pImpl->m_eState == STATE_INIT)
	{
		message_type *dest  = &(pImpl->dest.m_aMessage);
		sal_Char     *ident = &(pImpl->dest.m_pBuffer[sizeof (message_type)]);

		dest->m_nVersion = 4;
		dest->m_nCode    = eCommand;
		rtl_copyMemory (ident, "SOCKS", 6);

		if (__osl_socks_setSocketAddr (dest, hDestAddr))
			pImpl->m_eState = STATE_CONNECT;
		else
			return osl_Socket_Error;
	}
	return osl_Socket_Ok;
}

/*
 * __osl_socks_connectContext.
 */
oslSocketResult SAL_CALL __osl_socks_connectContext (oslSocksContext hContext)
{
	context_type *pImpl = (context_type*)hContext;
	if (!pImpl)
		return osl_Socket_Error;

	if (pImpl->m_eState == STATE_CONNECT)
	{
		oslSocketResult result = __osl_socks_connectSocket (pImpl);
		if (result == osl_Socket_Ok)
			pImpl->m_eState = STATE_REQUEST;
		else
			return (result);
		pImpl->m_pBuffer = pImpl->dest.m_pBuffer;
	}
	return osl_Socket_Ok;
}

/*
 * __osl_socks_sendContext.
 */
oslSocketResult SAL_CALL __osl_socks_sendContext (oslSocksContext hContext)
{
	context_type *pImpl = (context_type*)hContext;
	if (!pImpl)
		return osl_Socket_Error;

	if (pImpl->m_eState == STATE_REQUEST)
	{
		oslSocketResult result = __osl_socks_sendSocket (pImpl);
		if (result == osl_Socket_Ok)
			pImpl->m_eState = STATE_RESPONSE;
		else
			return (result);
		pImpl->m_pBuffer = pImpl->bind.m_pBuffer;
	}
	return osl_Socket_Ok;
}

/*
 * __osl_socks_recvContext.
 */
oslSocketResult SAL_CALL __osl_socks_recvContext (oslSocksContext hContext)
{
	context_type *pImpl = (context_type*)hContext;
	if (!pImpl)
		return osl_Socket_Error;

	if (pImpl->m_eState == STATE_RESPONSE)
	{
		oslSocketResult result = __osl_socks_recvSocket (pImpl);
		if (result == osl_Socket_Ok)
			pImpl->m_eState = STATE_DONE;
		else
			return (result);
		pImpl->m_pBuffer = pImpl->bind.m_pBuffer;
	}
	return osl_Socket_Ok;
}

/*
 * __osl_socks_getContext.
 */
oslSocketResult SAL_CALL __osl_socks_getContext (
	oslSocksContext hContext, oslSocketAddr *phBindAddr)
{
	context_type *pImpl = (context_type*)hContext;
	if (!pImpl)
		return osl_Socket_Error;

	if (pImpl->m_eState == STATE_DONE)
	{
		message_type *bind = &(pImpl->bind.m_aMessage);
		message_type *dest = &(pImpl->dest.m_aMessage);

		if (!(bind->m_nCode == 90))
			return osl_Socket_Error;

		if ((bind->m_nPort == dest->m_nPort) &&
			(bind->m_nHost == dest->m_nHost)    )
			return osl_Socket_Error;

		if (dest->m_nCode == osl_Socks_CommandBind)
			pImpl->m_eState = STATE_RESPONSE;
	}

	if (phBindAddr)
		*phBindAddr = __osl_socks_getSocketAddr (&(pImpl->bind.m_aMessage));

	return osl_Socket_Ok;
}

/*
 * __osl_socks_destroyContext.
 */
void SAL_CALL __osl_socks_destroyContext (oslSocksContext hContext)
{
	context_type *pImpl = (context_type*)hContext;
	if (pImpl)
	{
		osl_releaseSocket     (pImpl->m_hSocket);
		osl_destroySocketAddr (pImpl->m_hGwAddr);

		rtl_freeMemory (pImpl);
	}
}

/*
 * __osl_socks_connectSocketTo.
 */
oslSocketResult SAL_CALL __osl_socks_connectSocketTo (
	oslSocksContext hContext, oslSocketAddr hToAddr)
{
	oslSocketResult result;

	result = __osl_socks_initContext (
		hContext, osl_Socks_CommandConnect, hToAddr);
	if (!(result == osl_Socket_Ok))
		return result;

	result = __osl_socks_connectContext (hContext);
	if (!(result == osl_Socket_Ok))
		return result;

	result = __osl_socks_sendContext (hContext);
	if (!(result == osl_Socket_Ok))
		return result;

	result = __osl_socks_recvContext (hContext);
	if (!(result == osl_Socket_Ok))
		return result;

	result = __osl_socks_getContext (hContext, NULL);
	return (result);
}

/*
 * __osl_socks_listenOnSocket.
 */
oslSocketResult SAL_CALL __osl_socks_listenOnSocket (
	oslSocksContext hContext, oslSocketAddr hToAddr)
{
	oslSocketResult result;

	result = __osl_socks_initContext (
		hContext, osl_Socks_CommandBind, hToAddr);
	if (!(result == osl_Socket_Ok))
		return (result);

	result = __osl_socks_connectContext (hContext);
	if (!(result == osl_Socket_Ok))
		return (result);

	result = __osl_socks_sendContext (hContext);
	if (!(result == osl_Socket_Ok))
		return (result);

	result = __osl_socks_recvContext (hContext);
	if (!(result == osl_Socket_Ok))
		return (result);

	result = __osl_socks_getContext (hContext, NULL);
	return (result);
}

/*
 * __osl_socks_acceptConnectionOnSocket.
 */
oslSocket SAL_CALL __osl_socks_acceptConnectionOnSocket (
	oslSocksContext hContext, oslSocketAddr *phFromAddr)
{
	oslSocketResult result;

	result = __osl_socks_recvContext (hContext);
	if (!(result == osl_Socket_Ok))
		return ((oslSocket)NULL);

	result = __osl_socks_getContext (hContext, phFromAddr);
	if (result == osl_Socket_Ok)
	{
		context_type *pImpl = (context_type*)hContext;
		osl_acquireSocket (pImpl->m_hSocket);
		return (pImpl->m_hSocket);
	}
	return ((oslSocket)NULL);
}

