/*************************************************************************
 *
 *  $RCSfile: newsclnt.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: mhu $ $Date: 2001/03/19 11:06:59 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#ifndef __STL_USE_NEWALLOC
#define __STL_USE_NEWALLOC 1
#endif
#ifndef __UTILITY__
#include <utility>
#endif

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif
#ifndef _RTL_STRBUF_HXX
#include <rtl/strbuf.hxx>
#endif

#ifndef _VOS_MUTEX_HXX_
#include <vos/mutex.hxx>
#endif
#ifndef _VOS_REF_HXX_
#include <vos/ref.hxx>
#endif

#ifndef _LIST_HXX
#include <tools/list.hxx>
#endif
#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif

#ifndef _INETCORESTRM_HXX
#include <inetstrm.hxx>
#endif
#ifndef _INET_NEWSCLNT_HXX
#include <newsclnt.hxx>
#endif

using namespace rtl;

using namespace inet;
using namespace inet::mail;
using namespace inet::nntp;

/*========================================================================
 *
 * NewsClient_Impl internals.
 *
 *======================================================================*/
namespace inet
{
namespace nntp
{

/*========================================================================
 *
 * NewsContext_Impl.
 *
 *======================================================================*/
class NewsContext_Impl : public NAMESPACE_VOS(OReference)
{
public:
	/** Representation.
	 */
	typedef INetCoreMailerCallback callback_type;

	callback_type *m_pfnCB;
	void          *m_pDataCB;

	/** Construction.
	 */
	NewsContext_Impl (callback_type *pfnCB, void *pDataCB)
		: m_pfnCB (pfnCB), m_pDataCB (pDataCB)
	{}

protected:
	/** Destruction.
	 */
	virtual ~NewsContext_Impl (void);

private:
	/** Not implemented.
	 */
	NewsContext_Impl (const NewsContext_Impl&);
	NewsContext_Impl& operator= (const NewsContext_Impl&);
};

/*
 * ~NewsContext_Impl.
 */
NewsContext_Impl::~NewsContext_Impl (void)
{
}

/*========================================================================
 *
 * DecoderContext_Impl.
 *
 *======================================================================*/
class DecoderContext_Impl : public NewsContext_Impl
{
	/** Representation.
	 */
	INetCoreNewsMessageStream m_aDecoder;

public:
	/** Construction.
	 */
	DecoderContext_Impl (callback_type *pfnCB, void *pDataCB)
		: NewsContext_Impl (pfnCB, pDataCB)
	{}

	/** Initialization.
	 */
	sal_Bool initialize (INetCoreNewsMessage &rMessage)
	{
		m_aDecoder.SetTargetMessage (&rMessage);
		return sal_True;
	}

	/** Operation.
	 */
	INetCoreMessageOStream& operator() (void) { return m_aDecoder; }
};

/*========================================================================
 *
 * EncoderContext_Impl.
 *
 *======================================================================*/
class EncoderContext_Impl : public NewsContext_Impl
{
	/** Representation.
	 */
	INetCoreNewsMessageStream m_aEncoder;

public:
	/** Construction.
	 */
	EncoderContext_Impl (callback_type *pfnCB, void *pDataCB)
		: NewsContext_Impl (pfnCB, pDataCB)
	{}

	/** Initialization.
	 */
	sal_Bool initialize (
		NewsClient_Impl::ClientManager &rManager,
		INetCoreNewsMessage            &rMessage)
	{
		// Check Newsgroups.
		if (!rMessage.GetNewsgroups().Len())
			return sal_False;

		// Prepare message.
		if (!rManager.prepareMessage (rMessage))
			return sal_False;

		// Check UserAgent (X-Newsreader).
		if (!rMessage.GetXNewsreader().Len())
		{
			// Use X-Mailer, instead.
			rMessage.SetXNewsreader (rMessage.GetXMailer());
		}
		rMessage.SetXMailer (UniString());
		
		// Check document stream.
		SvStream *pDocStrm = rMessage.GetDocumentStream();
		if (pDocStrm)
		{
			// Rewind document stream.
			pDocStrm->Seek (STREAM_SEEK_TO_BEGIN);
		}

		// Done.
		m_aEncoder.SetSourceMessage (&rMessage);
		return sal_True;
	}

	/** Operation.
	 */
	INetCoreMessageIStream& operator() (void) { return m_aEncoder; }
};

/*========================================================================
 *
 * GroupContext_Impl.
 *
 *======================================================================*/
class GroupContext_Impl : public NewsContext_Impl
{
	/** Representation.
	 */
	INetCoreNNTPGroupListEntry *m_pEntry;

public:
	/** Construction.
	 */
	GroupContext_Impl (callback_type *pfnCB, void *pDataCB)
		: NewsContext_Impl (pfnCB, pDataCB), m_pEntry (NULL)
	{}

	/** Initialization.
	 */
	sal_Bool initialize (INetCoreNNTPGroupListEntry &rEntry)
	{
		m_pEntry = &rEntry;
		return (m_pEntry->m_aGroupName.getLength() > 0);
	}

	/** Operation.
	 */
	void operator() (const sal_Char *pReplyText)
	{
		ByteString aReply (pReplyText);
		aReply.EraseLeadingChars (' ');
		if (aReply.GetTokenCount (' ') > 3)
		{
			m_pEntry->m_nArticleCount =
				sal_uInt32 (aReply.GetToken(0, ' ').ToInt32());
			m_pEntry->m_nFirstArticleNumber =
				sal_uInt32 (aReply.GetToken(1, ' ').ToInt32());
			m_pEntry->m_nLastArticleNumber =
				sal_uInt32 (aReply.GetToken(2, ' ').ToInt32());
		}
	}
};

/*========================================================================
 *
 * GroupListContext_Impl.
 *
 *======================================================================*/
class GroupListContext_Impl; // NEWNEWS grouplist

/*========================================================================
 *
 * LoginContext_Impl.
 *
 *======================================================================*/
class LoginContext_Impl : public NewsContext_Impl
{
	/** Representation.
	 */
	OUString m_aPassword;

public:
	/** Construction.
	 */
	LoginContext_Impl (callback_type *pfnCB, void *pDataCB)
		: NewsContext_Impl (pfnCB, pDataCB)
	{}

	/** Initialization.
	 */
	sal_Bool initialize (const OUString &rPassword)
	{
		m_aPassword = rPassword;
		return sal_True;
	}

	/** Operation.
	 */
	OUString takePassword (void)
	{
		OUString aPassword (m_aPassword);
		m_aPassword = OUString();
		return aPassword;
	}
};

/*========================================================================
 *
 * OverviewContext_Impl.
 *
 *======================================================================*/
typedef std::pair<sal_uInt32, sal_uInt32> overview_argument_type;

class OverviewContext_Impl : public DecoderContext_Impl
{
	/** Representation.
	 */
	overview_argument_type m_aArguments;

public:
	/** Construction.
	 */
	OverviewContext_Impl (callback_type *pfnCB, void *pDataCB)
		: DecoderContext_Impl (pfnCB, pDataCB)
	{}

	/** Initialization.
	 */
	sal_Bool initialize (
		sal_uInt32           nFirst,
		sal_uInt32           nLast,
		INetCoreNewsMessage &rMessage)
	{
		m_aArguments = overview_argument_type (nFirst, nLast);
		return DecoderContext_Impl::initialize (rMessage);
	}

	/** Operation.
	 */
	const overview_argument_type& operator() (void) const
	{
		return m_aArguments;
	}

	void operator() (const ByteString &rFormat, const ByteString &rHeader);
};

/*
 * operator().
 */
void OverviewContext_Impl::operator() (
	const ByteString &rFormat, const ByteString &rHeader)
{
	INetCoreMessageOStream &rDecoder = DecoderContext_Impl::operator()();
	rDecoder.ParseHeader (TRUE);

	xub_StrLen i, n = rFormat.GetTokenCount ('\t');
	for (i = 0; i < n; i++)
	{
		ByteString aValue (rHeader.GetToken (i, '\t'));
		if (aValue.Len())
		{
			ByteString aLine (rFormat.GetToken (i, '\t'));
			if (aLine.Search (":full") == STRING_NOTFOUND)
				aLine += aValue;
			else
				aLine  = aValue;
			aLine += "\015\012";
			rDecoder.Write (aLine.GetBuffer(), aLine.Len(), NULL);
		}
	}
	rDecoder.Write ("\015\012", 2, NULL);
}

/*========================================================================
 *
 * OverviewFormatContext_Impl.
 *
 *======================================================================*/
class OverviewFormatContext_Impl : public OverviewContext_Impl
{
	/** Representation.
	 */
	List m_aList;

	/** getObject_Impl.
	 */
	OString* getObject_Impl (sal_uInt32 nIndex) const
	{
		return SAL_REINTERPRET_CAST(OString*, m_aList.GetObject (nIndex));
	}

public:
	/** Construction.
	 */
	OverviewFormatContext_Impl (callback_type *pfnCB, void *pDataCB)
		: OverviewContext_Impl (pfnCB, pDataCB)
	{}

	/** Operation.
	 */
	List& operator() (void) const
	{
		return ((SAL_CONST_CAST(OverviewFormatContext_Impl*, this))->m_aList);
	}

	OString getFormat (void) const;

protected:
	/** Destruction.
	 */
	virtual ~OverviewFormatContext_Impl (void);
};

/*
 * getFormat.
 */
OString OverviewFormatContext_Impl::getFormat (void) const
{
	ULONG i, n = m_aList.Count();
	if (n > 0)
	{
		OStringBuffer aFormat (*(getObject_Impl(0)));
		for (i = 1; i < n; i++)
		{
			aFormat.append (sal_Char ('\t'));
			aFormat.append (*(getObject_Impl(i)));
		}
		return aFormat.makeStringAndClear();
	}
	else
	{
		// Use common format.
		static const sal_Char pFormat[] =
			"Subject:\tFrom:\tDate:\tMessage-ID:\tReferences:\tBytes:\tLines:";
		return OString (pFormat);
	}
}

/*
 * ~OverviewFormatContext_Impl.
 */
OverviewFormatContext_Impl::~OverviewFormatContext_Impl (void)
{
	ULONG i, n = m_aList.Count();
	for (i = 0; i < n; i++)
		delete getObject_Impl(i);
	m_aList.Clear();
}

} // nntp
} // inet

/*========================================================================
 *
 * NewsClient_Impl implementation.
 *
 *======================================================================*/
/*
 * NewsClient_Impl.
 */
NewsClient_Impl::NewsClient_Impl (ClientManager &rManager)
	: m_rManager (rManager),
	  m_eState   (STATE_NONE)
{
}

/*
 * ~NewsClient_Impl.
 */
NewsClient_Impl::~NewsClient_Impl (void)
{
}

/*
 * getMutex_Impl.
 */
inline NAMESPACE_VOS(IMutex)& NewsClient_Impl::getMutex_Impl (void) const
{
	return ((SAL_CONST_CAST(NewsClient_Impl*, this))->m_aMutex);
}

/*
 * setState_Impl.
 */
inline void NewsClient_Impl::setState_Impl (State eState)
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	m_eState = eState;
}

/*
 * changeState_Impl.
 */
inline sal_Bool NewsClient_Impl::changeState_Impl (State ePrev, State eNext)
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	if (m_eState == ePrev)
	{
		m_eState = eNext;
		return sal_True;
	}
	return sal_False;
}

/*
 * createConnection_Impl.
 */
inline sal_Bool NewsClient_Impl::createConnection_Impl (
	NAMESPACE_VOS(ORef)<connection_type> &rxConnection)
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	if (!m_xConnection.isValid())
	{
		m_xConnection = new connection_type();
		m_xConnection->SetTransferCallback (onConnectionEvent, this);
		m_xConnection->SetTerminateCallback (onConnectionEvent, this);
	}
	rxConnection = m_xConnection;
	return (rxConnection.isValid());
}

/*
 * getConnection_Impl.
 */
inline sal_Bool NewsClient_Impl::getConnection_Impl (
	NAMESPACE_VOS(ORef)<connection_type> &rxConnection) const
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	rxConnection = m_xConnection;
	return (rxConnection.isValid());
}

/*
 * onConnectionEvent.
 */
sal_Bool NewsClient_Impl::onConnectionEvent (
	connection_type *pConnection,
	sal_Int32 nReplyCode, const sal_Char *pReplyText, void *pData)
{
	NewsClient_Impl *pThis = SAL_REINTERPRET_CAST(NewsClient_Impl*, pData);
	if (pThis)
	{
		NAMESPACE_VOS(ORef)<NewsClient_Impl> xThis (pThis);
		xThis->handleConnectionEvent (pConnection, nReplyCode, pReplyText);
	}
	return sal_True;
}

/*
 * handleConnectionEvent.
 */
void NewsClient_Impl::handleConnectionEvent (
	connection_type *pConnection,
	sal_Int32 nReplyCode, const sal_Char *pReplyText)
{
	if (changeState_Impl (STATE_OVER, STATE_EVENT))
	{
		NAMESPACE_VOS(OClearableGuard) aGuard (getMutex_Impl());

		NAMESPACE_VOS(ORef)<OverviewContext_Impl> xContext (
			SAL_REINTERPRET_CAST (OverviewContext_Impl*, &*m_xContext));
		if (xContext.isValid())
		{
			if (nReplyCode == INETCORENNTP_REPLY_TRANSFER_OVERVIEW)
			{
				// Analyze reply.
				INetCoreNNTPOverviewEntry *pEntry =
					(INetCoreNNTPOverviewEntry*)pReplyText;
				pReplyText = SAL_REINTERPRET_CAST(
					const sal_Char*, &(pEntry->m_nArticleNumber));

				(*xContext)(m_aOverviewFormat, pEntry->m_aOverviewHeader);

				// Wait for next event.
				setState_Impl (STATE_OVER);
			}
			else if (nReplyCode == INETCORENNTP_REPLY_TRANSFER_WAIT)
			{
				// Wait for next event.
				setState_Impl (STATE_OVER);
			}
			else
			{
				// Finished.
				setState_Impl (STATE_NONE);
			}
			aGuard.clear();

			// Notify caller.
			if (xContext->m_pfnCB) (xContext->m_pfnCB) (
				m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
		}
	}
	else if (changeState_Impl (STATE_REPLY_N, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<NewsContext_Impl> xContext (m_xContext);
		if (xContext.isValid())
		{
			if (nReplyCode == INETCORENNTP_REPLY_TRANSFER_WAIT)
			{
				// Wait for next event.
				changeState_Impl (STATE_EVENT, STATE_REPLY_N);
			}
			else
			{
				// Finished.
				setState_Impl (STATE_NONE);
			}

			// Notify caller.
			if (xContext->m_pfnCB) (xContext->m_pfnCB) (
				m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
		}
	}
	else if (changeState_Impl (STATE_GROUP, STATE_EVENT))
	{
		NAMESPACE_VOS(OClearableGuard) aGuard (getMutex_Impl());

		NAMESPACE_VOS(ORef)<GroupContext_Impl> xContext (
			SAL_REINTERPRET_CAST (GroupContext_Impl*, &*m_xContext));
		if (xContext.isValid())
		{
			if (nReplyCode == INETCORENNTP_REPLY_GROUP_SELECTED)
			{
				// GROUP command ok. Analyze reply.
				(*xContext)(pReplyText);
			}

			// Finished.
			setState_Impl (STATE_NONE);
			aGuard.clear();

			// Notify caller.
			if (xContext->m_pfnCB) (xContext->m_pfnCB) (
				m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
		}
	}
	else if (changeState_Impl (STATE_USER, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<LoginContext_Impl> xContext (
			SAL_REINTERPRET_CAST (LoginContext_Impl*, &*m_xContext));
		if (xContext.isValid())
		{
			if (nReplyCode == INETCORENNTP_REPLY_MORE_AUTHINFO_REQUIRED)
			{
				// Intermediate success, but AUTHINFO PASS needed.
				OUString aPassword (xContext->takePassword());
				if (!loginPass_Impl (aPassword))
				{
					// Failure.
					setState_Impl (STATE_NONE);

					// Notify caller.
					if (xContext->m_pfnCB) (xContext->m_pfnCB) (
						m_rManager, nReplyCode, pReplyText,
						xContext->m_pDataCB);
				}
			}
			else
			{
				// Finished with either ACCEPTED or REJECTED reply.
				setState_Impl (STATE_NONE);

				// Notify caller.
				if (xContext->m_pfnCB) (xContext->m_pfnCB) (
					m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
			}
		}
	}
	else if (changeState_Impl (STATE_OPEN, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<NewsContext_Impl> xContext (m_xContext);
		if (xContext.isValid())
		{
			if ((nReplyCode == INETCORENNTP_REPLY_RESOLVER_WAIT) ||
				(nReplyCode == INETCORENNTP_REPLY_CONNECT_WAIT )    )
			{
				// Wait for next event.
				changeState_Impl (STATE_EVENT, STATE_OPEN);
			}
			else
			{
				// Finished.
				setState_Impl (STATE_NONE);

				// Notify caller.
				if (xContext->m_pfnCB) (xContext->m_pfnCB) (
					m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
			}
		}
	}
	else if (changeState_Impl (STATE_OVERFMT, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<OverviewFormatContext_Impl> xContext (
			SAL_REINTERPRET_CAST (OverviewFormatContext_Impl*, &*m_xContext));
		if (xContext.isValid())
		{
			if (nReplyCode == INETCORENNTP_REPLY_PERMISSION_DENIED)
			{
				// Failure (502).
				setState_Impl (STATE_NONE);

				// Notify caller.
				if (xContext->m_pfnCB) (xContext->m_pfnCB) (
					m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
			}
			else if ((nReplyCode/100 == 2) || (nReplyCode/100 == 5))
			{
				m_aOverviewFormat = xContext->getFormat();
				if (!getOverview_Impl (&*xContext))
				{
					// Failure.
					setState_Impl (STATE_NONE);

					// Notify caller.
					if (xContext->m_pfnCB) (xContext->m_pfnCB) (
						m_rManager, INETCORENNTP_REPLY_NETWORK_ERROR, NULL,
						xContext->m_pDataCB);
				}
			}
			else
			{
				if (nReplyCode == INETCORENNTP_REPLY_TRANSFER_WAIT)
				{
					// Wait for next event.
					changeState_Impl (STATE_EVENT, STATE_OVERFMT);
				}
				else
				{
					// Finished.
					setState_Impl (STATE_NONE);
				}

				// Notify caller.
				if (xContext->m_pfnCB) (xContext->m_pfnCB) (
					m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
			}
		}
	}
	else if (changeState_Impl (STATE_REPLY_1, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<NewsContext_Impl> xContext (m_xContext);
		if (xContext.isValid())
		{
			// Finished.
			setState_Impl (STATE_NONE);

			// Notify caller.
			if (xContext->m_pfnCB) (xContext->m_pfnCB) (
				m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
		}
	}
	else
	{
		VOS_ASSERT(m_eState == STATE_NONE);
		if (nReplyCode == INETCORENNTP_REPLY_NETWORK_ERROR)
		{
			// Cleanup.
			m_aOverviewFormat = OString();

			// Notify manager.
			m_rManager.terminated (this);
		}
	}
}

/*
 * openConnection.
 */
sal_Bool NewsClient_Impl::openConnection (
	const OUString &rHost, sal_uInt16 nPort,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_OPEN))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!createConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext = new NewsContext_Impl (pfnCallback, pData);
		if (!xConnection->Open (rHost, nPort, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_OPEN, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * isOpen.
 */
sal_Bool NewsClient_Impl::isOpen (void) const
{
	NAMESPACE_VOS(ORef)<connection_type> xConnection;
	if (getConnection_Impl (xConnection))
		return xConnection->IsOpen();
	else
		return sal_False;
}

/*
 * closeConnection.
 */
sal_Bool NewsClient_Impl::closeConnection (
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_CLOSE))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext = new NewsContext_Impl (pfnCallback, pData);
		if (!xConnection->Close (onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_CLOSE, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * abortConnection.
 */
void NewsClient_Impl::abortConnection (void)
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	if (m_xConnection.isValid())
	{
		m_xConnection->Destroy();
		m_xConnection.unbind();
	}
	if (m_xContext.isValid())
	{
		m_xContext->m_pfnCB = 0;
		m_xContext.unbind();
	}
	m_eState = STATE_NONE;
}

/*
 * setModeReader.
 */
sal_Bool NewsClient_Impl::setModeReader (
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_READER))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext = new NewsContext_Impl (pfnCallback, pData);
		if (!xConnection->SetModeReader (onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_READER, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * authenticate.
 */
sal_Bool NewsClient_Impl::authenticate (
	const OUString &rUsername, const OUString &rPassword,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<LoginContext_Impl> xContext (
			new LoginContext_Impl (pfnCallback, pData));
		if (!xContext->initialize (rPassword))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext.bind (&*xContext);
		if (!loginUser_Impl (rUsername))
		{
			m_xContext.unbind();
			setState_Impl (STATE_NONE);
			return sal_False;
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * authenticatePassword.
 */
sal_Bool NewsClient_Impl::authenticatePassword (
	const OUString &rPassword,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_EVENT))
	{
		m_xContext = new NewsContext_Impl (pfnCallback, pData);
		if (!loginPass_Impl (rPassword))
		{
			m_xContext.unbind();
			setState_Impl (STATE_NONE);
			return sal_False;
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * getGroups.
 */
sal_Bool NewsClient_Impl::getGroups (
	List &rList,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_LIST))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext = new NewsContext_Impl (pfnCallback, pData);
		if (!xConnection->GetGroupList (rList, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_LIST, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * getNewGroups.
 */
sal_Bool NewsClient_Impl::getNewGroups (
	const DateTime &rSince, List &rList,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_NEWLIST))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext = new NewsContext_Impl (pfnCallback, pData);
		if (!xConnection->GetNewGroupsList (
			rSince, rList, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_NEWLIST, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * getNewNews.
 */
sal_Bool NewsClient_Impl::getNewNews (
	const OUString &rGroups, const DateTime &rSince, List &rList,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	// (NYI).
	return sal_False;
}

/*
 * selectGroup.
 */
sal_Bool NewsClient_Impl::selectGroup (
	INetCoreNNTPGroupListEntry &rEntry,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_GROUP))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		NAMESPACE_VOS(ORef)<GroupContext_Impl> xContext (
			new GroupContext_Impl (pfnCallback, pData));
		if (!xContext->initialize (rEntry))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext.bind (&*xContext);
		if (!xConnection->SelectGroup (
			rEntry.m_aGroupName, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_GROUP, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * getArticleOverview.
 */
sal_Bool NewsClient_Impl::getArticleOverview (
	sal_uInt32              nFirst,
	sal_uInt32              nLast,
	INetCoreNewsMessage    &rMessage,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_EVENT))
	{
		sal_Bool result = sal_False;
		if (!m_aOverviewFormat.getLength())
		{
			NAMESPACE_VOS(ORef)<OverviewFormatContext_Impl> xContext (
				new OverviewFormatContext_Impl (pfnCallback, pData));
			if (!xContext->initialize (nFirst, nLast, rMessage))
			{
				setState_Impl (STATE_NONE);
				return sal_False;
			}

			m_xContext.bind (&*xContext);
			result = getOverviewFormat_Impl (xContext);
		}
		else
		{
			NAMESPACE_VOS(ORef)<OverviewContext_Impl> xContext (
				new OverviewContext_Impl (pfnCallback, pData));
			if (!xContext->initialize (nFirst, nLast, rMessage))
			{
				setState_Impl (STATE_NONE);
				return sal_False;
			}

			m_xContext.bind (&*xContext);
			result = getOverview_Impl (xContext);
		}
		if (!result)
		{
			m_xContext.unbind();
			setState_Impl (STATE_NONE);
		}
		return result;
	}
	return sal_False;
}

/*
 * getArticleHead.
 */
sal_Bool NewsClient_Impl::getArticleHead (
	sal_uInt32              nArticle,
	const OUString         &rMessageID,
	INetCoreNewsMessage    &rMessage,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_RETR))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		NAMESPACE_VOS(ORef)<DecoderContext_Impl> xContext (
			new DecoderContext_Impl (pfnCallback, pData));
		if (!xContext->initialize (rMessage))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}
		m_xContext.bind (&*xContext);

		sal_Bool result;
		if (rMessageID.getLength())
			result = xConnection->GetArticleHeader (
				rMessageID, (*xContext)(), onConnectionEvent, this);
		else
			result = xConnection->GetArticleHeader (
				nArticle, (*xContext)(), onConnectionEvent, this);
		if (!result)
		{
			if (changeState_Impl (STATE_RETR, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * getArticle.
 */
sal_Bool NewsClient_Impl::getArticle (
	sal_uInt32              nArticle,
	const OUString         &rMessageID,
	INetCoreNewsMessage    &rMessage,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_RETR))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		NAMESPACE_VOS(ORef)<DecoderContext_Impl> xContext (
			new DecoderContext_Impl (pfnCallback, pData));
		if (!xContext->initialize (rMessage))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}
		m_xContext.bind (&*xContext);

		sal_Bool result;
		if (rMessageID.getLength())
			result = xConnection->RetrieveArticle (
				rMessageID, (*xContext)(), onConnectionEvent, this);
		else
			result = xConnection->RetrieveArticle (
				nArticle, (*xContext)(), onConnectionEvent, this);
		if (!result)
		{
			if (changeState_Impl (STATE_RETR, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * postArticle.
 */
sal_Bool NewsClient_Impl::postArticle (
	INetCoreNewsMessage    &rMessage,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_POST))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		NAMESPACE_VOS(ORef)<EncoderContext_Impl> xContext (
			new EncoderContext_Impl (pfnCallback, pData));
		if (!xContext->initialize (m_rManager, rMessage))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext.bind (&*xContext);
		if (!xConnection->PostArticle ((*xContext)(), onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_POST, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * isPostingAllowed.
 */
sal_Bool NewsClient_Impl::isPostingAllowed (void) const
{
	NAMESPACE_VOS(ORef)<connection_type> xConnection;
	if (getConnection_Impl (xConnection))
		return xConnection->IsPostingAllowed();
	else
		return sal_False;
}

/*
 * getTransferCount.
 */
sal_uInt32 NewsClient_Impl::getTransferCount (void) const
{
	NAMESPACE_VOS(ORef)<connection_type> xConnection;
	if (getConnection_Impl (xConnection))
		return xConnection->GetTransferCount();
	else
		return 0;
}

/*
 * getOverviewFormat_Impl.
 */
sal_Bool NewsClient_Impl::getOverviewFormat_Impl (
	const NAMESPACE_VOS(ORef)<OverviewFormatContext_Impl> &rxContext)
{
	if (changeState_Impl (STATE_EVENT, STATE_OVERFMT))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_EVENT);
			return sal_False;
		}
		if (!xConnection->GetArticleHeaderOverviewFormat (
			(*rxContext)(), onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_OVERFMT, STATE_EVENT))
			{
				// Failure.
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * getOverview_Impl.
 */
sal_Bool NewsClient_Impl::getOverview_Impl (
	const NAMESPACE_VOS(ORef)<OverviewContext_Impl> &rxContext)
{
	if (changeState_Impl (STATE_EVENT, STATE_OVER))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_EVENT);
			return sal_False;
		}

		const overview_argument_type &rArg = (*rxContext)();
		if (!xConnection->GetArticleHeaderOverview (
			rArg.first, rArg.second, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_OVER, STATE_EVENT))
			{
				// Failure.
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * loginUser_Impl.
 */
sal_Bool NewsClient_Impl::loginUser_Impl (const OUString &rUsername)
{
	if (changeState_Impl (STATE_EVENT, STATE_USER))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_EVENT);
			return sal_False;
		}
		if (!xConnection->AuthinfoUser (rUsername, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_USER, STATE_EVENT))
			{
				// Failure.
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * loginPass_Impl.
 */
sal_Bool NewsClient_Impl::loginPass_Impl (const OUString &rPassword)
{
	if (changeState_Impl (STATE_EVENT, STATE_PASS))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_EVENT);
			return sal_False;
		}
		if (!xConnection->AuthinfoPass (rPassword, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_PASS, STATE_EVENT))
			{
				// Failure.
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

