/*************************************************************************
 *
 *  $RCSfile: dir.c,v $
 *
 *  $Revision: 1.60.2.4 $
 *
 *  last change: $Author: mh $ $Date: 2003/03/26 08:50:30 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#define UNICODE
#include "system.h"


#define _UNICODE
#include <tchar.h>

#include "dirW9X.h"
#include <systools/win32/kernel9x.h>
#include <systools/win32/mpr9x.h>


#include <osl/diagnose.h>
#include <osl/file.h>
#include <rtl/string.h>
#include <rtl/ustring.h>
#include <rtl/alloc.h>
#include <rtl/tencinfo.h>
#include <osl/thread.h>
#include <osl/mutex.h>
#include "rtl/byteseq.h"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _OSL_TIME_H_
#include <osl/time.h>
#endif

#define ISVALIDHANDLE(handle) \
	 ((DWORD)(handle) + 1 > 1)

#define PATHTYPE_ERROR						0
#define PATHTYPE_RELATIVE					1
#define PATHTYPE_ABSOLUTE_UNC				2
#define PATHTYPE_ABSOLUTE_LOCAL				3
#define PATHTYPE_MASK_TYPE					0xFF
#define PATHTYPE_IS_VOLUME					0x0100
#define PATHTYPE_IS_SERVER					0x0200

#define VALIDATEPATH_NORMAL					0x0000
#define VALIDATEPATH_ALLOW_WILDCARDS		0x0001
#define VALIDATEPATH_ALLOW_ELLIPSE			0x0002
#define VALIDATEPATH_ALLOW_RELATIVE			0x0004
#define VALIDATEPATH_ALLOW_UNC				0x0008

DWORD WINAPI IsValidFilePath( LPCTSTR lpszPath, LPCTSTR *lppError, DWORD dwFlags );

HANDLE WINAPI OpenLogicalDrivesEnum();
BOOL WINAPI EnumLogicalDrives( HANDLE hEnum, LPTSTR lpBuffer );
BOOL WINAPI CloseLogicalDrivesEnum( HANDLE hEnum );


HANDLE WINAPI OpenDirectory( LPCTSTR lpszPath );
BOOL WINAPI CloseDirectory( HANDLE hDirectory );
BOOL WINAPI EnumDirectory( HANDLE hDirectory, LPWIN32_FIND_DATA pFindData );


DWORD WINAPI GetCaseCorrectPathName(
	LPCTSTR	lpszShortPath,	// file name
	LPTSTR	lpszLongPath,	// path buffer
	DWORD	cchBuffer		// size of path buffer 
);

typedef struct {
	UINT	uType;

	union {
		WIN32_FIND_DATA	FindData;
		TCHAR			cDriveString[4];
	};
	TCHAR			szFullPath[MAX_PATH];
	TCHAR			szLocation[MAX_PATH];
	int				nRefCount;
}DirectoryItem_Impl;

#define	DIRECTORYTYPE_LOCALROOT	0
#define	DIRECTORYTYPE_NETROOT		1
#define	DIRECTORYTYPE_NETRESORCE	2
#define	DIRECTORYTYPE_FILESYSTEM	3

#define DIRECTORYITEM_DRIVE		0
#define DIRECTORYITEM_FILE		1
#define DIRECTORYITEM_SERVER	2

typedef struct {
	UINT	uType;
	union {
		HANDLE	hDirectory;
		HANDLE	hEnumDrives;
	};
	TCHAR	szDirectoryPath[MAX_PATH];
} Directory_Impl;

/* Different types of paths */
typedef enum _PATHTYPE
{
	PATHTYPE_SYNTAXERROR = 0,
	PATHTYPE_NETROOT,
	PATHTYPE_NETSERVER,
	PATHTYPE_VOLUME,
	PATHTYPE_FILE
} PATHTYPE;

#define bufsizeof(a)	(sizeof(a) / sizeof((a)[0]))
static void _osl_warnFile( const char *message, rtl_uString *ustrFile );

#ifdef _DEBUG
#define OSL_ENSURE_FILE( cond, msg, file ) ( (cond) ?  (void)0 : _osl_warnFile( msg, file ) )
#else
#define OSL_ENSURE_FILE( cond, msg, file ) ((void)0)
#endif


static oslFileError SAL_CALL _osl_getFileURLFromSystemPath( rtl_uString* strPath, rtl_uString** pstrURL );
static oslFileError SAL_CALL _osl_getSystemPathFromFileURL( rtl_uString *strURL, rtl_uString **pustrPath, sal_Bool fAllowRelative );

/* This is the error table that defines the mapping between OS error
   codes and errno values */

struct errentry {
        unsigned long oscode;	/* OS return value */
        int errnocode;			/* System V error code */
};

static struct errentry errtable[] = {
        {  ERROR_SUCCESS,				 osl_File_E_None     },  /* 0 */
        {  ERROR_INVALID_FUNCTION,       osl_File_E_INVAL    },  /* 1 */
        {  ERROR_FILE_NOT_FOUND,         osl_File_E_NOENT    },  /* 2 */
        {  ERROR_PATH_NOT_FOUND,         osl_File_E_NOENT    },  /* 3 */
        {  ERROR_TOO_MANY_OPEN_FILES,    osl_File_E_MFILE    },  /* 4 */
        {  ERROR_ACCESS_DENIED,          osl_File_E_ACCES    },  /* 5 */
        {  ERROR_INVALID_HANDLE,         osl_File_E_BADF     },  /* 6 */
        {  ERROR_ARENA_TRASHED,          osl_File_E_NOMEM    },  /* 7 */
        {  ERROR_NOT_ENOUGH_MEMORY,      osl_File_E_NOMEM    },  /* 8 */
        {  ERROR_INVALID_BLOCK,          osl_File_E_NOMEM    },  /* 9 */
        {  ERROR_BAD_ENVIRONMENT,        osl_File_E_2BIG     },  /* 10 */
        {  ERROR_BAD_FORMAT,             osl_File_E_NOEXEC   },  /* 11 */
        {  ERROR_INVALID_ACCESS,         osl_File_E_INVAL    },  /* 12 */
        {  ERROR_INVALID_DATA,           osl_File_E_INVAL    },  /* 13 */
        {  ERROR_INVALID_DRIVE,          osl_File_E_NOENT    },  /* 15 */
        {  ERROR_CURRENT_DIRECTORY,      osl_File_E_ACCES    },  /* 16 */
        {  ERROR_NOT_SAME_DEVICE,        osl_File_E_XDEV     },  /* 17 */
        {  ERROR_NO_MORE_FILES,          osl_File_E_NOENT    },  /* 18 */
        {  ERROR_NOT_READY,              osl_File_E_NOTREADY },  /* 21 */
        {  ERROR_LOCK_VIOLATION,         osl_File_E_ACCES    },  /* 33 */
        {  ERROR_BAD_NETPATH,            osl_File_E_NOENT    },  /* 53 */
        {  ERROR_NETWORK_ACCESS_DENIED,  osl_File_E_ACCES    },  /* 65 */
        {  ERROR_BAD_NET_NAME,           osl_File_E_NOENT    },  /* 67 */
        {  ERROR_FILE_EXISTS,            osl_File_E_EXIST    },  /* 80 */
        {  ERROR_CANNOT_MAKE,            osl_File_E_ACCES    },  /* 82 */
        {  ERROR_FAIL_I24,               osl_File_E_ACCES    },  /* 83 */
        {  ERROR_INVALID_PARAMETER,      osl_File_E_INVAL    },  /* 87 */
        {  ERROR_NO_PROC_SLOTS,          osl_File_E_AGAIN    },  /* 89 */
        {  ERROR_DRIVE_LOCKED,           osl_File_E_ACCES    },  /* 108 */
        {  ERROR_BROKEN_PIPE,            osl_File_E_PIPE     },  /* 109 */
        {  ERROR_DISK_FULL,              osl_File_E_NOSPC    },  /* 112 */
        {  ERROR_INVALID_TARGET_HANDLE,  osl_File_E_BADF     },  /* 114 */
        {  ERROR_INVALID_HANDLE,         osl_File_E_INVAL    },  /* 124 */
        {  ERROR_WAIT_NO_CHILDREN,       osl_File_E_CHILD    },  /* 128 */
        {  ERROR_CHILD_NOT_COMPLETE,     osl_File_E_CHILD    },  /* 129 */
        {  ERROR_DIRECT_ACCESS_HANDLE,   osl_File_E_BADF     },  /* 130 */
        {  ERROR_NEGATIVE_SEEK,          osl_File_E_INVAL    },  /* 131 */
        {  ERROR_SEEK_ON_DEVICE,         osl_File_E_ACCES    },  /* 132 */
        {  ERROR_DIR_NOT_EMPTY,          osl_File_E_NOTEMPTY },  /* 145 */
        {  ERROR_NOT_LOCKED,             osl_File_E_ACCES    },  /* 158 */
        {  ERROR_BAD_PATHNAME,           osl_File_E_NOENT    },  /* 161 */
        {  ERROR_MAX_THRDS_REACHED,      osl_File_E_AGAIN    },  /* 164 */
        {  ERROR_LOCK_FAILED,            osl_File_E_ACCES    },  /* 167 */
        {  ERROR_ALREADY_EXISTS,         osl_File_E_EXIST    },  /* 183 */
        {  ERROR_FILENAME_EXCED_RANGE,   osl_File_E_NOENT    },  /* 206 */
        {  ERROR_NESTING_NOT_ALLOWED,    osl_File_E_AGAIN    },  /* 215 */
        {  ERROR_NOT_ENOUGH_QUOTA,       osl_File_E_NOMEM    }    /* 1816 */
};

/* size of the table */
#define ERRTABLESIZE (sizeof(errtable)/sizeof(errtable[0]))

/* The following two constants must be the minimum and maximum
   values in the (contiguous) range of osl_File_E_xec Failure errors. */
#define MIN_EXEC_ERROR ERROR_INVALID_STARTING_CODESEG
#define MAX_EXEC_ERROR ERROR_INFLOOP_IN_RELOC_CHAIN

/* These are the low and high value in the range of errors that are
   access violations */
#define MIN_EACCES_RANGE ERROR_WRITE_PROTECT
#define MAX_EACCES_RANGE ERROR_SHARING_BUFFER_EXCEEDED


/*******************************************************************************/

static oslFileError _mapError( DWORD dwError )
{
    int i;

    /* check the table for the OS error code */
    for ( i = 0; i < ERRTABLESIZE; ++i )
	{
		if ( dwError == errtable[i].oscode )
			return errtable[i].errnocode;
    }

    /* The error code wasn't in the table.  We check for a range of */
    /* osl_File_E_ACCES errors or exec failure errors (ENOEXEC).  Otherwise   */
    /* osl_File_E_INVAL is returned.                                          */

    if ( dwError >= MIN_EACCES_RANGE && dwError <= MAX_EACCES_RANGE)
		return osl_File_E_ACCES;
    else if ( dwError >= MIN_EXEC_ERROR && dwError <= MAX_EXEC_ERROR)
		return osl_File_E_NOEXEC;
    else
		return osl_File_E_INVAL;
}

#define MapError( oserror )	_mapError( oserror )

/****************************************************************************/
/*	osl_createFileHandleFromOSHandle	*/
/****************************************************************************/

oslFileHandle SAL_CALL osl_createFileHandleFromOSHandle( HANDLE hFile )
{
	if ( ISVALIDHANDLE(hFile) )
		return (oslFileHandle)hFile;
	else
		return NULL;
}

/****************************************************************************/
/*	osl_openFile	*/
/****************************************************************************/

oslFileError SAL_CALL osl_openFile( rtl_uString *strPath, oslFileHandle *pHandle, sal_uInt32 uFlags )
{
	rtl_uString		*strSysPath = NULL;
	oslFileError	error = _osl_getSystemPathFromFileURL( strPath, &strSysPath, sal_False );

	if ( osl_File_E_None == error )
	{
		DWORD	dwAccess = 0, dwShare = FILE_SHARE_READ, dwCreation = 0, dwAttributes = 0;
		HANDLE	hFile;

		if ( uFlags & osl_File_OpenFlag_Read )
			dwAccess |= GENERIC_READ;

		if ( uFlags & osl_File_OpenFlag_Write )
			dwAccess |= GENERIC_WRITE;
		else
			dwShare  |= FILE_SHARE_WRITE;

		if ( uFlags & osl_File_OpenFlag_Create )
			dwCreation |= CREATE_NEW;
		else
			dwCreation |= OPEN_EXISTING;

		hFile = CreateFileW( rtl_uString_getStr( strSysPath ), dwAccess, dwShare, NULL, dwCreation, dwAttributes, NULL );

		*pHandle = osl_createFileHandleFromOSHandle( hFile );

		if ( !ISVALIDHANDLE( hFile ) )
			error = MapError( GetLastError() );

		rtl_uString_release( strSysPath );
	}

	return error;
}


/****************************************************************************/
/*	osl_closeFile	*/
/****************************************************************************/

oslFileError SAL_CALL osl_closeFile( oslFileHandle Handle )
{
	oslFileError	error;
	HANDLE			hFile = (HANDLE)Handle;

	if ( ISVALIDHANDLE(hFile) )
		error = CloseHandle( hFile ) ? osl_File_E_None : MapError( GetLastError() );
	else
		error = osl_File_E_INVAL;

	return error;
}

/****************************************************************************/
/*	osl_isEndOfFile	*/
/****************************************************************************/

oslFileError SAL_CALL osl_isEndOfFile( oslFileHandle Handle, sal_Bool *pIsEOF )
{
	oslFileError	error = osl_File_E_INVAL;
	HANDLE			hFile = (HANDLE)Handle;

	if ( ISVALIDHANDLE(hFile) )
	{
		LONG		lDistanceToMove, lDistanceToMoveHigh;
		sal_uInt64	nCurPos;

		/*	Return value INVALID_SET_FILE_POINTER is no error indication and LastError could
			be set from previous IO call */

		SetLastError( NOERROR );

		lDistanceToMoveHigh = 0;
		lDistanceToMove = SetFilePointer( hFile, 0, &lDistanceToMoveHigh, FILE_CURRENT );

		error = MapError( GetLastError() );

		if ( osl_File_E_None == error )
		{
			nCurPos = (sal_uInt64)lDistanceToMove + ((sal_uInt64)lDistanceToMoveHigh << 32);

			lDistanceToMoveHigh = 0;
			lDistanceToMove = SetFilePointer( hFile, 0, &lDistanceToMoveHigh, FILE_END );

			error = MapError( GetLastError() );

			if ( osl_File_E_None == error )
			{
				sal_uInt64	nEndPos = (sal_uInt64)lDistanceToMove + ((sal_uInt64)lDistanceToMoveHigh << 32);

				*pIsEOF = nEndPos == nCurPos;

				lDistanceToMoveHigh = (LONG)(nCurPos >> 32);
				SetFilePointer( hFile, (LONG)(nCurPos & 0xFFFFFFFF), &lDistanceToMoveHigh, FILE_BEGIN );

				error = MapError( GetLastError() );
			}

		}
	}

	return error;
}

/****************************************************************************/
/*	osl_setFilePos	*/
/****************************************************************************/

oslFileError SAL_CALL osl_setFilePos( oslFileHandle Handle, sal_uInt32 uHow, sal_Int64 uPos )
{
	oslFileError	error;
	HANDLE			hFile = (HANDLE)Handle;

	if ( ISVALIDHANDLE(hFile) )
	{
		DWORD	dwMoveMethod;
		LONG	lDistanceToMove, lDistanceToMoveHigh;

		switch ( uHow )
		{
		case osl_Pos_Current:
			dwMoveMethod = FILE_CURRENT;
			break;
		case osl_Pos_End:
			dwMoveMethod = FILE_END;
			break;
		case osl_Pos_Absolut:
		default:
			dwMoveMethod = FILE_BEGIN;
			break;
		}

		lDistanceToMove = (LONG)(uPos & 0xFFFFFFFF);
		lDistanceToMoveHigh = (LONG)(uPos >> 32);


		SetLastError(0);
		SetFilePointer( hFile, lDistanceToMove, &lDistanceToMoveHigh, dwMoveMethod );

		error = MapError( GetLastError() );
	}
	else
		error = osl_File_E_INVAL;

	return error;
}

/****************************************************************************/
/*	osl_getFilePos	*/
/****************************************************************************/

oslFileError SAL_CALL osl_getFilePos( oslFileHandle Handle, sal_uInt64 *pPos )
{
	oslFileError	error;
	HANDLE			hFile = (HANDLE)Handle;

	if ( ISVALIDHANDLE(hFile) )
	{
		LONG	lDistanceToMove, lDistanceToMoveHigh;

		/*	Return value INVALID_SET_FILE_POINTER is no error indication and LastError could
			be set from previous IO call */

		SetLastError( NOERROR );

		lDistanceToMoveHigh = 0;
		lDistanceToMove = SetFilePointer( hFile, 0, &lDistanceToMoveHigh, FILE_CURRENT );

		error = MapError( GetLastError() );

		if ( osl_File_E_None == error )
			*pPos = (sal_uInt64)lDistanceToMove + ((sal_uInt64)lDistanceToMoveHigh << 32);
	}
	else
		error = osl_File_E_INVAL;


	return error;
}


/****************************************************************************/
/*	osl_setFileSize	*/
/****************************************************************************/

oslFileError SAL_CALL osl_setFileSize( oslFileHandle Handle, sal_uInt64 uSize )
{
	oslFileError	error = error = osl_setFilePos( Handle, osl_Pos_Absolut, uSize );
	if ( osl_File_E_None == error )
	{
		if ( !SetEndOfFile( (HANDLE)Handle ) )
			error = MapError( osl_File_E_None );
	}

	return error;
}

/****************************************************************************/
/*	osl_readFile	*/
/****************************************************************************/

oslFileError SAL_CALL osl_readFile( oslFileHandle Handle, void *pBuffer, sal_uInt64 uBytesRequested, sal_uInt64 *pBytesRead )
{
	oslFileError	error;
	HANDLE			hFile = (HANDLE)Handle;

	if ( ISVALIDHANDLE(hFile) )
	{
		DWORD	dwBytesToRead = (DWORD)uBytesRequested;
		DWORD	dwBytesRead;

		if ( ReadFile( hFile, pBuffer, dwBytesToRead, &dwBytesRead, NULL ) )
		{
			*pBytesRead = (sal_uInt64)dwBytesRead;
			error = osl_File_E_None;
		}
		else
			error = MapError( GetLastError() );
	}
	else
		error = osl_File_E_INVAL;

	return error;
}


/****************************************************************************/
/*	osl_writeFile	*/
/****************************************************************************/

oslFileError SAL_CALL osl_writeFile( oslFileHandle Handle, const void *pBuffer, sal_uInt64 uBytesToWrite, sal_uInt64 *pBytesWritten )
{
	oslFileError	error;
	HANDLE			hFile = (HANDLE)Handle;

	if ( ISVALIDHANDLE(hFile) )
	{
		DWORD	dwBytesToWrite = (DWORD)uBytesToWrite;
		DWORD	dwBytesWritten;

		if ( WriteFile( hFile, pBuffer, dwBytesToWrite, &dwBytesWritten, NULL ) )
		{
			*pBytesWritten = (sal_uInt64)dwBytesWritten;
			error = osl_File_E_None;
		}
		else
			error = MapError( GetLastError() );
	}
	else
		error = osl_File_E_INVAL;

	return error;
}

/****************************************************************************/
/*	osl_createDirectory	*/
/****************************************************************************/

oslFileError SAL_CALL osl_createDirectory( rtl_uString* strPath )
{
	rtl_uString	*strSysPath = NULL;
	oslFileError	error = _osl_getSystemPathFromFileURL( strPath, &strSysPath, sal_False );

	if ( osl_File_E_None == error )
	{
		if ( CreateDirectoryW( rtl_uString_getStr( strSysPath ), NULL ) )
			error = osl_File_E_None;
		else
		{
			const sal_Unicode	*pBuffer = rtl_uString_getStr( strSysPath );
			sal_Int32			nLen = rtl_uString_getLength( strSysPath );

			if (
				( pBuffer[0] >= 'A' && pBuffer[0] <= 'Z' ||
				  pBuffer[0] >= 'a' && pBuffer[0] <= 'z' ) &&
				pBuffer[1] == ':' && ( nLen ==2 || nLen == 3 && pBuffer[2] == '\\' )
				)
				SetLastError( ERROR_ALREADY_EXISTS );

			error = MapError( GetLastError() );
		}

		rtl_uString_release( strSysPath );
	}

	return error;
}


/****************************************************************************/
/*	osl_removeDirectory	*/
/****************************************************************************/

oslFileError SAL_CALL osl_removeDirectory( rtl_uString* strPath )
{
	rtl_uString	*strSysPath = NULL;
	oslFileError	error = _osl_getSystemPathFromFileURL( strPath, &strSysPath, sal_False );

	if ( osl_File_E_None == error )
	{
		if ( RemoveDirectory( rtl_uString_getStr( strSysPath ) ) )
			error = osl_File_E_None;
		else
			error = MapError( GetLastError() );

		rtl_uString_release( strSysPath );
	}

	return error;
}

/****************************************************************************/
/*	osl_removeFile	*/
/****************************************************************************/

oslFileError SAL_CALL osl_removeFile( rtl_uString* strPath )
{
	rtl_uString	*strSysPath = NULL;
	oslFileError	error = _osl_getSystemPathFromFileURL( strPath, &strSysPath, sal_False );

	if ( osl_File_E_None == error )
	{
		if ( DeleteFile( rtl_uString_getStr( strSysPath ) ) )
			error = osl_File_E_None;
		else
			error = MapError( GetLastError() );

		rtl_uString_release( strSysPath );
	}

	return error;
}

/****************************************************************************/
/*	osl_copyFile	*/
/****************************************************************************/

#define osl_File_CopyRecursive	0x0001
#define osl_File_CopyOverwrite	0x0002

oslFileError SAL_CALL osl_copyFile( rtl_uString* strPath, rtl_uString *strDestPath )
{
	rtl_uString	*strSysPath = NULL, *strSysDestPath = NULL;
	oslFileError	error = _osl_getSystemPathFromFileURL( strPath, &strSysPath, sal_False );

	if ( osl_File_E_None == error )
		error = _osl_getSystemPathFromFileURL( strDestPath, &strSysDestPath, sal_False );

	if ( osl_File_E_None == error )
	{
		if ( CopyFile( rtl_uString_getStr( strSysPath ), rtl_uString_getStr( strSysDestPath ), FALSE ) )
			error = osl_File_E_None;
		else
			error = MapError( GetLastError() );
	}

	if ( strSysPath )
		rtl_uString_release( strSysPath );
	if ( strSysDestPath )
		rtl_uString_release( strSysDestPath );

	return error;
}

/****************************************************************************/
/*	osl_moveFile	*/
/****************************************************************************/

oslFileError SAL_CALL osl_moveFile( rtl_uString* strPath, rtl_uString *strDestPath )
{
	rtl_uString	*strSysPath = NULL, *strSysDestPath = NULL;
	oslFileError	error = _osl_getSystemPathFromFileURL( strPath, &strSysPath, sal_False );

	if ( osl_File_E_None == error )
		error = _osl_getSystemPathFromFileURL( strDestPath, &strSysDestPath, sal_False );

	if ( osl_File_E_None == error )
	{
		if ( MoveFileEx( rtl_uString_getStr( strSysPath ), rtl_uString_getStr( strSysDestPath ), MOVEFILE_COPY_ALLOWED | MOVEFILE_WRITE_THROUGH | MOVEFILE_REPLACE_EXISTING ) )
			error = osl_File_E_None;
		else
			error = MapError( GetLastError() );
	}

	if ( strSysPath )
		rtl_uString_release( strSysPath );
	if ( strSysDestPath )
		rtl_uString_release( strSysDestPath );

	return error;
}


/****************************************************************************/
/****************************************************************************/
/*	*/
/*	Directory entry functions */
/*	*/
/****************************************************************************/
/****************************************************************************/



/****************************************************************************/
/*	osl_openDirectory	*/
/****************************************************************************/

oslFileError SAL_CALL osl_openLocalRoot(  rtl_uString *strDirectoryPath, oslDirectory *pDirectory )
{
	rtl_uString		*strSysPath = NULL;
	oslFileError	error;

	/* Assume failure */

	if ( !pDirectory )
		return osl_File_E_INVAL;

	*pDirectory = NULL;

	error = _osl_getSystemPathFromFileURL( strDirectoryPath, &strSysPath, sal_False );

	if ( osl_File_E_None == error )
	{
		Directory_Impl	*pDirImpl;

		pDirImpl = rtl_allocateMemory( sizeof(Directory_Impl) );
		_tcscpy( pDirImpl->szDirectoryPath, rtl_uString_getStr(strSysPath) );

		/* Append backslash if neccessary */

		if ( pDirImpl->szDirectoryPath[_tcslen(pDirImpl->szDirectoryPath) - 1] != L'\\' )
			_tcscat( pDirImpl->szDirectoryPath, L"\\" );

		pDirImpl->uType = DIRECTORYTYPE_LOCALROOT;
		pDirImpl->hEnumDrives = OpenLogicalDrivesEnum();

		if ( pDirImpl->hEnumDrives != INVALID_HANDLE_VALUE )
		{
			*pDirectory = (oslDirectory)pDirImpl;
			error = osl_File_E_None;
		}
		else
		{
			if ( pDirImpl )
				rtl_freeMemory(pDirImpl);

			error = MapError( GetLastError() );
		}

		rtl_uString_release( strSysPath );
	}

	return error;
}

oslFileError SAL_CALL osl_openFileDirectory( rtl_uString *strDirectoryPath, oslDirectory *pDirectory)
{
	rtl_uString		*strSysPath = NULL;
	oslFileError	error;

	/* Assume failure */

	if ( !pDirectory )
		return osl_File_E_INVAL;

	*pDirectory = NULL;

	error = _osl_getSystemPathFromFileURL( strDirectoryPath, &strSysPath, sal_False );
	if ( osl_File_E_None == error )
	{
		Directory_Impl	*pDirImpl;

		pDirImpl = rtl_allocateMemory( sizeof(Directory_Impl) );
		_tcscpy( pDirImpl->szDirectoryPath, rtl_uString_getStr(strSysPath) );

		/* Append backslash if neccessary */

		if ( pDirImpl->szDirectoryPath[_tcslen(pDirImpl->szDirectoryPath) - 1] != L'\\' )
			_tcscat( pDirImpl->szDirectoryPath, L"\\" );
		GetCaseCorrectPathName( pDirImpl->szDirectoryPath, pDirImpl->szDirectoryPath, sizeof(pDirImpl->szDirectoryPath) );

		pDirImpl->uType = DIRECTORYTYPE_FILESYSTEM;
		pDirImpl->hDirectory = OpenDirectory( pDirImpl->szDirectoryPath );

		if ( pDirImpl->hDirectory )
		{
			*pDirectory = (oslDirectory)pDirImpl;
			error = osl_File_E_None;
		}
		else
		{
			if ( pDirImpl )
				rtl_freeMemory(pDirImpl);

			error = MapError( GetLastError() );
		}

		rtl_uString_release( strSysPath );
	}

	return error;
}


oslFileError SAL_CALL osl_openDirectory( rtl_uString *strDirectoryPath, oslDirectory *pDirectory)
{
	oslFileError	error;
	rtl_uString*	strServer = NULL;
	rtl_uString*	strRoot = NULL;
	rtl_uString*	strDirectory = NULL;
	rtl_uString*	strFile = NULL;

	if ( 0 == rtl_ustr_compareIgnoreAsciiCase( strDirectoryPath->buffer, L"file:///" ) )
		error = osl_openLocalRoot( strDirectoryPath, pDirectory );
	else
		error = osl_openFileDirectory( strDirectoryPath, pDirectory );

	return error;
}

/****************************************************************************/
/*	osl_getNextDirectoryEntry	*/
/****************************************************************************/

oslFileError SAL_CALL osl_getNextDrive( oslDirectory Directory, oslDirectoryItem *pItem, sal_uInt32 uHint )
{
	Directory_Impl		*pDirImpl = (Directory_Impl *)Directory;
	DirectoryItem_Impl	*pItemImpl = NULL;
	BOOL				fSuccess;

	/* Assume failure */

	if ( !pItem )
		return osl_File_E_INVAL;

	*pItem = NULL;

	if ( !pDirImpl )
		return osl_File_E_INVAL;

	pItemImpl = rtl_allocateMemory( sizeof(DirectoryItem_Impl) );
	if ( !pItemImpl )
		return osl_File_E_NOMEM;


	ZeroMemory( pItemImpl, sizeof(DirectoryItem_Impl) );
	pItemImpl->uType = DIRECTORYITEM_DRIVE;
	osl_acquireDirectoryItem( (oslDirectoryItem)pItemImpl );
	fSuccess = EnumLogicalDrives( pDirImpl->hEnumDrives, pItemImpl->cDriveString );

	if ( fSuccess )
	{
		*pItem = pItemImpl;
		return osl_File_E_None;
	}
	else
	{
		rtl_freeMemory( pItemImpl );
		return MapError( GetLastError() );
	}
}

oslFileError SAL_CALL osl_getNextFileItem( oslDirectory Directory, oslDirectoryItem *pItem, sal_uInt32 uHint )
{
	Directory_Impl		*pDirImpl = (Directory_Impl *)Directory;
	DirectoryItem_Impl	*pItemImpl = NULL;
	BOOL				fFound;

	/* Assume failure */

	if ( !pItem )
		return osl_File_E_INVAL;

	*pItem = NULL;

	if ( !pDirImpl )
		return osl_File_E_INVAL;

	pItemImpl = rtl_allocateMemory( sizeof(DirectoryItem_Impl) );
	if ( !pItemImpl )
		return osl_File_E_NOMEM;

	memset( pItemImpl, 0, sizeof(DirectoryItem_Impl) );
	fFound = EnumDirectory( pDirImpl->hDirectory, &pItemImpl->FindData );

	if ( fFound )
	{
		pItemImpl->uType = DIRECTORYITEM_FILE;
		pItemImpl->nRefCount = 1;
		_tcscpy( pItemImpl->szFullPath, pDirImpl->szDirectoryPath );
		_tcscat( pItemImpl->szFullPath, pItemImpl->FindData.cFileName );
		*pItem = (oslDirectoryItem)pItemImpl;
		return osl_File_E_None;
	}
	else
	{
		rtl_freeMemory( pItemImpl );
		return MapError( GetLastError() );
	}
}


oslFileError SAL_CALL osl_getNextDirectoryItem( oslDirectory Directory, oslDirectoryItem *pItem, sal_uInt32 uHint )
{
	Directory_Impl		*pDirImpl = (Directory_Impl *)Directory;
	DirectoryItem_Impl	*pItemImpl = NULL;

	/* Assume failure */

	if ( !pItem )
		return osl_File_E_INVAL;

	*pItem = NULL;

	if ( !pDirImpl )
		return osl_File_E_INVAL;

	switch ( pDirImpl->uType )
	{
	case DIRECTORYTYPE_LOCALROOT:
		return osl_getNextDrive( Directory, pItem, uHint );
	case DIRECTORYTYPE_FILESYSTEM:
		return osl_getNextFileItem( Directory, pItem, uHint );
	default:
		return osl_File_E_INVAL;
	}
}


/****************************************************************************/
/*	osl_closeDirectory	*/
/****************************************************************************/

oslFileError SAL_CALL osl_closeDirectory( oslDirectory Directory )
{
	Directory_Impl	*pDirImpl = (Directory_Impl *)Directory;
	oslFileError	eError;

	if ( pDirImpl )
	{
		if ( pDirImpl->uType == DIRECTORYTYPE_FILESYSTEM )
			eError = CloseDirectory( pDirImpl->hDirectory ) ? osl_File_E_None : MapError( GetLastError() );
		else
			eError = CloseLogicalDrivesEnum( pDirImpl->hEnumDrives ) ? osl_File_E_None : MapError( GetLastError() );
		rtl_freeMemory(pDirImpl);
	}
	else
		eError = osl_File_E_INVAL;

	return eError;
}


/****************************************************************************/
/*	osl_getDirectoryItem */
/****************************************************************************/

oslFileError SAL_CALL osl_getDirectoryItem( rtl_uString *strFilePath, oslDirectoryItem *pItem )
{
	oslFileError	error = osl_File_E_None;
	rtl_uString*	strSysFilePath = NULL;
	PATHTYPE		type = PATHTYPE_FILE;
	DWORD			dwPathType;
	TCHAR			szCorrectedPathName[MAX_PATH];

	/* Assume failure */

	if ( !pItem )
		return osl_File_E_INVAL;

	*pItem = NULL;


	error = _osl_getSystemPathFromFileURL( strFilePath, &strSysFilePath, sal_False );

	if ( osl_File_E_None != error )
			return error;

	if ( GetCaseCorrectPathName( strSysFilePath->buffer, szCorrectedPathName, MAX_PATH ) )
	{
		rtl_uString_newFromStr( &strSysFilePath, szCorrectedPathName );
	}

	dwPathType = IsValidFilePath( strSysFilePath->buffer, NULL, VALIDATEPATH_NORMAL );

	if ( dwPathType & PATHTYPE_IS_VOLUME )
		type = PATHTYPE_VOLUME;
	else if ( dwPathType & PATHTYPE_IS_SERVER )
		type = PATHTYPE_NETSERVER;
	else
		type = PATHTYPE_FILE;

	switch ( type )
	{
	case PATHTYPE_NETSERVER:
		{
			DirectoryItem_Impl*	pItemImpl = rtl_allocateMemory( sizeof(DirectoryItem_Impl) );


			if ( !pItemImpl )
				error = osl_File_E_NOMEM;

			if ( osl_File_E_None == error )
			{
				ZeroMemory( pItemImpl, sizeof(DirectoryItem_Impl) );
				pItemImpl->uType = DIRECTORYITEM_SERVER;

				osl_acquireDirectoryItem( (oslDirectoryItem)pItemImpl );

				_tcscpy( pItemImpl->szFullPath, strSysFilePath->buffer );

				// Assign a title anyway
				{
					int iSrc = 2;
					int	iDst = 0;

					while( iSrc < strSysFilePath->length && strSysFilePath->buffer[iSrc] && strSysFilePath->buffer[iSrc] != '\\' )
					{
						pItemImpl->FindData.cFileName[iDst++] = strSysFilePath->buffer[iSrc++];
					}
				}

				*pItem = pItemImpl;
			}
		}
		break;
	case PATHTYPE_VOLUME:
		{
			DirectoryItem_Impl*	pItemImpl = rtl_allocateMemory( sizeof(DirectoryItem_Impl) );


			if ( !pItemImpl )
				error = osl_File_E_NOMEM;

			if ( osl_File_E_None == error )
			{
				ZeroMemory( pItemImpl, sizeof(DirectoryItem_Impl) );
				pItemImpl->uType = DIRECTORYITEM_DRIVE;

				osl_acquireDirectoryItem( (oslDirectoryItem)pItemImpl );

				_tcscpy( pItemImpl->cDriveString, strSysFilePath->buffer );

				if ( pItemImpl->cDriveString[_tcslen(pItemImpl->cDriveString) - 1] != '\\' )
					_tcscat( pItemImpl->cDriveString, TEXT( "\\" ) );

				*pItem = pItemImpl;
			}
		}
		break;
	default:
	case PATHTYPE_FILE:
		{
			HANDLE				hFind;
			WIN32_FIND_DATA		aFindData;

			if ( strSysFilePath->length > 0 && strSysFilePath->buffer[strSysFilePath->length - 1] == '\\' )
				rtl_uString_newFromStr_WithLength( &strSysFilePath, strSysFilePath->buffer, strSysFilePath->length - 1 );

			hFind = lpfnFindFirstFile( rtl_uString_getStr(strSysFilePath), &aFindData );

			if ( hFind != INVALID_HANDLE_VALUE )
			{
				DirectoryItem_Impl	*pItemImpl = rtl_allocateMemory( sizeof(DirectoryItem_Impl) );

				ZeroMemory( pItemImpl, sizeof(DirectoryItem_Impl) );
				osl_acquireDirectoryItem( (oslDirectoryItem)pItemImpl );

				CopyMemory( &pItemImpl->FindData, &aFindData, sizeof(WIN32_FIND_DATA) );
				_tcscpy( pItemImpl->szFullPath, rtl_uString_getStr(strSysFilePath) );
				GetCaseCorrectPathName( pItemImpl->szFullPath, pItemImpl->szFullPath, sizeof(pItemImpl->szFullPath) );

				pItemImpl->uType = DIRECTORYITEM_FILE;
				*pItem = pItemImpl;
				FindClose( hFind );
			}
			else
				error = MapError( GetLastError() );
		}
		break;
	}

	if ( strSysFilePath )
		rtl_uString_release( strSysFilePath );

	return error;
}


/****************************************************************************/
/*	osl_releaseDirectoryItem */
/****************************************************************************/

oslFileError SAL_CALL osl_acquireDirectoryItem( oslDirectoryItem Item )
{
	DirectoryItem_Impl	*pItemImpl = (DirectoryItem_Impl *)Item;

	if ( !pItemImpl )
		return osl_File_E_INVAL;

	pItemImpl->nRefCount++;
	return osl_File_E_None;
}

oslFileError SAL_CALL osl_releaseDirectoryItem( oslDirectoryItem Item )
{
	DirectoryItem_Impl	*pItemImpl = (DirectoryItem_Impl *)Item;

	if ( !pItemImpl )
		return osl_File_E_INVAL;

	if ( ! --pItemImpl->nRefCount )
		rtl_freeMemory( pItemImpl );
	return osl_File_E_None;
}


/****************************************************************************/
/*	osl_getFileStatus */
/****************************************************************************/

BOOL FileTimeToTimeValue( const FILETIME *cpFTime, TimeValue *pTimeVal )
{
	SYSTEMTIME	BaseSysTime;
	FILETIME	BaseFileTime;
	BOOL		fSuccess = FALSE;	/* Assume failure */

	BaseSysTime.wYear         = 1970;
	BaseSysTime.wMonth        = 1;
    BaseSysTime.wDayOfWeek    = 0;
    BaseSysTime.wDay          = 1;
    BaseSysTime.wHour         = 0;
    BaseSysTime.wMinute       = 0;
    BaseSysTime.wSecond       = 0;
    BaseSysTime.wMilliseconds = 0;

	if ( SystemTimeToFileTime(&BaseSysTime, &BaseFileTime) )
	{
		__int64		Value;

		fSuccess = 0 <= (Value = *((__int64 *)cpFTime) - *((__int64 *)&BaseFileTime));

		if ( fSuccess )
		{
			pTimeVal->Seconds  = (unsigned long) (Value / 10000000L);
			pTimeVal->Nanosec  = (unsigned long)((Value % 10000000L) * 100);
		}
	}

	return fSuccess;
}


/****************************************************************************/

static oslFileError SAL_CALL osl_getDriveInfo( oslDirectoryItem Item, oslFileStatus *pStatus, sal_uInt32 uFieldMask )
{
	DirectoryItem_Impl	*pItemImpl = (DirectoryItem_Impl *)Item;
	TCHAR				cDrive[3] = L"A:";
	TCHAR				cRoot[4] = L"A:\\";

	if ( !pItemImpl )
		return osl_File_E_INVAL;

	pStatus->uValidFields = 0;

	cDrive[0] = pItemImpl->cDriveString[0];
	cRoot[0] = pItemImpl->cDriveString[0];

	if ( uFieldMask & osl_FileStatus_Mask_FileName )
	{
		switch ( GetDriveType( cRoot ) )
		{
		case DRIVE_REMOTE:
			{
				TCHAR	szBuffer[1024];
				DWORD	dwBufsize = bufsizeof(szBuffer);
				DWORD	dwResult = WNetGetConnection( cDrive, szBuffer, &dwBufsize );

				if ( NO_ERROR == dwResult )
				{
					TCHAR	szFileName[bufsizeof(szBuffer) + 16];

					swprintf( szFileName, L"%s [%s]", cDrive, szBuffer );
					rtl_uString_newFromStr( &pStatus->ustrFileName, szFileName );
				}
				else
					rtl_uString_newFromStr( &pStatus->ustrFileName, cDrive );
			}
			break;
		case DRIVE_FIXED:
			{
				TCHAR	szVolumeNameBuffer[1024];

				if ( GetVolumeInformation( cRoot, szVolumeNameBuffer, bufsizeof(szVolumeNameBuffer), NULL, NULL, NULL, NULL, 0 ) )
				{
					TCHAR	szFileName[bufsizeof(szVolumeNameBuffer) + 16];

					swprintf( szFileName, L"%s [%s]", cDrive, szVolumeNameBuffer );
					rtl_uString_newFromStr( &pStatus->ustrFileName, szFileName );
				}
				else
					rtl_uString_newFromStr( &pStatus->ustrFileName, cDrive );
			}
			break;
		case DRIVE_CDROM:
		case DRIVE_REMOVABLE:
		case DRIVE_UNKNOWN:
		default:
			rtl_uString_newFromStr( &pStatus->ustrFileName, cDrive );
			break;
		}

		pStatus->uValidFields |= osl_FileStatus_Mask_FileName;
	}

	pStatus->eType = osl_File_Type_Volume;
	pStatus->uValidFields |= osl_FileStatus_Mask_Type;

	if ( uFieldMask & osl_FileStatus_Mask_FileURL )
	{
		rtl_uString	*ustrSystemPath = NULL;

		rtl_uString_newFromStr( &ustrSystemPath, pItemImpl->cDriveString );
		osl_getFileURLFromSystemPath( ustrSystemPath, &pStatus->ustrFileURL );
		rtl_uString_release( ustrSystemPath );
		pStatus->uValidFields |= osl_FileStatus_Mask_FileURL;
	}

	return osl_File_E_None;
}

/****************************************************************************/

static oslFileError SAL_CALL osl_getServerInfo( oslDirectoryItem Item, oslFileStatus *pStatus, sal_uInt32 uFieldMask )
{
	DirectoryItem_Impl	*pItemImpl = (DirectoryItem_Impl *)Item;

	if ( !pItemImpl )
		return osl_File_E_INVAL;

	pStatus->uValidFields = 0;

	pStatus->uValidFields |= osl_FileStatus_Mask_FileName;

	if ( _tcscmp( pItemImpl->FindData.cFileName, TEXT(".") ) == 0 )
		rtl_uString_newFromAscii( &pStatus->ustrFileName, "/" );
	else
		rtl_uString_newFromStr( &pStatus->ustrFileName, pItemImpl->FindData.cFileName );

	pStatus->eType = osl_File_Type_Directory;
	pStatus->uValidFields |= osl_FileStatus_Mask_Type;

	if ( uFieldMask & osl_FileStatus_Mask_FileURL )
	{
		rtl_uString	*ustrSystemPath = NULL;

		rtl_uString_newFromStr( &ustrSystemPath, pItemImpl->szFullPath );
		osl_getFileURLFromSystemPath( ustrSystemPath, &pStatus->ustrFileURL );
		rtl_uString_release( ustrSystemPath );
		pStatus->uValidFields |= osl_FileStatus_Mask_FileURL;
	}

	return osl_File_E_None;
}

/****************************************************************************/

oslFileError SAL_CALL osl_getFileStatus( oslDirectoryItem Item, oslFileStatus *pStatus, sal_uInt32 uFieldMask )
{
	DirectoryItem_Impl	*pItemImpl = (DirectoryItem_Impl *)Item;

	if ( !pItemImpl )
		return osl_File_E_INVAL;

	switch ( pItemImpl->uType  )
	{
	case DIRECTORYITEM_DRIVE:
		return osl_getDriveInfo( Item, pStatus, uFieldMask );
	case DIRECTORYITEM_SERVER:
		return osl_getServerInfo( Item, pStatus, uFieldMask );
	default:
		break;
	}

	if ( uFieldMask & osl_FileStatus_Mask_Validate )
	{
		HANDLE	hFind = lpfnFindFirstFile( pItemImpl->szFullPath, &pItemImpl->FindData );

		if ( hFind != INVALID_HANDLE_VALUE )
			FindClose( hFind );
		else
			return MapError( GetLastError() );

		uFieldMask &= ~	osl_FileStatus_Mask_Validate;
	}

	/* If no fields to retrieve left ignore pStatus */
	if ( !uFieldMask )
		return osl_File_E_None;

	/* Otherwise, this must be a valid pointer */
	if ( !pStatus )
		return osl_File_E_INVAL;

	if ( pStatus->uStructSize != sizeof(oslFileStatus) )
		return osl_File_E_INVAL;

	pStatus->uValidFields = 0;

	/* File time stamps */

	if ( (uFieldMask & osl_FileStatus_Mask_ModifyTime) &&
		FileTimeToTimeValue( &pItemImpl->FindData.ftLastWriteTime, &pStatus->aModifyTime ) )
		pStatus->uValidFields |= osl_FileStatus_Mask_ModifyTime;

	if ( (uFieldMask & osl_FileStatus_Mask_AccessTime) &&
		FileTimeToTimeValue( &pItemImpl->FindData.ftLastAccessTime, &pStatus->aAccessTime ) )
		pStatus->uValidFields |= osl_FileStatus_Mask_AccessTime;

	if ( (uFieldMask & osl_FileStatus_Mask_CreationTime) &&
		FileTimeToTimeValue( &pItemImpl->FindData.ftCreationTime, &pStatus->aCreationTime ) )
		pStatus->uValidFields |= osl_FileStatus_Mask_CreationTime;

	/* Most of the fields are already set, regardless of requiered fields */

	rtl_uString_newFromStr( &pStatus->ustrFileName, pItemImpl->FindData.cFileName );
	pStatus->uValidFields |= osl_FileStatus_Mask_FileName;

	pStatus->eType = pItemImpl->FindData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ? osl_File_Type_Directory : osl_File_Type_Regular;
	pStatus->uValidFields |= osl_FileStatus_Mask_Type;

	pStatus->uAttributes = pItemImpl->FindData.dwFileAttributes;
	pStatus->uValidFields |= osl_FileStatus_Mask_Attributes;

	pStatus->uFileSize = (sal_uInt64)pItemImpl->FindData.nFileSizeLow + ((sal_uInt64)pItemImpl->FindData.nFileSizeHigh << 32);
	pStatus->uValidFields |= osl_FileStatus_Mask_FileSize;

	if ( uFieldMask & osl_FileStatus_Mask_LinkTargetURL )
	{
		rtl_uString	*ustrFullPath = NULL;

		rtl_uString_newFromStr( &ustrFullPath, pItemImpl->szFullPath );
		osl_getFileURLFromSystemPath( ustrFullPath, &pStatus->ustrLinkTargetURL );
		rtl_uString_release( ustrFullPath );

		pStatus->uValidFields |= osl_FileStatus_Mask_LinkTargetURL;
	}

	if ( uFieldMask & osl_FileStatus_Mask_FileURL )
	{
		rtl_uString	*ustrFullPath = NULL;

		rtl_uString_newFromStr( &ustrFullPath, pItemImpl->szFullPath );
		osl_getFileURLFromSystemPath( ustrFullPath, &pStatus->ustrFileURL );
		rtl_uString_release( ustrFullPath );
		pStatus->uValidFields |= osl_FileStatus_Mask_FileURL;
	}

	return osl_File_E_None;
}

/****************************************************************************/
/*	osl_getVolumeInformation */
/****************************************************************************/

oslFileError SAL_CALL osl_getVolumeInformation( rtl_uString *ustrURL, oslVolumeInfo *pInfo, sal_uInt32 uFieldMask )
{
	rtl_uString		*ustrSystemPath = NULL;
	WCHAR			szVolumeNameBuffer[MAX_PATH];
	WCHAR			szFileSystemName[MAX_PATH];
	DWORD			dwSerial;
	DWORD			dwMaxComponentLength;
	DWORD			dwFlags;
	oslFileError	error;
	UINT			uDriveType;


	if ( !pInfo )
		return osl_File_E_INVAL;

	error = _osl_getSystemPathFromFileURL( ustrURL, &ustrSystemPath, sal_False );

	if ( osl_File_E_None == error )
	{
		sal_Unicode	*pszSystemPath = rtl_uString_getStr( ustrSystemPath );

		if ( rtl_uString_getLength( ustrSystemPath ) >= 2 &&
			 ( pszSystemPath[0] >= 'A' && pszSystemPath[0] <= 'Z' ||
			   pszSystemPath[0] >= 'a' && pszSystemPath[0] <= 'z' ) &&
			   pszSystemPath[1] == ':'
			)
		{
			WCHAR			szRootPath[4];

			szRootPath[0] = pszSystemPath[0];
			szRootPath[1] = pszSystemPath[1];
			szRootPath[2] = '\\';
			szRootPath[3] = '\0';

			rtl_uString_newFromStr( &ustrSystemPath, szRootPath );
		}
		else
			error = osl_File_E_INVAL;
	}

	if ( osl_File_E_None == error )
	{
		/* Erase valid fields */
		pInfo->uValidFields = 0;

		uDriveType = GetDriveType( rtl_uString_getStr( ustrSystemPath ) );

		/* Drive does not exist */
		if ( DRIVE_NO_ROOT_DIR == uDriveType )
			error = MapError( ERROR_INVALID_DRIVE );
	}

	if ( osl_File_E_None == error )
	{
		/* Detect, store and validate attributes if requested */
		if ( uFieldMask & osl_VolumeInfo_Mask_Attributes )
		{
			pInfo->uValidFields |= osl_VolumeInfo_Mask_Attributes;
			pInfo->uAttributes = 0;

			switch ( uDriveType )
			{
			case DRIVE_CDROM:
				pInfo->uAttributes |= osl_Volume_Attribute_CompactDisc | osl_Volume_Attribute_Removeable;
				break;
			case DRIVE_REMOVABLE:
				pInfo->uAttributes |= osl_Volume_Attribute_Removeable;
				if ( ustrSystemPath->buffer[1] == ':' &&
					(ustrSystemPath->buffer[0] == 'A' || ustrSystemPath->buffer[0] == 'B' 
					|| ustrSystemPath->buffer[0] == 'a' || ustrSystemPath->buffer[0] == 'b' )
					)
					pInfo->uAttributes |= osl_Volume_Attribute_FloppyDisk;
				break;
			case DRIVE_FIXED:
				pInfo->uAttributes |= osl_Volume_Attribute_FixedDisk;
				break;
			case DRIVE_RAMDISK:
				pInfo->uAttributes |= osl_Volume_Attribute_RAMDisk;
				break;
			case DRIVE_REMOTE:
				pInfo->uAttributes |= osl_Volume_Attribute_Remote;
				break;
			case DRIVE_UNKNOWN:
				pInfo->uAttributes = 0;
				break;
			case DRIVE_NO_ROOT_DIR:
			default:
				pInfo->uValidFields &= ~osl_VolumeInfo_Mask_Attributes;
				pInfo->uAttributes = 0;
				break;
			}
		}

		/* Detect, store and validate space info if requested */
		if ( uFieldMask & ( osl_VolumeInfo_Mask_TotalSpace | osl_VolumeInfo_Mask_UsedSpace | osl_VolumeInfo_Mask_FreeSpace ) )
		{
			if ( GetDiskFreeSpaceEx( rtl_uString_getStr( ustrSystemPath), (PULARGE_INTEGER)&pInfo->uFreeSpace, (PULARGE_INTEGER)&pInfo->uTotalSpace, NULL ) )
			{
				pInfo->uUsedSpace = pInfo->uTotalSpace - pInfo->uFreeSpace;

				pInfo->uValidFields |= osl_VolumeInfo_Mask_TotalSpace | osl_VolumeInfo_Mask_UsedSpace | osl_VolumeInfo_Mask_FreeSpace;
			}
		}

		if ( (uFieldMask & ( osl_VolumeInfo_Mask_MaxNameLength | osl_VolumeInfo_Mask_MaxPathLength | osl_VolumeInfo_Mask_FileSystemName )) &&
			GetVolumeInformation( rtl_uString_getStr( ustrSystemPath), szVolumeNameBuffer, MAX_PATH, &dwSerial, &dwMaxComponentLength, &dwFlags,
			 szFileSystemName, MAX_PATH ) )
		{
			/* Store and validate maximum component name length */
			pInfo->uValidFields |= osl_VolumeInfo_Mask_MaxNameLength;
			pInfo->uMaxNameLength = dwMaxComponentLength;

			/* Store and validate maximum path length */
			pInfo->uValidFields |= osl_VolumeInfo_Mask_MaxPathLength;
			pInfo->uMaxPathLength = MAX_PATH;

			/* Store and validate file system name length */
			pInfo->uValidFields |= osl_VolumeInfo_Mask_FileSystemName;
			rtl_uString_newFromStr( &pInfo->ustrFileSystemName, szFileSystemName );
		}

		if ( uFieldMask & osl_VolumeInfo_Mask_DeviceHandle )
		{
			/* Store and validate device handle */
			pInfo->uValidFields |= osl_VolumeInfo_Mask_DeviceHandle;
			osl_getFileURLFromSystemPath( ustrSystemPath, (rtl_uString **)pInfo->pDeviceHandle );
		}

	}


    // #90846# check for NULL pointer because
    // rtl_uString_release doesn't allow NULL
    // pointer
    if ( ustrSystemPath )
	    rtl_uString_release( ustrSystemPath );

	return error;
}

/****************************************************************************/
/*	osl_unmountVolumeDevice */
/****************************************************************************/

oslFileError SAL_CALL osl_unmountVolumeDevice( oslVolumeDeviceHandle Handle )
{
	if ( Handle )
		return osl_File_E_None;
	else
		return osl_File_E_INVAL;
}

/****************************************************************************/
/*	osl_automountVolumeDevice */
/****************************************************************************/

oslFileError SAL_CALL osl_automountVolumeDevice( oslVolumeDeviceHandle Handle )
{
	if ( Handle )
		return osl_File_E_None;
	else
		return osl_File_E_INVAL;
}

/****************************************************************************/
/*	osl_acquireVolumeDeviceHandle */
/****************************************************************************/

oslFileError SAL_CALL osl_acquireVolumeDeviceHandle( oslVolumeDeviceHandle Handle )
{
	if ( Handle )
	{
		rtl_uString_acquire( (rtl_uString *)Handle );
		return osl_File_E_None;
	}
	else
		return osl_File_E_INVAL;
}

/****************************************************************************/
/*	osl_releaseVolumeDeviceHandle */
/****************************************************************************/

oslFileError SAL_CALL osl_releaseVolumeDeviceHandle( oslVolumeDeviceHandle Handle )
{
	if ( Handle )
	{
		rtl_uString_release( (rtl_uString *)Handle );
		return osl_File_E_None;
	}
	else
		return osl_File_E_INVAL;
}

/****************************************************************************/
/*	osl_getVolumeDeviceMountPath */
/****************************************************************************/

oslFileError SAL_CALL osl_getVolumeDeviceMountPath( oslVolumeDeviceHandle Handle, rtl_uString **pstrPath )
{
	if ( Handle && pstrPath )
	{
		rtl_uString_assign( pstrPath, (rtl_uString *)Handle );
		return osl_File_E_None;
	}
	else
		return osl_File_E_INVAL;
}

/****************************************************************************/
/*	osl_getSystemPathFromFileURL */
/****************************************************************************/

static sal_Bool _osl_decodeURL( rtl_String* strUTF8, rtl_uString** pstrDecodedURL )
{
	sal_Char		*pBuffer;
	const sal_Char	*pSrcEnd;
	const sal_Char	*pSrc;
	sal_Char		*pDest;
	sal_Int32		nSrcLen;
	sal_Bool		bValidEncoded = sal_True;	/* Assume success */

	/* The resulting decoded string length is shorter or equal to the source length */

	nSrcLen = rtl_string_getLength(strUTF8);
	pBuffer = rtl_allocateMemory( nSrcLen + 1 );

	pDest = pBuffer;
	pSrc = rtl_string_getStr(strUTF8);
	pSrcEnd = pSrc + nSrcLen;

	/* Now decode the URL what should result in an UTF8 string */
	while ( bValidEncoded && pSrc < pSrcEnd )
	{
		switch ( *pSrc )
		{
		case '%':
			{
				sal_Char	aToken[3];
				sal_Char	aChar;

				pSrc++;
				aToken[0] = *pSrc++;
				aToken[1] = *pSrc++;
				aToken[2] = 0;

				aChar = (sal_Char)strtoul( aToken, NULL, 16 );

				/* The chars are path delimiters and must not be encoded */

				if ( '\\' == aChar || '/' == aChar || ':' == aChar )
					bValidEncoded = sal_False;
				else
					*pDest++ = aChar;
			}
			break;
		default:
			*pDest++ = *pSrc++;
			break;
		}
	}

	*pDest++ = 0;

	/* Convert the buffer data to UNICODE */

	if ( bValidEncoded )
		rtl_string2UString( pstrDecodedURL, pBuffer, rtl_str_getLength(pBuffer), RTL_TEXTENCODING_UTF8, OUSTRING_TO_OSTRING_CVTFLAGS );

	/* Release the buffer */

	rtl_freeMemory( pBuffer );

	return bValidEncoded;
}

/* Converts a FILE-URL into a system path */
static oslFileError SAL_CALL _osl_getSystemPathFromFileURL( rtl_uString *strURL, rtl_uString **pustrPath, sal_Bool bAllowRelative )
{
	rtl_String			*strUTF8 = NULL;
	rtl_uString			*strDecodedURL = NULL;
	rtl_uString			*strTempPath = NULL;
	const sal_Unicode	*pDecodedURL;
	sal_uInt32			nDecodedLen;
	sal_Bool			bValidEncoded;
	oslFileError		nError = osl_File_E_INVAL;	/* Assume failure */

	/*  If someone hasn't encoded the complete URL we convert it to UTF8 now to prevent from
		having a mixed encoded URL later */

	rtl_uString2String( &strUTF8, rtl_uString_getStr( strURL ), rtl_uString_getLength( strURL ), RTL_TEXTENCODING_UTF8, OUSTRING_TO_OSTRING_CVTFLAGS );

	/* If the length of strUTF8 and strURL differs it indicates that the URL was not correct encoded */

	OSL_ENSURE_FILE( strUTF8->length == strURL->length, "osl_getSystemPathFromFileURL: \"%s\" is not encoded !!!", strURL );

	bValidEncoded = _osl_decodeURL( strUTF8, &strDecodedURL );

	/* Release the encoded UTF8 string */

	rtl_string_release( strUTF8 );


	if ( bValidEncoded )
	{

		/* Replace backslashes and pipes */

		rtl_uString_newReplace( &strDecodedURL, strDecodedURL, '/', '\\' );
		rtl_uString_newReplace( &strDecodedURL, strDecodedURL, '|', ':' );

		pDecodedURL = rtl_uString_getStr( strDecodedURL );
		nDecodedLen = rtl_uString_getLength( strDecodedURL );

		/* Must start with "file://" */

		if ( 0 == rtl_ustr_ascii_shortenedCompare_WithLength( pDecodedURL, nDecodedLen, "file:\\\\", 7 ) )
		{
			sal_uInt32	nSkip;

			if ( 0 == rtl_ustr_ascii_shortenedCompare_WithLength( pDecodedURL, nDecodedLen, "file:\\\\\\", 8 ) )
				nSkip = 8;
			else if ( 
				0 == rtl_ustr_ascii_shortenedCompare_WithLength( pDecodedURL, nDecodedLen, "file:\\\\localhost\\", 17 ) ||
				0 == rtl_ustr_ascii_shortenedCompare_WithLength( pDecodedURL, nDecodedLen, "file:\\\\127.0.0.1\\", 17 ) 
			)
				nSkip = 17;
			else 
				nSkip = 5;

			/* Indicates local root */
			if ( nDecodedLen == nSkip )
				rtl_uString_newFromStr_WithLength( &strTempPath, L"\\\\.\\", 4 );
			else
				rtl_uString_newFromStr_WithLength( &strTempPath, pDecodedURL + nSkip, nDecodedLen - nSkip );

			if ( IsValidFilePath( strTempPath->buffer, NULL, VALIDATEPATH_ALLOW_ELLIPSE ) )
				nError = osl_File_E_None;
		}
		else if ( bAllowRelative )	/* This maybe a relative file URL */
		{
			rtl_uString_assign( &strTempPath, strDecodedURL );

			if ( IsValidFilePath( strTempPath->buffer, NULL, VALIDATEPATH_ALLOW_RELATIVE | VALIDATEPATH_ALLOW_ELLIPSE ) )
				nError = osl_File_E_None;
		}
/*
		else
			OSL_ENSURE_FILE( !nError, "osl_getSystemPathFromFileURL: \"%s\" is not an absolute FileURL !!!", strURL );
*/

	}

	/* Release the decoded URL */

	if ( strDecodedURL )
		rtl_uString_release( strDecodedURL );

	/* If no error occured assign output path */

	if ( osl_File_E_None == nError )
		rtl_uString_assign( pustrPath, strTempPath );

	/* release the temporary path */

	if ( strTempPath )
		rtl_uString_release( strTempPath );

	/*
	OSL_ENSURE_FILE( !nError, "osl_getSystemPathFromFileURL: \"%s\" is not a FileURL !!!", strURL );
	*/

	return nError;
}

/****************************************************************************/

oslFileError SAL_CALL osl_getSystemPathFromFileURL( rtl_uString *ustrURL, rtl_uString **pustrPath)
{
	oslFileError	eError = _osl_getSystemPathFromFileURL( ustrURL, pustrPath, sal_True );

/*
	if ( osl_File_E_None != eError )
	{
		rtl_uString	*ustrTemp = NULL;

		eError = _osl_getFileURLFromSystemPath( ustrURL, &ustrTemp );

		if ( osl_File_E_None == eError )
			rtl_uString_assign( pustrPath, ustrURL );

		if ( ustrTemp )
			rtl_uString_release( ustrTemp );
	}
*/

	return eError;
}

/****************************************************************************/
/*	osl_getFileURLFromSystemPath */
/****************************************************************************/

static void _osl_encodeURL( rtl_uString *strURL, rtl_String **pstrEncodedURL )
{
	/* Encode non ascii characters within the URL */

	rtl_String		*strUTF8 = NULL;
	sal_Char		*pszEncodedURL;
	const sal_Char	*pURLScan;
	sal_Char		*pURLDest;
	sal_Int32		nURLScanLen;
	sal_Int32		nURLScanCount;

	/* Now create an UTF8 encoded string */

	rtl_uString2String( &strUTF8, rtl_uString_getStr( strURL ), rtl_uString_getLength( strURL ), RTL_TEXTENCODING_UTF8, OUSTRING_TO_OSTRING_CVTFLAGS );

	/* Allocate Memory for destination buffer */

	pszEncodedURL = (sal_Char*) rtl_allocateMemory( (rtl_string_getLength( strUTF8 ) * 3 + 1)  * sizeof(sal_Char) );

	pURLDest = pszEncodedURL;
	pURLScan = rtl_string_getStr( strUTF8 );
	nURLScanLen = rtl_string_getLength( strUTF8 );
	nURLScanCount = 0;

	while ( nURLScanCount < nURLScanLen )
	{
		sal_Char	cCurrent = *pURLScan;

		switch ( cCurrent )
		{
		default:
			if (!( ( cCurrent >= 'a' && cCurrent <= 'z' ) || ( cCurrent >= 'A' && cCurrent <= 'Z' ) || ( cCurrent >= '0' && cCurrent <= '9' ) ) )
			{
				sprintf( pURLDest, "%%%02X", (unsigned char)cCurrent );
				pURLDest += 3;
				break;
			}
		case '!':
		case '\'':
		case '(':
		case ')':
		case '*':
		case '-':
		case '.':
		case '_':
		case '~':
		/* case '$': */
		case '&':
		case '+':
		case ',':
		case '=':
		case '@':
		case ':':
		case '/':
		case '\\':
		case '|':
			*pURLDest++ = cCurrent;
			break;
		case 0:
			break;
		}

		pURLScan++;
		nURLScanCount++;
	}


	*pURLDest = 0;

	/* Free the UTF8 encoded source buffer */

	rtl_string_release( strUTF8 );

    
	/* Create a rtl_string from buffer */

    rtl_string_newFromStr( pstrEncodedURL, pszEncodedURL );

	/* Release the destination buffer */

	rtl_freeMemory( pszEncodedURL );
}



/* Converts a system path into a FILE-URL */
static oslFileError SAL_CALL _osl_getFileURLFromSystemPath( rtl_uString* strPath, rtl_uString** pstrURL )
{
	oslFileError	nError = osl_File_E_INVAL; /* Assume failure */
	rtl_uString		*strTempURL = NULL;
	DWORD			dwPathType;

	if ( strPath && (dwPathType = IsValidFilePath( strPath->buffer, NULL, VALIDATEPATH_ALLOW_RELATIVE )) )
	{
		rtl_uString	*strTempPath = NULL;

		/* Replace backslashes */

		rtl_uString_newReplace( &strTempPath, strPath, '\\', '/' );

		switch ( dwPathType & PATHTYPE_MASK_TYPE )
		{
		case PATHTYPE_RELATIVE:
			rtl_uString_assign( &strTempURL, strTempPath );
			nError = osl_File_E_None;
			break;
		case PATHTYPE_ABSOLUTE_UNC:
			rtl_uString_newFromAscii( &strTempURL, "file:" );
			rtl_uString_newConcat( &strTempURL, strTempURL, strTempPath );
			nError = osl_File_E_None;
			break;
		case PATHTYPE_ABSOLUTE_LOCAL:
			rtl_uString_newFromAscii( &strTempURL, "file:///" );
			rtl_uString_newConcat( &strTempURL, strTempURL, strTempPath );
			nError = osl_File_E_None;
			break;
		default:
			break;
		}

		/* Release temp path */

		rtl_uString_release( strTempPath );
	}

	if ( osl_File_E_None == nError )
	{
		rtl_String	*strEncodedURL = NULL;

		/* Encode the URL */

		_osl_encodeURL( strTempURL, &strEncodedURL );

		/* Provide URL via unicode string */

		rtl_string2UString( pstrURL, rtl_string_getStr(strEncodedURL), rtl_string_getLength(strEncodedURL), RTL_TEXTENCODING_ASCII_US, OUSTRING_TO_OSTRING_CVTFLAGS );
		rtl_string_release( strEncodedURL );
	}

	/* Release temp URL */

	if ( strTempURL )
		rtl_uString_release( strTempURL );

	/*
	OSL_ENSURE_FILE( !nError, "osl_getFileURLFromSystemPath: \"%s\" is not a systemPath !!!", strPath );
	*/

	return nError;
}

/****************************************************************************/

oslFileError SAL_CALL osl_getFileURLFromSystemPath( rtl_uString* ustrPath, rtl_uString** pustrURL )
{
	oslFileError	eError = _osl_getFileURLFromSystemPath( ustrPath, pustrURL );

/*
	if ( osl_File_E_None != eError )
	{
		rtl_uString	*ustrTemp = NULL;

		eError = _osl_getSystemPathFromFileURL( ustrPath, &ustrTemp, sal_True );

		if ( osl_File_E_None == eError )
			rtl_uString_assign( pustrURL, ustrPath );

		if ( ustrTemp )
			rtl_uString_release( ustrTemp );
	}
*/

	return eError;
}

/****************************************************************************/
/*	osl_searchFileURL */
/****************************************************************************/

oslFileError SAL_CALL osl_searchFileURL( rtl_uString *ustrFileName, rtl_uString *ustrSystemSearchPath, rtl_uString **pustrPath)
{
	rtl_uString		*ustrUNCPath = NULL;
	rtl_uString		*ustrSysPath = NULL;
	oslFileError	error;

	/* First try to interpret the file name as an URL even a relative one */

	error = _osl_getSystemPathFromFileURL( ustrFileName, &ustrUNCPath, sal_True );

	/* So far we either have an UNC path or something invalid */

	/* Now create a system path */

	if ( osl_File_E_None == error )
		error = _osl_getSystemPathFromFileURL( ustrUNCPath, &ustrSysPath, sal_True );

	if ( osl_File_E_None == error )
	{
		DWORD	nBufferLength;
		DWORD	dwResult;		
		LPTSTR	lpBuffer = NULL;
		LPTSTR	lpszFilePart;


		/* Repeat calling SearchPath ... */

		/* Start with MAX_PATH for the buffer. In most cases this will be enough and does not force the loop to runtwice */

		dwResult = MAX_PATH;

		do
		{
			/* If search path is empty use a NULL pointer instead according to MSDN documentation of SearchPath */

			LPCTSTR	lpszSearchPath = ustrSystemSearchPath && ustrSystemSearchPath->length ? ustrSystemSearchPath->buffer : NULL;
			LPCTSTR	lpszSearchFile = ustrSysPath->buffer;

			/* Allocate space for buffer according to previous returned count of required chars */

			/* +1 is not neccessary if we follow MSDN documentation but for robustness we do so */

			nBufferLength = dwResult + 1;
			lpBuffer = lpBuffer ? rtl_reallocateMemory( lpBuffer, nBufferLength * sizeof(TCHAR) ) : rtl_allocateMemory( nBufferLength * sizeof(TCHAR) );

			dwResult = lpfnSearchPath( lpszSearchPath, lpszSearchFile, NULL, nBufferLength, lpBuffer, &lpszFilePart ); 
		} while ( dwResult && dwResult >= nBufferLength );

		/*	... until an error occures or buffer is large enough. 
			dwResult == nBufferLength can not happen according to documentation but lets be robust ;-) */

		if ( dwResult )
		{
			rtl_uString_newFromStr( &ustrSysPath, lpBuffer );
			error = osl_getFileURLFromSystemPath( ustrSysPath, pustrPath );
		}
		else
		{
			WIN32_FIND_DATA	aFindFileData;
			HANDLE	hFind;

			/* Somthing went wrong, perhaps the path was absolute */
			error = MapError( GetLastError() );

			hFind = lpfnFindFirstFile( ustrSysPath->buffer, &aFindFileData );

			if ( ISVALIDHANDLE(hFind) )
			{
				error = osl_getFileURLFromSystemPath( ustrSysPath, pustrPath );
				FindClose( hFind );
			}
		}

		rtl_freeMemory( lpBuffer );
	}

	/* Release temporary created path names */

	if ( ustrSysPath )
		rtl_uString_release( ustrSysPath );

	if ( ustrUNCPath )
		rtl_uString_release( ustrUNCPath );

	return error;
}


/****************************************************************************/
/*	osl_getAbsoluteFileURL */
/****************************************************************************/

oslMutex	g_CurrentDirectoryMutex; /* Initialized in dllentry.c */

oslFileError SAL_CALL osl_getAbsoluteFileURL( rtl_uString* ustrBaseURL, rtl_uString* ustrRelativeURL, rtl_uString** pustrAbsoluteURL )
{
	oslFileError	eError;
	rtl_uString		*ustrRelSysPath = NULL;
	
	eError = osl_getSystemPathFromFileURL( ustrRelativeURL, &ustrRelSysPath );

	if ( !eError )
	{
		rtl_uString	*ustrBaseSysPath = NULL;

		if ( !ustrBaseURL || !ustrBaseURL->length )
		{
			OSL_ENSURE( 0, "osl_getAbsoluteFileURL called with empty base URL. Defaulting to current working directory" );
		}
		else
		{
			eError = _osl_getSystemPathFromFileURL( ustrBaseURL, &ustrBaseSysPath, sal_False );
		}

		if ( !eError )
		{
			TCHAR	szBuffer[MAX_PATH];
			TCHAR	szCurrentDir[MAX_PATH];
			LPTSTR	lpFilePart = NULL;
			DWORD	dwResult;

			if ( ustrBaseSysPath )
			{
				osl_acquireMutex( g_CurrentDirectoryMutex );

				GetCurrentDirectory( MAX_PATH, szCurrentDir );
				SetCurrentDirectory( ustrBaseSysPath->buffer );
			}

			dwResult = GetFullPathName( ustrRelSysPath->buffer, MAX_PATH, szBuffer, &lpFilePart );

			if ( ustrBaseSysPath )
			{
				SetCurrentDirectory( szCurrentDir );

				osl_releaseMutex( g_CurrentDirectoryMutex );
			}

			if ( dwResult )
			{
				if ( dwResult >= MAX_PATH )
					eError = osl_File_E_INVAL;
				else
				{
					rtl_uString	*ustrAbsSysPath = NULL;

					rtl_uString_newFromStr( &ustrAbsSysPath, szBuffer );

					eError = osl_getFileURLFromSystemPath( ustrAbsSysPath, pustrAbsoluteURL );

					if ( ustrAbsSysPath )
						rtl_uString_release( ustrAbsSysPath );
				}
			}
			else
				eError = MapError( GetLastError() );
		}

		if ( ustrBaseSysPath )
			rtl_uString_release( ustrBaseSysPath );
	}

	if ( ustrRelSysPath )
		rtl_uString_release( ustrRelSysPath );

	return	eError;
}


/****************************************************************************/
/*	osl_getCanonicalName */
/****************************************************************************/

oslFileError SAL_CALL osl_getCanonicalName( rtl_uString *strRequested, rtl_uString **strValid )
{
	rtl_uString_newFromString(strValid, strRequested);

	return osl_File_E_None;
}


/****************************************************************************/
/*	osl_setFileAttributes */
/****************************************************************************/

// Sets attributes to a given file
oslFileError SAL_CALL osl_setFileAttributes( rtl_uString *ustrFileURL, sal_uInt64 uAttributes )
{
	oslFileError	error;
	rtl_uString		*ustrSysPath = NULL;
	DWORD			dwFileAttributes;
	BOOL			fSuccess;

	// Converts the normalized path into a systempath
	error = _osl_getSystemPathFromFileURL( ustrFileURL, &ustrSysPath, sal_False );

	if ( osl_File_E_None != error )
		return error;

	dwFileAttributes = GetFileAttributes( rtl_uString_getStr(ustrSysPath) );

	if ( (DWORD)-1 != dwFileAttributes )
	{
		dwFileAttributes &= ~(FILE_ATTRIBUTE_READONLY | FILE_ATTRIBUTE_HIDDEN);

		if ( uAttributes & osl_File_Attribute_ReadOnly )
			dwFileAttributes |= FILE_ATTRIBUTE_READONLY;

		if ( uAttributes & osl_File_Attribute_Hidden )
			dwFileAttributes |= FILE_ATTRIBUTE_HIDDEN;

		fSuccess = SetFileAttributes( rtl_uString_getStr(ustrSysPath), dwFileAttributes );
	}
	else
		fSuccess = FALSE;

	if ( !fSuccess )
		error = MapError( GetLastError() );

	rtl_uString_release( ustrSysPath );

	return error;
}

/****************************************************************************/
/*	osl_setFileTime */
/****************************************************************************/

// Converts a TimeValue into a FILETIME
BOOL TimeValueToFileTime(const TimeValue *cpTimeVal, FILETIME *pFTime)
{
	SYSTEMTIME	BaseSysTime;
	FILETIME	BaseFileTime;
	FILETIME	FTime;
	__int64		localTime;
	BOOL		fSuccess = FALSE;

	BaseSysTime.wYear         = 1970;
	BaseSysTime.wMonth        = 1;
    BaseSysTime.wDayOfWeek    = 0;
    BaseSysTime.wDay          = 1;
    BaseSysTime.wHour         = 0;
    BaseSysTime.wMinute       = 0;
    BaseSysTime.wSecond       = 0;
    BaseSysTime.wMilliseconds = 0;

	if (cpTimeVal==NULL)
		return fSuccess;

	if ( SystemTimeToFileTime(&BaseSysTime, &BaseFileTime) )
	{
		__int64 timeValue;
		localTime=cpTimeVal->Seconds*(__int64)10000000+cpTimeVal->Nanosec/100;
		*(__int64 *)&FTime=localTime;
		fSuccess = 0 <= (timeValue= *((__int64 *)&BaseFileTime) + *((__int64 *) &FTime));
		if (fSuccess)
			*(__int64 *)pFTime=timeValue;
	}

	return fSuccess;
}

// Sets the time parameters to a given file
oslFileError SAL_CALL osl_setFileTime( rtl_uString *filePath, TimeValue *aCreationTime, TimeValue *aLastAccessTime, TimeValue *aLastWriteTime)
{
	oslFileError error;
	rtl_uString *sysPath=NULL;
	FILETIME *lpCreationTime=NULL;
	FILETIME *lpLastAccessTime=NULL;
	FILETIME *lpLastWriteTime=NULL;
	FILETIME ftCreationTime;
	FILETIME ftLastAccessTime;
	FILETIME ftLastWriteTime;
	HANDLE hFile;
	BOOL fSuccess;


	error=_osl_getSystemPathFromFileURL(filePath, &sysPath, sal_False);

	if (error==osl_File_E_INVAL)
		return error;

	hFile=CreateFileW(rtl_uString_getStr(sysPath), GENERIC_WRITE, 0, NULL , OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	rtl_uString_release(sysPath);

	if (hFile==INVALID_HANDLE_VALUE)
		return osl_File_E_NOENT;

	if (TimeValueToFileTime(aCreationTime, &ftCreationTime))
		lpCreationTime=&ftCreationTime;

	if (TimeValueToFileTime(aLastAccessTime, &ftLastAccessTime))
		lpLastAccessTime=&ftLastAccessTime;

	if (TimeValueToFileTime(aLastWriteTime, &ftLastWriteTime))
		lpLastWriteTime=&ftLastWriteTime;

	fSuccess=SetFileTime(hFile, lpCreationTime, lpLastAccessTime, lpLastWriteTime);

	CloseHandle(hFile);

	if (!fSuccess)
		return osl_File_E_INVAL;
	else
		return osl_File_E_None;
}


/****************************************************************************/

/*----------------------------------------*/
/* Undocumented in SHELL32.DLL ordinal 35 */
/*----------------------------------------*/

BOOL WINAPI PathRemoveFileSpec( LPTSTR lpPath )
{
	BOOL	fSuccess = FALSE;	// Assume failure
	LPTSTR	lpLastBkSlash = _tcsrchr( lpPath, '\\' );
	LPTSTR	lpLastSlash = _tcsrchr( lpPath, '/' );
	LPTSTR	lpLastDelimiter = lpLastSlash > lpLastBkSlash ? lpLastSlash : lpLastBkSlash;

	if ( lpLastDelimiter )
	{
		if ( 0 == *(lpLastDelimiter + 1) )
		{
			if ( lpLastDelimiter > lpPath && *(lpLastDelimiter - 1) != ':' )
			{
				*lpLastDelimiter = 0;
				fSuccess = TRUE;
			}
		}
		else
		{
			*(++lpLastDelimiter) = 0;
			fSuccess = TRUE;
		}
	}

	return fSuccess;
}

/*----------------------------------------*/
/* Undocumented in SHELL32.DLL ordinal 32 */
/*----------------------------------------*/

LPTSTR WINAPI PathAddBackslash( LPTSTR lpPath )
{
	LPTSTR	lpEndPath = NULL;

	if ( lpPath )
	{
		int		nLen = _tcslen(lpPath);

		if ( !nLen || lpPath[nLen-1] != '\\' && lpPath[nLen-1] != '/' && nLen < MAX_PATH - 1 )
		{
			lpEndPath = lpPath + nLen;
			*lpEndPath++ = '\\';
			*lpEndPath = 0;
		}
	}

	return lpEndPath;
}


/*-----------------------------------------*/
/* Same as GetLongPathName but also 95/NT4 */
/*-----------------------------------------*/

DWORD WINAPI GetCaseCorrectPathNameEx(
	LPCTSTR	lpszShortPath,	// file name
	LPTSTR	lpszLongPath,	// path buffer
	DWORD	cchBuffer,		// size of path buffer 
	DWORD	nSkipLevels
)
{
	TCHAR	szPath[MAX_PATH];
	BOOL	fSuccess;

	_tcscpy( szPath, lpszShortPath );

	fSuccess = PathRemoveFileSpec( szPath );

	if ( fSuccess )
	{
		int nLenShort = _tcslen( lpszShortPath );
		int nLen = _tcslen( szPath );
		LPCTSTR	lpszFileSpec = lpszShortPath + nLen;
		BOOL	bSkipThis;

		if ( 0 == _tcscmp( lpszFileSpec, TEXT("..") ) )
		{
			bSkipThis = TRUE;
			nSkipLevels += 1;
		}
		else if ( 
			0 == _tcscmp( lpszFileSpec, TEXT(".") ) || 
			0 == _tcscmp( lpszFileSpec, TEXT("\\") ) ||
			0 == _tcscmp( lpszFileSpec, TEXT("/") ) 
			)
		{
			bSkipThis = TRUE;
		}
		else if ( nSkipLevels )
		{
			bSkipThis = TRUE;
			nSkipLevels--;
		}
		else
			bSkipThis = FALSE;

		GetCaseCorrectPathNameEx( szPath, szPath, MAX_PATH, nSkipLevels );

		PathAddBackslash( szPath );

		/* Analyze parent if not only a trailing backslash was cutted but a real file spec */

		if ( !bSkipThis )
		{
			WIN32_FIND_DATA	aFindFileData;
			HANDLE	hFind = lpfnFindFirstFile( lpszShortPath, &aFindFileData );

			if ( ISVALIDHANDLE(hFind) )
			{
				_tcscat( szPath, aFindFileData.cFileName[0] ? aFindFileData.cFileName : aFindFileData.cAlternateFileName );

				FindClose( hFind );
			}
			else
				return 0;
		}

	}
	else
	{
		/* File specification can't be removed therefore the short path is either a drive
			or a network share */

		/* If still levels to skip are left, the path specification tries to travel below the
			file system root */

		if ( nSkipLevels )
			return 0;

		_tcsupr( szPath );
	}

	_tcscpy( lpszLongPath, szPath );

	return _tcslen( lpszLongPath );
}


DWORD WINAPI GetCaseCorrectPathName(
	LPCTSTR	lpszShortPath,	// file name
	LPTSTR	lpszLongPath,	// path buffer
	DWORD	cchBuffer		// size of path buffer 
)
{
	return GetCaseCorrectPathNameEx( lpszShortPath, lpszLongPath, cchBuffer, 0 );
}

/****************************************************************************/

typedef struct tagDRIVEENUM {
	LPCTSTR	lpIdent;
	TCHAR	cBuffer[/*('Z' - 'A' + 1) * sizeof("A:\\") + 1*/256];
	LPCTSTR	lpCurrent;
} DRIVEENUM, * PDRIVEENUM, FAR * LPDRIVEENUM;

/****************************************************************************/

HANDLE WINAPI OpenLogicalDrivesEnum()
{
	LPDRIVEENUM	pEnum = NULL;

	pEnum = (LPDRIVEENUM)HeapAlloc( GetProcessHeap(), 0, sizeof(DRIVEENUM) );

	if ( pEnum )
	{
		DWORD	dwNumCopied = GetLogicalDriveStrings( (sizeof(pEnum->cBuffer) - 1) / sizeof(TCHAR), pEnum->cBuffer );

		if ( dwNumCopied && dwNumCopied < sizeof(pEnum->cBuffer) / sizeof(TCHAR) )
		{
			pEnum->lpCurrent = pEnum->cBuffer;
			pEnum->lpIdent = L"tagDRIVEENUM";
		}
		else
		{
			HeapFree( GetProcessHeap(), 0, pEnum );
			pEnum = NULL;
		}
	}

	return pEnum ? (HANDLE)pEnum : INVALID_HANDLE_VALUE;
}

/****************************************************************************/

BOOL WINAPI EnumLogicalDrives( HANDLE hEnum, LPTSTR lpBuffer )
{
	BOOL		fSuccess = FALSE;
	LPDRIVEENUM	pEnum = (LPDRIVEENUM)hEnum;

	if ( pEnum )
	{
		int	nLen = _tcslen( pEnum->lpCurrent );

		if ( nLen )
		{
			CopyMemory( lpBuffer, pEnum->lpCurrent, (nLen + 1) * sizeof(TCHAR) );
			pEnum->lpCurrent += nLen + 1;
			fSuccess = TRUE;
		}
		else
			SetLastError( ERROR_NO_MORE_FILES );
	}
	else
		SetLastError( ERROR_INVALID_HANDLE );

	return fSuccess;
}

/****************************************************************************/

BOOL WINAPI CloseLogicalDrivesEnum( HANDLE hEnum )
{
	BOOL		fSuccess = FALSE;
	LPDRIVEENUM	pEnum = (LPDRIVEENUM)hEnum;

	if ( pEnum )
	{
		HeapFree( GetProcessHeap(), 0, pEnum );
		fSuccess = TRUE;
	}
	else
		SetLastError( ERROR_INVALID_HANDLE );

	return fSuccess;
}


/****************************************************************************/

typedef struct tagDIRECTORY
{
	HANDLE			hFind;
	WIN32_FIND_DATA	aFirstData;
} DIRECTORY, *PDIRECTORY, FAR *LPDIRECTORY;


/****************************************************************************/

HANDLE WINAPI OpenDirectory( LPCTSTR lpszPath )
{
	LPDIRECTORY	pDirectory = (LPDIRECTORY)HeapAlloc( GetProcessHeap(), 0, sizeof(DIRECTORY) );

	if ( pDirectory )
	{
		TCHAR	szFileMask[MAX_PATH];
		int		nLen;

		_tcscpy( szFileMask, lpszPath );
		nLen = _tcslen( szFileMask );

		if ( nLen && szFileMask[nLen-1] != '\\' )
			_tcscat( szFileMask, TEXT("\\*.*") );
		else
			_tcscat( szFileMask, TEXT("*.*") );
		pDirectory->hFind = lpfnFindFirstFile( szFileMask, &pDirectory->aFirstData );

		if ( ISVALIDHANDLE(pDirectory->hFind) )
		{
			/* Everything O.K. */
		}
		else
		{
			HeapFree( GetProcessHeap(), 0, pDirectory );
			pDirectory = NULL;
		}
	}

	return (HANDLE)pDirectory;
}

/****************************************************************************/

BOOL WINAPI CloseDirectory( HANDLE hDirectory )
{
	BOOL		fSuccess = FALSE;
	LPDIRECTORY	pDirectory = (LPDIRECTORY)hDirectory;

	if ( pDirectory )
	{
		if ( ISVALIDHANDLE(pDirectory->hFind) )
			fSuccess = FindClose( pDirectory->hFind );

		fSuccess = HeapFree( GetProcessHeap(), 0, pDirectory ) && fSuccess;
	}
	else
		SetLastError( ERROR_INVALID_HANDLE );

	return fSuccess;
}

/****************************************************************************/

BOOL WINAPI EnumDirectory( HANDLE hDirectory, LPWIN32_FIND_DATA pFindData )
{
	BOOL		fSuccess = FALSE;
	LPDIRECTORY	pDirectory = (LPDIRECTORY)hDirectory;

	if ( pDirectory )
	{
		BOOL	fValid;

		do
		{
			if ( pDirectory->aFirstData.cFileName[0] )
			{
				*pFindData = pDirectory->aFirstData;
				fSuccess = TRUE;
				pDirectory->aFirstData.cFileName[0] = 0;
			}
			else
				fSuccess = lpfnFindNextFile( pDirectory->hFind, pFindData );

			fValid = fSuccess && _tcscmp( TEXT("."), pFindData->cFileName ) != 0 && _tcscmp( TEXT(".."), pFindData->cFileName ) != 0;

		} while( fSuccess && !fValid );
	}
	else
		SetLastError( ERROR_INVALID_HANDLE );

	return fSuccess;
}

/****************************************************************************/

#define	CHARSET_SEPARATOR	TEXT("\\/")


static BOOL WINAPI IsValidFilePathComponent( LPCTSTR lpComponent, LPCTSTR *lppComponentEnd, DWORD dwFlags )
{
	LPCTSTR	lpComponentEnd = NULL;
	LPCTSTR	lpCurrent = lpComponent;
	BOOL	fValid = TRUE;	/* Assume success */
	TCHAR	cLast = 0;

	/* Path component length must not exceed MAX_PATH */

	while ( !lpComponentEnd && lpCurrent && lpCurrent - lpComponent < MAX_PATH )
	{
		switch ( *lpCurrent )
		{
			/* Both backslash and slash determine the end of a path component */
		case '\0':
		case '/':
		case '\\':
			switch ( cLast )
			{
				/* Component must not end with '.' or blank and can't be empty */

			case '.':
				if ( dwFlags & VALIDATEPATH_ALLOW_ELLIPSE )
				{
					if ( 1 == lpCurrent - lpComponent )
					{
						/* Current directory is O.K. */
						lpComponentEnd = lpCurrent;
						break;
					}
					else if ( 2 == lpCurrent - lpComponent && '.' == *lpComponent )
					{
						/* Parent directory is O.K. */
						lpComponentEnd = lpCurrent;
						break;
					}
				}
			case 0:
			case ' ':
				lpComponentEnd = lpCurrent - 1;
				fValid = FALSE;
				break;
			default:
				lpComponentEnd = lpCurrent;
				break;
			}
			break;
			/* '?' and '*' are valid wildcards but not valid file name characters */
		case '?':
		case '*':
			if ( dwFlags & VALIDATEPATH_ALLOW_WILDCARDS )
				break;
			/* The following characters are reserved */
		case '<':
		case '>':
		case '\"':
		case '|':
		case ':':
			lpComponentEnd = lpCurrent;
			fValid = FALSE;
			break;
		default:
			/* Characters below ASCII 32 are not allowed */
			if ( *lpCurrent < ' ' )
			{
				lpComponentEnd = lpCurrent;
				fValid = FALSE;
			}
			break;
		}


		cLast = *lpCurrent++;
	}


	/*	If we don't reached the end of the component the length of the component was to long 
		( See condition of while loop ) */

	if ( !lpComponentEnd )
	{
		fValid = FALSE;
		lpComponentEnd = lpCurrent;
	}


	/* Test wether the component specifies a device name what is not allowed */

	if ( fValid )
	{
		LPCTSTR	alpDeviceNames[] =
		{
			TEXT("CON"),
			TEXT("PRN"),
			TEXT("AUX"),
			TEXT("CLOCK$"),
			TEXT("NUL"),
			TEXT("LPT1"),
			TEXT("LPT2"),
			TEXT("LPT3"),
			TEXT("LPT4"),
			TEXT("LPT5"),
			TEXT("LPT6"),
			TEXT("LPT7"),
			TEXT("LPT8"),
			TEXT("LPT9"),
			TEXT("COM1"),
			TEXT("COM2"),
			TEXT("COM3"),
			TEXT("COM4"),
			TEXT("COM5"),
			TEXT("COM6"),
			TEXT("COM7"),
			TEXT("COM8"),
			TEXT("COM9")
		};
		
		TCHAR	szComponent[MAX_PATH];
		int		nComponentLength;
		LPCTSTR	lpDot;
		int		i;

		/* A device name with an extension is also invalid */

		lpDot = _tcschr( lpComponent, '.' );

		if ( !lpDot || lpDot > lpComponentEnd )
			nComponentLength = lpComponentEnd - lpComponent;
		else
			nComponentLength = lpDot - lpComponent;

		_tcsncpy( szComponent, lpComponent, nComponentLength );
		szComponent[nComponentLength] = 0;

		for ( i = 0; i < sizeof( alpDeviceNames ) / sizeof(LPCTSTR); i++ )
		{
			if ( 0 == _tcsicmp( szComponent, alpDeviceNames[i] ) )
			{
				lpComponentEnd = lpComponent;
				fValid = FALSE;
				break;
			}
		}

		if ( fValid )
		{
			/* Empty components are not allowed */

			if ( lpComponentEnd - lpComponent < 1 )
				fValid = FALSE;

			/* If we reached the end of the string NULL is returned */

			else if ( !*lpComponentEnd )
				lpComponentEnd = NULL;

		}
	}


	if ( lppComponentEnd )
		*lppComponentEnd = lpComponentEnd;

	return fValid;
}

/****************************************************************************/

static DWORD WINAPI IsValidFilePath( LPCTSTR lpszPath, LPCTSTR *lppError, DWORD dwFlags )
{
	LPCTSTR	lpComponent;
	BOOL	fValid = TRUE;
	DWORD	dwPathType = PATHTYPE_ERROR;

	if ( dwFlags & VALIDATEPATH_ALLOW_RELATIVE )
		dwFlags |= VALIDATEPATH_ALLOW_ELLIPSE;

	if ( !lpszPath )
	{
		fValid = FALSE;
		lpComponent = lpszPath;
	}

	/* Test for UNC path notation */

	if ( 2 == _tcsspn( lpszPath, CHARSET_SEPARATOR ) )
	{
		/* Place the pointer behind the leading to backslashes */

		lpComponent = lpszPath + 2;

		fValid = IsValidFilePathComponent( lpComponent, &lpComponent, VALIDATEPATH_ALLOW_ELLIPSE );

		/* So far we have a valid servername. Now let's see if we also have a network resource */

		dwPathType = PATHTYPE_ABSOLUTE_UNC;

		if ( fValid )
		{
			if ( lpComponent &&	 !*++lpComponent )
				lpComponent = NULL;
			
			if ( !lpComponent )
			{
#if 0
				/* We only have a Server specification what is invalid */

				lpComponent = lpszPath;
				fValid = FALSE;
#else
				dwPathType |= PATHTYPE_IS_SERVER;
#endif
			}
			else
			{

				/* Now test the network resource */

				fValid = IsValidFilePathComponent( lpComponent, &lpComponent, 0 );

				/* If we now reached the end of the path, everything is O.K. */


				if ( fValid && lpComponent && !*++lpComponent )
				{
					lpComponent = NULL;
					dwPathType |= PATHTYPE_IS_VOLUME;
				}
			}

		}
	}

	/* Local path verification. Must start with <drive>: */

	else if ( _istalpha( lpszPath[0] ) && ':' == lpszPath[1] )
	{
		/* Place pointer behind correct drive specification */

		lpComponent = lpszPath + 2;

		if ( 1 == _tcsspn( lpComponent, CHARSET_SEPARATOR ) )
			lpComponent++;
		else if ( *lpComponent )
			fValid = FALSE;

		dwPathType = PATHTYPE_ABSOLUTE_LOCAL;

		/* Now we are behind the backslash or it was a simple drive without backslash */

		if ( fValid && !*lpComponent )
		{
			lpComponent = NULL;
			dwPathType |= PATHTYPE_IS_VOLUME;
		}

	}

	/* Can be a relative path */

	else if ( dwFlags & VALIDATEPATH_ALLOW_RELATIVE )
	{
		lpComponent = lpszPath;

		/* Relative path can start with a backslash */

		if ( 1 == _tcsspn( lpComponent, CHARSET_SEPARATOR ) )
		{
			lpComponent++;
			if ( !*lpComponent )
				lpComponent = NULL;
		}

		dwPathType = PATHTYPE_RELATIVE;
	}

	/* Anything else is an error */
	else
	{
		fValid = FALSE;
		lpComponent = lpszPath;
	}

	/* Now validate each component of the path */

	while ( fValid && lpComponent )
	{
		fValid = IsValidFilePathComponent( lpComponent, &lpComponent, dwFlags );

		if ( fValid && lpComponent )
		{
			lpComponent++;

			/* If the string behind the backslash is empty, we've done */

			if ( !*lpComponent )
				lpComponent = NULL;
		}
	}

	/* Last we have to check if the path exceeds MAX_PATH characters */

	if ( fValid && _tcslen( lpszPath ) >= MAX_PATH )
	{
		fValid = FALSE;
		lpComponent = lpszPath + MAX_PATH;
	}

	/* Store the error if possible */

	if ( lppError )
		*lppError = lpComponent;

	return fValid ? dwPathType : PATHTYPE_ERROR;
}



static void _osl_warnFile( const char *message, rtl_uString *ustrFile )
{
	char	szBuffer[2048];

	if ( ustrFile )
	{
		rtl_String	*strFile = NULL;

		rtl_uString2String( &strFile, rtl_uString_getStr( ustrFile ), rtl_uString_getLength( ustrFile ), osl_getThreadTextEncoding(), OUSTRING_TO_OSTRING_CVTFLAGS );
		sprintf( szBuffer, message, strFile->buffer );
		rtl_string_release( strFile );

		message = szBuffer;
	}

	OSL_ENSURE( 0, message );
}
