/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: xechart.hxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2007/07/03 13:25:30 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef SC_XECHART_HXX
#define SC_XECHART_HXX

#ifndef SC_XLCHART_HXX
#include "xlchart.hxx"
#endif

#ifndef SC_XLFORMULA_HXX
#include "xlformula.hxx"
#endif
#ifndef SC_XLSTYLE_HXX
#include "xlstyle.hxx"
#endif
#ifndef SC_XEROOT_HXX
#include "xeroot.hxx"
#endif
#ifndef SC_XERECORD_HXX
#include "xerecord.hxx"
#endif
#ifndef SC_XESTRING_HXX
#include "xestring.hxx"
#endif

class Size;

namespace com { namespace sun { namespace star {
    namespace frame
    {
        class XModel;
    }
    namespace chart2
    {
        struct ScaleData;
        class XChartDocument;
        class XDiagram;
        class XCoordinateSystem;
        class XChartType;
        class XDataSeries;
        class XAxis;
        class XTitle;
        class XFormattedString;
        namespace data
        {
            class XDataSequence;
            class XLabeledDataSequence;
        }
    }
} } }

// Common =====================================================================

class XclExpChRootData;
class XclExpChChart;

/** Base class for complex chart classes, provides access to other components of the chart. */
class XclExpChRoot : public XclExpRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XChartDocument > XChartDocRef;

public:
    explicit            XclExpChRoot( const XclExpRoot& rRoot, XclExpChChart* pChartData );
    virtual             ~XclExpChRoot();

    /** Returns this root instance - for code readability in derived classes. */
    inline const XclExpChRoot& GetChRoot() const { return *this; }
    /** Returns a reference to the parent chart data object. */
    XclExpChChart&      GetChartData() const;
    /** Returns chart type info for a unique chart type identifier. */
    const XclChTypeInfo& GetChartTypeInfo( XclChTypeId eType ) const;
    /** Returns the first fitting chart type info for the passed service name. */
    const XclChTypeInfo& GetChartTypeInfo( const ::rtl::OUString& rServiceName ) const;

    /** Returns an info struct about auto formatting for the passed object type. */
    const XclChFormatInfo& GetFormatInfo( XclChObjectType eObjType ) const;

    /** Starts the API chart document conversion. Must be called once before all API conversion. */
    void                InitConversion( XChartDocRef xChartDoc ) const;
    /** Finishes the API chart document conversion. Must be called once after all API conversion. */
    void                FinishConversion() const;

    /** Returns true, if the passed color equals to the specified system color. */
    bool                IsSystemColor( const Color& rColor, sal_uInt16 nSysColorIdx ) const;
    /** Sets a system color and the respective color identifier. */
    void                SetSystemColor( Color& rColor, sal_uInt32& rnColorId, sal_uInt16 nSysColorIdx ) const;

    /** Reads all line properties from the passed property set. */
    void                ConvertLineFormat(
                            XclChLineFormat& rLineFmt,
                            const ScfPropertySet& rPropSet,
                            XclChPropertyMode ePropMode ) const;
    /** Reads solid area properties from the passed property set. */
    void                ConvertAreaFormat(
                            XclChAreaFormat& rAreaFmt,
                            const ScfPropertySet& rPropSet,
                            XclChPropertyMode ePropMode ) const;
    /** Reads gradient or bitmap area properties from the passed property set. */
    void                ConvertEscherFormat(
                            XclChEscherFormat& rEscherFmt,
                            XclChPicFormat& rPicFmt,
                            const ScfPropertySet& rPropSet,
                            XclChPropertyMode ePropMode ) const;
    /** Reads font properties from the passed property set. */
    sal_uInt16          ConvertFont(
                            const ScfPropertySet& rPropSet,
                            sal_Int16 nScript ) const;

private:
    typedef ScfRef< XclExpChRootData > XclExpChRootDataRef;
    XclExpChRootDataRef mxChData;           /// Reference to the root data object.
};

// ----------------------------------------------------------------------------

/** Base class for chart record groups. Provides helper functions to write sub records.

    A chart record group consists of a header record, followed by a CHBEGIN
    record, followed by group sub records, and finished with a CHEND record.
 */
class XclExpChGroupBase : public XclExpRecord
{
public:
    /** @param bWriteBeginEnd  true = write CHBEGIN/CHEND records enclosing sub records. */
    explicit            XclExpChGroupBase(
                            sal_uInt16 nRecId, sal_Size nRecSize = 0 );
    virtual             ~XclExpChGroupBase();

    /** Saves the header record. Calls WriteSubRecords() to let derived classes write sub records. */
    virtual void        Save( XclExpStream& rStrm );
    /** Derived classes return whether there are any records embedded in this group. */
    virtual bool        HasSubRecords() const;
    /** Derived classes implement writing any records embedded in this group. */
    virtual void        WriteSubRecords( XclExpStream& rStrm ) = 0;
};

// Frame formatting ===========================================================

class XclExpChLineFormat : public XclExpRecord
{
public:
    explicit            XclExpChLineFormat( const XclExpChRoot& rRoot );

    /** Converts line formatting properties from the passed property set. */
    void                Convert( const XclExpChRoot& rRoot,
                            const ScfPropertySet& rPropSet, XclChObjectType eObjType );
    /** Sets or clears the automatic flag. */
    inline void         SetAuto( bool bAuto ) { ::set_flag( maData.mnFlags, EXC_CHLINEFORMAT_AUTO, bAuto ); }
    /** Sets flag to show or hide an axis. */
    inline void         SetShowAxis( bool bShowAxis )
                            { ::set_flag( maData.mnFlags, EXC_CHLINEFORMAT_SHOWAXIS, bShowAxis ); }
    /** Sets the line format to the specified default type. */
    void                SetDefault( XclChFrameType eDefFrameType );

    /** Returns true, if the line format is set to automatic. */
    inline bool         IsAuto() const { return ::get_flag( maData.mnFlags, EXC_CHLINEFORMAT_AUTO ); }
    /** Returns true, if the line style is set to something visible. */
    inline bool         HasLine() const { return maData.mnPattern != EXC_CHLINEFORMAT_NONE; }
    /** Returns true, if the line contains default formatting according to the passed frame type. */
    bool                IsDefault( XclChFrameType eDefFrameType ) const;

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChLineFormat     maData;             /// Contents of the CHLINEFORMAT record.
    sal_uInt32          mnColorId;          /// Line color identifier.
};

typedef ScfRef< XclExpChLineFormat > XclExpChLineFormatRef;

// ----------------------------------------------------------------------------

class XclExpChAreaFormat : public XclExpRecord
{
public:
    explicit            XclExpChAreaFormat( const XclExpChRoot& rRoot );

    /** Converts area formatting properties from the passed property set. */
    void                Convert( const XclExpChRoot& rRoot,
                            const ScfPropertySet& rPropSet, XclChObjectType eObjType );
    /** Sets or clears the automatic flag. */
    inline void         SetAuto( bool bAuto ) { ::set_flag( maData.mnFlags, EXC_CHAREAFORMAT_AUTO, bAuto ); }
    /** Sets the area format to the specified default type. */
    void                SetDefault( XclChFrameType eDefFrameType );

    /** Returns true, if the area format is set to automatic. */
    inline bool         IsAuto() const { return ::get_flag( maData.mnFlags, EXC_CHAREAFORMAT_AUTO ); }
    /** Returns true, if the area style is set to something visible. */
    inline bool         HasArea() const { return maData.mnPattern != EXC_PATT_NONE; }
    /** Returns true, if the area contains default formatting according to the passed frame type. */
    bool                IsDefault( XclChFrameType eDefFrameType ) const;

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChAreaFormat     maData;             /// Contents of the CHAREAFORMAT record.
    sal_uInt32          mnPattColorId;      /// Pattern color identifier.
    sal_uInt32          mnBackColorId;      /// Pattern background color identifier.
};

typedef ScfRef< XclExpChAreaFormat > XclExpChAreaFormatRef;

// ----------------------------------------------------------------------------

class XclExpChEscherFormat : public XclExpChGroupBase
{
public:
    explicit            XclExpChEscherFormat( const XclExpChRoot& rRoot );

    /** Converts complex area formatting from the passed property set. */
    void                Convert( const XclExpChRoot& rRoot,
                            const ScfPropertySet& rPropSet, XclChObjectType eObjType );

    /** Returns true, if the object contains valid formatting data. */
    bool                IsValid() const;

    /** Writes the CHESCHERFORMAT record group to the stream, if complex formatting is extant. */
    virtual void        Save( XclExpStream& rStrm );
    /** Returns true, if this record group contains a CHPICFORMAT record. */
    virtual bool        HasSubRecords() const;
    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    /** Inserts a color from the contained Escher property set into the color palette. */
    sal_uInt32          RegisterColor( const XclExpChRoot& rRoot, sal_uInt16 nPropId );

    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChEscherFormat   maData;             /// Fill properties for complex areas (CHESCHERFORMAT record).
    XclChPicFormat      maPicFmt;           /// Image options, e.g. stretched, stacked (CHPICFORMAT record).
    sal_uInt32          mnColor1Id;         /// First fill color identifier.
    sal_uInt32          mnColor2Id;         /// Second fill color identifier.
};

typedef ScfRef< XclExpChEscherFormat > XclExpChEscherFormatRef;

// ----------------------------------------------------------------------------

/** Base class for record groups containing frame formatting.

    Frame formatting can be part of several record groups, e.g. CHFRAME,
    CHDATAFORMAT, CHDROPBAR. It consists of CHLINEFORMAT, CHAREAFORMAT, and
    CHESCHERFORMAT group.
 */
class XclExpChFrameBase
{
public:
    explicit            XclExpChFrameBase();
    virtual             ~XclExpChFrameBase();

protected:
    /** Converts frame formatting properties from the passed property set. */
    void                ConvertFrameBase( const XclExpChRoot& rRoot,
                            const ScfPropertySet& rPropSet, XclChObjectType eObjType );
    /** Sets the frame formatting to the specified default type. */
    void                SetDefaultFrameBase( const XclExpChRoot& rRoot,
                            XclChFrameType eDefFrameType, bool bIsFrame );

    /** Returns true, if the frame contains default formatting (as if the frame is missing). */
    bool                IsDefaultFrameBase( XclChFrameType eDefFrameType ) const;

    /** Writes all contained frame records to the passed stream. */
    void                WriteFrameRecords( XclExpStream& rStrm );

private:
    XclExpChLineFormatRef mxLineFmt;        /// Line format (CHLINEFORMAT record).
    XclExpChAreaFormatRef mxAreaFmt;        /// Area format (CHAREAFORMAT record).
    XclExpChEscherFormatRef mxEscherFmt;    /// Complex area format (CHESCHERFORMAT record).
};

// ----------------------------------------------------------------------------

/** Represents the CHFRAME record group containing object frame properties.

    The CHFRAME group consists of: CHFRAME, CHBEGIN, CHLINEFORMAT,
    CHAREAFORMAT, CHESCHERFORMAT group, CHEND.
 */
class XclExpChFrame : public XclExpChGroupBase, public XclExpChFrameBase, protected XclExpChRoot
{
public:
    explicit            XclExpChFrame( const XclExpChRoot& rRoot, XclChObjectType eObjType );

    /** Converts frame formatting properties from the passed property set. */
    void                Convert( const ScfPropertySet& rPropSet );

    /** Returns true, if the frame object contains default formats. */
    bool                IsDefault() const;
    /** Returns true, if the frame object can be deleted because it contains default formats. */
    bool                IsDeleteable() const;

    /** Writes the entire record group. */
    virtual void        Save( XclExpStream& rStrm );
    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChFrame          maData;             /// Contents of the CHFRAME record.
    XclChObjectType     meObjType;          /// Type of the represented object.
};

typedef ScfRef< XclExpChFrame > XclExpChFrameRef;

// Source links ===============================================================

class XclExpChSourceLink : public XclExpRecord, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::data::XDataSequence >   XDataSequenceRef;
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XFormattedString >      XFormattedStringRef;
    typedef ::com::sun::star::uno::Sequence< XFormattedStringRef >                              XFormattedStringSeq;

public:
    explicit            XclExpChSourceLink( const XclExpChRoot& rRoot, sal_uInt8 nDestType );

    /** Converts the passed source link, returns the number of linked values. */
    sal_uInt16          ConvertDataSequence( XDataSequenceRef xDataSeq, bool bSplitToColumns );
    /** Converts the passed sequence of formatted string objects, returns leading font index. */
    sal_uInt16          ConvertStringSequence( const XFormattedStringSeq& rStringSeq );

    /** Returns true, if this source link contains explicit string data. */
    inline bool         HasString() const { return mxString.is() && !mxString->IsEmpty(); }

    /** Writes the CHSOURCELINK record and optionally a CHSTRING record with explicit string data. */
    virtual void        Save( XclExpStream& rStrm );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChSourceLink     maData;             /// Contents of the CHSOURCELINK record.
    XclTokenArrayRef    mxLinkFmla;         /// Formula with link to source data.
    XclExpStringRef     mxString;           /// Text data (CHSTRING record).
};

typedef ScfRef< XclExpChSourceLink > XclExpChSourceLinkRef;

// Text =======================================================================

/** The CHFONT record containing a font index for text objects. */
class XclExpChFont : public XclExpUInt16Record
{
public:
    explicit            XclExpChFont( sal_uInt16 nFontIdx );
};

typedef ScfRef< XclExpChFont > XclExpChFontRef;

// ----------------------------------------------------------------------------

/** The CHOBJECTLINK record linking a text object to a specific chart object. */
class XclExpChObjectLink : public XclExpRecord
{
public:
    explicit            XclExpChObjectLink( sal_uInt16 nLinkTarget,
                            sal_uInt16 nSeriesIdx, sal_uInt16 nPointIdx );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChObjectLink     maData;             /// Contents of the CHOBJECTLINK record.
};

typedef ScfRef< XclExpChObjectLink > XclExpChObjectLinkRef;

// ----------------------------------------------------------------------------

/** Base class for objects with font settings. Provides font conversion helper functions. */
class XclExpChFontBase
{
public:
    virtual             ~XclExpChFontBase();

    /** Derived classes set font color and color identifier to internal data structures. */
    virtual void        SetFont( XclExpChFontRef xFont, const Color& rColor, sal_uInt32 nColorId ) = 0;
    /** Derived classes set text rotation to internal data structures. */
    virtual void        SetRotation( sal_uInt16 nRotation ) = 0;

    /** Creates a CHFONT record from the passed font index, calls virtual function SetFont(). */
    void                ConvertFontBase( const XclExpChRoot& rRoot, sal_uInt16 nFontIdx );
    /** Creates a CHFONT record from the passed font index, calls virtual function SetFont(). */
    void                ConvertFontBase( const XclExpChRoot& rRoot, const ScfPropertySet& rPropSet );
    /** Converts rotation settings, calls virtual function SetRotation(). */
    void                ConvertRotationBase( const XclExpChRoot& rRoot, const ScfPropertySet& rPropSet );
};

// ----------------------------------------------------------------------------

/** Represents the CHTEXT record group containing text object properties.

    The CHTEXT group consists of: CHTEXT, CHBEGIN, CHFRAMEPOS, CHFONT,
    CHFORMATRUNS, CHSOURCELINK, CHSTRING, CHFRAME group, CHOBJECTLINK, and CHEND.
 */
class XclExpChText : public XclExpChGroupBase, public XclExpChFontBase, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XTitle > XTitleRef;

public:
    explicit            XclExpChText( const XclExpChRoot& rRoot );

    /** Sets font color and color identifier to internal data structures. */
    virtual void        SetFont( XclExpChFontRef xFont, const Color& rColor, sal_uInt32 nColorId );
    /** Sets text rotation to internal data structures. */
    virtual void        SetRotation( sal_uInt16 nRotation );

    /** Converts all text settings of the passed title text object. */
    void                ConvertTitle( XTitleRef xTitle, sal_uInt16 nTarget );
    /** Converts all text settings of the passed legend. */
    void                ConvertLegend( const ScfPropertySet& rPropSet );
    /** Converts all settings of the passed data point caption text object. */
    bool                ConvertDataLabel( const ScfPropertySet& rPropSet,
                            const XclChTypeInfo& rTypeInfo, const XclChDataPointPos& rPointPos );

    /** Returns true, if the string object does not contain any text data. */
    inline bool         HasString() const { return mxSrcLink.is() && mxSrcLink->HasString(); }
    /** Returns the flags needed for the CHATTACHEDLABEL record. */
    sal_uInt16          GetAttLabelFlags() const;

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChText           maData;             /// Contents of the CHTEXT record.
    XclExpChSourceLinkRef mxSrcLink;        /// Linked data (CHSOURCELINK with CHSTRING record).
    XclExpChFrameRef    mxFrame;            /// Text object frame properties (CHFRAME group).
    XclExpChFontRef     mxFont;             /// Index into font buffer (CHFONT record).
    XclExpChObjectLinkRef mxObjLink;        /// Link target for this text object.
    sal_uInt32          mnTextColorId;      /// Text color identifier.
};

typedef ScfRef< XclExpChText > XclExpChTextRef;

// Data series ================================================================

/** The CHMARKERFORMAT record containing data point marker formatting data. */
class XclExpChMarkerFormat : public XclExpRecord
{
public:
    explicit            XclExpChMarkerFormat( const XclExpChRoot& rRoot );

    /** Converts symbol properties from the passed property set. */
    void                Convert( const XclExpChRoot& rRoot,
                            const ScfPropertySet& rPropSet, sal_uInt16 nFormatIdx );
    /** Converts symbol properties for stock charts from the passed property set. */
    void                ConvertStockSymbol( const XclExpChRoot& rRoot,
                            const ScfPropertySet& rPropSet, bool bCloseSymbol );

    /** Returns true, if markers are enabled. */
    inline bool         HasMarker() const { return maData.mnMarkerType != EXC_CHMARKERFORMAT_NOSYMBOL; }
    /** Returns true, if border line of markers is visible. */
    inline bool         HasLineColor() const { return !::get_flag( maData.mnFlags, EXC_CHMARKERFORMAT_NOLINE ); }
    /** Returns true, if fill area of markers is visible. */
    inline bool         HasFillColor() const { return !::get_flag( maData.mnFlags, EXC_CHMARKERFORMAT_NOFILL ); }

private:
    /** Registers marker colors in palette and stores color identifiers. */
    void                RegisterColors( const XclExpChRoot& rRoot );

    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChMarkerFormat   maData;             /// Contents of the CHMARKERFORMAT record.
    sal_uInt32          mnLineColorId;      /// Border line color identifier.
    sal_uInt32          mnFillColorId;      /// Fill color identifier.
};

typedef ScfRef< XclExpChMarkerFormat > XclExpChMarkerFormatRef;

// ----------------------------------------------------------------------------

/** The CHPIEFORMAT record containing data point formatting data for pie segments. */
class XclExpChPieFormat : public XclExpUInt16Record
{
public:
    explicit            XclExpChPieFormat();

    /** Sets pie segment properties from the passed property set. */
    void                Convert( const ScfPropertySet& rPropSet );
};

typedef ScfRef< XclExpChPieFormat > XclExpChPieFormatRef;

// ----------------------------------------------------------------------------

/** The CH3DDATAFORMAT record containing the bar type in 3D bar charts. */
class XclExpCh3dDataFormat : public XclExpRecord
{
public:
    explicit            XclExpCh3dDataFormat();

    /** Sets 3d bar properties from the passed property set. */
    void                Convert( const ScfPropertySet& rPropSet );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclCh3dDataFormat   maData;             /// Contents of the CH3DDATAFORMAT record.
};

typedef ScfRef< XclExpCh3dDataFormat > XclExpCh3dDataFormatRef;

// ----------------------------------------------------------------------------

/** The CHATTACHEDLABEL record that contains the type of a data point label. */
class XclExpChAttachedLabel : public XclExpUInt16Record
{
public:
    explicit            XclExpChAttachedLabel( sal_uInt16 nFlags );
};

typedef ScfRef< XclExpChAttachedLabel > XclExpChAttLabelRef;

// ----------------------------------------------------------------------------

/** Represents the CHDATAFORMAT record group containing data point properties.

    The CHDATAFORMAT group consists of: CHDATAFORMAT, CHBEGIN, CHFRAME group,
    CHMARKERFORMAT, CHPIEFORMAT, CH3DDATAFORMAT, CHSERIESFORMAT,
    CHATTACHEDLABEL, CHEND.
 */
class XclExpChDataFormat : public XclExpChGroupBase, public XclExpChFrameBase, protected XclExpChRoot
{
public:
    explicit            XclExpChDataFormat( const XclExpChRoot& rRoot,
                            sal_uInt16 nSeriesIdx, sal_uInt16 nPointIdx, sal_uInt16 nFormatIdx );

    /** Converts the passed data series or data point formatting. */
    void                ConvertDataSeries( const ScfPropertySet& rPropSet, const XclChExtTypeInfo& rTypeInfo );
    /** Sets default formatting for a series in a stock chart. */
    void                ConvertStockSeries( const ScfPropertySet& rPropSet, bool bCloseSymbol );
    /** Converts line formatting for the specified object (e.g. trend lines, error bars). */
    void                ConvertLine( const ScfPropertySet& rPropSet, XclChObjectType eObjType );

    /** Returns true, if this objects describes the formatting of an entire series. */
    inline bool         IsSeriesFormat() const { return maData.maPointPos.mnPointIdx == EXC_CHDATAFORMAT_ALLPOINTS; }

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChDataFormat     maData;             /// Contents of the CHDATAFORMAT record.
    XclExpChMarkerFormatRef mxMarkerFmt;    /// Data point marker (CHMARKERFORMAT record).
    XclExpChPieFormatRef mxPieFmt;          /// Pie segment format (CHPIEFORMAT record).
    XclExpRecordRef     mxSeriesFmt;        /// Series properties (CHSERIESFORMAT record).
    XclExpCh3dDataFormatRef mx3dDataFmt;    /// 3D bar format (CH3DDATAFORMAT record).
    XclExpChAttLabelRef mxAttLabel;         /// Data point label type (CHATTACHEDLABEL record).
};

typedef ScfRef< XclExpChDataFormat > XclExpChDataFormatRef;

// ----------------------------------------------------------------------------

/** Represents the CHSERTRENDLINE record containing settings for a trend line. */
class XclExpChSerTrendLine : public XclExpRecord, protected XclExpChRoot
{
public:
    explicit            XclExpChSerTrendLine( const XclExpChRoot& rRoot );

    /** Converts the passed trend line, returns true if trend line type is supported. */
    bool                Convert( const ScfPropertySet& rPropSet );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChSerTrendLine   maData;             /// Contents of the CHSERTRENDLINE record.
};

typedef ScfRef< XclExpChSerTrendLine > XclExpChSerTrendLineRef;

// ----------------------------------------------------------------------------

/** Represents the CHSERERRORBAR record containing settings for error bars. */
class XclExpChSerErrorBar : public XclExpRecord, protected XclExpChRoot
{
public:
    explicit            XclExpChSerErrorBar( const XclExpChRoot& rRoot, sal_uInt8 nBarType );

    /** Converts the passed error bar settings, returns true if error bar type is supported. */
    bool                Convert( const ScfPropertySet& rPropSet );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChSerErrorBar    maData;             /// Contents of the CHSERERRORBAR record.
};

typedef ScfRef< XclExpChSerErrorBar > XclExpChSerErrorBarRef;

// ----------------------------------------------------------------------------

/** Represents the CHSERIES record group describing a data series in a chart.

    The CHSERIES group consists of: CHSERIES, CHBEGIN, CHSOURCELINK groups,
    CHDATAFORMAT groups, CHSERGROUP, CHSERPARENT, CHSERERRORBAR,
    CHSERTRENDLINE, CHEND.
 */
class XclExpChSeries : public XclExpChGroupBase, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XDataSeries >                   XDataSeriesRef;
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::data::XLabeledDataSequence >    XLabeledDataSeqRef;

public:
    explicit            XclExpChSeries( const XclExpChRoot& rRoot, sal_uInt16 nSeriesIdx );

    /** Converts the passed data series (source links and formatting). */
    bool                ConvertDataSeries( XDataSeriesRef xDataSeries,
                            const XclChExtTypeInfo& rTypeInfo,
                            sal_uInt16 nGroupIdx, sal_uInt16 nFormatIdx );
    /** Converts the passed data series for stock charts. */
    bool                ConvertStockSeries( XDataSeriesRef xDataSeries,
                            const ::rtl::OUString& rValueRole,
                            sal_uInt16 nGroupIdx, sal_uInt16 nFormatIdx, bool bCloseSymbol );
    /** Converts the passed error bar settings (called at trend line child series). */
    bool                ConvertTrendLine( const ScfPropertySet& rPropSet, sal_uInt16 nParentIdx );
    /** Converts the passed error bar settings (called at error bar child series). */
    bool                ConvertErrorBar( const ScfPropertySet& rPropSet, sal_uInt16 nParentIdx, sal_uInt8 nBarId );
    /** Converts and inserts category ranges for all inserted series. */
    void                ConvertCategSequence( XLabeledDataSeqRef xCategSeq );

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    /** Tries to create trend line series objects (called at parent series). */
    void                CreateTrendLines( XDataSeriesRef xDataSeries );
    /** Tries to create positive and negative error bar series objects (called at parent series). */
    void                CreateErrorBars( const ScfPropertySet& rPropSet,
                            const ::rtl::OUString& rBarPropName,
                            sal_uInt8 nPosBarId, sal_uInt8 nNegBarId );
    /** Tries to create an error bar series object (called at parent series). */
    void                CreateErrorBar( const ScfPropertySet& rPropSet,
                            const ::rtl::OUString& rShowPropName, sal_uInt8 nBarId );

    virtual void        WriteBody( XclExpStream& rStrm );

private:
    typedef XclExpRecordList< XclExpChDataFormat > XclExpChDataFormatList;

private:
    XclChSeries         maData;             /// Contents of the CHSERIES record.
    XclExpChSourceLinkRef mxTitleLink;      /// Link data for series title.
    XclExpChSourceLinkRef mxValueLink;      /// Link data for series values.
    XclExpChSourceLinkRef mxCategLink;      /// Link data for series category names.
    XclExpChSourceLinkRef mxBubbleLink;     /// Link data for series bubble sizes.
    XclExpChDataFormatRef mxSeriesFmt;      /// CHDATAFORMAT group for series format.
    XclExpChDataFormatList maPointFmts;     /// CHDATAFORMAT groups for data point formats.
    XclExpChSerTrendLineRef mxTrendLine;    /// Trend line settings (CHSERTRENDLINE record).
    XclExpChSerErrorBarRef mxErrorBar;      /// Error bar settings (CHSERERRORBAR record).
    sal_uInt16          mnGroupIdx;         /// Chart type group (CHTYPEGROUP group) this series is assigned to.
    sal_uInt16          mnSeriesIdx;        /// 0-based series index.
    sal_uInt16          mnParentIdx;        /// 0-based index of parent series (trend lines and error bars).
};

typedef ScfRef< XclExpChSeries > XclExpChSeriesRef;

// Chart type groups ==========================================================

/** Represents the chart type record for all supported chart types. */
class XclExpChType : public XclExpRecord, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XChartType > XChartTypeRef;

public:
    explicit            XclExpChType( const XclExpChRoot& rRoot );

    /** Converts the passed chart type and the contained data series. */
    void                Convert( XChartTypeRef xChartType,
                            sal_Int32 nApiAxesSetIdx, bool bSwappedAxesSet, bool bHasXLabels );
    /** Sets stacking mode (standard or percent) for the series in this chart type group. */
    void                SetStacked( bool bPercent );

    /** Returns true, if this is object represents a valid chart type. */
    inline bool         IsValidType() const { return maTypeInfo.meTypeId != EXC_CHTYPEID_UNKNOWN; }
    /** Returns the chart type info struct for the contained chart type. */
    inline const XclChTypeInfo& GetTypeInfo() const { return maTypeInfo; }

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChType           maData;             /// Contents of the chart type record.
    XclChTypeInfo       maTypeInfo;         /// Chart type info for the contained type.
};

// ----------------------------------------------------------------------------

/** Represents the CHCHART3D record that contains 3D view settings. */
class XclExpChChart3d : public XclExpRecord
{
public:
    explicit            XclExpChChart3d();

    /** Converts 3d settings for the passed chart type. */
    void                Convert( const ScfPropertySet& rPropSet, const XclChTypeInfo& rTypeInfo );
    /** Sets flag that the data points are clustered on the X axis. */
    inline void         SetClustered() { ::set_flag( maData.mnFlags, EXC_CHCHART3D_CLUSTER ); }

    /** Returns true, if the data points are clustered on the X axis. */
    inline bool         IsClustered() const { return ::get_flag( maData.mnFlags, EXC_CHCHART3D_CLUSTER ); }

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChChart3d        maData;             /// Contents of the CHCHART3D record.
};

typedef ScfRef< XclExpChChart3d > XclExpChChart3dRef;

// ----------------------------------------------------------------------------

/** Represents the CHLEGEND record group describing the chart legend.

    The CHLEGEND group consists of: CHLEGEND, CHBEGIN, CHFRAME group,
    CHTEXT group, CHEND.
 */
class XclExpChLegend : public XclExpChGroupBase, protected XclExpChRoot
{
public:
    explicit            XclExpChLegend( const XclExpChRoot& rRoot );

    /** Converts all legend settings from the passed property set. */
    void                Convert( const ScfPropertySet& rPropSet );

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChLegend         maData;             /// Contents of the CHLEGEND record.
    XclExpChTextRef     mxText;             /// Legend text format (CHTEXT group).
    XclExpChFrameRef    mxFrame;            /// Legend frame format (CHFRAME group).
};

typedef ScfRef< XclExpChLegend > XclExpChLegendRef;

// ----------------------------------------------------------------------------

/** Represents the CHDROPBAR record group describing pos/neg bars in line charts.

    The CHDROPBAR group consists of: CHDROPBAR, CHBEGIN, CHLINEFORMAT,
    CHAREAFORMAT, CHESCHERFORMAT group, CHEND.
 */
class XclExpChDropBar : public XclExpChGroupBase, public XclExpChFrameBase
{
public:
    explicit            XclExpChDropBar( XclChObjectType eObjType );

    /** Converts and writes the contained frame data to the passed property set. */
    void                Convert( const XclExpChRoot& rRoot, const ScfPropertySet& rPropSet );

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChObjectType     meObjType;          /// Type of the dropbar.
    sal_uInt16          mnBarDist;          /// Distance between bars (CHDROPBAR record).
};

typedef ScfRef< XclExpChDropBar > XclExpChDropBarRef;

// ----------------------------------------------------------------------------

/** Represents the CHTYPEGROUP record group describing a group of series.

    The CHTYPEGROUP group consists of: CHTYPEGROUP, CHBEGIN, a chart type
    record (e.g. CHBAR, CHLINE, CHAREA, CHPIE, ...), CHCHART3D, CHLEGEND group,
    CHDROPBAR groups, CHCHARTLINE groups (CHCHARTLINE with CHLINEFORMAT),
    CHDATAFORMAT group, CHEND.
 */
class XclExpChTypeGroup : public XclExpChGroupBase, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XDiagram >                      XDiagramRef;
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XChartType >                    XChartTypeRef;
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XDataSeries >                   XDataSeriesRef;
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::data::XLabeledDataSequence >    XLabeledDataSeqRef;

public:
    explicit            XclExpChTypeGroup( const XclExpChRoot& rRoot, sal_uInt16 nGroupIdx );

    /** Converts the passed chart type to Excel type settings. */
    void                ConvertType( XDiagramRef xDiagram, XChartTypeRef xChartType,
                            sal_Int32 nApiAxesSetIdx, bool b3dChart, bool bSwappedAxesSet, bool bHasXLabels );
    /** Converts and inserts all series from the passed chart type. */
    void                ConvertSeries( XChartTypeRef xChartType,
                            sal_Int32 nGroupAxesSetIdx, bool bPercent, bool bConnectorLines );
    /** Converts and inserts category ranges for all inserted series. */
    void                ConvertCategSequence( XLabeledDataSeqRef xCategSeq );
    /** Creates a legend object and converts all legend settings. */
    void                ConvertLegend( const ScfPropertySet& rPropSet );

    /** Returns true, if this chart type group contains at least one valid series. */
    inline bool         IsValidGroup() const { return !maSeries.IsEmpty() && maType.IsValidType(); }
    /** Returns the index of this chart type group format. */
    inline sal_uInt16   GetGroupIdx() const { return maData.mnGroupIdx; }
    /** Returns the chart type info struct for the contained chart type. */
    inline const XclChExtTypeInfo& GetTypeInfo() const { return maTypeInfo; }
    /** Returns true, if the chart is three-dimensional. */
    inline bool         Is3dChart() const { return maTypeInfo.mb3dChart; }
    /** Returns true, if chart type supports wall and floor format. */
    inline bool         Is3dWallChart() const { return Is3dChart() && maTypeInfo.mb3dWalls; }
    /** Returns true, if the series in this chart type group are ordered on the Z axis. */
    inline bool         Is3dDeepChart() const { return Is3dWallChart() && mxChart3d.is() && !mxChart3d->IsClustered(); }
    /** Returns true, if this chart type can be combined with other types. */
    inline bool         IsCombinable2d() const { return !Is3dChart() && maTypeInfo.mbCombinable2d; }

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    /** Returns an unused format index to be used for the next created series. */
    sal_uInt16          GetFreeFormatIdx() const;
    /** Creates all data series of any chart type except stock charts. */
    void                CreateDataSeries( XDataSeriesRef xDataSeries );
    /** Creates all data series of a stock chart. */
    void                CreateAllStockSeries( XChartTypeRef xChartType,
                            XDataSeriesRef xDataSeries );
    /** Creates a single data series of a stock chart. */
    bool                CreateStockSeries( XDataSeriesRef xDataSeries,
                            const ::rtl::OUString& rValueRole, bool bCloseSymbol );

    virtual void        WriteBody( XclExpStream& rStrm );

private:
    typedef XclExpRecordList< XclExpChSeries >          XclExpChSeriesList;
    typedef ScfRefMap< sal_uInt16, XclExpChLineFormat > XclExpChLineFormatMap;

    XclChTypeGroup      maData;             /// Contents of the CHTYPEGROUP record.
    XclExpChType        maType;             /// Chart type (e.g. CHBAR, CHLINE, ...).
    XclChExtTypeInfo    maTypeInfo;         /// Extended chart type info.
    XclExpChSeriesList  maSeries;           /// List of series data (CHSERIES groups).
    XclExpChChart3dRef  mxChart3d;          /// 3D settings (CHCHART3D record).
    XclExpChLegendRef   mxLegend;           /// Chart legend (CHLEGEND group).
    XclExpChDropBarRef  mxUpBar;            /// White dropbars (CHDROPBAR group).
    XclExpChDropBarRef  mxDownBar;          /// Black dropbars (CHDROPBAR group).
    XclExpChLineFormatMap maChartLines;     /// Global line formats (CHCHARTLINE group).
};

typedef ScfRef< XclExpChTypeGroup > XclExpChTypeGroupRef;

// Axes =======================================================================

class XclExpChLabelRange : public XclExpRecord, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::chart2::ScaleData ScaleData;

public:
    explicit            XclExpChLabelRange( const XclExpChRoot& rRoot );

    /** Converts category axis scaling settings. */
    void                Convert( const ScaleData& rScaleData );
    /** Sets flag for tickmark position between categories or on categories. */
    inline void         SetTicksBetweenCateg( bool bTicksBetween )
                            { ::set_flag( maData.mnFlags, EXC_CHLABELRANGE_BETWEEN, bTicksBetween ); }
    /** Swaps flag for crossing axis at maximum position. */
    inline void         SwapAxisMaxCross()
                            { ::set_flag( maData.mnFlags, EXC_CHLABELRANGE_MAXCROSS, !::get_flag( maData.mnFlags, EXC_CHLABELRANGE_MAXCROSS ) ); }

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChLabelRange     maData;             /// Contents of the CHLABELRANGE record.
};

typedef ScfRef< XclExpChLabelRange > XclExpChLabelRangeRef;

// ----------------------------------------------------------------------------

class XclExpChValueRange : public XclExpRecord, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::chart2::ScaleData ScaleData;

public:
    explicit            XclExpChValueRange( const XclExpChRoot& rRoot );

    /** Converts value axis scaling settings. */
    void                Convert( const ScaleData& rScaleData );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChValueRange     maData;             /// Contents of the CHVALUERANGE record.
};

typedef ScfRef< XclExpChValueRange > XclExpChValueRangeRef;

// ----------------------------------------------------------------------------

class XclExpChTick : public XclExpRecord, protected XclExpChRoot
{
public:
    explicit            XclExpChTick( const XclExpChRoot& rRoot );

    /** Converts axis tick mark settings. */
    void                Convert( const ScfPropertySet& rPropSet );
    /** Sets font color and color identifier to internal data structures. */
    void                SetFontColor( const Color& rColor, sal_uInt32 nColorId );
    /** Sets text rotation to internal data structures. */
    void                SetRotation( sal_uInt16 nRotation );
    /** Sets position of axis labels relative to axis. */
    inline void         SetLabelPos( sal_uInt8 nLabelPos ) { maData.mnLabelPos = nLabelPos; }

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChTick           maData;             /// Contents of the CHTICK record.
    sal_uInt32          mnTextColorId;      /// Axis labels text color identifier.
};

typedef ScfRef< XclExpChTick > XclExpChTickRef;

// ----------------------------------------------------------------------------

/** Represents the CHAXIS record group describing an entire chart axis.

    The CHAXIS group consists of: CHAXIS, CHBEGIN, CHLABELRANGE, CHEXTRANGE,
    CHVALUERANGE, CHFORMAT, CHTICK, CHFONT, CHAXISLINE groups (CHAXISLINE with
    CHLINEFORMAT, CHAREAFORMAT, and CHESCHERFORMAT group), CHEND.
 */
class XclExpChAxis : public XclExpChGroupBase, public XclExpChFontBase, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XDiagram >  XDiagramRef;
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XAxis >     XAxisRef;

public:
    explicit            XclExpChAxis( const XclExpChRoot& rRoot, sal_uInt16 nAxisType );

    /** Sets font color and color identifier to internal data structures. */
    virtual void        SetFont( XclExpChFontRef xFont, const Color& rColor, sal_uInt32 nColorId );
    /** Sets text rotation to internal data structures. */
    virtual void        SetRotation( sal_uInt16 nRotation );

    /** Converts formatting and scaling settings from the passed axis. */
    void                Convert( XAxisRef xAxis, const XclChExtTypeInfo& rTypeInfo, sal_Int32 nApiAxesSetIdx );
    /** Converts and writes 3D wall/floor properties from the passed diagram. */
    void                ConvertWall( XDiagramRef xDiagram );

    /** Returns the type of this axis. */
    inline sal_uInt16   GetAxisType() const { return maData.mnType; }
    /** Returns the axis dimension index used by the chart API. */
    inline sal_Int32    GetApiAxisDimension() const { return maData.GetApiAxisDimension(); }

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    XclChAxis           maData;             /// Contents of the CHAXIS record.
    XclExpChLabelRangeRef mxLabelRange;     /// Category scaling (CHLABELRANGE record).
    XclExpChValueRangeRef mxValueRange;     /// Value scaling (CHVALUERANGE record).
    XclExpChTickRef     mxTick;             /// Axis ticks (CHTICK record).
    XclExpChFontRef     mxFont;             /// Index into font buffer (CHFONT record).
    XclExpChLineFormatRef mxAxisLine;       /// Axis line format (CHLINEFORMAT record).
    XclExpChLineFormatRef mxMajorGrid;      /// Major grid line format (CHLINEFORMAT record).
    XclExpChLineFormatRef mxMinorGrid;      /// Minor grid line format (CHLINEFORMAT record).
    XclExpChFrameRef    mxWallFrame;        /// Wall/floor format (sub records of CHFRAME group).
    sal_uInt16          mnNumFmtIdx;        /// Index into number format buffer (CHFORMAT record).
};

typedef ScfRef< XclExpChAxis > XclExpChAxisRef;

// ----------------------------------------------------------------------------

/** Represents the CHAXESSET record group describing an axes set (X/Y/Z axes).

    The CHAXESSET group consists of: CHAXESSET, CHBEGIN, CHFRAMEPOS, CHAXIS
    groups, CHTEXT groups, CHPLOTFRAME group (CHPLOTFRAME with CHFRAME group),
    CHTYPEGROUP group, CHEND.
 */
class XclExpChAxesSet : public XclExpChGroupBase, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XDiagram >          XDiagramRef;
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XCoordinateSystem > XCoordSystemRef;

public:
    explicit            XclExpChAxesSet( const XclExpChRoot& rRoot, sal_uInt16 nAxesSetId );

    /** Converts the passed diagram to chart record data.
        @return  First unused chart type group index. */
    sal_uInt16          Convert( XDiagramRef xDiagram, sal_uInt16 nFirstGroupIdx );

    /** Returns true, if this axes set exists (returns false if this is a dummy object). */
    inline bool         IsValidAxesSet() const { return !maTypeGroups.IsEmpty(); }
    /** Returns the index of the axes set (primary/secondary). */
    inline sal_uInt16   GetAxesSetId() const { return maData.mnAxesSetId; }
    /** Returns the axes set index used by the chart API. */
    inline sal_Int32    GetApiAxesSetIndex() const { return maData.GetApiAxesSetIndex(); }
    /** Returns true, if the chart is three-dimensional. */
    bool                Is3dChart() const;

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    /** Returns first inserted chart type group. */
    XclExpChTypeGroupRef GetFirstTypeGroup() const;
    /** Returns last inserted chart type group. */
    XclExpChTypeGroupRef GetLastTypeGroup() const;

    /** Converts a complete axis object including axis title. */
    void                ConvertAxis( XclExpChAxisRef& rxChAxis, sal_uInt16 nAxisType,
                            XclExpChTextRef& rxChAxisTitle, sal_uInt16 nTitleTarget,
                            XCoordSystemRef xCoordSystem, const XclChExtTypeInfo& rTypeInfo );

    virtual void        WriteBody( XclExpStream& rStrm );

private:
    typedef XclExpRecordList< XclExpChTypeGroup > XclExpChTypeGroupList;

    XclChAxesSet        maData;             /// Contents of the CHAXESSET record.
    XclExpChAxisRef     mxXAxis;            /// The X axis (CHAXIS group).
    XclExpChAxisRef     mxYAxis;            /// The Y axis (CHAXIS group).
    XclExpChAxisRef     mxZAxis;            /// The Z axis (CHAXIS group).
    XclExpChTextRef     mxXAxisTitle;       /// The X axis title (CHTEXT group).
    XclExpChTextRef     mxYAxisTitle;       /// The Y axis title (CHTEXT group).
    XclExpChTextRef     mxZAxisTitle;       /// The Z axis title (CHTEXT group).
    XclExpChFrameRef    mxPlotFrame;        /// Plot area (CHPLOTFRAME group).
    XclExpChTypeGroupList maTypeGroups;     /// Chart type groups (CHTYPEGROUP group).
};

typedef ScfRef< XclExpChAxesSet > XclExpChAxesSetRef;

// The chart object ===========================================================

/** Represents the CHCHART record group describing the chart contents.

    The CHCHART group consists of: CHCHART, CHBEGIN, SCL, CHPLOTGROWTH, CHFRAME
    group, CHSERIES groups, CHPROPERTIES, CHDEFAULTTEXT groups (CHDEFAULTTEXT
    with CHTEXT groups), CHUSEDAXESSETS, CHAXESSET groups, CHTEXT groups, CHEND.
 */
class XclExpChChart : public XclExpChGroupBase, protected XclExpChRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::chart2::XChartDocument > XChartDocRef;

public:
    explicit            XclExpChChart( const XclExpRoot& rRoot,
                            XChartDocRef xChartDoc, const Size& rSize );

    /** Creates, registers and returns a new data series object. */
    XclExpChSeriesRef   CreateSeries();
    /** Removes the last created data series object from the series list. */
    void                RemoveLastSeries();
    /** Stores a CHTEXT group that describes a data point label. */
    void                SetDataLabel( XclExpChTextRef xText );

    /** Writes all embedded records. */
    virtual void        WriteSubRecords( XclExpStream& rStrm );

private:
    virtual void        WriteBody( XclExpStream& rStrm );

private:
    typedef XclExpRecordList< XclExpChSeries >  XclExpChSeriesList;
    typedef XclExpRecordList< XclExpChText >    XclExpChTextList;

    XclChRectangle      maRect;             /// Position of the chart on the sheet (CHCHART record).
    XclExpChSeriesList  maSeries;           /// List of series data (CHSERIES groups).
    XclExpChFrameRef    mxFrame;            /// Chart frame format (CHFRAME group).
    XclChProperties     maProps;            /// Chart properties (CHPROPERTIES record).
    XclExpChAxesSetRef  mxPrimAxesSet;      /// Primary axes set (CHAXESSET group).
    XclExpChAxesSetRef  mxSecnAxesSet;      /// Secondary axes set (CHAXESSET group).
    XclExpChTextRef     mxTitle;            /// Chart title (CHTEXT group).
    XclExpChTextList    maLabels;           /// Data point labels (CHTEXT groups).
};

// ----------------------------------------------------------------------------

/** Represents the entire chart substream (all records in BOF/EOF block). */
class XclExpChart : public XclExpSubStream, protected XclExpRoot
{
public:
    typedef ::com::sun::star::uno::Reference< ::com::sun::star::frame::XModel > XModelRef;

public:
    explicit            XclExpChart( const XclExpRoot& rRoot,
                            XModelRef xModel, const Size& rSize );
};

// ============================================================================

#endif

