/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2007, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * $Id: SumFunction.java 3186 2007-08-15 16:54:01Z mimil $
 * ------------
 * (C) Copyright 2006-2007, by Pentaho Corporation.
 */
package org.jfree.formula.function.math;

import org.jfree.formula.EvaluationException;
import org.jfree.formula.FormulaContext;
import org.jfree.formula.LibFormulaErrorValue;
import org.jfree.formula.function.Function;
import org.jfree.formula.function.ParameterCallback;
import org.jfree.formula.lvalues.TypeValuePair;
import org.jfree.formula.typing.Type;
import org.jfree.formula.typing.coretypes.NumberType;
import org.jfree.formula.typing.sequence.NumberSequence;
import org.jfree.formula.util.NumberUtil;

import java.math.BigDecimal;

/**
 * Creation-Date: 31.10.2006, 17:39:19
 *
 * @author Thomas Morgner
 */
public class SumFunction implements Function
{
  public static final BigDecimal ZERO = new BigDecimal(0);

  public SumFunction()
  {
  }

  public String getCanonicalName()
  {
    return "SUM";
  }

  public TypeValuePair evaluate(final FormulaContext context,
                                final ParameterCallback parameters)
      throws EvaluationException
  {
    BigDecimal computedResult = ZERO;
    final int parameterCount = parameters.getParameterCount();

    if (parameterCount == 0)
    {
      throw new EvaluationException(LibFormulaErrorValue.ERROR_ARGUMENTS_VALUE);
    }

    for (int paramIdx = 0; paramIdx < parameterCount; paramIdx++)
    {
      final Type type = parameters.getType(paramIdx);
      final Object value = parameters.getValue(paramIdx);
      final NumberSequence sequence = context.getTypeRegistry().convertToNumberSequence(type, value);

      while (sequence.hasNext())
      {
        computedResult = compute(sequence.nextNumber(), computedResult);
      }
    }

    return new TypeValuePair(NumberType.GENERIC_NUMBER, computedResult);
  }

  private BigDecimal compute(final Number value,
                             final BigDecimal computedResult) throws EvaluationException
  {
    if (value == null)
    {
      // no-op ..
      return computedResult;
    }

    return computedResult.add(NumberUtil.getAsBigDecimal(value));
  }
}
