#!/bin/sh

# Copyright (C) 2006  Kel Modderman <kelrin@tpg.com.au>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# On Debian GNU/Linux systems, the text of the GPL license can be
# found in /usr/share/common-licenses/GPL.

# Quit if we're called for the loopback
if [ "$IFACE" = lo ]; then
	exit 0
fi

WPA_SUP_BIN="/sbin/wpa_supplicant"
WPA_SUP_PNAME="wpa_supplicant"
WPA_SUP_PIDFILE="/var/run/wpa_supplicant.$IFACE.pid"
WPA_SUP_OPTIONS="-B -P $WPA_SUP_PIDFILE -i $IFACE"

WPA_CLI_BIN="/sbin/wpa_cli"
WPA_CLI_PNAME="wpa_cli"
WPA_CLI_PIDFILE="/var/run/wpa_cli.$IFACE.pid"
WPA_CLI_ACTFILE="/var/run/wpa_cli.$IFACE.action"
WPA_CLI_OPTIONS="-B -P $WPA_CLI_PIDFILE -i $IFACE"

WPA_COMMON_CTRL_IFACE="/var/run/wpa_supplicant"

if [ ! -x "$WPA_SUP_BIN" -o ! -x "$WPA_CLI_BIN" ]; then
	exit 0
fi

if [ -s "$IF_WPA_CONF" ]; then
	WPA_SUP_CONF="-c $IF_WPA_CONF"
	WPA_SUP_CONF_CTRL_IFACE=$(sed -n -e 's/[ \t]*#.*//g;s/[ \t]*$//g;s/^ctrl_interface=//p' $IF_WPA_CONF)

	if [ -n "$WPA_SUP_CONF_CTRL_IFACE" ]; then
		WPA_COMMON_CTRL_IFACE="$WPA_SUP_CONF_CTRL_IFACE"
	fi

elif env | grep -q "^IF_WPA"; then
	WPA_SUP_CONF="-C $WPA_COMMON_CTRL_IFACE"
else
	exit 0
fi

if [ "$VERBOSITY" = "1" ]; then
	echo "$WPA_SUP_PNAME: ctrl_interface=$WPA_COMMON_CTRL_IFACE"
	# hide wpa_cli messages unless VERBOSITY has been set
	TO_NULL="/dev/stdout"
	# deafult options for start-stop-daemon
	DAEMON_VEROSITY="--verbose"
else
	TO_NULL="/dev/null"
	DAEMON_VERBOSITY="--quiet"
fi

case "$MODE" in 
	start)
	if [ -f "$WPA_SUP_PIDFILE" ]; then
		
		if [ "$VERBOSITY" = "1" ]; then
			echo "$WPA_SUP_PNAME: terminating orphaned $WPA_SUP_PNAME daemon"
		fi
		
		start-stop-daemon --stop --oknodo $DAEMON_VEROSITY \
			--name $WPA_SUP_PNAME --pidfile	$WPA_SUP_PIDFILE
		
		if [ -f "$WPA_SUP_PIDFILE" ]; then
			rm -f $WPA_SUP_PIDFILE
		fi

		if [ -S "$WPA_COMMON_CTRL_IFACE/$IFACE" ]; then
			rm -f $WPA_COMMON_CTRL_IFACE/$IFACE
		fi
	fi
	
	if [ ! -S "$WPA_COMMON_CTRL_IFACE/$IFACE" ]; then
		
		# driver type of interface, defaults to wext when undefined
		if [ -n "$IF_WPA_DRIVER" ]; then
			WPA_DRIVER="$IF_WPA_DRIVER"
		else
			if ! grep -q "^[ \t]*$IFACE:" /proc/net/wireless 2>/dev/null; then
				# no wireless extensions
				WPA_DRIVER="wired"
			else
				# wext is default/fallback
				WPA_DRIVER="wext"
			fi
		
			if [ "$VERBOSITY" = "1" ]; then
				echo "$WPA_SUP_PNAME: guessing wpa-driver type of \"$WPA_DRIVER\" for $IFACE"
			fi
		fi

		if [ -x "$IF_WPA_ACTION" ]; then
			# wait for wpa_cli to attach
			WPA_SUP_OPTIONS="-W $WPA_SUP_OPTIONS"	
			if [ "$VERBOSITY" = "1" ]; then
				echo "$WPA_SUP_PNAME: waiting for wpa_cli to attach before starting"
			fi
		elif [ -n "$IF_WPA_ACTION" -a ! -x "$IF_WPA_ACTION" ]; then
			echo "$WPA_SUP_PNAME: action script not executable, not launchng wpa_cli action daemon" >/dev/stderr
			exit 1
		fi

		# if we have passed the criteria, start wpa_supplicant
		if [ -n "$WPA_SUP_CONF" ]; then
			
			if [ "$VERBOSITY" = "1" ]; then
				echo "$WPA_SUP_PNAME: $WPA_SUP_BIN $WPA_SUP_OPTIONS $WPA_SUP_CONF -D $WPA_DRIVER"
			fi
			
			start-stop-daemon --start --oknodo $DAEMON_VEROSITY \
				--name $WPA_SUP_PNAME --startas $WPA_SUP_BIN --pidfile $WPA_SUP_PIDFILE \
				-- $WPA_SUP_OPTIONS $WPA_SUP_CONF -D $WPA_DRIVER
		fi
	else
		echo "$WPA_SUP_PNAME: unknown wpa_supplicant process created $WPA_COMMON_CTRL_IFACE/$IFACE, exiting" >/dev/stderr
		exit 1
	fi

	# if the interface socket exists, then wpa_supplicant was invoked successfully
	if [ -S "$WPA_COMMON_CTRL_IFACE/$IFACE" ]; then
	
		if [ "$VERBOSITY" = "1" ]; then
			echo "$WPA_SUP_PNAME: ctrl_interface socket located at $WPA_COMMON_CTRL_IFACE/$IFACE"
		fi
	
		# global settings
		if [ -n "$IF_WPA_AP_SCAN" ]; then

			if [ "$VERBOSITY" = "1" ]; then
				echo -n "$WPA_SUP_PNAME: wpa-ap-scan : "
			fi
			
			$WPA_CLI_BIN  -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
				ap_scan "$IF_WPA_AP_SCAN" >$TO_NULL
		elif [ -n "$IF_WPA_APSCAN" ]; then
			
			if [ "$VERBOSITY" = "1" ]; then
				echo -n "$WPA_SUP_PNAME: wpa-apscan : "
			fi
			
			$WPA_CLI_BIN  -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
				ap_scan "$IF_WPA_APSCAN" >$TO_NULL
		fi

		if [ -n "$IF_WPA_PREAUTHENTICATE" ]; then

			if [ "$VERBOSITY" = "1" ]; then
				echo -n "$WPA_SUP_PNAME: wpa-preauthenticate : "
			fi
		
			$WPA_CLI_BIN  -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
				preauthenticate "$IF_WPA_PREAUTHENTICATE" >$TO_NULL
		fi
			
		if [ -n "$IF_WPA_SSID" ]; then
	
			# append/create network block on first unused network_id
			NW_ID=$($WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE list_networks | \
				awk '/^[0-9]+/{ f++ ; if ($1 >= max) max = $1 } END { if (f) print max+1 ; else print 0 }')
			
			if [ "$VERBOSITY" = "1" ]; then
				echo -n "$WPA_SUP_PNAME: configuring new network block, guessing network_id of \"$NW_ID\"? : "
			fi
		
			# initialize a new network block
			$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
				add_network >$TO_NULL

			if [ "$VERBOSITY" = "1" ]; then
				echo -n "$WPA_SUP_PNAME: wpa-ssid : "
			fi
			
			# add ssid to new network block
			$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
				set_network $NW_ID ssid "\"$IF_WPA_SSID\"" >$TO_NULL
		
			if [ -n "$IF_WPA_PRIORITY" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-priority : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID priority "$IF_WPA_PRIORITY" >$TO_NULL
			fi
			
			# configure the new network block
			if [ -n "$IF_WPA_BSSID" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-bssid : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID bssid "$IF_WPA_BSSID" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PSK" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-psk : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID psk "$IF_WPA_PSK" >$TO_NULL
			elif [ -s "$IF_WPA_PSK_FILE" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-psk-file : "
				fi
				
				IF_WPA_PSK=$(cat "$IF_WPA_PSK_FILE")
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID psk "$IF_WPA_PSK" >$TO_NULL
			elif [ -n "$IF_WPA_PASSPHRASE" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-passphrase : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID psk "\"$IF_WPA_PASSPHRASE\"" >$TO_NULL
			elif [ -s "$IF_WPA_PASSPHRASE_FILE" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-passphrase-file : "
				fi
				
				IF_WPA_PASSPHRASE=$(cat "$IF_WPA_PASSPHRASE_FILE")
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID psk "\"$IF_WPA_PASSPHRASE\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PAIRWISE" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-pairwise : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID pairwise "$IF_WPA_PAIRWISE" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_GROUP" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-group : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID group "$IF_WPA_GROUP" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_KEY_MGMT" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-key-mgmt : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID key_mgmt "$IF_WPA_KEY_MGMT" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PROTO" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-proto : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID proto "$IF_WPA_PROTO" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_AUTH_ALG" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-auth-alg : "
				fi
			
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID auth_alg "$IF_WPA_AUTH_ALG" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_SCAN_SSID" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-scan-ssid : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID scan_ssid "$IF_WPA_SCAN_SSID" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_IDENTITY" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-identity : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID identity "\"$IF_WPA_IDENTITY\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_ANONYMOUS_IDENTITY" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-anonymous-identity : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID anonymous_identity "\"$IF_WPA_ANONYMOUS_IDENTITY\"" >$TO_NULL
			fi
			
			if [ -n "$IF_WPA_EAP" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-eap : "
				fi
			
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID eap "$IF_WPA_EAP" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_EAPPSK" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-eappsk : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID eappsk "$IF_WPA_EAPPSK" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_NAI" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-nai : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID nai "\"$IF_WPA_NAI\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PASSWORD" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-password : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID password "\"$IF_WPA_PASSWORD\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_CA_CERT" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-ca-cert : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID ca_cert "\"$IF_WPA_CA_CERT\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_CA_PATH" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-ca-path"
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID ca_path "\"$IF_WPA_CA_PATH\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_CLIENT_CERT" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-client-cert : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID client_cert "\"$IF_WPA_CLIENT_CERT\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PRIVATE_KEY" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-private-key : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID private_key "\"$IF_WPA_PRIVATE_KEY\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PRIVATE_KEY_PASSWD" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-private-key-passwd : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID private_key_passwd "\"$IF_WPA_PRIVATE_KEY_PASSWD\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_DH_FILE"  ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-dh-file : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID dh_file "\"$IF_WPA_DH_FILE\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_SUBJECT_MATCH" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-subject-match : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID subject_match "\"$IF_WPA_SUBJECT_MATCH\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_ALTSUBJECT_MATCH" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-altsubject-match : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID subject_match "\"$IF_WPA_ALTSUBJECT_MATCH\"" >$TO_NULL
			fi
			
			if [ -n "$IF_WPA_CA_CERT2" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-ca-cert2 : "
				fi
			
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID ca_cert "\"$IF_WPA_CA_CERT2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_CA_PATH2" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-ca-path2 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID ca_path "\"$IF_WPA_CA_PATH2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_CLIENT_CERT2" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-client-cert2 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID client_cert "\"$IF_WPA_CLIENT_CERT2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PRIVATE_KEY2" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-private-key2 : "
				fi
			
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID private_key "\"$IF_WPA_PRIVATE_KEY2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PRIVATE_KEY_PASSWD2" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-private-key-passwd2 : "
				fi
			
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID private_key_passwd "\"$IF_WPA_PRIVATE_KEY_PASSWD2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_DH_FILE2"  ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-dh-file2 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID dh_file "\"$IF_WPA_DH_FILE2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_SUBJECT_MATCH2" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-subject-match2 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID subject_match "\"$IF_WPA_SUBJECT_MATCH2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_ALTSUBJECT_MATCH2" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-altsubject-match2 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID subject_match "\"$IF_WPA_ALTSUBJECT_MATCH2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_EAP_METHODS" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-eap-methods : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID eap_methods "$IF_WPA_EAP_METHODS" >$TO_NULL
			fi
			
			if [ -n "$IF_WPA_PHASE1" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-phase1 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID phase1 "\"$IF_WPA_PHASE1\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PHASE2" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-phase2 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID phase2 "\"$IF_WPA_PHASE2\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PCSC" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-pcsc : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID pcsc "$IF_WPA_PCSC" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PIN" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-pin : "
				fi
			
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID pin "\"$IF_WPA_PIN\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_ENGINE" ]; then
			
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-engine : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID engine "$IF_WPA_ENGINE" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_ENGINE_ID" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-engine-id : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID engine "\"$IF_WPA_ENGINE_ID\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_KEY_ID" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-key-id : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID engine "\"$IF_WPA_KEY_ID\"" >$TO_NULL
			fi
			
			if [ -n "$IF_WPA_EAPOL_FLAGS" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-eapol-flags : "
				fi
			
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID eapol_flags "$IF_WPA_EAPOL_FLAGS" >$TO_NULL
			fi
			
			if [ -n "$IF_WPA_WEP_KEY0" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-wep-key0 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID wep_key0 "$IF_WPA_WEP_KEY0" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_WEP_KEY1" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-wep-key1 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID wep_key1 "$IF_WPA_WEP_KEY1" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_WEP_KEY2" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-wep-key2 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
				set_network $NW_ID wep_key2 "$IF_WPA_WEP_KEY2" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_WEP_KEY3" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-wep-key3 : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID wep_key3 "$IF_WPA_WEP_KEY3" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_WEP_TX_KEYIDX" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-wep-tx-keyidx : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID wep_tx_keyidx "$IF_WPA_WEP_TX_KEYIDX" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PROACTIVE_KEY_CACHING" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-proactive-key-caching : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID proactive_key_caching "$IF_WPA_PROACTIVE_KEY_CACHING" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_EAP_WORKAROUND" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-eap-workaround : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID eap_workaround "$IF_WPA_EAP_WORKAROUND" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_PAC_FILE" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-pac-file : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID pac_file "\"$IF_WPA_PAC_FILE\"" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_MODE" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-mode : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID mode "$IF_WPA_MODE" >$TO_NULL
			fi
		
			if [ -n "$IF_WPA_STAKEY" ]; then
				
				if [ "$VERBOSITY" = "1" ]; then
					echo -n "$WPA_SUP_PNAME: wpa-stakey : "
				fi
				
				$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
					set_network $NW_ID stakey "$IF_WPA_STAKEY" >$TO_NULL
			fi
		
			if [ "$VERBOSITY" = "1" ]; then
				echo -n "$WPA_SUP_PNAME: enabling network $NW_ID : "
			fi
			
			# finality: activate the new network block
			$WPA_CLI_BIN -p $WPA_COMMON_CTRL_IFACE -i $IFACE \
				enable_network $NW_ID >$TO_NULL
		fi
		
		# start wpa_cli action daemon if asked for
		if [ -x "$IF_WPA_ACTION" -a ! -f "$WPA_CLI_PIDFILE" ]; then
			
			# ensure the actfile does not exist
			if [ -f "$WPA_CLI_ACTFILE" ]; then
				rm -f $WPA_CLI_ACTFILE
			fi
			
			if [ "$VERBOSITY" = "1" ]; then
				echo "$WPA_SUP_PNAME: $WPA_CLI_BIN $WPA_CLI_OPTIONS -p $WPA_COMMON_CTRL_IFACE -a $IF_WPA_ACTION"
			fi
			
			start-stop-daemon --start --oknodo $DAEMON_VEROSITY \
				--name $WPA_CLI_PNAME --startas $WPA_CLI_BIN --pid $WPA_CLI_PIDFILE \
				-- $WPA_CLI_OPTIONS -p $WPA_COMMON_CTRL_IFACE -a $IF_WPA_ACTION
			
			if [ "$METHOD" = "manual" ]; then
				# IF_WPA_ACTION timeout loop
				WPA_ACTION_TIME="0"
				if [ -z "$IF_WPA_ACTION_TIMEOUT" ]; then
					# default: allow one minute for action script to succeed
					IF_WPA_ACTION_TIMEOUT="60"
				fi
				
				while true; do
				
					if [ "$IF_WPA_ACTION_TIMEOUT" -lt "0" ]; then
						# dont wait
						echo "$WPA_SUP_PNAME: backgrounding $IF_WPA_ACTION action script" >/dev/stdout
						break
					elif [ -e "$WPA_CLI_ACTFILE" ]; then
						# success
						if [ "$VERBOSITY" = "1" ]; then
							echo "$WPA_SUP_PNAME: action script $IF_WPA_ACTION succeeded" >/dev/stdout
						fi
						break
					elif [ "$IF_WPA_ACTION_TIMEOUT" = "0" ]; then
						# infinite wait
						continue
					elif [ "$WPA_ACTION_TIME" -ge "$IF_WPA_ACTION_TIMEOUT" ]; then
						# NB: it is safe to check for wpa_cli's pidfile only after sufficient time has passed
						if [ ! -e "$WPA_CLI_PIDFILE" ]; then
							# failure
							echo "$WPA_SUP_PNAME: action script failed" >/dev/stderr
							# don't exit, or else we leave wpa_supplicant mess behind
							break
						else 
							# timeout
							echo "$WPA_SUP_PNAME: action script timed out, backgrounding" >/dev/stdout
							break
						fi
					fi
					
					WPA_ACTION_TIME=$(($WPA_ACTION_TIME+1))
					sleep 1
				done
			fi
		fi
	else
		# wpa_supplicant socket does not exist; wpa_supplicant failed to start
		echo "$WPA_SUP_PNAME: failed to locate ctrl_interface socket, exiting" >/dev/stderr
		exit 1
	fi
	;;

	stop)
	case "$PHASE" in
		pre-down)
		# kill wpa_cli before wpa_supplicant
		# future versions of wpa_supplicant may send a disconnect signal to wpa_cli
		if [ -f "$WPA_CLI_PIDFILE" ]; then
		
			if [ "$VERBOSITY" = "1" ]; then
				echo "$WPA_SUP_PNAME: sending disconnect signal and terminating $WPA_CLI_PNAME daemon"
				echo -n "$WPA_SUP_PNAME: $WPA_CLI_BIN  -p $WPA_COMMON_CTRL_IFACE -i $IFACE disconnect : "
			fi
		
			$WPA_CLI_BIN  -p $WPA_COMMON_CTRL_IFACE -i $IFACE disconnect >$TO_NULL
			
			start-stop-daemon --stop --oknodo $DAEMON_VEROSITY \
				--name $WPA_CLI_PNAME --pidfile	$WPA_CLI_PIDFILE
		
			if [ -f "$WPA_CLI_PIDFILE" ]; then
				rm -f $WPA_CLI_PIDFILE
			fi

			if [ -f "$WPA_CLI_ACTFILE" ]; then
				rm -f $WPA_CLI_ACTFILE
			fi
		fi
		;;

		post-down)
		# Shutdown wpa_supplicant
		if [ -f "$WPA_SUP_PIDFILE" ]; then
		
			if [ "$VERBOSITY" = "1" ]; then
				echo "$WPA_SUP_PNAME: terminating $WPA_SUP_PNAME daemon"
			fi
		
			start-stop-daemon --stop --oknodo $DAEMON_VEROSITY \
				--name $WPA_SUP_PNAME --pidfile	$WPA_SUP_PIDFILE
			
			if [ -f "$WPA_SUP_PIDFILE" ]; then
				rm -f $WPA_SUP_PIDFILE
			fi

			if [ -S "$WPA_COMMON_CTRL_IFACE/$IFACE" ]; then
				rm -f $WPA_COMMON_CTRL_IFACE/$IFACE
			fi
		fi
		;;

		*)
		echo "$WPA_SUP_PNAME: wrong phase : \"$PHASE\"" >/dev/stderr
		exit 1
		;;
	esac
	;;

	*)
	echo "$WPA_SUP_PNAME: unknown mode : \"$MODE\"" >/dev/stderr
	exit 1
	;;
esac
