# Copyright 2012-2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Model definition for NodeGroupInterface."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    'NodeGroupInterface',
    ]


from django.db.models import (
    CharField,
    ForeignKey,
    IntegerField,
    PROTECT,
)
from maasserver import DefaultMeta
from maasserver.enum import (
    NODEGROUPINTERFACE_MANAGEMENT,
    NODEGROUPINTERFACE_MANAGEMENT_CHOICES,
)
from maasserver.fields import (
    MAASIPAddressField,
    VerboseRegexValidator,
)
from maasserver.models.cleansave import CleanSave
from maasserver.models.nodegroup import NodeGroup
from maasserver.models.timestampedmodel import TimestampedModel


def get_default_vlan():
    from maasserver.models.vlan import VLAN
    return VLAN.objects.get_default_vlan().id


class NodeGroupInterface(CleanSave, TimestampedModel):
    """Cluster interface.

    Represents a network to which a given cluster controller is connected.
    These interfaces are discovered automatically, but an admin can also
    add/edit/remove them.

    This class duplicates some of :class:`Network`, and adds settings for
    managing DHCP.  Some day we hope to delegate the duplicated fields, and
    have auto-discovery populate the :class:`Network` model along the way.
    """

    class Meta(DefaultMeta):
        # The API identifies a NodeGroupInterface by cluster and name.
        unique_together = ('nodegroup', 'name')

    # Static IP of the network interface.
    ip = MAASIPAddressField(
        null=False, editable=True,
        help_text="Static IP Address of the interface",
        verbose_name="IP")

    # The `NodeGroup` this interface belongs to.
    nodegroup = ForeignKey(
        'maasserver.NodeGroup', editable=True, null=False, blank=False)

    vlan = ForeignKey(
        'VLAN', default=get_default_vlan, editable=True, blank=False,
        null=False, on_delete=PROTECT)

    # Note: adding null=True temporarily; will be adjusted after a data
    # migration occurs to create each subnet link.
    subnet = ForeignKey(
        'Subnet', editable=True, blank=True, null=True, on_delete=PROTECT)

    # Name for this interface.  It must be unique within the cluster.
    # The code ensures that this is never an empty string, but we do allow
    # an empty string on the form.  The field defaults to a unique name based
    # on the network interface name.
    name = CharField(
        blank=True, null=False, editable=True, max_length=255, default='',
        validators=[VerboseRegexValidator('^[\w:.-]+$')],
        help_text=(
            "Identifying name for this cluster interface.  "
            "Must be unique within the cluster, and consist only of letters, "
            "digits, dashes, and colons."))

    management = IntegerField(
        choices=NODEGROUPINTERFACE_MANAGEMENT_CHOICES, editable=True,
        default=NODEGROUPINTERFACE_MANAGEMENT.DEFAULT)

    # DHCP server settings.
    interface = CharField(
        blank=True, editable=True, max_length=255, default='',
        help_text="Network interface (e.g. 'eth1').")

    ip_range_low = MAASIPAddressField(
        editable=True, unique=False, blank=True, null=True, default=None,
        verbose_name="DHCP dynamic IP range low value",
        help_text="Lowest IP number of the range for dynamic IPs, used for "
                  "enlistment, commissioning and unknown devices.")
    ip_range_high = MAASIPAddressField(
        editable=True, unique=False, blank=True, null=True, default=None,
        verbose_name="DHCP dynamic IP range high value",
        help_text="Highest IP number of the range for dynamic IPs, used for "
                  "enlistment, commissioning and unknown devices.")
    static_ip_range_low = MAASIPAddressField(
        editable=True, unique=False, blank=True, null=True, default=None,
        verbose_name="Static IP range low value",
        help_text="Lowest IP number of the range for IPs given to allocated "
                  "nodes, must be in same network as dynamic range.")
    static_ip_range_high = MAASIPAddressField(
        editable=True, unique=False, blank=True, null=True, default=None,
        verbose_name="Static IP range high value",
        help_text="Highest IP number of the range for IPs given to allocated "
                  "nodes, must be in same network as dynamic range.")

    # Foreign DHCP server address, if any, that was detected on this
    # interface.
    foreign_dhcp_ip = MAASIPAddressField(
        null=True, default=None, editable=True, blank=True, unique=False)
