// ---------------------------------------------------------------------------
// - Codec.cpp                                                               -
// - afnix:sec module - abstract codec class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2019 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Codec.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "QuarkZone.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default codec

  Codec::Codec (void) {
    d_rflg = false;
    d_snum = 0L;
    d_cnum = 0L;
  }

  // create a codec by name

  Codec::Codec (const String& name) {
    d_name = name;
    d_rflg = false;
    d_snum = 0L;
    d_cnum = 0L;
  }

  // return the codec name
    
  String Codec::getname (void) const {
    rdlock ();
    try {
      String result = d_name;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // reset this codec

  void Codec::reset (void) {
    wrlock ();
    try {
      d_snum = 0;
      d_cnum = 0L;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }
  
  // clear this codec

  void Codec::clear (void) {
    wrlock ();
    try {
      d_snum = 0;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the codec reverse flag

  void Codec::setrflg (const bool rflg) {
    wrlock ();
    try {
      d_rflg = rflg;
      reset  ();
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the codec reverse flag

  bool Codec::getrflg (void) const {
    rdlock ();
    try {
      bool result = d_rflg;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // stream an input buffer into an output buffer

  long Codec::stream (Buffer& ob, Buffer& ib) {
    wrlock ();
    try {
      // clear the stream
      clear ();
      // stream in mode
      long result = 0L;
      while (ib.empty() == false) {
	result += d_rflg ? decode (ob, ib) : encode (ob, ib);
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // stream an input stream into an output buffer

  long Codec::stream (Buffer& ob, InputStream& is) {
    wrlock ();
    try {
      // clear the stream
      clear ();
      // stream in mode
      long result = 0L;
      while (is.valid () == true) {
	result += d_rflg ? decode (ob, is) : encode (ob, is);
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // stream an input stream into an output stream

  long Codec::stream (OutputStream& os, InputStream& is) {
    wrlock ();
    try {
      // clear the stream
      clear ();
      // stream in mode
      long result = 0;
      while (is.valid () == true) {
	result += d_rflg ? decode (os, is) : encode (os, is);
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 5;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_RESET   = zone.intern ("reset");
  static const long QUARK_CLEAR   = zone.intern ("clear");
  static const long QUARK_STREAM  = zone.intern ("stream");
  static const long QUARK_SETRFLG = zone.intern ("set-reverse");
  static const long QUARK_GETRFLG = zone.intern ("get-reverse");

  // return true if the given quark is defined

  bool Codec::isquark (const long quark, const bool hflg) const {
    rdlock ();
    try {
      if (zone.exists (quark) == true) {
	unlock ();
	return true;
      }
      bool result = hflg ? Nameable::isquark (quark, hflg) : false;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this object with a set of arguments and a quark
  
  Object* Codec::apply (Runnable* robj, Nameset* nset, const long quark,
			Vector* argv) {
    // get the number of arguments
    long argc = (argv == nullptr) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETRFLG) return new Boolean (getrflg ());
      if (quark == QUARK_RESET) {
	reset ();
	return nullptr;
      }
      if (quark == QUARK_CLEAR) {
	clear ();
	return nullptr;
      }
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETRFLG) {
	bool rflg = argv->getbool (0);
	setrflg (rflg);
	return nullptr;
      }
    }
    // check for 2 arguments
    if (argc == 2) {
      if (quark == QUARK_STREAM) {
	// get the first object
	Object* obj = argv->get (0);
	// check for a buffer
	Buffer* ob = dynamic_cast <Buffer*> (obj);
	if (ob != nullptr) {
	  // get the second object
	  obj = argv->get (1);
	  // check for a buffer
	  Buffer* ib = dynamic_cast <Buffer*> (obj);
	  if (ib != nullptr) return new Integer (stream (*ob, *ib));
	  // check for an input stream
	  InputStream* is = dynamic_cast <InputStream*> (obj);
	  if (is != nullptr) return new Integer (stream (*ob, *is));
	  // type error
	  throw Exception ("type-error", "invalid object for codec stream",
			   Object::repr (obj));
	}
	// check for a stream
	OutputStream* os = dynamic_cast <OutputStream*> (obj);
	if (os != nullptr) {
	  // get the second object
	  obj = argv->get (1);
	  // check for an input stream
	  InputStream* is = dynamic_cast <InputStream*> (obj);
	  if (is != nullptr) return new Integer (stream (*os, *is));
	  // type error
	  throw Exception ("type-error", "invalid object for codec stream",
			   Object::repr (obj));
	}
	// invalid object
	throw Exception ("type-error", "invalid object for codec stream",
			 Object::repr (obj));
      }
    }
    // call the nameable method
    return Nameable::apply (robj, nset, quark, argv);
  }
}
