//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Tests/Suite/GUI/Check.cpp
//! @brief     Implements function checkSimulation for GUI standard test
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "BABuild.h"
#include "BATesting.h"
#include "Base/Util/PathUtil.h"
#include "Device/Data/Datafield.h"
#include "Device/Histo/DiffUtil.h"
#include "Device/Histo/SimulationResult.h"
#include "Device/IO/IOFactory.h"
#include "GUI/Model/Device/InstrumentModel.h"
#include "GUI/Model/FromCore/ItemizeSample.h"
#include "GUI/Model/FromCore/ItemizeSimulation.h"
#include "GUI/Model/Sample/SampleItem.h"
#include "GUI/Model/ToCore/SimulationToCore.h"
#include "GUI/Support/Data/SimulationOptionsItem.h"
#include "Sim/Simulation/ISimulation.h"
#include <filesystem>
#include <iostream>

namespace fs = std::filesystem;

namespace {

//! Converts ISimulation from core to GUI representation and back.
std::unique_ptr<ISimulation> indirectSimulation(const ISimulation& sim)
{
    std::unique_ptr<SampleItem> sampleItem(GUI::FromCore::itemizeSample(*sim.sample()));
    std::unique_ptr<InstrumentModel> instrumentItems(GUI::FromCore::itemizeInstruments(sim));
    std::unique_ptr<SimulationOptionsItem> optionsItem(GUI::FromCore::itemizeOptions(sim));

    return GUI::ToCore::itemsToSimulation(*sampleItem, *instrumentItems->instrumentItems().front(),
                                          *optionsItem);
}

} // namespace

//! Run simulation directly (in core) and through GUI model, and compare results.

bool checkSimulation(const std::string& name, ISimulation& sim, const double limit)
{
    std::cout << "Tests/Suite/GUI:"
              << " Validate round trip (simulation -> GUI items -> simulation)"
              << " against direct simulation." << std::endl;

    std::unique_ptr<ISimulation> sim2(indirectSimulation(sim));

    const SimulationResult data2 = sim2->simulate();
    const SimulationResult ref_data = sim.simulate();

    bool ok = DiffUtil::checkRelativeDifference(data2.flatVector(), ref_data.flatVector(), limit);

    if (ok)
        return true;

    // Save simulation, as it differs from reference.
    const auto dir = fs::path(BATesting::TestOutDir()) / fs::path("Suite") / fs::path("GUI");
    Base::Path::createDirectories(dir.string());

    const std::map<const std::string, const SimulationResult&> tosave{{"gui", data2},
                                                                      {"std", ref_data}};
    for (const auto& [kind, data] : tosave) {
        const std::string out_fname = (dir / fs::path(name + "." + kind + ".int.gz")).string();
        IO::writeDatafield(data, out_fname);
        std::cout << "Saved " << out_fname << "\n";
    }
    std::cout << "Notes:\n"
              << "- to visualize an intensity map, use " << BABuild::srcDir()
              << "/devtools/view/plot_int.py\n"
              << "- to plot a difference image, use " << BABuild::srcDir()
              << "/devtools/view/plot_diff_int.py\n";
    return false;
}
