/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      CMORlite      cmorlite        CMOR lite
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "param_conversion.h"
#include "cdo_cmor.h"
#include "pmlist.h"
#include "convert_units.h"
#include "cdo_cdiWrapper.h"
#include "cdi_lockedIO.h"

void
cdo_define_var_units(CmorVar &var, int vlistID2, int varID, const char *units)
{
  char units_old[CDI_MAX_NAME];

  vlistInqVarUnits(vlistID2, varID, units_old);
  const auto len1 = strlen(units_old);
  const auto len2 = strlen(units);

  if (strcmp(units, units_old) != 0)
    {
      if (len1 > 0 && len2 > 0)
        {
          var.changeunits = true;
          strcpy(var.units_old, units_old);
          strcpy(var.units, units);
        }

      vlistDefVarUnits(vlistID2, varID, units);
      cdiDefAttTxt(vlistID2, varID, "original_units", (int) strlen(units_old), units_old);
    }
}

void
cmor_check_init(int nvars, std::vector<CmorVar> &vars)
{
  for (int varID = 0; varID < nvars; ++varID)
    {
      auto &var = vars[varID];
      if (var.checkvalid || var.check_min_mean_abs || var.check_max_mean_abs)
        {
          var.amean = 0;
          var.nvals = 0;
          var.n_lower_min = 0;
          var.n_greater_max = 0;
        }
    }
}

void
cmor_check_eval(int vlistID, int nvars, std::vector<CmorVar> &vars)
{
  char varname[CDI_MAX_NAME];

  for (int varID = 0; varID < nvars; ++varID)
    {
      const auto &var = vars[varID];
      if (var.checkvalid || var.check_min_mean_abs || var.check_max_mean_abs)
        {
          auto amean = var.amean;
          const auto nvals = var.nvals;

          if (nvals > 0) amean /= nvals;

          const auto n_lower_min = var.n_lower_min;
          const auto n_greater_max = var.n_greater_max;

          vlistInqVarName(vlistID, varID, varname);

          if (n_lower_min > 0)
            cdoWarning("Invalid value(s) detected for variable '%s': %ld values were lower than minimum valid value (%.4g).",
                       varname, n_lower_min, var.valid_min);
          if (n_greater_max > 0)
            cdoWarning("Invalid value(s) detected for variable '%s': %ld values were greater than maximum valid value (%.4g).",
                       varname, n_greater_max, var.valid_max);

          if (var.check_min_mean_abs)
            {
              if (amean < .1 * var.ok_min_mean_abs)
                cdoWarning("Invalid Absolute Mean for variable '%s' (%.5g) is lower by more than an order of magnitude than "
                           "minimum allowed: %.4g",
                           varname, amean, var.ok_min_mean_abs);

              if (amean < var.ok_min_mean_abs)
                cdoWarning("Invalid Absolute Mean for variable '%s' (%.5g) is lower than minimum allowed: %.4g", varname, amean,
                           var.ok_min_mean_abs);
            }

          if (var.check_max_mean_abs)
            {
              if (amean > 10. * var.ok_max_mean_abs)
                cdoWarning("Invalid Absolute Mean for variable '%s' (%.5g) is greater by more than an order of magnitude than "
                           "maximum allowed: %.4g",
                           varname, amean, var.ok_max_mean_abs);

              if (amean > var.ok_max_mean_abs)
                cdoWarning("Invalid Absolute Mean for variable '%s' (%.5g) is greater than maximum allowed: %.4g", varname, amean,
                           var.ok_max_mean_abs);
            }
        }
    }
}

void
cmor_check_prep(CmorVar &var, long gridsize, double missval, double *array)
{
  if (var.checkvalid || var.check_min_mean_abs || var.check_max_mean_abs)
    {
      double aval;
      double amean = 0;
      long nvals = 0;

      for (long i = 0; i < gridsize; ++i)
        {
          aval = array[i];
          if (!DBL_IS_EQUAL(aval, missval))
            {
              amean += std::fabs(aval);
              nvals++;
            }
        }

      var.amean += amean;
      var.nvals += nvals;

      long n_lower_min = 0;
      long n_greater_max = 0;

      for (long i = 0; i < gridsize; ++i)
        {
          aval = array[i];
          if (!DBL_IS_EQUAL(aval, missval))
            {
              if (aval < var.valid_min) n_lower_min++;
              if (aval > var.valid_max) n_greater_max++;
            }
        }

      var.n_lower_min += n_lower_min;
      var.n_greater_max += n_greater_max;
    }
}

static void
applyCmorList(PMList &pmlist, int nvars, int vlistID2, std::vector<CmorVar> &vars)
{
  const std::vector<std::string> hentry = { "Header" };
  const std::vector<std::string> ventry = { "variable_entry", "parameter" };
  char varname[CDI_MAX_NAME];

  // search for global missing value
  bool lmissval = false;
  double missval = 0;

  {
    auto kvlist = pmlist.getKVListVentry(hentry);
    if (kvlist)
      {
        for (const auto &kv : *kvlist)
          {
            const auto &key = kv.key;
            const auto &value = kv.values[0];
            if (kv.nvalues != 1 || value.empty()) continue;

            if (key == "missing_value")
              {
                lmissval = true;
                missval = parameter2double(value);
              }
            else if (key == "table_id" || key == "modeling_realm" || key == "realm" || key == "project_id" || key == "frequency")
              {
                cdiDefAttTxt(vlistID2, CDI_GLOBAL, key.c_str(), (int) value.size(), value.c_str());
              }
          }
      }
  }

  for (int varID = 0; varID < nvars; varID++)
    {
      auto &var = vars[varID];
      vlistInqVarName(vlistID2, varID, varname);

      strcpy(var.name, varname);
      if (lmissval)
        {
          const double missval_old = vlistInqVarMissval(vlistID2, varID);
          if (!DBL_IS_EQUAL(missval, missval_old))
            {
              var.changemissval = true;
              var.missval_old = missval_old;
              vlistDefVarMissval(vlistID2, varID, missval);
            }
        }

      auto kvlist = pmlist.searchKVListVentry("name", varname, ventry);
      if (kvlist)
        {
          bool lvalid_min = false, lvalid_max = false;

          for (const auto &kv : *kvlist)
            {
              const auto &key = kv.key;
              const auto &value = kv.values[0];
              if (kv.nvalues != 1 || value.empty()) continue;
              auto value_cstr = value.c_str();

              // printf("key=%s  value=>%s<\n", key.c_str(), value.c_str());

              // clang-format off
              if      (key == "standard_name") vlistDefVarStdname(vlistID2, varID, value_cstr);
              else if (key == "long_name") vlistDefVarLongname(vlistID2, varID, value_cstr);
              else if (key == "units") cdo_define_var_units(var, vlistID2, varID, value_cstr);
              else if (key == "name") ; // vlistDefVarName(vlistID2, varID, parameter2word(value.c_str()));
              else if (key == "out_name")
                {
                  const char *outname = parameter2word(value.c_str());
                  if (!cstrIsEqual(var.name, outname))
                    {
                      vlistDefVarName(vlistID2, varID, outname);
                      cdiDefAttTxt(vlistID2, varID, "original_name", (int) strlen(var.name), var.name);
                    }
                }
              else if (key == "param") vlistDefVarParam(vlistID2, varID, stringToParam(parameter2word(value.c_str())));
              else if (key == "out_param") vlistDefVarParam(vlistID2, varID, stringToParam(parameter2word(value.c_str())));
              else if (key == "comment") cdiDefAttTxt(vlistID2, varID, key.c_str(), (int) value.size(), value_cstr);
              else if (key == "cell_methods") cdiDefAttTxt(vlistID2, varID, key.c_str(), (int) value.size(), value_cstr);
              else if (key == "cell_measures") cdiDefAttTxt(vlistID2, varID, key.c_str(), (int) value.size(), value_cstr);
              else if (key == "delete") var.remove = parameter2bool(value);
              else if (key == "convert") var.convert = parameter2bool(value);
              else if (key == "factor")
                {
                  var.lfactor = true;
                  var.factor = parameter2double(value);
                  if (Options::cdoVerbose) cdoPrint("%s - scale factor %g", varname, var.factor);
                }
              else if (key == "missval" || key == "missing_value")
                {
                  auto missval = parameter2double(value);
                  auto missval_old = vlistInqVarMissval(vlistID2, varID);
                  if (!DBL_IS_EQUAL(missval, missval_old))
                    {
                      if (Options::cdoVerbose) cdoPrint("%s - change missval from %g to %g", varname, missval_old, missval);
                      var.changemissval = true;
                      var.missval_old = missval_old;
                      vlistDefVarMissval(vlistID2, varID, missval);
                    }
                }
              else if (key == "valid_min")
                {
                  lvalid_min = true;
                  var.valid_min = parameter2double(value);
                }
              else if (key == "valid_max")
                {
                  lvalid_max = true;
                  var.valid_max = parameter2double(value);
                }
              else if (key == "ok_min_mean_abs")
                {
                  var.check_min_mean_abs = true;
                  var.ok_min_mean_abs = parameter2double(value);
                }
              else if (key == "ok_max_mean_abs")
                {
                  var.check_max_mean_abs = true;
                  var.ok_max_mean_abs = parameter2double(value);
                }
              else if (key == "datatype" || key == "type")
                {
                  int datatype = str2datatype(parameter2word(value.c_str()));
                  if (datatype != -1) vlistDefVarDatatype(vlistID2, varID, datatype);
                }
              else
                {
                  if (Options::cdoVerbose) cdoPrint("Attribute %s:%s not supported!", varname, key.c_str());
                }
              // clang-format on
            }

          if (lvalid_min && lvalid_max) var.checkvalid = true;
        }
      else
        {
          cdoPrint("Variable %s not found in CMOR table!", varname);
        }
    }
}

void *
CMOR_lite(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;
  bool delvars = false;

  cdoInitialize(process);

  Options::CMOR_Mode = 1;
  if (Options::CMOR_Mode) cdiDefGlobal("CMOR_MODE", Options::CMOR_Mode);

  cdoOperatorAdd("cmorlite", 0, 0, "parameter table name");

  const auto operatorID = cdoOperatorID();

  operatorInputArg(cdoOperatorEnter(operatorID));

  if (operatorArgc() < 1) cdoAbort("Too few arguments!");

  bool convert_data = false;
  if (operatorArgc() == 2)
    {
      if (cdoOperatorArgv(1).compare("convert") == 0)
        convert_data = true;
      else
        cdoAbort("Unknown parameter: >%s<", cdoOperatorArgv(1).c_str());
    }

  if (operatorArgc() > 2) cdoAbort("Too many arguments!");

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  auto vlistID2 = vlistDuplicate(vlistID1);
  /* vlistPrint(vlistID2);*/

  const auto nvars = vlistNvars(vlistID2);
  std::vector<CmorVar> vars(nvars);
  memset(vars.data(), 0, nvars * sizeof(CmorVar));

  if (convert_data)
    for (varID = 0; varID < nvars; ++varID) vars[varID].convert = true;

  const auto filename = cdoOperatorArgv(0).c_str();
  auto fp = fopen(filename, "r");
  if (fp == nullptr) cdoAbort("Open failed on: %s\n", filename);

  PMList pmlist;
  pmlist.readCmortable(fp, filename);
  fclose(fp);

  applyCmorList(pmlist, nvars, vlistID2, vars);

  for (int varID = 0; varID < nvars; ++varID)
    if (vars[varID].remove)
      {
        delvars = true;
        break;
      }

  if (delvars)
    {
      vlistClearFlag(vlistID1);
      vlistClearFlag(vlistID2);

      for (int varID = 0; varID < nvars; varID++)
        {
          const auto zaxisID = vlistInqVarZaxis(vlistID2, varID);
          const auto nlevs = zaxisInqSize(zaxisID);
          for (int levID = 0; levID < nlevs; levID++)
            {
              vlistDefFlag(vlistID1, varID, levID, true);
              vlistDefFlag(vlistID2, varID, levID, true);
              if (vars[varID].remove)
                {
                  vlistDefFlag(vlistID1, varID, levID, false);
                  vlistDefFlag(vlistID2, varID, levID, false);
                }
            }
        }

      auto vlistIDx = vlistCreate();
      cdoVlistCopyFlag(vlistIDx, vlistID2);

      vlistDestroy(vlistID2);

      vlistID2 = vlistIDx;
      if (vlistNvars(vlistID2) == 0) cdoAbort("No variable selected!");
    }

  for (int varID = 0; varID < nvars; ++varID)
    {
      auto &var = vars[varID];
      if (!var.convert) var.changeunits = false;
      if (var.changeunits)
        cdoConvertUnits(&var.ut_converter, &var.changeunits, (char *) &var.units, (char *) &var.units_old, var.name);
    }

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  /* vlistPrint(vlistID2);*/
  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  auto gridsizemax = vlistGridsizeMax(vlistID1);
  if (vlistNumber(vlistID1) != CDI_REAL) gridsizemax *= 2;
  Varray<double> array(gridsizemax);

  int tsID1 = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID1)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);

      cdoDefTimestep(streamID2, tsID1);

      cmor_check_init(nvars, vars);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          auto &var = vars[varID];
          auto varID2 = varID;
          auto levelID2 = levelID;

          if (delvars)
            {
              if (var.remove) continue;

              if (vlistInqFlag(vlistID1, varID, levelID) == true)
                {
                  varID2 = vlistFindVar(vlistID2, varID);
                  levelID2 = vlistFindLevel(vlistID2, varID, levelID);
                }
            }

          cdoDefRecord(streamID2, varID2, levelID2);

          cdoReadRecord(streamID1, array.data(), &nmiss);

          const auto missval = vlistInqVarMissval(vlistID2, varID2);
          auto gridsize = gridInqSize(vlistInqVarGrid(vlistID2, varID2));
          if (vlistInqVarNumber(vlistID2, varID2) != CDI_REAL) gridsize *= 2;

          if (nmiss && var.changemissval)
            {
              for (size_t i = 0; i < gridsize; ++i)
                {
                  if (DBL_IS_EQUAL(array[i], var.missval_old)) array[i] = missval;
                }
            }

          if (var.lfactor)
            {
              for (size_t i = 0; i < gridsize; ++i)
                {
                  if (!DBL_IS_EQUAL(array[i], missval)) array[i] *= var.factor;
                }
            }

#ifdef HAVE_UDUNITS2
          if (var.changeunits)
            {
              int nerr = 0;
              for (size_t i = 0; i < gridsize; ++i)
                {
                  if (!DBL_IS_EQUAL(array[i], missval))
                    {
                      array[i] = cv_convert_double((const cv_converter *) var.ut_converter, array[i]);
                      if (ut_get_status() != UT_SUCCESS) nerr++;
                    }
                }
              if (nerr)
                {
                  cdoWarning("Udunits: Error converting units from [%s] to [%s], parameter: %s", var.units_old, var.units,
                             var.name);
                  var.changeunits = false;
                }
            }
#endif

          cdoWriteRecord(streamID2, array.data(), nmiss);

          cmor_check_prep(var, gridsize, missval, array.data());
        }

      cmor_check_eval(vlistID2, nvars, vars);

      tsID1++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

#ifdef HAVE_UDUNITS2
  for (int varID = 0; varID < nvars; varID++)
    if (vars[varID].changeunits) cdoConvertFree(vars[varID].ut_converter);

  cdoConvertDestroy();
#endif

  cdoFinish();

  return 0;
}
