"""Helpers for mapping Nova api results to the juju machine abstraction"""

from juju.machine import ProviderMachine


_SERVER_STATE_MAP = {
    None: 'pending',
    'ACTIVE': 'running',
    'BUILD': 'pending',
    'BUILDING': 'pending',
    'REBUILDING': 'pending',
    'DELETED': 'terminated',
    'STOPPED': 'stopped',
    }


class NovaProviderMachine(ProviderMachine):
    """Nova-specific ProviderMachine implementation"""


def get_server_status(server):
    status = server.get('status')
    if status is not None and "(" in status:
        status = status.split("(", 1)[0]
    return _SERVER_STATE_MAP.get(status, 'unknown')


def get_server_addresses(server):
    private_addr = public_addr = None
    addresses = server.get("addresses")
    if addresses is not None:
        # Issue with some setups, have custom network only, use as private
        network = ()
        for name in sorted(addresses):
            if name not in ("private", "public"):
                network = addresses[name]
                if network:
                    break
        network = addresses.get("private", network)
        for address in network:
            if address.get("version", 0) == 4:
                private_addr = address['addr']
                break
        # Issue with HP cloud, public address is second in private network
        network = addresses.get("public", network[1:])
        for address in network:
            if address.get("version", 0) == 4:
                public_addr = address['addr']
    return private_addr, public_addr


def machine_from_instance(server):
    """Create an :class:`NovaProviderMachine` from a server details dict

    :param server: a dictionary of server info as given by the Nova api

    :return: a matching :class:`NovaProviderMachine`
    """
    private_addr, public_addr = get_server_addresses(server)
    # Juju assumes it always needs a public address and loops waiting for one.
    # In fact a private address is generally fine provided it can be sshed to.
    if public_addr is None and private_addr is not None:
        public_addr = private_addr
    return NovaProviderMachine(
        server['id'],
        public_addr,
        private_addr,
        get_server_status(server))
