"""Manage port access to machines using Nova security group extension

The mechanism is based on the existing scheme used by the EC2 provider.

Each machine is launched with two security groups, a juju group that is shared
across all machines and allows access to 22/tcp for ssh, and a machine group
just for that server so ports can be opened and closed on an individual level.

There is some mismatch between the port hole poking and security group models:
* A new security group is created for every machine
* Rules are not shared between service units but set up again each launch
* Support for port ranges is not exposed

The Nova security group module follows the EC2 example quite closely, but as
of Essex it's still under contrib and has a number of quirks:
* To run a server with, or add or remove groups from a server, 'name' is used
* To get details, delete, or add or remove rules from a group, 'id' is needed

The only way of getting 'id' if 'name' is known is by listing all groups then
looking at the details of the one with the matching name.
"""

from twisted.internet.defer import inlineCallbacks, returnValue

from juju import errors

from .client import log


class NovaPortManager(object):
    """Mapping of port-based juju interface to Nova security group actions

    There is the potential to record some state on the instance to reduce api
    round-trips when, for instance, launching multiple machines at once, but
    for now
    """

    def __init__(self, nova, environment_name):
        self.nova = nova
        self.tag = environment_name

    def _juju_group_name(self):
        return "juju-%s" % (self.tag,)

    def _machine_group_name(self, machine_id):
        return "juju-%s-%s" % (self.tag, machine_id)

    @inlineCallbacks
    def _get_machine_group(self, machine, machine_id):
        """Get details of the machine specific security group

        As only the name of the group can be derived, this means listing every
        security group for that server and seeing which has a matching name.
        """
        group_name = self._machine_group_name(machine_id)
        server_id = machine.instance_id
        groups = yield self.nova.get_server_security_groups(server_id)

        found = False
        for group in groups:
            if group['name'] == group_name:
                found = group
                break

        # 2012/12/19: kt diablo/hpcloud compatibility
        if found and not 'rules' in group:
            group = yield self.nova.get_security_group_details(group['id'])
            found = True

        if found:
            returnValue(group)

        raise errors.ProviderInteractionError(
            "Missing security group %r for machine %r" %
            (group_name, server_id))

    @inlineCallbacks
    def open_port(self, machine, machine_id, port, protocol="tcp"):
        """Allow access to a port for the given machine only"""
        group = yield self._get_machine_group(machine, machine_id)
        yield self.nova.add_security_group_rule(
            group['id'],
            ip_protocol=protocol, from_port=port, to_port=port)
        log.debug("Opened %s/%s on machine %r",
                  port, protocol, machine.instance_id)

    @inlineCallbacks
    def close_port(self, machine, machine_id, port, protocol="tcp"):
        """Revoke access to a port for the given machine only"""
        group = yield self._get_machine_group(machine, machine_id)
        for rule in group["rules"]:
            if (port == rule["from_port"] == rule["to_port"] and
                rule["ip_protocol"] == protocol):

                yield self.nova.delete_security_group_rule(rule["id"])
                log.debug("Closed %s/%s on machine %r",
                          port, protocol, machine.instance_id)
                return
        raise errors.ProviderInteractionError(
            "Couldn't close unopened %s/%s on machine %r",
            port, protocol, machine.instance_id)

    @inlineCallbacks
    def get_opened_ports(self, machine, machine_id):
        """Get a set of opened port/protocol pairs for a machine"""
        group = yield self._get_machine_group(machine, machine_id)
        opened_ports = set()
        for rule in group.get("rules", []):
            if not rule.get("group"):
                protocol = rule["ip_protocol"]
                from_port = rule["from_port"]
                to_port = rule["to_port"]
                if from_port == to_port:
                    opened_ports.add((from_port, protocol))
        returnValue(opened_ports)

    @inlineCallbacks
    def ensure_groups(self, machine_id):
        """Get names of the security groups for a machine, creating if needed

        If the juju group already exists, it is assumed to be correctly set up.
        If the machine group already exists, it is deleted then recreated.
        """
        security_groups = yield self.nova.list_security_groups()
        groups_by_name = dict((sg['name'], sg['id']) for sg in security_groups)

        juju_group = self._juju_group_name()
        if not juju_group in groups_by_name:
            log.debug("Creating juju security group %s", juju_group)
            sg = yield self.nova.create_security_group(
                juju_group,
                "juju group for %s" % (self.tag,))
            # Add external ssh access
            yield self.nova.add_security_group_rule(
                sg['id'], ip_protocol="tcp", from_port=22, to_port=22)
            # Add internal group access
            yield self.nova.add_security_group_rule(
                parent_group_id=sg['id'], group_id=sg['id'],
                ip_protocol="tcp", from_port=1, to_port=65535)
            yield self.nova.add_security_group_rule(
                parent_group_id=sg['id'], group_id=sg['id'],
                ip_protocol="udp", from_port=1, to_port=65535)

        machine_group = self._machine_group_name(machine_id)
        if machine_group in groups_by_name:
            yield self.nova.delete_security_group(
                groups_by_name[machine_group])
        log.debug("Creating machine security group %s", machine_group)
        yield self.nova.create_security_group(
            machine_group,
            "juju group for %s machine %s" % (self.tag, machine_id))

        returnValue([juju_group, machine_group])

    @inlineCallbacks
    def get_machine_groups(self, machine, with_juju_group=False):
        try:
            ret = yield self.get_machine_groups_pure(machine, with_juju_group)
        except errors.ProviderInteractionError, e:
            # XXX: Need to wire up treatment of 500s properly in client
            if getattr(e, "kind", None) == "computeError":
                try:
                    yield self.nova.get_server(machine.instance_id)
                except errors.ProviderInteractionError, e:
                    pass  # just rebinding e
            if True or getattr(e, "kind", None) == "itemNotFound":
                returnValue(None)
            raise
        returnValue(ret)

    @inlineCallbacks
    def get_machine_groups_pure(self, machine, with_juju_group=False):
        server_id = machine.instance_id
        groups = yield self.nova.get_server_security_groups(server_id)
        juju_group = self._juju_group_name()
        groups_by_name = dict(
            (g['name'], g['id']) for g in groups
            if g['name'].startswith(juju_group))
        if juju_group not in groups_by_name:
            # Not a juju machine, shouldn't touch
            returnValue(None)
        if not with_juju_group:
            groups_by_name.pop(juju_group)
        # else assumption: only one remaining group, is the machine group
        returnValue(groups_by_name)

    @inlineCallbacks
    def delete_juju_group(self):
        security_groups = yield self.nova.list_security_groups()
        juju_group = self._juju_group_name()
        for group in security_groups:
            if group['name'] == juju_group:
                break
        else:
            log.debug("Can't delete missing juju group")
            return
        yield self.nova.delete_security_group(group['id'])
