"""Tests for OpenStack provider method for listing live juju machines"""

from juju.lib import testing
from juju.providers.openstack.machine import NovaProviderMachine
from juju.providers.openstack.tests import OpenStackTestMixin


class GetMachinesTests(OpenStackTestMixin, testing.TestCase):

    def check_machines(self, machines, expected):
        machine_details = []
        for m in machines:
            self.assertTrue(isinstance(m, NovaProviderMachine))
            machine_details.append((m.instance_id, m.state))
        machine_details.sort()
        expected.sort()
        self.assertEqual(expected, machine_details)

    def test_all_none(self):
        self.expect_nova_get("servers/detail", response={"servers": []})
        self.mocker.replay()
        return self.get_provider().get_machines().addCallback(
            self.check_machines, [])

    def test_all_single(self):
        self.expect_nova_get("servers/detail", response={"servers": [
            self.make_server(1000),
            ]})
        self.mocker.replay()
        return self.get_provider().get_machines().addCallback(
            self.check_machines, [(1000, 'running')])

    def test_all_multiple(self):
        self.expect_nova_get("servers/detail", response={"servers": [
            self.make_server(1001),
            self.make_server(1002, status="BUILDING"),
            ]})
        self.mocker.replay()
        return self.get_provider().get_machines().addCallback(
            self.check_machines, [(1001, 'running'), (1002, 'pending')])

    def test_all_some_dead(self):
        self.expect_nova_get("servers/detail", response={"servers": [
            self.make_server(1001, status="BUILDING"),
            self.make_server(1002, status="DELETED"),
            ]})
        self.mocker.replay()
        return self.get_provider().get_machines().addCallback(
            self.check_machines, [(1001, 'pending')])

    def test_all_some_other(self):
        self.expect_nova_get("servers/detail", response={"servers": [
            self.make_server(1001, name="nova started server"),
            self.make_server(1002),
            ]})
        self.mocker.replay()
        return self.get_provider().get_machines().addCallback(
            self.check_machines, [(1002, 'running')])

    def test_two_none(self):
        self.expect_nova_get("servers/detail", response={"servers": []})
        self.mocker.replay()
        deferred = self.get_provider().get_machines([1001, 1002])
        return self.assert_not_found(deferred, [1001, 1002])

    def test_two_some_dead(self):
        self.expect_nova_get("servers/detail", response={"servers": [
            self.make_server(1001, status="BUILDING"),
            self.make_server(1002, status="DELETED"),
            ]})
        self.mocker.replay()
        deferred = self.get_provider().get_machines([1001, 1002])
        return self.assert_not_found(deferred, [1002])

    def test_two_some_other(self):
        self.expect_nova_get("servers/detail", response={"servers": [
            self.make_server(1001, name="nova started server"),
            self.make_server(1002),
            ]})
        self.mocker.replay()
        deferred = self.get_provider().get_machines([1001, 1002])
        return self.assert_not_found(deferred, [1001])

    def test_one_running(self):
        self.expect_nova_get("servers/1000",
            response={"server": self.make_server(1000)})
        self.mocker.replay()
        deferred = self.get_provider().get_machines([1000])
        return deferred.addCallback(self.check_machines, [(1000, 'running')])

    def test_one_dead(self):
        self.expect_nova_get("servers/1000",
            response={"server": self.make_server(1000, status="DELETED")})
        self.mocker.replay()
        deferred = self.get_provider().get_machines([1000])
        return self.assert_not_found(deferred, [1000])

    def test_one_other(self):
        self.expect_nova_get("servers/1000",
            response={"server": self.make_server(1000, name="testing")})
        self.mocker.replay()
        deferred = self.get_provider().get_machines([1000])
        return self.assert_not_found(deferred, [1000])

    def test_one_missing(self):
        self.expect_nova_get("servers/1000",
            code=404, response={"itemNotFound":
                {"message": "The resource could not be found.", "code": 404}})
        self.mocker.replay()
        deferred = self.get_provider().get_machines([1000])
        return self.assert_not_found(deferred, [1000])
    # XXX: Need to do error handling properly in client still
    test_one_missing.skip = True

    # XXX: EC2 also has some tests for wrapping unexpected errors from backend
