(ns leiningen.pom
  "Write a pom.xml file to disk for Maven interop."
  (:use [leiningen.util.maven :only [make-model]]
        [leiningen.core :only [abort]]
        [clojure.java.io :only [copy file]])
  (:import (java.io StringWriter ByteArrayOutputStream)
           (java.util Properties)
           (org.apache.maven.project MavenProject)))

(def #^{:doc "A notice to place at the bottom of generated files."} disclaimer
     "\n<!-- This file was autogenerated by Leiningen.
  Please do not edit it directly; instead edit project.clj and regenerate it.
  It should not be considered canonical data. For more information see
  https://github.com/technomancy/leiningen -->\n")

(defn snapshot? [project]
  (re-find #"SNAPSHOT" (:version project)))

(def ^{:doc "For backwards-compatibility"} default-repos
  leiningen.core/default-repos)

(defn check-for-snapshot-deps [project]
  (when (and (not (snapshot? project))
             (not (System/getenv "LEIN_SNAPSHOTS_IN_RELEASE"))
             (some #(re-find #"SNAPSHOT" (second %)) (:dependencies project)))
    (abort "Release versions may not depend upon snapshots."
           "\nFreeze snapshots to dated versions or set the"
           "LEIN_SNAPSHOTS_IN_RELEASE environment variable to override this.")))

(defn make-pom
  ([project] (make-pom project false))
  ([project disclaimer?]
     (check-for-snapshot-deps project)
     (with-open [w (StringWriter.)]
       (.writeModel (MavenProject. (make-model project)) w)
       (when disclaimer?
         (.write w disclaimer))
       (.getBytes (str w)))))

(defn make-pom-properties [project]
  (with-open [baos (ByteArrayOutputStream.)]
    (.store (doto (Properties.)
              (.setProperty "version" (:version project))
              (.setProperty "groupId" (:group project))
              (.setProperty "artifactId" (:name project)))
              baos "Leiningen")
    (.getBytes (str baos))))

(defn ^{:help-arglists '([])} pom
  "Write a pom.xml file to disk for Maven interop."
  ([project pom-location silently?]
     (let [pom-file (file (:root project) pom-location)]
       (copy (make-pom project true) pom-file)
       (when-not silently? (println "Wrote" (.getName pom-file)))
       (.getAbsolutePath pom-file)))
  ([project pom-location] (pom project pom-location false))
  ([project] (pom project "pom.xml")))
