// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright The Music Player Daemon Project

#include "Bindings.hxx"
#include "Command.hxx"
#include "KeyName.hxx"
#include "i18n.h"

#include <assert.h>
#include <stdio.h>

const char *
KeyBindings::GetFirstKeyName(Command command) const noexcept
{
	const auto &b = key_bindings[size_t(command)];
	return b.keys.front() != 0
		? GetLocalizedKeyName(b.keys[0])
		: nullptr;
}

std::string
KeyBindings::GetKeyNames(Command command) const noexcept
{
	const auto &b = key_bindings[size_t(command)];

	std::string s = GetLocalizedKeyName(b.keys[0]);

	for (size_t i = 1; i < b.keys.size() && b.keys[i] != 0; ++i) {
		s.push_back(' ');
		s.append(GetLocalizedKeyName(b.keys[i]));
	}
	return s;
}

Command
KeyBindings::FindKey(int key) const noexcept
{
	assert(key != 0);

	for (size_t i = 0; i < size_t(Command::NONE); ++i) {
		if (key_bindings[i].HasKey(key))
			return Command(i);
	}

	return Command::NONE;
}

#ifndef NCMPC_MINI

bool
KeyBindings::Check(char *buf, size_t bufsize) const noexcept
{
	bool success = true;

	for (size_t i = 0; i < size_t(Command::NONE); ++i) {
		for (const auto key : key_bindings[i].keys) {
			if (key == 0)
				break;

			Command cmd;
			if ((cmd = FindKey(key)) != Command(i)) {
				if (buf) {
					snprintf(buf, bufsize,
						 _("Key %s assigned to %s and %s"),
						 GetLocalizedKeyName(key),
						 get_key_command_name(Command(i)),
						 get_key_command_name(cmd));
				} else {
					fprintf(stderr,
						_("Key %s assigned to %s and %s"),
						GetLocalizedKeyName(key),
						get_key_command_name(Command(i)),
						get_key_command_name(cmd));
					fputc('\n', stderr);
				}
				success = false;
			}
		}
	}

	return success;
}

void
KeyBinding::WriteToFile(FILE *f, const command_definition_t &cmd,
			bool comment) const noexcept
{
	fprintf(f, "## %s\n", cmd.description);
	if (comment)
		fprintf(f, "#");
	fprintf(f, "key %s = ", cmd.name);

	if (keys.front() == 0) {
		fputs("0\n\n", f);
		return;
	}

	bool first = true;
	for (const auto key : keys) {
		if (key == 0)
			break;

		if (first)
			first = false;
		else
			fprintf(f, ",  ");

		fputs(GetKeyName(key), f);
	}
	fprintf(f,"\n\n");
}

bool
KeyBindings::WriteToFile(FILE *f, int flags) const noexcept
{
	const auto *cmds = get_command_definitions();

	if (flags & KEYDEF_WRITE_HEADER)
		fprintf(f, "## Key bindings for ncmpc (generated by ncmpc)\n\n");

	for (size_t i = 0; i < size_t(Command::NONE) && !ferror(f); ++i) {
		if (key_bindings[i].modified || flags & KEYDEF_WRITE_ALL) {
			key_bindings[i].WriteToFile(f, cmds[i],
						    flags & KEYDEF_COMMENT_ALL);
		}
	}

	return ferror(f) == 0;
}

#endif /* NCMPC_MINI */
