-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Ada.Characters.Handling;
with Ada.Integer_Text_IO;
with Ada.Text_IO;

with CMD;

package body Utility is

   function Format_Int (Item  : in Integer;
                        Width : in Integer) return String is
      Temp_String : String (1 .. Width);
   begin
      Ada.Integer_Text_IO.Put (To   => Temp_String,
                               Item => Item,
                               Base => 10);
      return Temp_String;
   end Format_Int;

   function Get_File_Size (File : in String) return Ada.Streams.Stream_IO.Count is
      package SIO renames Ada.Streams.Stream_IO;
      S : SIO.Count;
      F : SIO.File_Type;
   begin
      SIO.Open (F, SIO.In_File, File, "");
      S := SIO.Size (F);
      SIO.Close (F);
      return S;
   exception
      when others =>
         if SIO.Is_Open (F) then
            SIO.Close (F);
         end if;
         return 0;
   end Get_File_Size;

   procedure Put_Message_With_Duration (Message : in String;
                                        D       : in Duration) is
      package Duration_IO is new Ada.Text_IO.Fixed_IO (Duration);
      Hours   : Natural;
      Minutes : Natural;
      Seconds : Duration;
      Residue : Long_Float;
      LD      : Long_Float;
   begin
      LD      := Long_Float (D);
      Hours   := Natural (Long_Float'Floor (LD / 3600.0));
      Residue := LD - (Long_Float (Hours) * 3600.0);
      Minutes := Natural (Long_Float'Floor (Residue / 60.0));
      Seconds := Duration (Residue - (Long_Float (Minutes) * 60.0));
      Ada.Text_IO.Put (Message);
      Ada.Integer_Text_IO.Put (Item  => Hours,
                               Width => 4,
                               Base  => 10);
      Ada.Text_IO.Put (':');
      Ada.Integer_Text_IO.Put (Item  => Minutes,
                               Width => 2,
                               Base  => 10);
      Ada.Text_IO.Put (':');
      Duration_IO.Put (Item => Seconds,
                       Fore => 2,
                       Aft  => 2,
                       Exp  => 0);
      Ada.Text_IO.New_Line;
   end Put_Message_With_Duration;

   procedure Debug (Str : in String) is
   begin
      if CMD.Verbose then
         Ada.Text_IO.Put_Line (Str);
      end if;
   end Debug;

   function Is_A_VCG_File (File : in String) return Boolean is
      T : constant String := Ada.Characters.Handling.To_Lower (File);
   begin
      return (T'Length >= 5)
        and then (T (T'Last - 3) = '.' and T (T'Last - 2) = 'v' and T (T'Last - 1) = 'c' and T (T'Last) = 'g');
   end Is_A_VCG_File;

   function Is_A_DPC_File (File : in String) return Boolean is
      T : constant String := Ada.Characters.Handling.To_Lower (File);
   begin
      return (T'Length >= 5)
        and then (T (T'Last - 3) = '.' and T (T'Last - 2) = 'd' and T (T'Last - 1) = 'p' and T (T'Last) = 'c');
   end Is_A_DPC_File;

end Utility;
