/************************************************************************

The TASS 1.7 theory of the Saturnian satellites including HYPERION
by Alain VIENNE and Luc DURIEZ can be found at
ftp://ftp.imcce.fr/pub/ephem/satel/tass17

I (Johannes Gajdosik) have just taken the Fortran code and data
obtained from above and rearranged it into this piece of software.

I can neither allow nor forbid the usage of the TASS 1.7 theory.
The copyright notice below covers not the work of Alain VIENNE and Luc DURIEZ
but just my work, that is the compilation of the TASS 1.7 theory
into the software supplied in this file.


Copyright (c) 2005 Johannes Gajdosik

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

This is an implementation of the TASS 1.7 theory.
My modifications are:
1) do not calculate constant terms at runtime but beforehand
2) unite terms with the same frequencies
3) rearrange the terms so that calculation of the argument becomes easier
4) substitute so that the independent variable becomes T=jd-2444240
   for all satellites (including hyperion)
5) do not calculate a satellites "rmu" at runtime but beforehand
6) use a rotation matrix for Transformation to J2000
   instead of AIA, OMA and inclination of earth axis
7) exchange indices of hyperion and iapetus
8) calculate the orbital elements not for every new jd but rather reuse
   the previousely calculated elements if possible

****************************************************************/

#include "tass17.h"
#include "calc_interpolated_elements.h"
#include "elliptic_to_rectangular.h"

#include <math.h>

struct Tass17Term
{
	double s[3];
};

struct Tass17MultiTerm
{
	signed char i[8];
	int nr_of_terms;
	const struct Tass17Term *terms;
};

struct Tass17Series
{
	int nr_of_multi_terms;
	const struct Tass17MultiTerm *multi_terms;
};

struct Tass17Body
{
	const char *name;
	double mu;
	double aam;
	double s0[6];
	struct Tass17Series series[4];
};

static const struct Tass17Term MIMAS_0_0[2] = {
  {{ 2.760763800054540e-05, 6.863463000141887e-01, 2.437929552050393e-04}},
  {{ 9.347313993693880e-06, 2.209688858348459e+00, 2.791965241219151e-02}},
};

static const struct Tass17Term MIMAS_0_1[5] = {
  {{ 1.247132122206555e-04, 3.384553386830940e+00, 6.649590915041870e+00}},
  {{ 5.773002078925660e-05, 9.293070332553311e-01, 6.649834707997076e+00}},
  {{ 5.767710349652186e-05, 2.698207086816744e+00, 6.649347122086664e+00}},
  {{ 1.211438560239529e-05, 2.011860786802310e+00, 6.649103329131461e+00}},
  {{ 1.201719293977645e-05, 4.757245986859100e+00, 6.650078500952279e+00}},
};

static const struct Tass17MultiTerm MIMAS_0[2] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  2,MIMAS_0_0},
  {{ 2, 0,-2, 0, 0, 0, 0, 0},  5,MIMAS_0_1},
};

static const struct Tass17Term MIMAS_1_0[29] = {
  {{ 9.283150504689616e-06, 1.804019257682701e+00, 1.990850081500073e-05}},
  {{ 9.230818991264319e-05, 2.435908416203692e+00, 8.594659879963006e-05}},
  {{ 7.345436389839680e-05, 4.239927673886392e+00, 1.058550996146308e-04}},
  {{ 5.060827138159436e-04, 4.533623190990083e+00, 1.578463564054092e-04}},
  {{ 2.628008938901560e-04, 5.445714149319756e-02, 1.777548572204099e-04}},
  {{ 1.096551493377600e-05, 2.023919695921281e+00, 2.238844543900385e-04}},
  {{ 7.574073228630605e-01, 6.863463000141887e-01, 2.437929552050393e-04}},
  {{ 1.317556115144799e-05, 2.490365557696890e+00, 2.637014560200400e-04}},
  {{ 1.969761970496516e-04, 4.459828112124973e+00, 3.098310531896687e-04}},
  {{ 2.459408712169538e-04, 3.122254716217880e+00, 3.297395540046694e-04}},
  {{ 2.799611810201398e-05, 5.219969491004272e+00, 4.016393116104485e-04}},
  {{ 7.266042399548720e-04, 1.372692600028377e+00, 4.875859104100786e-04}},
  {{ 9.274943256038984e-06, 6.670083626422765e-01, 5.735325092097087e-04}},
  {{ 1.284412109690001e-05, 2.764723137428667e+00, 6.454322668154877e-04}},
  {{ 1.243299165183161e-02, 2.059038900042566e+00, 7.313788656151179e-04}},
  {{ 8.715614709370008e-06, 4.494947316246257e+00, 8.173254644147480e-04}},
  {{ 3.590283311520520e-04, 3.431731500070944e+00, 1.218964776025196e-03}},
  {{ 3.847028724666396e-06, 6.152515926191327e+00, 1.366310570108340e-02}},
  {{ 1.276023939118606e-04, 5.446831688805226e+00, 1.374905229988303e-02}},
  {{ 9.505947727365781e-06, 3.697269572615722e+00, 1.390689865628844e-02}},
  {{ 5.198697062661200e-06, 4.383615872629911e+00, 1.415069161149348e-02}},
  {{ 1.163953693117700e-04, 3.677931635243811e+00, 1.423663821029311e-02}},
  {{ 4.870003920011340e-05, 3.292242611895686e+00, 2.718827354657639e-02}},
  {{ 2.237171274873740e-04, 8.369962583200818e-01, 2.743206650178144e-02}},
  {{ 1.059900648786554e-03, 4.664935211924063e+00, 2.767585945698648e-02}},
  {{ 2.266397525680950e-03, 2.209688858348459e+00, 2.791965241219151e-02}},
  {{ 1.022783782146454e-03, 2.896035158362648e+00, 2.816344536739655e-02}},
  {{ 2.097481794022580e-04, 3.582381458376836e+00, 2.840723832260159e-02}},
  {{ 4.473029935270700e-05, 4.268727758391025e+00, 2.865103127780663e-02}},
};

static const struct Tass17Term MIMAS_1_1[1] = {
  {{ 8.436544361747722e-06, 5.237122972947232e+00, 4.050159832983069e-01}},
};

static const struct Tass17Term MIMAS_1_2[5] = {
  {{ 1.456357118609923e-04, 2.429607332411430e-01, 6.649590915041870e+00}},
  {{ 6.713447572711532e-05, 4.070899686845125e+00, 6.649834707997076e+00}},
  {{ 6.681066992467156e-05, 5.839799740406537e+00, 6.649347122086664e+00}},
  {{ 1.422191153382851e-05, 5.153453440392031e+00, 6.649103329131461e+00}},
  {{ 1.410806846233703e-05, 1.615653333269246e+00, 6.650078500952279e+00}},
};

static const struct Tass17MultiTerm MIMAS_1[3] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 29,MIMAS_1_0},
  {{-1, 3,-2, 0, 0, 0, 0, 0},  1,MIMAS_1_1},
  {{ 2, 0,-2, 0, 0, 0, 0, 0},  5,MIMAS_1_2},
};

static const struct Tass17Term MIMAS_2_0[31] = {
  {{ 1.607455685492625e-04, 4.012828055947553e+00,-1.044883867247364e-02}},
  {{ 1.578422930662670e-05, 5.932334220063532e+00, 1.625184896369267e-02}},
  {{ 7.291096595471061e-05, 3.477087866487927e+00, 1.649564191889771e-02}},
  {{ 6.526439278535280e-06, 4.869118403888217e+00, 1.665348827530312e-02}},
  {{ 3.336219324045869e-04, 1.021841512912324e+00, 1.673943487410275e-02}},
  {{ 8.915309088647409e-06, 3.161572755262240e-01, 1.682538147290238e-02}},
  {{ 1.733062364950220e-05, 2.413872050312614e+00, 1.689728123050816e-02}},
  {{ 1.511549643267883e-03, 4.849780466516306e+00, 1.698322782930779e-02}},
  {{ 2.716644193630320e-05, 4.144096229130206e+00, 1.706917442810742e-02}},
  {{ 3.214169596116360e-05, 6.241811003916594e+00, 1.714107418571320e-02}},
  {{ 7.314708595483528e-03, 2.394534112940702e+00, 1.722702078451283e-02}},
  {{ 4.732749947219130e-05, 1.688849875554602e+00, 1.731296738331245e-02}},
  {{ 2.051066737888520e-05, 6.449719967511965e-01, 1.738486714091824e-02}},
  {{ 1.598170384010833e-02, 6.222473066544683e+00, 1.747081373971786e-02}},
  {{ 3.701056581056820e-05, 2.375196175568790e+00, 1.755676033851749e-02}},
  {{ 7.741070178691200e-05, 1.331318296765385e+00, 1.762866009612328e-02}},
  {{ 7.111400179101439e-03, 6.256340593792847e-01, 1.771460669492290e-02}},
  {{ 6.027636670891800e-05, 3.061542475582978e+00, 1.780055329372254e-02}},
  {{ 2.628589083368250e-05, 2.017664596779575e+00, 1.787245305132832e-02}},
  {{ 1.462159663847020e-03, 1.311980359393475e+00, 1.795839965012795e-02}},
  {{ 2.018714468805330e-05, 3.747888775597167e+00, 1.804434624892758e-02}},
  {{ 7.427257818452120e-06, 2.704010896793763e+00, 1.811624600653336e-02}},
  {{ 3.306740201503266e-04, 1.998326659407663e+00, 1.820219260533298e-02}},
  {{ 6.793672394573960e-06, 4.434235075611354e+00, 1.828813920413261e-02}},
  {{ 7.462286917233880e-05, 2.684672959421853e+00, 1.844598556053802e-02}},
  {{ 1.636165431356800e-05, 3.371019259436040e+00, 1.868977851574306e-02}},
  {{ 7.716155921462711e-06, 7.762840176851795e-01, 4.490288024149930e-02}},
  {{ 1.650028552141730e-05, 4.604222971289160e+00, 4.514667319670434e-02}},
  {{ 1.155548887648690e-05, 2.148976617713557e+00, 4.539046615190938e-02}},
  {{ 1.620277455112710e-05, 2.835322917727745e+00, 4.563425910711441e-02}},
  {{ 7.443474433517800e-06, 3.521669217741933e+00, 4.587805206231945e-02}},
};

static const struct Tass17Term MIMAS_2_1[1] = {
  {{ 2.602700419616530e-03, 1.822484926062486e-01, 6.667061728781588e+00}},
};

static const struct Tass17Term MIMAS_2_2[1] = {
  {{ 5.322831922763783e-06, 5.176410732312329e+00, 4.224867970380249e-01}},
};

static const struct Tass17Term MIMAS_2_3[5] = {
  {{ 6.248133126576452e-05, 4.252092258474006e-01, 1.331665264382346e+01}},
  {{ 2.892274500639771e-05, 4.253148179451381e+00, 1.331689643677866e+01}},
  {{ 2.889623530439334e-05, 6.022048233012799e+00, 1.331640885086825e+01}},
  {{ 6.069308702586342e-06, 5.335701932998608e+00, 1.331616505791305e+01}},
  {{ 6.020614391923135e-06, 1.797901825875778e+00, 1.331714022973387e+01}},
};

static const struct Tass17MultiTerm MIMAS_2[4] = {
  {{-1, 0, 2, 0, 0, 0, 0, 0}, 31,MIMAS_2_0},
  {{ 1, 0, 0, 0, 0, 0, 0, 0},  1,MIMAS_2_1},
  {{-2, 3, 0, 0, 0, 0, 0, 0},  1,MIMAS_2_2},
  {{ 3, 0,-2, 0, 0, 0, 0, 0},  5,MIMAS_2_3},
};

static const struct Tass17Term MIMAS_3_0[29] = {
  {{ 5.339008498818640e-06, 4.924784125696391e+00, 9.986805201763754e-03}},
  {{ 1.140613650769890e-05, 2.469537772120787e+00, 1.023059815696879e-02}},
  {{ 8.188197469152401e-06, 1.429141854518208e-02, 1.047439111217383e-02}},
  {{ 1.141748624638240e-05, 7.006377185593701e-01, 1.071818406737887e-02}},
  {{ 5.211706783516070e-06, 1.386984018573560e+00, 1.096197702258391e-02}},
  {{ 1.139717898986310e-05, 1.236334060267667e+00,-1.622629652399248e-02}},
  {{ 5.206791451508880e-05, 5.499877602534791e-01,-1.647008947919752e-02}},
  {{ 2.328430197381575e-04, 6.146826767418875e+00,-1.671388243440256e-02}},
  {{ 5.186530053942100e-06, 5.693256976253909e-01,-1.679982903320219e-02}},
  {{ 1.412387702084000e-05, 1.613203576428794e+00,-1.687172879080797e-02}},
  {{ 1.074089419341231e-03, 5.460480467404687e+00,-1.695767538960760e-02}},
  {{ 1.838941226078010e-05, 6.166164704790789e+00,-1.704362198840723e-02}},
  {{ 4.248311992240600e-05, 9.268572764146046e-01,-1.711552174601300e-02}},
  {{ 5.301694964646620e-03, 4.774134167390498e+00,-1.720146834481264e-02}},
  {{ 5.463249591300440e-05, 5.479818404776600e+00,-1.728741494361227e-02}},
  {{ 2.612362501347000e-05, 2.405109764004165e-01,-1.735931470121805e-02}},
  {{ 1.188963618162444e-02, 4.087787867376310e+00,-1.744526130001768e-02}},
  {{ 1.417196313981720e-05, 4.793472104762412e+00,-1.753120789881731e-02}},
  {{ 3.275262890701040e-05, 5.837349983565814e+00,-1.760310765642309e-02}},
  {{ 5.317666807877856e-03, 2.598489137723283e-01,-1.768905425522272e-02}},
  {{ 2.254194030722800e-05, 4.107125804748224e+00,-1.777500085402235e-02}},
  {{ 1.876210282323130e-05, 2.009411029961832e+00,-1.784690061162813e-02}},
  {{ 1.092196251660480e-03, 2.715095267347932e+00,-1.793284721042776e-02}},
  {{ 1.200955094991380e-05, 2.791868511442401e-01,-1.801879380922739e-02}},
  {{ 6.138904676788960e-06, 4.464657383537437e+00,-1.809069356683316e-02}},
  {{ 2.224213393203463e-04, 5.170341620923537e+00,-1.817664016563279e-02}},
  {{ 5.011145194424460e-05, 1.342402667319555e+00,-1.842043312083783e-02}},
  {{ 1.083732897435760e-05, 3.797649020895159e+00,-1.866422607604287e-02}},
  {{ 1.978445332406060e-05, 5.019691662617644e+00,-4.536491371220917e-02}},
};

static const struct Tass17Term MIMAS_3_1[3] = {
  {{ 1.477125534949241e-05, 2.559894425015774e+00, 1.335156871886320e+01}},
  {{ 6.813829579894155e-06, 1.873548125001586e+00, 1.335132492590799e+01}},
  {{ 6.707536779238787e-06, 1.046480714401697e-01, 1.335181251181840e+01}},
};

static const struct Tass17MultiTerm MIMAS_3[2] = {
  {{-1, 0, 2, 0, 0, 0, 0, 0}, 29,MIMAS_3_0},
  {{ 3, 0,-2, 0, 0, 0, 0, 0},  3,MIMAS_3_1},
};

static const struct Tass17Term ENCELADUS_0_0[1] = {
  {{ 2.265054397578757e-05, 7.519405621081341e-01, 4.579638210202045e+00}},
};

static const struct Tass17Term ENCELADUS_0_1[1] = {
  {{ 7.116593169805980e-06, 5.461033287440088e-01, 2.514460612958353e+00}},
};

static const struct Tass17MultiTerm ENCELADUS_0[2] = {
  {{ 0, 2, 0,-2, 0, 0, 0, 0},  1,ENCELADUS_0_0},
  {{ 0, 2,-2, 0, 0, 0, 0, 0},  1,ENCELADUS_0_1},
};

static const struct Tass17Term ENCELADUS_1_0[3] = {
  {{ 4.496393702552367e-03, 2.342959364982154e+00, 1.549375812779411e-03}},
  {{ 3.354575501528797e-03, 4.597833882505114e+00, 4.427150026633873e-03}},
  {{ 3.106536996299520e-05, 2.912482457830643e+00, 8.854300053267745e-03}},
};

static const struct Tass17Term ENCELADUS_1_1[1] = {
  {{ 4.253124471669380e-06, 1.946766624913765e+00, 2.289819105101023e+00}},
};

static const struct Tass17Term ENCELADUS_1_2[1] = {
  {{ 2.407327778886120e-05, 4.985440644756694e+00, 1.257230306479177e+00}},
};

static const struct Tass17Term ENCELADUS_1_3[1] = {
  {{ 5.359800347726300e-06, 1.150736720732397e+00, 8.382988370771400e+00}},
};

static const struct Tass17Term ENCELADUS_1_4[1] = {
  {{ 2.157848301674358e-05, 3.893533207807963e+00, 4.579638210202045e+00}},
};

static const struct Tass17Term ENCELADUS_1_5[1] = {
  {{ 2.107499273982886e-05, 5.461033287440088e-01, 2.514460612958353e+00}},
};

static const struct Tass17Term ENCELADUS_1_6[1] = {
  {{ 1.204571746494518e-05, 3.481858748969675e+00, 4.492830157146606e-01}},
};

static const struct Tass17Term ENCELADUS_1_7[1] = {
  {{ 1.082902927586888e-05, 2.389951319910909e+00, 3.771690919437530e+00}},
};

static const struct Tass17Term ENCELADUS_1_8[1] = {
  {{ 6.457229782189520e-06, 4.233799311077810e+00, 5.028921225916706e+00}},
};

static const struct Tass17Term ENCELADUS_1_9[1] = {
  {{ 4.159628279141040e-06, 6.077647302244714e+00, 6.286151532395882e+00}},
};

static const struct Tass17MultiTerm ENCELADUS_1[10] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  3,ENCELADUS_1_0},
  {{ 0, 1, 0,-1, 0, 0, 0, 0},  1,ENCELADUS_1_1},
  {{ 0, 1,-1, 0, 0, 0, 0, 0},  1,ENCELADUS_1_2},
  {{ 0, 2, 0, 0, 0,-2, 0, 0},  1,ENCELADUS_1_3},
  {{ 0, 2, 0,-2, 0, 0, 0, 0},  1,ENCELADUS_1_4},
  {{ 0, 2,-2, 0, 0, 0, 0, 0},  1,ENCELADUS_1_5},
  {{ 0, 2,-4, 2, 0, 0, 0, 0},  1,ENCELADUS_1_6},
  {{ 0, 3,-3, 0, 0, 0, 0, 0},  1,ENCELADUS_1_7},
  {{ 0, 4,-4, 0, 0, 0, 0, 0},  1,ENCELADUS_1_8},
  {{ 0, 5,-5, 0, 0, 0, 0, 0},  1,ENCELADUS_1_9},
};

static const struct Tass17Term ENCELADUS_2_0[1] = {
  {{ 6.526400221336371e-06, 2.681979744993624e+00, 2.445958739557666e-05}},
};

static const struct Tass17Term ENCELADUS_2_1[1] = {
  {{ 4.864126392950970e-06, 2.790627620835740e+00,-3.797451619237667e+00}},
};

static const struct Tass17Term ENCELADUS_2_2[5] = {
  {{ 5.778112672027477e-06, 1.733182516415079e+00, 1.471391305053998e-03}},
  {{ 7.443719437241270e-06, 3.988057033938039e+00, 4.349165518908460e-03}},
  {{ 4.803805197845248e-03, 3.189423405738944e+00, 5.898541331687871e-03}},
  {{ 1.097719996101334e-04, 5.532383110312554e+00, 7.447917144467280e-03}},
  {{ 6.715628349206410e-06, 4.645664974245721e+00, 1.032569135832174e-02}},
};

static const struct Tass17Term ENCELADUS_2_3[1] = {
  {{ 5.311492417496450e-06, 2.536683592343353e-01, 2.071076138575381e+00}},
};

static const struct Tass17Term ENCELADUS_2_4[1] = {
  {{ 1.576763094428542e-03, 7.997716879783442e-01, 4.585536751533733e+00}},
};

static const struct Tass17Term ENCELADUS_2_5[1] = {
  {{ 1.305558563149728e-05, 4.693005675247022e+00, 8.138458320962040e-01}},
};

static const struct Tass17Term ENCELADUS_2_6[1] = {
  {{ 2.328380239517312e-05, 5.990750337669914e+00,-4.433844743829719e-01}},
};

static const struct Tass17Term ENCELADUS_2_7[1] = {
  {{ 1.134789737634680e-05, 4.693304903676271e+00, 9.165174961735778e+00}},
};

static const struct Tass17Term ENCELADUS_2_8[1] = {
  {{ 5.297748739729408e-06, 4.146902346503011e+00,-1.700614780862149e+00}},
};

static const struct Tass17MultiTerm ENCELADUS_2[9] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  1,ENCELADUS_2_0},
  {{ 0,-1, 0, 0, 0, 2, 0, 0},  1,ENCELADUS_2_1},
  {{ 0,-1, 0, 2, 0, 0, 0, 0},  5,ENCELADUS_2_2},
  {{ 0,-1, 2, 0, 0, 0, 0, 0},  1,ENCELADUS_2_3},
  {{ 0, 1, 0, 0, 0, 0, 0, 0},  1,ENCELADUS_2_4},
  {{ 0,-2, 3, 0, 0, 0, 0, 0},  1,ENCELADUS_2_5},
  {{ 0,-3, 4, 0, 0, 0, 0, 0},  1,ENCELADUS_2_6},
  {{ 0, 3, 0,-2, 0, 0, 0, 0},  1,ENCELADUS_2_7},
  {{ 0,-4, 5, 0, 0, 0, 0, 0},  1,ENCELADUS_2_8},
};

static const struct Tass17Term ENCELADUS_3_0[1] = {
  {{ 1.280938903343441e-04, 1.983149608096680e+00,-7.280483472742215e-03}},
};

static const struct Tass17Term ENCELADUS_3_1[3] = {
  {{ 3.085667117081219e-05, 3.937780549016139e+00,-3.452416044929610e-03}},
  {{ 1.421727517964700e-05, 4.624126849030327e+00,-3.208623089724569e-03}},
  {{ 1.420735399816070e-05, 1.098415954121576e-01,-3.696209000134648e-03}},
};

static const struct Tass17MultiTerm ENCELADUS_3[2] = {
  {{ 0,-1, 0, 2, 0, 0, 0, 0},  1,ENCELADUS_3_0},
  {{-1, 0, 2, 0, 0, 0, 0, 0},  3,ENCELADUS_3_1},
};

static const struct Tass17Term TETHYS_0_0[1] = {
  {{ 8.692361446792260e-06, 3.746226045578181e+00, 5.868527757813049e+00}},
};

static const struct Tass17Term TETHYS_0_1[1] = {
  {{ 9.773647178903700e-06, 3.347429921083522e+00, 2.065177597243693e+00}},
};

static const struct Tass17Term TETHYS_0_2[1] = {
  {{ 6.341490354446320e-06, 3.087559012405942e-01, 3.097766395865539e+00}},
};

static const struct Tass17MultiTerm TETHYS_0[3] = {
  {{ 0, 0, 2, 0, 0,-2, 0, 0},  1,TETHYS_0_0},
  {{ 0, 0, 2,-2, 0, 0, 0, 0},  1,TETHYS_0_1},
  {{ 0, 0, 3,-3, 0, 0, 0, 0},  1,TETHYS_0_2},
};

static const struct Tass17Term TETHYS_1_0[28] = {
  {{ 3.094782117290198e-06, 1.804019257682701e+00, 1.990850081500073e-05}},
  {{ 1.001557835312868e-05, 6.318891585209909e-01, 6.603809798462935e-05}},
  {{ 1.080021159829085e-05, 5.577501069793485e+00, 8.594659879963006e-05}},
  {{ 4.021385382444620e-06, 4.239927673886392e+00, 1.058550996146308e-04}},
  {{ 2.555821179926381e-05, 1.392030537400290e+00, 1.578463564054092e-04}},
  {{ 1.267528991884128e-05, 3.196049795082991e+00, 1.777548572204099e-04}},
  {{ 1.157195347749026e-05, 5.165512349511074e+00, 2.238844543900385e-04}},
  {{ 3.597193003482037e-02, 3.827938953603982e+00, 2.437929552050393e-04}},
  {{ 9.845954273310880e-06, 5.631958211286683e+00, 2.637014560200400e-04}},
  {{ 9.194949884833879e-06, 1.318235458535180e+00, 3.098310531896687e-04}},
  {{ 9.815613284418477e-06, 6.263847369807674e+00, 3.297395540046694e-04}},
  {{ 1.717434823483564e-05, 2.078376837414479e+00, 4.016393116104485e-04}},
  {{ 3.432162123697680e-05, 4.514285253618171e+00, 4.875859104100786e-04}},
  {{ 7.218512993756520e-06, 3.808601016232069e+00, 5.735325092097087e-04}},
  {{ 1.266151790256666e-05, 5.906315791018461e+00, 6.454322668154877e-04}},
  {{ 5.891806577851199e-04, 5.200631553632359e+00, 7.313788656151179e-04}},
  {{ 1.323271931613666e-05, 1.353354662656465e+00, 8.173254644147480e-04}},
  {{ 1.697093421514454e-05, 2.901388464811505e-01, 1.218964776025196e-03}},
  {{ 3.767845663771360e-06, 3.010923272601534e+00, 1.366310570108340e-02}},
  {{ 6.001489632871980e-06, 2.305239035215433e+00, 1.374905229988303e-02}},
  {{ 8.037140287447099e-06, 5.556769190259295e-01, 1.390689865628844e-02}},
  {{ 3.600130988680400e-06, 1.242023219040118e+00, 1.415069161149348e-02}},
  {{ 5.489567383712060e-06, 5.363389816540173e-01, 1.423663821029311e-02}},
  {{ 1.052724821924284e-05, 3.978588911909874e+00, 2.743206650178144e-02}},
  {{ 4.944669618681220e-05, 1.523342558334271e+00, 2.767585945698648e-02}},
  {{ 1.049859753280688e-04, 5.351281511938252e+00, 2.791965241219151e-02}},
  {{ 4.772345678736420e-05, 6.037627811952441e+00, 2.816344536739655e-02}},
  {{ 9.867772453097199e-06, 4.407888047870434e-01, 2.840723832260159e-02}},
};

static const struct Tass17Term TETHYS_1_1[1] = {
  {{ 3.381153874922860e-06, 3.443909349583986e+00, 2.934263878906524e+00}},
};

static const struct Tass17Term TETHYS_1_2[1] = {
  {{ 4.252829449661740e-06, 2.159362918305770e+00, 1.937452727948935e+00}},
};

static const struct Tass17Term TETHYS_1_3[1] = {
  {{ 2.857487995253398e-05, 1.029186337468635e-01, 1.032588798621846e+00}},
};

static const struct Tass17Term TETHYS_1_4[1] = {
  {{ 1.086312362376316e-05, 3.746226045578181e+00, 5.868527757813049e+00}},
};

static const struct Tass17Term TETHYS_1_5[1] = {
  {{ 5.706936513245540e-06, 1.177133183021748e+00, 3.874905455897870e+00}},
};

static const struct Tass17Term TETHYS_1_6[1] = {
  {{ 2.612226809076254e-05, 3.347429921083522e+00, 2.065177597243693e+00}},
};

static const struct Tass17Term TETHYS_1_7[1] = {
  {{ 1.309044788609482e-05, 3.087559012405942e-01, 3.097766395865539e+00}},
};

static const struct Tass17Term TETHYS_1_8[1] = {
  {{ 7.589604761197260e-06, 3.553267188577251e+00, 4.130355194487387e+00}},
};

static const struct Tass17Term TETHYS_1_9[1] = {
  {{ 4.749981493034660e-06, 5.145931687343214e-01, 5.162943993109232e+00}},
};

static const struct Tass17Term TETHYS_1_10[1] = {
  {{ 4.209916715651780e-06, 1.843847991166901e+00, 1.257230306479177e+00}},
};

static const struct Tass17Term TETHYS_1_11[1] = {
  {{ 9.634177151388563e-06, 3.197170586335357e+00, 2.290686578839635e-01}},
};

static const struct Tass17Term TETHYS_1_12[1] = {
  {{ 3.367826063379420e-06, 3.687695982333802e+00, 2.514460612958353e+00}},
};

static const struct Tass17MultiTerm TETHYS_1[13] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 28,TETHYS_1_0},
  {{ 0, 0, 1, 0, 0,-1, 0, 0},  1,TETHYS_1_1},
  {{ 0, 0, 1, 0,-1, 0, 0, 0},  1,TETHYS_1_2},
  {{ 0, 0, 1,-1, 0, 0, 0, 0},  1,TETHYS_1_3},
  {{ 0, 0, 2, 0, 0,-2, 0, 0},  1,TETHYS_1_4},
  {{ 0, 0, 2, 0,-2, 0, 0, 0},  1,TETHYS_1_5},
  {{ 0, 0, 2,-2, 0, 0, 0, 0},  1,TETHYS_1_6},
  {{ 0, 0, 3,-3, 0, 0, 0, 0},  1,TETHYS_1_7},
  {{ 0, 0, 4,-4, 0, 0, 0, 0},  1,TETHYS_1_8},
  {{ 0, 0, 5,-5, 0, 0, 0, 0},  1,TETHYS_1_9},
  {{ 0, 1,-1, 0, 0, 0, 0, 0},  1,TETHYS_1_10},
  {{ 0, 1,-2, 1, 0, 0, 0, 0},  1,TETHYS_1_11},
  {{ 0, 2,-2, 0, 0, 0, 0, 0},  1,TETHYS_1_12},
};

static const struct Tass17Term TETHYS_2_0[1] = {
  {{ 2.463670320328344e-05, 2.681979744993623e+00, 2.445958739557666e-05}},
};

static const struct Tass17Term TETHYS_2_1[1] = {
  {{ 6.736048525042149e-06, 1.795199654407042e+00, 3.940425661480334e-01}},
};

static const struct Tass17Term TETHYS_2_2[1] = {
  {{ 1.012648120942390e-05, 4.634475612002642e+00,-2.540221312758490e+00}},
};

static const struct Tass17Term TETHYS_2_3[1] = {
  {{ 1.316230344933185e-05, 9.203831673794880e-01,-5.465990108433128e-01}},
};

static const struct Tass17Term TETHYS_2_4[1] = {
  {{ 9.537957898915770e-06, 1.891679082907508e+00, 1.263128847810865e+00}},
};

static const struct Tass17Term TETHYS_2_5[1] = {
  {{ 1.026432488829525e-03, 5.239109003991030e+00, 3.328306445054557e+00}},
};

static const struct Tass17Term TETHYS_2_6[1] = {
  {{ 4.858676635900159e-05, 4.930353102750437e+00, 2.305400491890175e-01}},
};

static const struct Tass17Term TETHYS_2_7[1] = {
  {{ 1.274174154891934e-05, 4.827434469003571e+00,-8.020487494328282e-01}},
};

static const struct Tass17Term TETHYS_2_8[1] = {
  {{ 4.893120486700722e-06, 1.582923181666916e+00,-1.834637548054674e+00}},
};

static const struct Tass17Term TETHYS_2_9[1] = {
  {{ 1.379536465955919e-05, 4.874775170004872e+00, 1.471391305053998e-03}},
};

static const struct Tass17Term TETHYS_2_10[1] = {
  {{ 4.496122761411780e-06, 2.849157684080121e+00,-4.433844743829719e-01}},
};

static const struct Tass17Term TETHYS_2_11[12] = {
  {{ 1.564767415994558e-04, 4.568461127222154e+00, 3.458059983814795e-03}},
  {{ 8.681007704489939e-05, 7.405221736181721e-01, 3.214267028609755e-03}},
  {{ 8.168879255357250e-05, 5.941153727250531e+00, 3.945645894224873e-03}},
  {{ 8.101165266469901e-05, 3.195768527193775e+00, 2.970474073404717e-03}},
  {{ 7.081435508752854e-05, 5.254807427236342e+00, 3.701852939019836e-03}},
  {{ 1.795872905935810e-05, 3.443147200851330e-01, 4.189438849429914e-03}},
  {{ 1.765915793094640e-05, 5.651014880769380e+00, 2.726681118199677e-03}},
  {{ 6.448693043220393e-06, 8.713108941844011e-01,-1.044883867247364e-02}},
  {{ 4.134187193316490e-06, 3.483793623207228e-02, 3.300213627409385e-03}},
  {{ 4.064250964938940e-06, 5.960491664622441e+00, 3.615906340220205e-03}},
  {{ 3.921367852619880e-06, 1.030661020099323e+00, 4.433231804634951e-03}},
  {{ 3.847793440490170e-06, 1.823075927165398e+00, 2.482888162994638e-03}},
};

static const struct Tass17MultiTerm TETHYS_2[12] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  1,TETHYS_2_0},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  1,TETHYS_2_1},
  {{ 0, 0,-1, 0, 0, 2, 0, 0},  1,TETHYS_2_2},
  {{ 0, 0,-1, 0, 2, 0, 0, 0},  1,TETHYS_2_3},
  {{ 0, 0,-1, 2, 0, 0, 0, 0},  1,TETHYS_2_4},
  {{ 0, 0, 1, 0, 0, 0, 0, 0},  1,TETHYS_2_5},
  {{ 0, 0,-2, 3, 0, 0, 0, 0},  1,TETHYS_2_6},
  {{ 0, 0,-3, 4, 0, 0, 0, 0},  1,TETHYS_2_7},
  {{ 0, 0,-4, 5, 0, 0, 0, 0},  1,TETHYS_2_8},
  {{ 0,-1, 0, 2, 0, 0, 0, 0},  1,TETHYS_2_9},
  {{ 0,-3, 4, 0, 0, 0, 0, 0},  1,TETHYS_2_10},
  {{-1, 0, 2, 0, 0, 0, 0, 0}, 12,TETHYS_2_11},
};

static const struct Tass17Term TETHYS_3_0[2] = {
  {{ 6.111685892210941e-06, 6.135349392018898e+00,-2.445240135601595e-05}},
  {{ 4.341737854415172e-06, 2.696600432337302e+00,-4.804041710792879e-04}},
};

static const struct Tass17Term TETHYS_3_1[30] = {
  {{ 8.097194747293030e-06, 1.086326741907496e+00,-2.233451268904413e-03}},
  {{ 3.696660017446820e-05, 3.999804418933084e-01,-2.477244224109452e-03}},
  {{ 3.369179308695820e-06, 2.149542558082810e+00,-2.635090580514859e-03}},
  {{ 1.634061837956512e-04, 5.996819449058704e+00,-2.721037179314491e-03}},
  {{ 3.688282109438380e-06, 4.193183792652183e-01,-2.806983778114122e-03}},
  {{ 1.001531421510340e-05, 1.463196258068622e+00,-2.878883535719899e-03}},
  {{ 7.269088792971872e-04, 5.310473149044516e+00,-2.964830134519532e-03}},
  {{ 1.307143245256740e-05, 6.016157386430616e+00,-3.050776733319159e-03}},
  {{ 3.003084312634590e-05, 7.768499580544321e-01,-3.122676490924936e-03}},
  {{ 3.578626570900766e-03, 4.624126849030327e+00,-3.208623089724569e-03}},
  {{ 3.863423686243950e-05, 5.329811086416426e+00,-3.294569688524200e-03}},
  {{ 1.853853482238710e-05, 9.050365804024403e-02,-3.366469446129977e-03}},
  {{ 7.978986672131195e-03, 3.937780549016139e+00,-3.452416044929610e-03}},
  {{ 1.026902134395760e-05, 4.643464786402239e+00,-3.538362643729240e-03}},
  {{ 2.358139037174070e-05, 5.687342665205642e+00,-3.610262401335017e-03}},
  {{ 3.586811056529552e-03, 1.098415954121576e-01,-3.696209000134648e-03}},
  {{ 1.598764340805630e-05, 3.957118486388051e+00,-3.782155598934278e-03}},
  {{ 1.348141163501630e-05, 1.859403711601660e+00,-3.854055356540055e-03}},
  {{ 7.456079964789274e-04, 2.565087948987761e+00,-3.940001955339688e-03}},
  {{ 8.589176280796499e-06, 1.291795327840677e-01,-4.025948554139318e-03}},
  {{ 4.417994658707790e-06, 4.314650065177265e+00,-4.097848311745095e-03}},
  {{ 1.629278673196500e-04, 5.020334302563366e+00,-4.183794910544728e-03}},
  {{ 3.605229442958720e-05, 1.192395348959384e+00,-4.427587865749766e-03}},
  {{ 7.804425391545350e-06, 3.647641702534988e+00,-4.671380820954806e-03}},
  {{ 3.824550961047250e-06, 4.774776807336220e+00, 2.397965045685182e-02}},
  {{ 8.194661069081041e-06, 2.319530453760617e+00, 2.422344341205686e-02}},
  {{ 5.770004540143910e-06, 6.147469407364597e+00, 2.446723636726190e-02}},
  {{ 8.055438816212310e-06, 5.506304001991995e-01, 2.471102932246693e-02}},
  {{ 3.690275690503300e-06, 1.236976700213389e+00, 2.495482227767198e-02}},
  {{ 1.403252488381090e-05, 4.869684344257474e+00,-3.137206845712112e-02}},
};

static const struct Tass17Term TETHYS_3_2[1] = {
  {{ 4.066369173620947e-06, 2.572521517863344e-01, 6.660065306154045e+00}},
};

static const struct Tass17MultiTerm TETHYS_3[3] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  2,TETHYS_3_0},
  {{-1, 0, 2, 0, 0, 0, 0, 0}, 30,TETHYS_3_1},
  {{ 1, 0, 0, 0, 0, 0, 0, 0},  1,TETHYS_3_2},
};

static const struct Tass17Term DIONE_0_0[1] = {
  {{ 5.044474622853420e-06, 5.198036938148699e+00, 9.048639293270888e-01}},
};

static const struct Tass17Term DIONE_0_1[1] = {
  {{ 1.975589414013096e-05, 3.540388778084452e+00, 3.803350160569355e+00}},
};

static const struct Tass17Term DIONE_0_2[1] = {
  {{ 1.231028575289072e-05, 9.712959155280194e-01, 1.809727858654177e+00}},
};

static const struct Tass17Term DIONE_0_3[1] = {
  {{ 5.098846661514180e-06, 5.981941867419887e-01, 5.705025240854031e+00}},
};

static const struct Tass17Term DIONE_0_4[1] = {
  {{ 7.467512730405980e-06, 3.027740200086925e+00, 2.714591787981265e+00}},
};

static const struct Tass17Term DIONE_0_5[1] = {
  {{ 4.727673963003260e-06, 5.084184484645832e+00, 3.619455717308356e+00}},
};

static const struct Tass17Term DIONE_0_6[1] = {
  {{ 3.991422799327240e-06, 1.029186337468635e-01, 1.032588798621846e+00}},
};

static const struct Tass17Term DIONE_0_7[1] = {
  {{ 4.809588755287700e-06, 2.058372674937290e-01, 2.065177597243693e+00}},
};

static const struct Tass17Term DIONE_0_8[1] = {
  {{ 4.044589508141894e-06, 2.614152002392929e-01, 2.294246255127657e+00}},
};

static const struct Tass17MultiTerm DIONE_0[9] = {
  {{ 0, 0, 0, 1,-1, 0, 0, 0},  1,DIONE_0_0},
  {{ 0, 0, 0, 2, 0,-2, 0, 0},  1,DIONE_0_1},
  {{ 0, 0, 0, 2,-2, 0, 0, 0},  1,DIONE_0_2},
  {{ 0, 0, 0, 3, 0,-3, 0, 0},  1,DIONE_0_3},
  {{ 0, 0, 0, 3,-3, 0, 0, 0},  1,DIONE_0_4},
  {{ 0, 0, 0, 4,-4, 0, 0, 0},  1,DIONE_0_5},
  {{ 0, 0, 1,-1, 0, 0, 0, 0},  1,DIONE_0_6},
  {{ 0, 0, 2,-2, 0, 0, 0, 0},  1,DIONE_0_7},
  {{ 0, 1, 0,-1, 0, 0, 0, 0},  1,DIONE_0_8},
};

static const struct Tass17Term DIONE_1_0[3] = {
  {{ 1.253214092917414e-04, 5.484552018571947e+00, 1.549375812779411e-03}},
  {{ 9.470863297623297e-05, 1.456241228915321e+00, 4.427150026633873e-03}},
  {{ 2.711677399803780e-06, 6.054075111420435e+00, 8.854300053267745e-03}},
};

static const struct Tass17Term DIONE_1_1[1] = {
  {{ 4.269998249018742e-06, 5.396405225690409e+00, 3.940181065606379e-01}},
};

static const struct Tass17Term DIONE_1_2[1] = {
  {{ 1.025994897473446e-05, 1.993980622473295e-01, 1.901675080284677e+00}},
};

static const struct Tass17Term DIONE_1_3[1] = {
  {{ 2.682461086510494e-05, 5.198036938148699e+00, 9.048639293270888e-01}},
};

static const struct Tass17Term DIONE_1_4[1] = {
  {{ 2.565388997294820e-05, 3.540388778084452e+00, 3.803350160569355e+00}},
};

static const struct Tass17Term DIONE_1_5[1] = {
  {{ 3.125192040346257e-06, 1.285576205981393e+00, 3.409332054008717e+00}},
};

static const struct Tass17Term DIONE_1_6[1] = {
  {{ 2.717321774931640e-05, 9.712959155280194e-01, 1.809727858654177e+00}},
};

static const struct Tass17Term DIONE_1_7[1] = {
  {{ 5.552232090659460e-06, 5.981941867419887e-01, 5.705025240854031e+00}},
};

static const struct Tass17Term DIONE_1_8[1] = {
  {{ 1.270721017173096e-05, 3.027740200086925e+00, 2.714591787981265e+00}},
};

static const struct Tass17Term DIONE_1_9[1] = {
  {{ 6.847120194215620e-06, 5.084184484645832e+00, 3.619455717308356e+00}},
};

static const struct Tass17Term DIONE_1_10[1] = {
  {{ 3.974354213288560e-06, 8.574434620251531e-01, 4.524319646635444e+00}},
};

static const struct Tass17Term DIONE_1_11[1] = {
  {{ 1.499025754719972e-05, 3.244511287336658e+00, 1.032588798621846e+00}},
};

static const struct Tass17Term DIONE_1_12[1] = {
  {{ 1.201329949530382e-05, 2.058372674937290e-01, 2.065177597243693e+00}},
};

static const struct Tass17Term DIONE_1_13[1] = {
  {{ 6.090479694942740e-06, 3.450348554830387e+00, 3.097766395865539e+00}},
};

static const struct Tass17Term DIONE_1_14[1] = {
  {{ 3.552297626424840e-06, 4.116745349874580e-01, 4.130355194487387e+00}},
};

static const struct Tass17Term DIONE_1_15[1] = {
  {{ 4.723715451200404e-06, 3.403007853829086e+00, 2.294246255127657e+00}},
};

static const struct Tass17Term DIONE_1_16[1] = {
  {{ 4.770404124767720e-06, 5.557793274556413e-02, 2.290686578839635e-01}},
};

static const struct Tass17MultiTerm DIONE_1[17] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  3,DIONE_1_0},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  1,DIONE_1_1},
  {{ 0, 0, 0, 1, 0,-1, 0, 0},  1,DIONE_1_2},
  {{ 0, 0, 0, 1,-1, 0, 0, 0},  1,DIONE_1_3},
  {{ 0, 0, 0, 2, 0,-2, 0, 0},  1,DIONE_1_4},
  {{ 0, 0, 0, 2, 0,-3, 0, 0},  1,DIONE_1_5},
  {{ 0, 0, 0, 2,-2, 0, 0, 0},  1,DIONE_1_6},
  {{ 0, 0, 0, 3, 0,-3, 0, 0},  1,DIONE_1_7},
  {{ 0, 0, 0, 3,-3, 0, 0, 0},  1,DIONE_1_8},
  {{ 0, 0, 0, 4,-4, 0, 0, 0},  1,DIONE_1_9},
  {{ 0, 0, 0, 5,-5, 0, 0, 0},  1,DIONE_1_10},
  {{ 0, 0, 1,-1, 0, 0, 0, 0},  1,DIONE_1_11},
  {{ 0, 0, 2,-2, 0, 0, 0, 0},  1,DIONE_1_12},
  {{ 0, 0, 3,-3, 0, 0, 0, 0},  1,DIONE_1_13},
  {{ 0, 0, 4,-4, 0, 0, 0, 0},  1,DIONE_1_14},
  {{ 0, 1, 0,-1, 0, 0, 0, 0},  1,DIONE_1_15},
  {{ 0, 1,-2, 1, 0, 0, 0, 0},  1,DIONE_1_16},
};

static const struct Tass17Term DIONE_2_0[1] = {
  {{ 1.172252747692422e-04, 2.681979734714067e+00, 2.445958739557666e-05}},
};

static const struct Tass17Term DIONE_2_1[1] = {
  {{ 1.288509519128533e-05, 1.795199654407042e+00, 3.940425661480334e-01}},
};

static const struct Tass17Term DIONE_2_2[1] = {
  {{ 3.368317726971210e-06, 3.079746085685258e+00, 1.390853717105622e+00}},
};

static const struct Tass17Term DIONE_2_3[1] = {
  {{ 2.527837703075836e-05, 1.595801592159713e+00,-1.507632514136644e+00}},
};

static const struct Tass17Term DIONE_2_4[1] = {
  {{ 3.539705714890319e-06, 3.850614164262772e+00,-1.113614407576006e+00}},
};

static const struct Tass17Term DIONE_2_5[1] = {
  {{ 2.594271531157734e-05, 1.023301801126352e+00, 4.859897877785333e-01}},
};

static const struct Tass17Term DIONE_2_6[1] = {
  {{ 6.245540270289302e-04, 1.994597716654372e+00, 2.295717646432711e+00}},
};

static const struct Tass17Term DIONE_2_7[1] = {
  {{ 4.309530620212000e-06, 4.537996183502177e+00,-3.409307594421322e+00}},
};

static const struct Tass17Term DIONE_2_8[1] = {
  {{ 2.669607656360495e-05, 2.108450170157240e+00,-4.188741415485552e-01}},
};

static const struct Tass17Term DIONE_2_9[1] = {
  {{ 6.968783491136024e-06, 5.200588559833291e-02,-1.323738070875643e+00}},
};

static const struct Tass17Term DIONE_2_10[1] = {
  {{ 3.304505798253070e-06, 4.278746908219012e+00,-2.228602000202733e+00}},
};

static const struct Tass17Term DIONE_2_11[1] = {
  {{ 3.787380702785700e-06, 5.033271736497301e+00, 1.263128847810865e+00}},
};

static const struct Tass17Term DIONE_2_12[1] = {
  {{ 2.859852996917539e-05, 1.788760449160644e+00, 2.305400491890175e-01}},
};

static const struct Tass17Term DIONE_2_13[1] = {
  {{ 7.799304255363336e-06, 1.685841815413778e+00,-8.020487494328282e-01}},
};

static const struct Tass17Term DIONE_2_14[1] = {
  {{ 2.969852804959130e-06, 4.724515835256708e+00,-1.834637548054674e+00}},
};

static const struct Tass17Term DIONE_2_15[5] = {
  {{ 5.709737153001810e-06, 5.673408458612511e+00,-7.798450772541216e-05}},
  {{ 2.203368656279073e-03, 4.874775170004872e+00, 1.471391305053998e-03}},
  {{ 4.177951671700660e-06, 3.418533941089551e+00,-2.955758721579874e-03}},
  {{ 5.709763882914900e-06, 9.345492278074399e-01, 3.020767117833410e-03}},
  {{ 1.886080065709498e-05, 3.189421946098056e+00, 5.898541331687871e-03}},
};

static const struct Tass17Term DIONE_2_16[1] = {
  {{ 2.618379490210864e-06, 1.426868473632360e+00, 3.458059983814795e-03}},
};

static const struct Tass17MultiTerm DIONE_2[17] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  1,DIONE_2_0},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  1,DIONE_2_1},
  {{ 0, 0, 0, 0, 1, 0, 0, 0},  1,DIONE_2_2},
  {{ 0, 0, 0,-1, 0, 2, 0, 0},  1,DIONE_2_3},
  {{ 0, 0, 0,-1, 0, 3, 0, 0},  1,DIONE_2_4},
  {{ 0, 0, 0,-1, 2, 0, 0, 0},  1,DIONE_2_5},
  {{ 0, 0, 0, 1, 0, 0, 0, 0},  1,DIONE_2_6},
  {{ 0, 0, 0,-2, 0, 3, 0, 0},  1,DIONE_2_7},
  {{ 0, 0, 0,-2, 3, 0, 0, 0},  1,DIONE_2_8},
  {{ 0, 0, 0,-3, 4, 0, 0, 0},  1,DIONE_2_9},
  {{ 0, 0, 0,-4, 5, 0, 0, 0},  1,DIONE_2_10},
  {{ 0, 0,-1, 2, 0, 0, 0, 0},  1,DIONE_2_11},
  {{ 0, 0,-2, 3, 0, 0, 0, 0},  1,DIONE_2_12},
  {{ 0, 0,-3, 4, 0, 0, 0, 0},  1,DIONE_2_13},
  {{ 0, 0,-4, 5, 0, 0, 0, 0},  1,DIONE_2_14},
  {{ 0,-1, 0, 2, 0, 0, 0, 0},  5,DIONE_2_15},
  {{-1, 0, 2, 0, 0, 0, 0, 0},  1,DIONE_2_16},
};

static const struct Tass17Term DIONE_3_0[2] = {
  {{ 2.282739732511724e-05, 6.135349401370048e+00,-2.445240135601595e-05}},
  {{ 3.529922124449878e-05, 2.632866665182736e+00,-4.804041710792879e-04}},
};

static const struct Tass17Term DIONE_3_1[1] = {
  {{ 1.655363859165119e-04, 1.556643393089241e+00,-1.471954902036816e-03}},
};

static const struct Tass17Term DIONE_3_2[5] = {
  {{ 2.754688236937855e-05, 7.961878954263457e-01,-3.452416044929610e-03}},
  {{ 1.269228313121738e-05, 1.482534195440534e+00,-3.208623089724569e-03}},
  {{ 1.268342612853373e-05, 3.251434249001951e+00,-3.696209000134648e-03}},
  {{ 2.655183882413472e-06, 2.168880495454724e+00,-2.964830134519532e-03}},
  {{ 2.629764340505144e-06, 5.706680602577554e+00,-3.940001955339688e-03}},
};

static const struct Tass17MultiTerm DIONE_3[3] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  2,DIONE_3_0},
  {{ 0,-1, 0, 2, 0, 0, 0, 0},  1,DIONE_3_1},
  {{-1, 0, 2, 0, 0, 0, 0, 0},  5,DIONE_3_2},
};

static const struct Tass17Term RHEA_0_0[1] = {
  {{ 1.460129617898124e-05, 4.426139084868009e+00, 9.968111509575888e-01}},
};

static const struct Tass17Term RHEA_0_1[1] = {
  {{ 6.497476875291102e-05, 5.710685516146226e+00, 1.993622301915178e+00}},
};

static const struct Tass17Term RHEA_0_2[1] = {
  {{ 8.448119776705509e-06, 3.455872944043167e+00, 1.599604195354539e+00}},
};

static const struct Tass17Term RHEA_0_3[1] = {
  {{ 2.344587498942920e-05, 7.120466402448569e-01, 2.990433452872767e+00}},
};

static const struct Tass17Term RHEA_0_4[1] = {
  {{ 4.000365153622274e-06, 4.740419375321384e+00, 2.596415346312128e+00}},
};

static const struct Tass17Term RHEA_0_5[1] = {
  {{ 8.875319141540999e-06, 1.996593071523072e+00, 3.987244603830355e+00}},
};

static const struct Tass17Term RHEA_0_6[1] = {
  {{ 3.452681160707960e-06, 3.281139502801291e+00, 4.984055754787943e+00}},
};

static const struct Tass17Term RHEA_0_7[1] = {
  {{ 9.207121596242679e-06, 5.198036938148699e+00, 9.048639293270888e-01}},
};

static const struct Tass17Term RHEA_0_8[1] = {
  {{ 4.621278522602440e-06, 4.112888569117812e+00, 1.809727858654177e+00}},
};

static const struct Tass17Term RHEA_0_9[1] = {
  {{ 5.977877370825760e-06, 2.159362918305770e+00, 1.937452727948935e+00}},
};

static const struct Tass17MultiTerm RHEA_0[10] = {
  {{ 0, 0, 0, 0, 1,-1, 0, 0},  1,RHEA_0_0},
  {{ 0, 0, 0, 0, 2,-2, 0, 0},  1,RHEA_0_1},
  {{ 0, 0, 0, 0, 2,-3, 0, 0},  1,RHEA_0_2},
  {{ 0, 0, 0, 0, 3,-3, 0, 0},  1,RHEA_0_3},
  {{ 0, 0, 0, 0, 3,-4, 0, 0},  1,RHEA_0_4},
  {{ 0, 0, 0, 0, 4,-4, 0, 0},  1,RHEA_0_5},
  {{ 0, 0, 0, 0, 5,-5, 0, 0},  1,RHEA_0_6},
  {{ 0, 0, 0, 1,-1, 0, 0, 0},  1,RHEA_0_7},
  {{ 0, 0, 0, 2,-2, 0, 0, 0},  1,RHEA_0_8},
  {{ 0, 0, 1, 0,-1, 0, 0, 0},  1,RHEA_0_9},
};

static const struct Tass17Term RHEA_1_0[11] = {
  {{ 4.982624562626507e-04, 4.765136030335644e+00, 5.271851040460129e-06}},
  {{ 3.475126795324123e-06, 5.905021737251245e+00, 1.054370208092026e-05}},
  {{ 4.462723299012653e-06, 4.222312533161678e+00, 1.898637155826853e-05}},
  {{ 3.255047679977468e-04, 3.303959081334276e+00, 2.445240135601595e-05}},
  {{ 1.772508422160692e-05, 5.196471838140832e+00, 4.891917479115332e-05}},
  {{ 9.471405209345812e-06, 7.450246191142529e-01, 1.035769927204739e-04}},
  {{ 1.830753059305353e-05, 2.440869213255669e-01, 4.559517697232720e-04}},
  {{ 6.147405445926997e-05, 3.666743633441163e+00, 4.804041710792879e-04}},
  {{ 4.693769190005650e-05, 4.409577074506501e+00, 5.839811637997618e-04}},
  {{ 6.390530214147782e-05, 2.864957498258305e+00, 1.167962327599524e-03}},
  {{ 8.699581713814392e-06, 4.133277437711403e+00, 1.751943491399285e-03}},
};

static const struct Tass17Term RHEA_1_1[1] = {
  {{ 8.614402961042187e-06, 5.396405225692853e+00, 3.940181065606379e-01}},
};

static const struct Tass17Term RHEA_1_2[1] = {
  {{ 5.229341727885530e-05, 4.426139084868009e+00, 9.968111509575888e-01}},
};

static const struct Tass17Term RHEA_1_3[1] = {
  {{ 1.003942988473349e-05, 2.171326512764949e+00, 6.027930443969509e-01}},
};

static const struct Tass17Term RHEA_1_4[1] = {
  {{ 3.825560625917272e-06, 6.199699253192279e+00, 2.087749378363131e-01}},
};

static const struct Tass17Term RHEA_1_5[1] = {
  {{ 9.265555925994737e-05, 5.710685516146226e+00, 1.993622301915178e+00}},
};

static const struct Tass17Term RHEA_1_6[1] = {
  {{ 1.367398797935688e-05, 3.455872944043167e+00, 1.599604195354539e+00}},
};

static const struct Tass17Term RHEA_1_7[1] = {
  {{ 2.755180293512720e-05, 7.120466402448569e-01, 2.990433452872767e+00}},
};

static const struct Tass17Term RHEA_1_8[1] = {
  {{ 5.024449380692616e-06, 4.740419375321384e+00, 2.596415346312128e+00}},
};

static const struct Tass17Term RHEA_1_9[1] = {
  {{ 9.310491515462421e-06, 1.996593071523072e+00, 3.987244603830355e+00}},
};

static const struct Tass17Term RHEA_1_10[1] = {
  {{ 2.065629848458831e-06, 6.024965806599599e+00, 3.593226497269717e+00}},
};

static const struct Tass17Term RHEA_1_11[1] = {
  {{ 3.359851109236900e-06, 3.281139502801291e+00, 4.984055754787943e+00}},
};

static const struct Tass17Term RHEA_1_12[1] = {
  {{ 1.266911496422784e-05, 2.056444284558907e+00, 9.048639293270888e-01}},
};

static const struct Tass17Term RHEA_1_13[1] = {
  {{ 9.413368380851320e-06, 4.112888569117812e+00, 1.809727858654177e+00}},
};

static const struct Tass17Term RHEA_1_14[1] = {
  {{ 4.460853124280260e-06, 6.169332853676718e+00, 2.714591787981265e+00}},
};

static const struct Tass17Term RHEA_1_15[1] = {
  {{ 2.423482201792380e-06, 1.942591831056039e+00, 3.619455717308356e+00}},
};

static const struct Tass17Term RHEA_1_16[1] = {
  {{ 3.028875527833520e-06, 5.300955571895564e+00, 1.937452727948935e+00}},
};

static const struct Tass17MultiTerm RHEA_1[17] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 11,RHEA_1_0},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  1,RHEA_1_1},
  {{ 0, 0, 0, 0, 1,-1, 0, 0},  1,RHEA_1_2},
  {{ 0, 0, 0, 0, 1,-2, 0, 0},  1,RHEA_1_3},
  {{ 0, 0, 0, 0, 1,-3, 0, 0},  1,RHEA_1_4},
  {{ 0, 0, 0, 0, 2,-2, 0, 0},  1,RHEA_1_5},
  {{ 0, 0, 0, 0, 2,-3, 0, 0},  1,RHEA_1_6},
  {{ 0, 0, 0, 0, 3,-3, 0, 0},  1,RHEA_1_7},
  {{ 0, 0, 0, 0, 3,-4, 0, 0},  1,RHEA_1_8},
  {{ 0, 0, 0, 0, 4,-4, 0, 0},  1,RHEA_1_9},
  {{ 0, 0, 0, 0, 4,-5, 0, 0},  1,RHEA_1_10},
  {{ 0, 0, 0, 0, 5,-5, 0, 0},  1,RHEA_1_11},
  {{ 0, 0, 0, 1,-1, 0, 0, 0},  1,RHEA_1_12},
  {{ 0, 0, 0, 2,-2, 0, 0, 0},  1,RHEA_1_13},
  {{ 0, 0, 0, 3,-3, 0, 0, 0},  1,RHEA_1_14},
  {{ 0, 0, 0, 4,-4, 0, 0, 0},  1,RHEA_1_15},
  {{ 0, 0, 1, 0,-1, 0, 0, 0},  1,RHEA_1_16},
};

static const struct Tass17Term RHEA_2_0[4] = {
  {{ 6.496268041362011e-06, 6.195159743999218e-01,-2.445958739557666e-05}},
  {{ 9.712611923476502e-04, 2.687921917028628e+00, 2.445958739557666e-05}},
  {{ 1.671852669066537e-04, 5.358703148577743e-02, 4.806275624808988e-04}},
  {{ 2.256764766375221e-06, 3.428726874313563e-01, 1.143502740203947e-03}},
};

static const struct Tass17Term RHEA_2_1[1] = {
  {{ 3.248040932920730e-05, 1.795199654407042e+00, 3.940425661480334e-01}},
};

static const struct Tass17Term RHEA_2_2[1] = {
  {{ 1.108384911297719e-04, 3.652245876718619e+00,-6.027685848095554e-01}},
};

static const struct Tass17Term RHEA_2_3[1] = {
  {{ 3.382899984196108e-05, 5.907058448821680e+00,-2.087504782489175e-01}},
};

static const struct Tass17Term RHEA_2_4[1] = {
  {{ 2.778461053446649e-06, 5.020278361984143e+00, 1.852676283117203e-01}},
};

static const struct Tass17Term RHEA_2_5[1] = {
  {{ 3.116043009304567e-04, 6.221338739275052e+00, 1.390853717105622e+00}},
};

static const struct Tass17Term RHEA_2_6[1] = {
  {{ 2.241664081352878e-05, 2.367699445440403e+00,-1.599579735767144e+00}},
};

static const struct Tass17Term RHEA_2_7[1] = {
  {{ 4.424102595490495e-06, 4.622512017543462e+00,-1.205561629206506e+00}},
};

static const struct Tass17Term RHEA_2_8[1] = {
  {{ 6.929884985440310e-06, 1.083153014162185e+00,-2.596390886724733e+00}},
};

static const struct Tass17Term RHEA_2_9[1] = {
  {{ 5.779542899092500e-06, 2.507246294651899e+00, 3.384476019020799e+00}},
};

static const struct Tass17Term RHEA_2_10[1] = {
  {{ 2.427569664020370e-06, 6.081791890063556e+00,-3.593202037682322e+00}},
};

static const struct Tass17Term RHEA_2_11[1] = {
  {{ 2.492530233817800e-06, 3.791792725930115e+00, 4.381287169978387e+00}},
};

static const struct Tass17Term RHEA_2_12[1] = {
  {{ 6.766246868422730e-06, 4.164894454716146e+00, 4.859897877785333e-01}},
};

static const struct Tass17Term RHEA_2_13[1] = {
  {{ 3.851717115480820e-06, 1.994597716654372e+00, 2.295717646432711e+00}},
};

static const struct Tass17Term RHEA_2_14[1] = {
  {{ 1.308501254327346e-05, 5.250042823747033e+00,-4.188741415485552e-01}},
};

static const struct Tass17Term RHEA_2_15[1] = {
  {{ 3.467317924478190e-06, 3.193598539188126e+00,-1.323738070875643e+00}},
};

static const struct Tass17Term RHEA_2_16[1] = {
  {{ 2.224849328232030e-06, 5.239109003991030e+00, 3.328306445054557e+00}},
};

static const struct Tass17Term RHEA_2_17[1] = {
  {{ 7.641483476943351e-06, 1.733182516415079e+00, 1.471391305053998e-03}},
};

static const struct Tass17MultiTerm RHEA_2[18] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  4,RHEA_2_0},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  1,RHEA_2_1},
  {{ 0, 0, 0, 0,-1, 2, 0, 0},  1,RHEA_2_2},
  {{ 0, 0, 0, 0,-1, 3, 0, 0},  1,RHEA_2_3},
  {{ 0, 0, 0, 0,-1, 4, 0, 0},  1,RHEA_2_4},
  {{ 0, 0, 0, 0, 1, 0, 0, 0},  1,RHEA_2_5},
  {{ 0, 0, 0, 0,-2, 3, 0, 0},  1,RHEA_2_6},
  {{ 0, 0, 0, 0,-2, 4, 0, 0},  1,RHEA_2_7},
  {{ 0, 0, 0, 0,-3, 4, 0, 0},  1,RHEA_2_8},
  {{ 0, 0, 0, 0, 3,-2, 0, 0},  1,RHEA_2_9},
  {{ 0, 0, 0, 0,-4, 5, 0, 0},  1,RHEA_2_10},
  {{ 0, 0, 0, 0, 4,-3, 0, 0},  1,RHEA_2_11},
  {{ 0, 0, 0,-1, 2, 0, 0, 0},  1,RHEA_2_12},
  {{ 0, 0, 0, 1, 0, 0, 0, 0},  1,RHEA_2_13},
  {{ 0, 0, 0,-2, 3, 0, 0, 0},  1,RHEA_2_14},
  {{ 0, 0, 0,-3, 4, 0, 0, 0},  1,RHEA_2_15},
  {{ 0, 0, 1, 0, 0, 0, 0, 0},  1,RHEA_2_16},
  {{ 0,-1, 0, 2, 0, 0, 0, 0},  1,RHEA_2_17},
};

static const struct Tass17Term RHEA_3_0[8] = {
  {{ 2.970532433280590e-03, 2.626878648804450e+00,-4.804041710792879e-04}},
  {{ 1.787509410723081e-04, 6.204551285275548e+00,-2.445240135601595e-05}},
  {{ 2.723114438070935e-05, 6.085981723978948e+00, 1.167962327599524e-03}},
  {{ 2.414440346500943e-05, 5.096853638619743e+00,-5.839811637997618e-04}},
  {{ 9.525093659923903e-06, 5.044236237492881e+00,-5.271851040460129e-06}},
  {{ 3.807917651740949e-06, 1.070489829428804e+00, 1.751943491399285e-03}},
  {{ 3.291355004303486e-06, 4.446713203128226e+00, 5.839811637997618e-04}},
  {{ 2.944165615658908e-06, 3.500661322372421e-01,-1.167962327599524e-03}},
};

static const struct Tass17MultiTerm RHEA_3[1] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},  8,RHEA_3_0},
};

static const struct Tass17Term TITAN_0_0[1] = {
  {{ 5.193372353628770e-06, 5.396405225692853e+00, 3.940181065606379e-01}},
};

static const struct Tass17Term TITAN_0_1[1] = {
  {{ 4.923588320704660e-06, 5.786945864275197e-01, 7.869171700036950e-01}},
};

static const struct Tass17Term TITAN_0_2[1] = {
  {{ 2.514051816697911e-05, 4.426139084868009e+00, 9.968111509575888e-01}},
};

static const struct Tass17Term TITAN_0_3[1] = {
  {{ 1.234070873751902e-05, 1.993980622473295e-01, 1.901675080284677e+00}},
};

static const struct Tass17Term TITAN_0_4[1] = {
  {{ 7.428005944912294e-06, 3.443909349583986e+00, 2.934263878906524e+00}},
};

static const struct Tass17Term TITAN_0_5[1] = {
  {{ 1.514267591217036e-06, 5.287757340750888e+00, 4.191494185385700e+00}},
};

static const struct Tass17MultiTerm TITAN_0[6] = {
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  1,TITAN_0_0},
  {{ 0, 0, 0, 0, 0, 2, 0, 0},  1,TITAN_0_1},
  {{ 0, 0, 0, 0, 1,-1, 0, 0},  1,TITAN_0_2},
  {{ 0, 0, 0, 1, 0,-1, 0, 0},  1,TITAN_0_3},
  {{ 0, 0, 1, 0, 0,-1, 0, 0},  1,TITAN_0_4},
  {{ 0, 1, 0, 0, 0,-1, 0, 0},  1,TITAN_0_5},
};

static const struct Tass17Term TITAN_1_0[18] = {
  {{ 1.489184840960493e-03, 4.482905304402765e+00, 5.271851040460129e-06}},
  {{ 9.861735003619475e-06, 5.820876772324451e+00, 1.054370208092026e-05}},
  {{ 3.205222547434483e-05, 2.421050292728566e+00, 1.880354218803814e-05}},
  {{ 6.277976937099409e-04, 1.641703069309713e-01, 2.445240135601595e-05}},
  {{ 2.782832942434251e-05, 2.085876834041031e+00, 4.891917479115332e-05}},
  {{ 3.898032222540717e-06, 2.691928895904259e+00, 2.825103078658793e-04}},
  {{ 3.675185983863103e-06, 3.132180625569890e+00, 3.017286310072802e-04}},
  {{ 6.328208168488662e-06, 1.099595805258099e+00, 5.652783908051565e-04}},
  {{ 1.839936992682087e-04, 4.432539236156053e+00, 5.839811637997618e-04}},
  {{ 1.012886232301891e-06, 3.824011892951753e+00, 6.026839367943671e-04}},
  {{ 1.807246533364200e-06, 5.897892147256107e+00, 8.662336965922434e-04}},
  {{ 3.340295455803047e-06, 3.949193776134189e+00, 1.119043152808370e-03}},
  {{ 2.671699222038788e-06, 1.780723999304769e-02, 1.149259554604918e-03}},
  {{ 2.064532509871716e-04, 2.875895084979086e+00, 1.167962327599524e-03}},
  {{ 2.919923956899900e-06, 2.618006049618430e+00, 1.186665100594129e-03}},
  {{ 3.918483314907732e-06, 3.102869921028134e+00, 1.192414728955539e-03}},
  {{ 2.910646188137447e-05, 4.133511734209461e+00, 1.751943491399285e-03}},
  {{ 3.080473554041300e-06, 5.390964128071579e+00, 2.335924655199047e-03}},
};

static const struct Tass17Term TITAN_1_1[9] = {
  {{ 2.177885623014171e-06, 5.338524981350306e+00, 1.937456015378634e-03}},
  {{ 2.229788591597716e-06, 3.516065572948062e+00, 1.942727866419093e-03}},
  {{ 3.240885104844179e-06, 1.456814599107305e+00, 1.956509193942217e-03}},
  {{ 3.317890554439667e-06, 5.917531821903471e+00, 1.961781044982677e-03}},
  {{ 8.515242483441068e-07, 4.094907555655027e+00, 1.967052896023137e-03}},
  {{ 1.583596726276704e-06, 1.857323310303026e+00, 1.967322011606203e-03}},
  {{ 3.183322410669546e-06, 3.483641352342939e-02, 1.972593862646663e-03}},
  {{ 2.439949469786914e-06, 4.495498981484839e+00, 1.977865713687124e-03}},
  {{ 8.331051796244125e-07, 2.672845480380433e+00, 1.983137564727584e-03}},
};

static const struct Tass17Term TITAN_1_2[2] = {
  {{ 9.325252754097003e-07, 1.465474667914254e+00, 3.928990634430571e-01}},
  {{ 6.059629840959338e-06, 2.254812572103060e+00, 3.940181065606379e-01}},
};

static const struct Tass17Term TITAN_1_3[2] = {
  {{ 1.119995323971813e-06, 5.646772130696905e+00, 7.863331888575091e-01}},
  {{ 5.747511089543420e-06, 5.786945864275197e-01, 7.869171700036950e-01}},
};

static const struct Tass17Term TITAN_1_4[1] = {
  {{ 1.208825245395646e-05, 1.284546431278216e+00, 9.968111509575888e-01}},
};

static const struct Tass17Term TITAN_1_5[1] = {
  {{ 9.999688627568780e-07, 2.569092862556433e+00, 1.993622301915178e+00}},
};

static const struct Tass17Term TITAN_1_6[1] = {
  {{ 6.461479160208660e-06, 3.340990715837123e+00, 1.901675080284677e+00}},
};

static const struct Tass17Term TITAN_1_7[1] = {
  {{ 4.165211007064940e-06, 3.023166959941940e-01, 2.934263878906524e+00}},
};

static const struct Tass17Term TITAN_1_8[1] = {
  {{ 8.903963476562940e-07, 2.146164687161095e+00, 4.191494185385700e+00}},
};

static const struct Tass17MultiTerm TITAN_1[9] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 18,TITAN_1_0},
  {{ 0, 0, 0, 0, 0,-1, 5, 0},  9,TITAN_1_1},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  2,TITAN_1_2},
  {{ 0, 0, 0, 0, 0, 2, 0, 0},  2,TITAN_1_3},
  {{ 0, 0, 0, 0, 1,-1, 0, 0},  1,TITAN_1_4},
  {{ 0, 0, 0, 0, 2,-2, 0, 0},  1,TITAN_1_5},
  {{ 0, 0, 0, 1, 0,-1, 0, 0},  1,TITAN_1_6},
  {{ 0, 0, 1, 0, 0,-1, 0, 0},  1,TITAN_1_7},
  {{ 0, 1, 0, 0, 0,-1, 0, 0},  1,TITAN_1_8},
};

static const struct Tass17Term TITAN_2_0[24] = {
  {{ 2.030439241690526e-06, 3.465392760946876e+00,-5.406408831993331e-06}},
  {{ 1.720731211496869e-05, 3.423177098527374e+00, 5.406408831993331e-06}},
  {{ 2.297673200703965e-06, 4.851969044359338e+00,-1.067825987245346e-05}},
  {{ 5.578526024329263e-06, 4.763368404218852e+00, 1.067825987245346e-05}},
  {{ 2.429402127836536e-05, 4.497027349870592e+00, 1.918773635511653e-05}},
  {{ 1.921261556124706e-04, 6.049785198226078e-01,-2.445958739557666e-05}},
  {{ 2.892653650392732e-02, 2.687601928257754e+00, 2.445958739557666e-05}},
  {{ 5.704537193893060e-06, 2.441938711199521e+00,-2.973143843603680e-05}},
  {{ 2.391668026442592e-05, 4.004671318127588e+00, 2.973143843603680e-05}},
  {{ 1.273579323673662e-06, 4.271696076744320e+00,-3.500328947649692e-05}},
  {{ 2.219325143437113e-06, 3.927500699309107e-01,-4.891198875159261e-05}},
  {{ 1.391334470388962e-06, 3.792208194559931e+00, 2.777039784926735e-04}},
  {{ 1.840440310193102e-06, 4.554834665120421e+00,-5.595215764041851e-04}},
  {{ 6.349090434417589e-06, 5.278059950506263e+00, 5.595215764041851e-04}},
  {{ 1.359141741251292e-06, 6.222686370138073e+00, 5.839811637997618e-04}},
  {{ 1.313668933428861e-06, 5.635555579481489e+00,-6.084407511953385e-04}},
  {{ 1.818913752278572e-06, 3.943058897661665e+00, 6.084407511953385e-04}},
  {{ 1.054968278903017e-06, 6.106129826283189e-01, 1.124788312095915e-03}},
  {{ 2.599144808715373e-06, 6.100137782540312e+00,-1.143502740203947e-03}},
  {{ 7.446564554937326e-05, 3.484557927409254e+00, 1.143502740203947e-03}},
  {{ 1.030286270548740e-06, 3.243359475417337e+00, 1.162179719401369e-03}},
  {{ 2.100942202805606e-06, 2.406466000093305e+00, 1.192421914995100e-03}},
  {{ 9.603481577558545e-06, 4.751548744302498e+00, 1.727483904003709e-03}},
  {{ 1.071090644258225e-06, 6.019617650063695e+00, 2.311465067803471e-03}},
};

static const struct Tass17Term TITAN_2_1[2] = {
  {{ 9.612769517722884e-07, 2.431612830889724e+00,-3.922906227094757e-01}},
  {{ 4.930907396569310e-06, 1.216505067979523e+00,-3.928746038556616e-01}},
};

static const struct Tass17Term TITAN_2_2[1] = {
  {{ 6.687871156591785e-05, 4.936792307996836e+00, 3.940425661480334e-01}},
};

static const struct Tass17Term TITAN_2_3[1] = {
  {{ 2.596686176814385e-06, 9.084195729203095e-01, 7.880606727086713e-01}},
};

static const struct Tass17Term TITAN_2_4[1] = {
  {{ 4.938696934118670e-06, 3.652245876718619e+00,-6.027685848095554e-01}},
};

static const struct Tass17Term TITAN_2_5[1] = {
  {{ 1.010084616284730e-05, 6.221338739275052e+00, 1.390853717105622e+00}},
};

static const struct Tass17Term TITAN_2_6[1] = {
  {{ 2.338446911029510e-06, 1.595801592159713e+00,-1.507632514136644e+00}},
};

static const struct Tass17Term TITAN_2_7[1] = {
  {{ 5.269296403563900e-06, 1.994597716654372e+00, 2.295717646432711e+00}},
};

static const struct Tass17Term TITAN_2_8[1] = {
  {{ 1.363323740924180e-06, 4.634475612002642e+00,-2.540221312758490e+00}},
};

static const struct Tass17Term TITAN_2_9[1] = {
  {{ 3.315192749055180e-06, 5.239109003991030e+00, 3.328306445054557e+00}},
};

static const struct Tass17MultiTerm TITAN_2[10] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 24,TITAN_2_0},
  {{ 0, 0, 0, 0, 0,-1, 0, 0},  2,TITAN_2_1},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  1,TITAN_2_2},
  {{ 0, 0, 0, 0, 0, 2, 0, 0},  1,TITAN_2_3},
  {{ 0, 0, 0, 0,-1, 2, 0, 0},  1,TITAN_2_4},
  {{ 0, 0, 0, 0, 1, 0, 0, 0},  1,TITAN_2_5},
  {{ 0, 0, 0,-1, 0, 2, 0, 0},  1,TITAN_2_6},
  {{ 0, 0, 0, 1, 0, 0, 0, 0},  1,TITAN_2_7},
  {{ 0, 0,-1, 0, 0, 2, 0, 0},  1,TITAN_2_8},
  {{ 0, 0, 1, 0, 0, 0, 0, 0},  1,TITAN_2_9},
};

static const struct Tass17Term TITAN_3_0[21] = {
  {{ 2.789942947721349e-03, 6.204694901726296e+00,-2.445240135601595e-05}},
  {{ 1.312363309291625e-04, 5.044269214427242e+00,-5.271851040460129e-06}},
  {{ 1.125670790406430e-04, 6.084205141557698e+00, 1.167962327599524e-03}},
  {{ 1.916668518784865e-05, 5.094974746907165e+00,-5.839811637997618e-04}},
  {{ 1.497943297409488e-05, 1.070503341744811e+00, 1.751943491399285e-03}},
  {{ 1.144622908335464e-05, 4.298020835227772e+00, 5.839811637997618e-04}},
  {{ 1.105889373330841e-05, 3.817225181469991e+00,-1.870277299460525e-05}},
  {{ 9.469793088277916e-06, 3.725645429506557e+00,-1.054370208092026e-05}},
  {{ 6.878606841089768e-06, 4.544829611068419e+00, 5.271851040460129e-06}},
  {{ 6.072279735075281e-06, 3.088021134790829e-01,-1.167962327599524e-03}},
  {{ 3.672628251015072e-06, 5.291779482141999e+00, 4.891917479115332e-05}},
  {{ 3.001306766151942e-06, 2.631624338898295e-01, 2.445240135601595e-05}},
  {{ 2.508038681269874e-06, 5.838193085927095e+00,-4.804041710792879e-04}},
  {{ 2.453965972048404e-06, 4.844097342411929e+00,-2.972425239647608e-05}},
  {{ 1.777443612458054e-06, 1.373143790052677e-01, 3.017286310072802e-04}},
  {{ 1.638627359387858e-06, 2.338023546843110e+00, 2.335924655199047e-03}},
  {{ 1.619813827555800e-06, 3.201683403167966e+00, 1.149259554604918e-03}},
  {{ 1.585471208377748e-06, 5.833093965190200e+00, 1.186665100594129e-03}},
  {{ 1.518975368269692e-06, 3.329608496168708e+00, 1.870277299460525e-05}},
  {{ 1.151714556320695e-06, 2.625783233864042e-02, 1.192414728955539e-03}},
  {{ 8.830894877995539e-07, 5.372467623946739e+00,-1.751943491399285e-03}},
};

static const struct Tass17MultiTerm TITAN_3[1] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 21,TITAN_3_0},
};

static const struct Tass17Term IAPETUS_0_0[7] = {
  {{ 2.055029475492061e-06, 2.502670659577745e+00, 7.745544060220222e-02}},
  {{ 1.017421707635384e-05,-2.565419768024297e+00, 7.803942174838812e-02}},
  {{ 6.585278743116177e-07, 1.895225923201359e+00, 7.804469359942858e-02}},
  {{ 1.178034971556374e-05, 3.767316688774966e+00, 7.917751804453235e-02}},
  {{ 2.109797808204992e-05, 6.169137529926802e+00, 7.919657122309593e-02}},
  {{ 5.616037270279334e-07,-2.864068502197524e+00, 7.920738404075992e-02}},
  {{ 1.210381264018071e-06,-1.536414322676488e+00, 7.921265589180036e-02}},
};

static const struct Tass17Term IAPETUS_0_1[17] = {
  {{ 3.248586724047530e-06, 1.173515119422923e+00, 1.560680306791122e-01}},
  {{ 2.327804697388999e-05, 2.388610005279730e+00, 1.566520118252981e-01}},
  {{ 1.506081447204865e-06, 5.660579539606869e-01, 1.566572836763386e-01}},
  {{ 2.219668063709623e-06, 2.284631982621601e+00, 1.572307211204436e-01}},
  {{ 1.177813487687222e-04,-2.679467329715666e+00, 1.572359929714840e-01}},
  {{ 5.121363233915475e-06, 1.781122928927975e+00, 1.572412648225245e-01}},
  {{ 3.258205784517132e-06, 1.657988726564947e+00, 1.578199741176699e-01}},
  {{ 6.406895152069824e-07, 3.418104295086557e+00, 1.578252459687104e-01}},
  {{ 7.233685566797959e-07, 5.117268855773154e-01, 1.583740892676283e-01}},
  {{ 6.553814036784504e-07, 2.913580261178442e+00, 1.583931424461918e-01}},
  {{ 5.886085615859691e-06,-2.969003559963930e+00, 1.584039552638558e-01}},
  {{ 3.846693748791777e-07, 1.489841855036775e+00, 1.584092271148963e-01}},
  {{ 1.849113934605446e-06, 5.952077755114058e+00, 1.584144989659368e-01}},
  {{ 1.523920218717224e-07, 1.386868830832936e+00, 1.589879364100418e-01}},
  {{ 6.359828738920669e-07, 5.848319820906959e+00, 1.589932082610822e-01}},
  {{ 1.105572785393772e-07,-1.164606940957312e-01, 1.595719175562276e-01}},
  {{ 6.957733951844473e-07, 1.201970451738299e+00, 1.595771894072681e-01}},
};

static const struct Tass17Term IAPETUS_0_2[2] = {
  {{ 6.748302380368017e-07, 5.416167758678300e+00, 2.358485830483941e-01}},
  {{ 3.458158840283551e-06, 3.480902144089144e-01, 2.364325641945799e-01}},
};

static const struct Tass17Term IAPETUS_0_3[13] = {
  {{ 4.946872838108632e-07, 6.077953687111643e+00, 1.918402836815050e-03}},
  {{ 5.065090050950890e-07, 4.255501070028474e+00, 1.923674687855510e-03}},
  {{ 5.354701955778437e-07, 1.796193334454873e+00, 1.926643197714647e-03}},
  {{ 2.368323695654563e-06, 2.196618138085234e+00, 1.937456015378634e-03}},
  {{ 2.424687512275475e-06, 3.741587517563335e-01, 1.942727866419093e-03}},
  {{ 6.223980017737441e-07, 4.834746214664873e+00, 1.947999717459554e-03}},
  {{ 3.515734994495499e-06, 4.598407354081068e+00, 1.956509193942217e-03}},
  {{ 3.599154632304914e-06, 2.775939218332986e+00, 1.961781044982677e-03}},
  {{ 9.236794241568811e-07, 9.533149020652341e-01, 1.967052896023137e-03}},
  {{ 1.730312424825915e-06, 4.998916170228774e+00, 1.967322011606203e-03}},
  {{ 3.478141777876746e-06, 3.176429222382267e+00, 1.972593862646663e-03}},
  {{ 2.665836454376912e-06, 1.353906429920962e+00, 1.977865713687124e-03}},
  {{ 9.101734358180643e-07, 5.814438133970226e+00, 1.983137564727584e-03}},
};

static const struct Tass17Term IAPETUS_0_4[5] = {
  {{ 1.173808537373678e-05, 3.099779317070623e-01, 3.148353166650650e-01}},
  {{ 1.422582505896866e-03, 1.629074651441286e+00, 3.148405885161054e-01}},
  {{ 1.282760621734726e-05, 2.948174134620363e+00, 3.148458603671461e-01}},
  {{ 4.431013807987148e-07, 5.376468530544894e-01, 3.148597690664211e-01}},
  {{ 8.613770793559659e-07, 4.998444131476766e+00, 3.148650409174615e-01}},
};

static const struct Tass17Term IAPETUS_0_5[4] = {
  {{ 1.529823976752019e-05, 4.943058008729855e+00, 4.732445437799614e-01}},
  {{ 1.638556362699022e-05, 3.120590560712127e+00, 4.732498156310018e-01}},
  {{ 4.393561481041129e-06, 1.297974022507647e+00, 4.732550874820422e-01}},
  {{ 5.965719593399348e-07, 5.170692394242208e+00, 4.732689961813175e-01}},
};

static const struct Tass17Term IAPETUS_0_6[7] = {
  {{ 1.205192931267955e-06, 2.446685865249762e-02, 2.356279326243051e-01}},
  {{ 1.539010606612464e-06, 1.349751065508437e+00, 2.356332044753456e-01}},
  {{ 5.125234375859194e-07, 6.347454751688458e-01, 2.356386108841776e-01}},
  {{ 4.815890873838023e-05, 4.857380043863779e+00, 2.356440172930096e-01}},
  {{ 1.049405056557275e-06, 6.205868207753926e+00, 2.356492891440501e-01}},
  {{ 2.261000099791683e-06, 4.144773749632801e+00, 2.356630704715732e-01}},
  {{ 6.346855400965112e-07, 4.208242392372787e+00, 2.368011668029451e-01}},
};

static const struct Tass17Term IAPETUS_0_7[5] = {
  {{ 4.677155865229010e-07, 1.960182176247418e+00, 1.564260895501687e-01}},
  {{ 1.822472901105999e-06, 1.375578599796670e-01, 1.564313614012092e-01}},
  {{ 1.780232418309521e-06, 4.598275031411171e+00, 1.564366332522496e-01}},
  {{ 1.648047287035982e-06, 1.857191416947010e+00, 1.564474460699136e-01}},
  {{ 7.820099241529981e-07, 1.117332070337260e+00, 1.564664992484772e-01}},
};

static const struct Tass17Term IAPETUS_0_8[2] = {
  {{ 7.283955691107278e-07, 3.393198290814080e+00, 7.723479017811326e-02}},
  {{ 7.115131339933769e-07, 1.570730155065997e+00, 7.724006202915373e-02}},
};

static const struct Tass17Term IAPETUS_0_9[1] = {
  {{ 3.791140650423246e-05, 3.877892946988424e+00, 7.088586950767433e-01}},
};

static const struct Tass17Term IAPETUS_0_10[2] = {
  {{ 5.570980365205931e-07, 9.146867890556700e-01, 8.672626503405993e-01}},
  {{ 5.703165858434648e-07, 5.375403960487173e+00, 8.672679221916397e-01}},
};

static const struct Tass17Term IAPETUS_0_11[2] = {
  {{ 1.199902334914149e-06, 8.563423202419649e-01, 6.296621238536474e-01}},
  {{ 1.557003369506093e-05, 3.258149248363778e+00, 6.296811770322111e-01}},
};

static const struct Tass17Term IAPETUS_0_12[2] = {
  {{ 2.861462370993798e-06, 2.306043720186055e-01, 5.504846058091149e-01}},
  {{ 1.182264841666031e-06, 2.632429282009304e+00, 5.505036589876786e-01}},
};

static const struct Tass17Term IAPETUS_0_13[1] = {
  {{ 1.228324287730070e-06, 6.138699763037207e+00, 1.102876801637381e+00}},
};

static const struct Tass17Term IAPETUS_0_14[1] = {
  {{ 4.329731043745186e-06, 1.745631218955876e+00, 9.445217655483165e-01}},
};

static const struct Tass17Term IAPETUS_0_15[1] = {
  {{ 9.898839492003675e-07, 5.001271649790288e+00, 8.653251943252205e-01}},
};

static const struct Tass17Term IAPETUS_0_16[1] = {
  {{ 1.430384079869284e-06, 2.331131895479707e-01, 1.259362354064422e+00}},
};

static const struct Tass17Term IAPETUS_0_17[1] = {
  {{ 4.421327735883800e-07, 5.003780467319657e+00, 1.574202942580528e+00}},
};

static const struct Tass17Term IAPETUS_0_18[1] = {
  {{ 3.479701758905445e-05, 2.913621055460106e+00, 1.311651739473694e+00}},
};

static const struct Tass17Term IAPETUS_0_19[2] = {
  {{ 5.667000670514980e-07, 6.227605955410412e+00, 1.470055694737550e+00}},
  {{ 5.801464486513693e-07, 4.405137819662329e+00, 1.470060966588590e+00}},
};

static const struct Tass17Term IAPETUS_0_20[1] = {
  {{ 1.080732052994442e-06, 6.169261486294519e+00, 1.232455168250598e+00}},
};

static const struct Tass17Term IAPETUS_0_21[1] = {
  {{ 1.870537893668915e-05, 4.970065340019012e+00, 2.216515668800783e+00}},
};

static const struct Tass17Term IAPETUS_0_22[1] = {
  {{ 5.433130776641635e-07, 1.942520463673839e+00, 2.137319097577687e+00}},
};

static const struct Tass17Term IAPETUS_0_23[1] = {
  {{ 1.129835111020686e-05, 1.931391320176082e+00, 3.249104467422628e+00}},
};

static const struct Tass17Term IAPETUS_0_24[1] = {
  {{ 8.163925621658264e-07, 3.775239311342985e+00, 4.506334773901807e+00}},
};

static const struct Tass17Term IAPETUS_0_25[1] = {
  {{ 8.563952066234920e-07, 3.157716115970889e+00, 6.587859751149659e+00}},
};

static const struct Tass17MultiTerm IAPETUS_0[26] = {
  {{ 0, 0, 0, 0, 0, 0, 1, 0},  7,IAPETUS_0_0},
  {{ 0, 0, 0, 0, 0, 0, 2, 0}, 17,IAPETUS_0_1},
  {{ 0, 0, 0, 0, 0, 0, 3, 0},  2,IAPETUS_0_2},
  {{ 0, 0, 0, 0, 0,-1, 5, 0}, 13,IAPETUS_0_3},
  {{ 0, 0, 0, 0, 0, 1,-1, 0},  5,IAPETUS_0_4},
  {{ 0, 0, 0, 0, 0, 1, 1, 0},  4,IAPETUS_0_5},
  {{ 0, 0, 0, 0, 0, 1,-2, 0},  7,IAPETUS_0_6},
  {{ 0, 0, 0, 0, 0, 1,-3, 0},  5,IAPETUS_0_7},
  {{ 0, 0, 0, 0, 0, 1,-4, 0},  2,IAPETUS_0_8},
  {{ 0, 0, 0, 0, 0, 2,-1, 0},  1,IAPETUS_0_9},
  {{ 0, 0, 0, 0, 0, 2, 1, 0},  2,IAPETUS_0_10},
  {{ 0, 0, 0, 0, 0, 2,-2, 0},  2,IAPETUS_0_11},
  {{ 0, 0, 0, 0, 0, 2,-3, 0},  2,IAPETUS_0_12},
  {{ 0, 0, 0, 0, 0, 3,-1, 0},  1,IAPETUS_0_13},
  {{ 0, 0, 0, 0, 0, 3,-3, 0},  1,IAPETUS_0_14},
  {{ 0, 0, 0, 0, 0, 3,-4, 0},  1,IAPETUS_0_15},
  {{ 0, 0, 0, 0, 0, 4,-4, 0},  1,IAPETUS_0_16},
  {{ 0, 0, 0, 0, 0, 5,-5, 0},  1,IAPETUS_0_17},
  {{ 0, 0, 0, 0, 1, 0,-1, 0},  1,IAPETUS_0_18},
  {{ 0, 0, 0, 0, 1, 0, 1, 0},  2,IAPETUS_0_19},
  {{ 0, 0, 0, 0, 1, 0,-2, 0},  1,IAPETUS_0_20},
  {{ 0, 0, 0, 1, 0, 0,-1, 0},  1,IAPETUS_0_21},
  {{ 0, 0, 0, 1, 0, 0,-2, 0},  1,IAPETUS_0_22},
  {{ 0, 0, 1, 0, 0, 0,-1, 0},  1,IAPETUS_0_23},
  {{ 0, 1, 0, 0, 0, 0,-1, 0},  1,IAPETUS_0_24},
  {{ 1, 0, 0, 0, 0, 0,-1, 0},  1,IAPETUS_0_25},
};

static const struct Tass17Term IAPETUS_1_0[22] = {
  {{ 1.928386916598716e-01, 1.316743285171985e+00, 5.271851040460129e-06}},
  {{ 1.197700127979115e-03, 2.905816296579413e+00, 1.054370208092026e-05}},
  {{ 1.785444267015284e-04, 3.897159948941672e+00, 1.894532180866185e-05}},
  {{ 7.466410913376219e-04, 3.411600296951643e+00, 2.445240135601595e-05}},
  {{ 4.076652529480639e-05, 3.776409831511328e+00, 2.966566038404764e-05}},
  {{ 3.620254646473473e-05, 6.177614480595389e-01, 3.510481004211992e-05}},
  {{ 2.023635272434820e-05, 5.342652755830533e+00, 4.891917479115332e-05}},
  {{ 2.148685811756090e-05, 3.324109814821024e+00, 5.483417106242491e-05}},
  {{ 2.377227624569038e-05, 3.239254695447169e+00, 2.825103078658793e-04}},
  {{ 4.028991254502335e-05, 1.095989272623461e+00, 5.652783908051565e-04}},
  {{ 5.283085924212965e-05, 6.218635238921890e+00, 5.787093127593016e-04}},
  {{ 1.125807495942191e-03, 4.414124516924016e+00, 5.839811637997618e-04}},
  {{ 3.586013723690129e-05, 2.436062381089369e+00, 5.892530148402220e-04}},
  {{ 7.459887995646048e-06, 4.087393199562610e+00, 6.026839367943671e-04}},
  {{ 1.314054216207817e-05, 5.873663787672587e+00, 8.662336965922434e-04}},
  {{ 1.603086203623188e-05, 2.485710326866144e+00, 1.157149509935537e-03}},
  {{ 2.400461005136443e-04, 2.926349976058209e+00, 1.167962327599524e-03}},
  {{ 3.004295676379663e-04, 4.182760326808072e+00, 1.173234178639984e-03}},
  {{ 7.393695567545098e-05, 5.503253064658139e+00, 1.178506029680444e-03}},
  {{ 3.490441837523331e-05, 4.159104921593594e+00, 1.751943491399285e-03}},
  {{ 3.811816549252109e-05, 5.449835843337831e+00, 1.757215342439746e-03}},
  {{ 9.878270828624305e-06, 4.692950541308903e-01, 1.762487193480206e-03}},
};

static const struct Tass17Term IAPETUS_1_1[24] = {
  {{ 6.143353968708967e-07, 1.287562896667545e+00, 7.687145945601632e-02}},
  {{ 4.523919915777730e-06, 2.502659168377380e+00, 7.745544060220222e-02}},
  {{ 3.714286018866130e-07,-1.141689927113436e+00, 7.802333707968367e-02}},
  {{ 5.608496749491884e-07, 1.852697585028197e-01, 7.802860893072413e-02}},
  {{ 3.504383346317222e-07, 2.398679759866219e+00, 7.803414989734765e-02}},
  {{ 2.295038701669439e-05,-2.593118006894055e+00, 7.803942174838812e-02}},
  {{ 6.404009664458889e-07, 1.893523997409873e+00, 7.804469359942858e-02}},
  {{ 8.091467686328637e-07, 6.119791853597793e+00, 7.805847492695170e-02}},
  {{ 4.190958483159967e-07, 1.812437113434972e+00, 7.861259007691003e-02}},
  {{ 6.286284677985618e-07, 1.791293531808356e+00, 7.862340289457402e-02}},
  {{ 3.164931584142412e-07, 5.589825798207451e+00, 7.917224619349188e-02}},
  {{ 4.353564226390001e-06, 6.257381704689877e-01, 7.917751804453235e-02}},
  {{ 3.164896582547253e-07, 1.944834709464376e+00, 7.918278989557279e-02}},
  {{ 1.404022491486175e-07,-1.433272645194262e+00, 7.919129937205547e-02}},
  {{ 3.794706462471952e-06, 3.027544876345173e+00, 7.919657122309593e-02}},
  {{ 3.059127278557355e-07, 3.041311214209734e+00, 7.919657125832364e-02}},
  {{ 1.407650236154023e-07, 1.204832936646125e+00, 7.920184307413637e-02}},
  {{ 8.733847339265118e-07,-2.854848402830646e+00, 7.920738404075992e-02}},
  {{ 6.190372619937466e-07,-1.535786959595905e+00, 7.921265589180036e-02}},
  {{ 3.845413058184980e-07, 5.829842020175415e+00, 7.922643721932350e-02}},
  {{ 3.936684188721603e-07, 4.007371902635274e+00, 7.923170907036394e-02}},
  {{ 4.101604037784085e-07, 4.242652639255375e+00, 7.978055236928183e-02}},
  {{ 4.086365824538882e-08,-4.056548501884276e-01, 8.036453351546773e-02}},
  {{ 4.231066732027226e-07, 4.057026289098603e+00, 8.036980536650817e-02}},
};

static const struct Tass17Term IAPETUS_1_2[16] = {
  {{ 4.416345379733718e-07, 6.241592663692312e+00, 1.554840495329263e-01}},
  {{ 3.837882772948628e-06, 1.173503571344250e+00, 1.560680306791122e-01}},
  {{ 4.360169268782757e-07, 1.069524219711398e+00, 1.566467399742577e-01}},
  {{ 2.721985372704609e-05, 2.388610893433953e+00, 1.566520118252981e-01}},
  {{ 1.203630670831745e-06, 5.660322567848149e-01, 1.566572836763386e-01}},
  {{ 2.227388462533576e-06, 2.284631982621601e+00, 1.572307211204436e-01}},
  {{ 1.378064515172981e-04,-2.679466414745420e+00, 1.572359929714840e-01}},
  {{ 5.768343947193972e-06, 1.781127439794527e+00, 1.572412648225245e-01}},
  {{ 4.253374565827929e-06, 1.704388717616716e+00, 1.578199741176699e-01}},
  {{ 3.291231434148502e-07,-2.864966874148501e+00, 1.578252459687104e-01}},
  {{ 4.141402947439892e-07,-2.965677063355761e+00, 1.584039552638558e-01}},
  {{ 6.393990743929588e-06,-1.649960073803633e+00, 1.584092271148963e-01}},
  {{ 2.790880384447099e-07,-1.753655473493772e+00, 1.589879364100418e-01}},
  {{ 3.256983864383599e-07,-4.347514166563534e-01, 1.589932082610822e-01}},
  {{ 2.267895350910583e-07, 3.024427904422583e+00, 1.595719175562276e-01}},
  {{ 1.090721316940875e-07,-1.939541945846773e+00, 1.595771894072681e-01}},
};

static const struct Tass17Term IAPETUS_1_3[2] = {
  {{ 4.889359735634015e-07, 5.416167758678300e+00, 2.358485830483941e-01}},
  {{ 2.502679246525132e-06, 3.480902144089144e-01, 2.364325641945799e-01}},
};

static const struct Tass17Term IAPETUS_1_4[75] = {
  {{ 5.227495470767936e-07, 4.005663121141773e+00, 2.153785202069178e-04}},
  {{ 5.262843436434110e-07, 2.183142522975625e+00, 2.206503712473779e-04}},
  {{ 3.815771898913724e-07, 7.105474585412466e-02, 7.504405092155264e-04}},
  {{ 9.196878775751337e-07, 2.473094150998370e+00, 7.694936877791099e-04}},
  {{ 1.352657464923885e-06, 2.873044246870902e+00, 7.803065054430962e-04}},
  {{ 3.973724314685021e-07, 4.140593867861234e-01, 7.832750153022327e-04}},
  {{ 1.368009711983665e-06, 1.050560920047950e+00, 7.855783564835565e-04}},
  {{ 3.472239690061985e-07, 1.735637572758064e+00, 7.885468663426930e-04}},
  {{ 3.480378535316622e-07, 5.511199893486148e+00, 7.908502075240164e-04}},
  {{ 3.033297996664303e-07, 6.195403829536804e+00, 7.938187173831529e-04}},
  {{ 3.418187177414854e-07, 8.141915154020281e-01, 7.940878329662196e-04}},
  {{ 2.688244928253836e-06, 5.275014486169524e+00, 7.993596840066795e-04}},
  {{ 2.569260340418719e-06, 3.452633751557773e+00, 8.046315350471398e-04}},
  {{ 1.036863039010391e-06, 1.630081068326518e+00, 8.099033860875997e-04}},
  {{ 3.969393749935304e-07, 8.738771354003039e-01, 1.869483662023897e-03}},
  {{ 3.985256112150306e-06, 3.275634661668579e+00, 1.888536840587480e-03}},
  {{ 1.511339171430308e-06, 4.356355818907831e+00, 1.902318168110603e-03}},
  {{ 1.475074400837350e-05, 5.719796115875051e+00, 1.907590019151063e-03}},
  {{ 1.216707168547837e-06, 7.209469663619112e-01, 1.912861870191523e-03}},
  {{ 3.075011314359275e-07, 3.644290442329441e+00, 1.912996427983057e-03}},
  {{ 1.415742853315232e-06, 4.757124368944763e+00, 1.913130985774590e-03}},
  {{ 3.814316630052462e-07, 3.288005910176154e+00, 1.918275465063078e-03}},
  {{ 1.179423115340822e-05, 6.061718986149902e+00, 1.918402836815050e-03}},
  {{ 1.876817015958310e-06, 4.754449361520011e-01, 1.921371346674187e-03}},
  {{ 1.399238708843930e-05, 4.239759216446882e+00, 1.923674687855510e-03}},
  {{ 2.186693735333124e-05, 1.995274806149824e+00, 1.926643197714647e-03}},
  {{ 2.330447358044871e-06, 2.432875200820899e+00, 1.928946538895970e-03}},
  {{ 8.755936386396388e-07, 3.115210303371935e+00, 1.931915048755107e-03}},
  {{ 2.254199615438521e-06, 6.141816327520342e+00, 1.932049606546640e-03}},
  {{ 5.971965491672709e-06, 8.753490785501354e-01, 1.932184164338173e-03}},
  {{ 2.941694888718857e-07, 3.751626473924576e+00, 1.934218389936430e-03}},
  {{ 4.077887373315127e-07, 5.689822989499520e+00, 1.937328643626661e-03}},
  {{ 7.393168775867132e-05, 2.280321547311384e+00, 1.937456015378634e-03}},
  {{ 8.171736433893184e-05, 4.510335354461255e-01, 1.942727866419093e-03}},
  {{ 6.144966414607280e-07, 2.242399219818416e-02, 1.942855238171066e-03}},
  {{ 3.223198173228014e-06, 2.430574184290058e-01, 1.942862424210627e-03}},
  {{ 8.603158836863158e-07, 4.197919449469832e+00, 1.945696376278230e-03}},
  {{ 1.704559011257133e-05,-1.417339665945397e+00, 1.947999717459554e-03}},
  {{ 3.140486507800807e-07, 4.482050013324328e+00, 1.948127089211526e-03}},
  {{ 3.299741975330665e-06, 4.703710238329596e+00, 1.948134275251087e-03}},
  {{ 4.531105084562628e-07, 6.090353248906123e+00, 1.948141461290647e-03}},
  {{ 2.574587837723788e-06, 5.732598943257583e+00, 1.948268833042620e-03}},
  {{ 5.336777443382800e-06, 3.277820916335360e+00, 1.951237342901756e-03}},
  {{ 1.915584168637660e-06, 6.159110785889812e+00, 1.953271568500014e-03}},
  {{ 4.614651554236785e-07, 4.267748340354923e+00, 1.953413312331107e-03}},
  {{ 1.540440893862750e-06,-2.380019622661880e+00, 1.953540684083080e-03}},
  {{ 1.101769834711451e-04,-1.556727729554004e+00, 1.956509193942217e-03}},
  {{ 3.070012836628351e-07, 5.229794370811791e+00, 1.958812535123540e-03}},
  {{ 1.189228383255110e-04, 2.899805052106753e+00, 1.961781044982677e-03}},
  {{ 2.613243661620941e-06, 2.424238476169638e+00, 1.961908416734649e-03}},
  {{ 8.507522187621048e-06, 3.014198965814164e+00, 1.961915602774210e-03}},
  {{ 5.610129338334513e-07, 3.684122942793509e+00, 1.962050160565743e-03}},
  {{ 3.066646802040954e-05, 1.120589496342319e+00, 1.967052896023137e-03}},
  {{ 9.616747267908099e-07, 6.000060288823683e-01, 1.967180267775110e-03}},
  {{ 8.812132213666628e-06, 1.185788445220376e+00, 1.967187453814670e-03}},
  {{ 3.612477825551320e-05,-1.345764111856863e+00, 1.967322011606203e-03}},
  {{ 1.191488030171526e-06, 2.272066757023249e+00, 1.972324747063597e-03}},
  {{ 2.549155479604930e-06, 5.291014601965657e+00, 1.972459304855130e-03}},
  {{ 9.027685330072385e-05,-3.132006983447723e+00, 1.972593862646663e-03}},
  {{ 5.273506284484312e-07, 7.170338045570530e-01, 1.975562372505800e-03}},
  {{ 7.318838264333341e-05, 1.353835293153985e+00, 1.977865713687124e-03}},
  {{ 5.398713831902744e-07, 5.177746978176391e+00, 1.980834223546260e-03}},
  {{ 3.608057298728251e-06, 4.863591915141639e+00, 1.980961595298232e-03}},
  {{ 2.328751291389269e-05, 5.814424528754509e+00, 1.983137564727584e-03}},
  {{ 1.576239906932032e-06, 3.002295562826074e+00, 1.986233446338693e-03}},
  {{ 3.799033641308746e-07, 4.259130445062984e+00, 1.986375190169786e-03}},
  {{ 3.242171175776111e-07, 3.907553066682691e+00, 1.986502561921759e-03}},
  {{ 1.499811815760443e-06, 3.988002288677764e+00, 1.988409415768044e-03}},
  {{ 5.627708927914463e-07, 5.539620791805293e+00, 1.991647041210247e-03}},
  {{ 2.042389544816678e-06, 5.291730688852117e+00, 1.991774412962219e-03}},
  {{ 1.460797854077857e-06, 3.748973474201665e+00, 1.996918892250707e-03}},
  {{ 6.197295348292014e-06, 3.290476399266866e+00, 1.997046264002680e-03}},
  {{ 1.095526621882867e-06, 1.933069997683063e+00, 2.002190743291167e-03}},
  {{ 2.089500807780357e-06, 1.580270469414437e+00, 2.002318115043139e-03}},
  {{ 3.187588107072781e-07, 6.035967805962264e+00, 2.007589966083600e-03}},
};

static const struct Tass17Term IAPETUS_1_5[5] = {
  {{ 3.414474608395144e-07, 5.333092134829300e+00, 3.928800102292660e-01}},
  {{ 3.741206194503089e-07, 3.334420556287444e+00, 3.940318878881610e-01}},
  {{ 2.637183491724404e-05, 4.683474110615003e+00, 3.940371597392014e-01}},
  {{ 8.279548593975801e-07, 1.908398224503419e+00, 3.940479725568654e-01}},
  {{ 6.484210285032441e-07, 3.233671623826318e+00, 3.940532444079058e-01}},
};

static const struct Tass17Term IAPETUS_1_6[4] = {
  {{ 2.940137697333849e-06, 3.451571632901731e+00, 3.148353166650650e-01}},
  {{ 7.282549735894645e-04, 4.770667297372782e+00, 3.148405885161054e-01}},
  {{ 3.358151723062283e-06, 6.089767182911175e+00, 3.148458603671461e-01}},
  {{ 3.257749851873783e-07, 1.856841516027055e+00, 3.148650409174615e-01}},
};

static const struct Tass17Term IAPETUS_1_7[4] = {
  {{ 9.488209354297983e-06, 4.943057563632108e+00, 4.732445437799614e-01}},
  {{ 1.029453478899256e-05, 3.120590331336322e+00, 4.732498156310018e-01}},
  {{ 2.792746040615356e-06, 1.297967072116361e+00, 4.732550874820422e-01}},
  {{ 4.374967901448417e-07, 5.170827648758129e+00, 4.732689961813175e-01}},
};

static const struct Tass17Term IAPETUS_1_8[2] = {
  {{ 6.013024933189873e-06, 1.743122401426509e+00, 2.356440172930096e-01}},
  {{ 4.958075120563441e-06, 4.144929329850513e+00, 2.356630704715732e-01}},
};

static const struct Tass17Term IAPETUS_1_9[4] = {
  {{ 7.533074112851135e-07, 1.960182176247418e+00, 1.564260895501687e-01}},
  {{ 2.517101053299125e-06, 1.375551717917236e-01, 1.564313614012092e-01}},
  {{ 2.480836668098538e-06, 4.598277210434070e+00, 1.564366332522496e-01}},
  {{ 1.143486543010702e-06, 1.117389755399236e+00, 1.564664992484772e-01}},
};

static const struct Tass17Term IAPETUS_1_10[5] = {
  {{ 3.574500970589592e-07, 5.215822607081832e+00, 7.722951832707283e-02}},
  {{ 1.392818914226552e-06, 3.393198290814080e+00, 7.723479017811326e-02}},
  {{ 1.360536764325014e-06, 1.570730155065997e+00, 7.724006202915373e-02}},
  {{ 4.891864753374598e-07, 5.795005219238083e+00, 7.725384335667685e-02}},
  {{ 4.778483243651071e-07, 3.972537083490000e+00, 7.725911520771732e-02}},
};

static const struct Tass17Term IAPETUS_1_11[3] = {
  {{ 7.489157248963648e-07, 6.282589894747257e-01, 7.880552662998393e-01}},
  {{ 7.443531196416904e-07, 3.430541494724291e+00, 7.880851322960668e-01}},
  {{ 7.620147658522374e-07, 1.608073358976210e+00, 7.880904041471072e-01}},
};

static const struct Tass17Term IAPETUS_1_12[1] = {
  {{ 2.222879678616121e-05, 7.364334077752746e-01, 7.088586950767433e-01}},
};

static const struct Tass17Term IAPETUS_1_13[3] = {
  {{ 6.546227444825897e-07, 1.939050585742052e+00, 6.296759051811704e-01}},
  {{ 1.769563612146425e-05, 3.258149248363778e+00, 6.296811770322111e-01}},
  {{ 6.546227444825897e-07, 4.577247910985506e+00, 6.296864488832515e-01}},
};

static const struct Tass17Term IAPETUS_1_14[2] = {
  {{ 2.569819239848606e-06, 2.306043720186055e-01, 5.504846058091149e-01}},
  {{ 1.150895741388962e-06, 2.632411300442608e+00, 5.505036589876786e-01}},
};

static const struct Tass17Term IAPETUS_1_15[2] = {
  {{ 3.716648744882634e-07, 4.908225137751350e+00, 4.712719499173147e-01}},
  {{ 3.630505879727552e-07, 3.085757002003267e+00, 4.712772217683551e-01}},
};

static const struct Tass17Term IAPETUS_1_16[2] = {
  {{ 7.410458807790901e-07, 2.997108078146744e+00, 1.102876801637381e+00}},
  {{ 2.917099551383937e-07, 9.555532956830555e-02, 1.102930992663213e+00}},
};

static const struct Tass17Term IAPETUS_1_17[1] = {
  {{ 3.001367588499240e-07, 5.512961820466838e+00, 1.023699283592849e+00}},
};

static const struct Tass17Term IAPETUS_1_18[1] = {
  {{ 4.216532478702710e-06, 1.745631218955876e+00, 9.445217655483165e-01}},
};

static const struct Tass17Term IAPETUS_1_19[2] = {
  {{ 8.149891095115382e-07, 5.001271649790290e+00, 8.653251943252205e-01}},
  {{ 3.764251887678508e-07, 1.119909356269040e+00, 8.653442475037843e-01}},
};

static const struct Tass17Term IAPETUS_1_20[1] = {
  {{ 1.310246341606266e-06, 2.331131895479707e-01, 1.259362354064422e+00}},
};

static const struct Tass17Term IAPETUS_1_21[1] = {
  {{ 3.830879278957400e-07, 5.003780467319657e+00, 1.574202942580528e+00}},
};

static const struct Tass17Term IAPETUS_1_22[1] = {
  {{ 7.277204919188508e-07, 5.941165931805280e+00, 1.390848310696790e+00}},
};

static const struct Tass17Term IAPETUS_1_23[1] = {
  {{ 2.181721459549961e-05, 6.055213709049899e+00, 1.311651739473694e+00}},
};

static const struct Tass17Term IAPETUS_1_24[1] = {
  {{ 3.811901873161304e-07, 1.714424909184600e+00, 2.295712240023879e+00}},
};

static const struct Tass17Term IAPETUS_1_25[1] = {
  {{ 1.184715424009358e-05, 1.828472686429220e+00, 2.216515668800783e+00}},
};

static const struct Tass17Term IAPETUS_1_26[1] = {
  {{ 7.269853312811282e-06, 5.072983973765877e+00, 3.249104467422628e+00}},
};

static const struct Tass17Term IAPETUS_1_27[1] = {
  {{ 5.287253161661598e-07, 6.336466577531914e-01, 4.506334773901807e+00}},
};

static const struct Tass17Term IAPETUS_1_28[1] = {
  {{ 5.608837306250120e-07, 1.612346238109583e-02, 6.587859751149659e+00}},
};

static const struct Tass17MultiTerm IAPETUS_1[29] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 22,IAPETUS_1_0},
  {{ 0, 0, 0, 0, 0, 0, 1, 0}, 24,IAPETUS_1_1},
  {{ 0, 0, 0, 0, 0, 0, 2, 0}, 16,IAPETUS_1_2},
  {{ 0, 0, 0, 0, 0, 0, 3, 0},  2,IAPETUS_1_3},
  {{ 0, 0, 0, 0, 0,-1, 5, 0}, 75,IAPETUS_1_4},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  5,IAPETUS_1_5},
  {{ 0, 0, 0, 0, 0, 1,-1, 0},  4,IAPETUS_1_6},
  {{ 0, 0, 0, 0, 0, 1, 1, 0},  4,IAPETUS_1_7},
  {{ 0, 0, 0, 0, 0, 1,-2, 0},  2,IAPETUS_1_8},
  {{ 0, 0, 0, 0, 0, 1,-3, 0},  4,IAPETUS_1_9},
  {{ 0, 0, 0, 0, 0, 1,-4, 0},  5,IAPETUS_1_10},
  {{ 0, 0, 0, 0, 0, 2, 0, 0},  3,IAPETUS_1_11},
  {{ 0, 0, 0, 0, 0, 2,-1, 0},  1,IAPETUS_1_12},
  {{ 0, 0, 0, 0, 0, 2,-2, 0},  3,IAPETUS_1_13},
  {{ 0, 0, 0, 0, 0, 2,-3, 0},  2,IAPETUS_1_14},
  {{ 0, 0, 0, 0, 0, 2,-4, 0},  2,IAPETUS_1_15},
  {{ 0, 0, 0, 0, 0, 3,-1, 0},  2,IAPETUS_1_16},
  {{ 0, 0, 0, 0, 0, 3,-2, 0},  1,IAPETUS_1_17},
  {{ 0, 0, 0, 0, 0, 3,-3, 0},  1,IAPETUS_1_18},
  {{ 0, 0, 0, 0, 0, 3,-4, 0},  2,IAPETUS_1_19},
  {{ 0, 0, 0, 0, 0, 4,-4, 0},  1,IAPETUS_1_20},
  {{ 0, 0, 0, 0, 0, 5,-5, 0},  1,IAPETUS_1_21},
  {{ 0, 0, 0, 0, 1, 0, 0, 0},  1,IAPETUS_1_22},
  {{ 0, 0, 0, 0, 1, 0,-1, 0},  1,IAPETUS_1_23},
  {{ 0, 0, 0, 1, 0, 0, 0, 0},  1,IAPETUS_1_24},
  {{ 0, 0, 0, 1, 0, 0,-1, 0},  1,IAPETUS_1_25},
  {{ 0, 0, 1, 0, 0, 0,-1, 0},  1,IAPETUS_1_26},
  {{ 0, 1, 0, 0, 0, 0,-1, 0},  1,IAPETUS_1_27},
  {{ 1, 0, 0, 0, 0, 0,-1, 0},  1,IAPETUS_1_28},
};

static const struct Tass17Term IAPETUS_2_0[24] = {
  {{ 7.357194732163894e-04, 6.035829207568037e+00,-5.406408831993331e-06}},
  {{ 2.935648107313708e-02, 3.358649794029711e+00, 5.406408831993331e-06}},
  {{ 6.699050371985383e-04, 4.753014210656826e+00,-1.067825987245346e-05}},
  {{ 4.151869133102979e-04, 4.743964405299185e+00, 1.067825987245346e-05}},
  {{ 1.899854046059933e-04, 3.517567007545812e+00,-1.595011091291359e-05}},
  {{ 1.991669418726220e-04, 4.485476749277378e+00, 1.918773635511653e-05}},
  {{ 1.011804925675502e-04, 3.745348713034283e+00,-2.445958739557666e-05}},
  {{ 9.953584143544754e-04, 5.831603333509536e+00, 2.445958739557666e-05}},
  {{ 6.928098928797385e-05, 5.576141930376416e+00,-2.973143843603680e-05}},
  {{ 1.234894258008271e-04, 8.652949986394700e-01, 2.973143843603680e-05}},
  {{ 2.082728796332944e-05, 1.173870037237461e+00,-3.500328947649692e-05}},
  {{ 3.040421176914689e-07, 2.136467296916470e+00, 5.595215764041851e-04}},
  {{ 1.287853944279400e-05, 5.287782933428181e+00,-5.785747549677684e-04}},
  {{ 2.039385297517835e-05, 4.585155128610154e+00, 5.785747549677684e-04}},
  {{ 1.595251648310000e-05, 6.234432051555658e+00, 5.839811637997618e-04}},
  {{ 1.343153395169566e-05, 4.649306306086897e+00, 5.893875726317552e-04}},
  {{ 1.799640976047236e-05, 2.341543068636097e-01, 1.143502740203947e-03}},
  {{ 1.130469268372695e-05, 4.574612374891347e+00, 1.157149509935537e-03}},
  {{ 3.789273029962630e-04, 2.745292826757329e+00, 1.162555918767530e-03}},
  {{ 1.065311292483293e-05, 3.513635458424760e-03, 1.173234178639984e-03}},
  {{ 6.594412507279831e-06, 1.160392236116490e+00, 1.178640587471977e-03}},
  {{ 4.598867989379571e-07, 1.609956090712706e+00, 1.727483904003709e-03}},
  {{ 4.889687732101663e-05, 4.014982830991590e+00, 1.746537082567292e-03}},
  {{ 5.041791242559473e-06, 5.286783389563643e+00, 2.330518246367054e-03}},
};

static const struct Tass17Term IAPETUS_2_1[17] = {
  {{ 3.827903648104360e-07, 3.349310327302221e+00,-7.628207190099841e-02}},
  {{ 3.347616463036283e-06, 2.134214888660969e+00,-7.686605304718431e-02}},
  {{ 4.386017871369636e-07, 2.238193464103548e+00,-7.744476234232975e-02}},
  {{ 2.330306527936928e-05, 9.191067654693918e-01,-7.745003419337021e-02}},
  {{ 9.621927111300821e-07, 2.741692065655550e+00,-7.745530604441068e-02}},
  {{ 2.236294716863211e-06, 1.023085701193345e+00,-7.802874348851565e-02}},
  {{ 1.188902959997339e-04,-2.960003180568126e-01,-7.803401533955611e-02}},
  {{ 5.159571361902864e-06, 1.526594756217454e+00,-7.803928719059658e-02}},
  {{ 3.648905338577408e-06, 1.607558302204549e+00,-7.861799648574201e-02}},
  {{ 2.763121546924847e-07,-1.105228500517151e-01,-7.862326833678248e-02}},
  {{ 2.374053234088113e-06,-6.755268368608213e-03,-7.920197763192791e-02}},
  {{ 2.809879517690395e-06,-1.325635285498895e+00,-7.920724948296838e-02}},
  {{ 7.601575967574296e-07, 3.638825235880475e+00,-7.921252133400884e-02}},
  {{ 2.011807388354464e-07,-1.221851342565206e+00,-7.978595877811381e-02}},
  {{ 2.722673162134422e-07,-2.540738375872117e+00,-7.979123062915428e-02}},
  {{ 2.341140383006404e-07, 2.833112279003631e-01,-8.036993992429971e-02}},
  {{ 1.079744827006783e-07,-1.035863524313422e+00,-8.037521177534018e-02}},
};

static const struct Tass17Term IAPETUS_2_2[15] = {
  {{ 5.077127357111450e-07, 1.063128508151314e+00, 7.744476234232975e-02}},
  {{ 1.216563739390478e-07,-7.586058089542492e-01, 7.745003419337021e-02}},
  {{ 3.788966811408025e-07, 9.593705739442162e-01, 7.802347163747521e-02}},
  {{ 8.559499914533230e-07, 2.278052169500489e+00, 7.802874348851565e-02}},
  {{ 9.792124873256762e-07,-2.686078783286550e+00, 7.803401533955611e-02}},
  {{ 2.075828194269492e-06, 2.092609920904744e+00, 7.861799648574201e-02}},
  {{ 3.015291948812891e-06,-1.152929578102659e+00, 7.919670578088745e-02}},
  {{ 2.453612858406647e-04, 1.661250302251527e-01, 7.920197763192791e-02}},
  {{ 3.267848713606678e-06, 1.485175738282932e+00, 7.920724948296838e-02}},
  {{ 2.045440860185655e-06,-1.760359860454440e+00, 7.978595877811381e-02}},
  {{ 9.507520895249895e-07, 3.018328843736854e+00, 8.036993992429971e-02}},
  {{ 8.310721660889963e-07,-1.945802109050186e+00, 8.037521177534018e-02}},
  {{ 3.678842089652509e-07, 5.656064793685675e+00, 8.038048362638064e-02}},
  {{ 1.163907976076402e-07, 1.090855869404554e+00, 8.095392107048561e-02}},
  {{ 4.857377246389406e-07, 5.552306859478579e+00, 8.095919292152608e-02}},
};

static const struct Tass17Term IAPETUS_2_3[1] = {
  {{ 1.155571530124003e-06, 2.959627469406032e+00,-1.572305865626520e-01}},
};

static const struct Tass17Term IAPETUS_2_4[4] = {
  {{ 3.386727736182448e-07, 5.810388343392691e+00, 1.566574182341301e-01}},
  {{ 1.733359910757746e-06, 7.423107991233067e-01, 1.572413993803160e-01}},
  {{ 4.262641945379126e-06, 7.918495104173626e-01, 1.583794956764603e-01}},
  {{ 8.024442052834091e-06, 3.193669906570326e+00, 1.583985488550238e-01}},
};

static const struct Tass17Term IAPETUS_2_5[8] = {
  {{ 3.591607352372491e-07, 4.481232803237869e+00, 2.352700083110402e-01}},
  {{ 2.976455752240285e-06,-5.868558890738737e-01, 2.358539894572260e-01}},
  {{ 1.320515754572473e-05, 6.282501451704030e-01, 2.364379706034119e-01}},
  {{ 8.543461484774715e-07, 5.088883400685836e+00, 2.364432424544524e-01}},
  {{ 3.770599260080091e-07, 4.984963438378681e+00, 2.370219517495979e-01}},
  {{ 1.426804754726213e-06, 3.386073918171003e-01, 2.376059328957838e-01}},
  {{ 1.112281761662677e-06,-1.484029309805413e+00, 2.376112047468242e-01}},
  {{ 3.039469315779643e-07, 2.976610131749417e+00, 2.376164765978647e-01}},
};

static const struct Tass17Term IAPETUS_2_6[1] = {
  {{ 5.756485350981315e-07, 3.655807898223860e+00, 3.156345418265079e-01}},
};

static const struct Tass17Term IAPETUS_2_7[4] = {
  {{ 9.807167381341471e-06, 4.647844982264678e+00,-3.940425661480334e-01}},
  {{ 1.050421274404971e-05, 1.871271231028191e-01,-3.940478379990738e-01}},
  {{ 2.816558865567409e-06, 2.009743661307299e+00,-3.940531098501144e-01}},
  {{ 3.824414539954584e-07, 4.420210596752325e+00,-3.940670185493895e-01}},
};

static const struct Tass17Term IAPETUS_2_8[3] = {
  {{ 3.660318350154349e-07, 4.194707709203449e+00,-3.148161289287099e-01}},
  {{ 1.034350804286399e-05, 1.792690836877676e+00,-3.148351821072734e-01}},
  {{ 3.362081661163648e-07, 4.540912112901320e+00,-3.148459949249374e-01}},
};

static const struct Tass17Term IAPETUS_2_9[3] = {
  {{ 1.891655464023067e-06, 2.997740360570593e+00,-2.356333390331371e-01}},
  {{ 2.738557431501198e-04, 1.678643034250733e+00,-2.356386108841776e-01}},
  {{ 2.085092385794607e-06, 3.595434709071512e-01,-2.356438827352181e-01}},
};

static const struct Tass17Term IAPETUS_2_10[4] = {
  {{ 3.297148610970001e-07, 3.283240059277484e+00,-1.564259549923772e-01}},
  {{ 4.210058117501252e-07, 1.957966659954585e+00,-1.564312268434177e-01}},
  {{ 1.295232370537576e-05, 4.706187935978230e+00,-1.564420396610816e-01}},
  {{ 3.074615033071044e-06, 5.446004958756377e+00,-1.564610928396452e-01}},
};

static const struct Tass17Term IAPETUS_2_11[5] = {
  {{ 5.503284015527491e-07, 1.347535507567527e+00,-7.722411191824082e-02}},
  {{ 2.144377111729538e-06, 3.170159823835280e+00,-7.722938376928129e-02}},
  {{ 2.094675673402415e-06, 4.992627959583361e+00,-7.723465562032175e-02}},
  {{ 4.509478629698950e-07, 1.450547484355915e+00,-7.724546843798574e-02}},
  {{ 9.711472430656398e-07, 2.190338602286141e+00,-7.726452161654932e-02}},
};

static const struct Tass17Term IAPETUS_2_12[14] = {
  {{ 1.580459593848266e-06, 5.957697356265878e+00, 1.912996427983057e-03}},
  {{ 3.059086093929264e-06, 2.076274581959904e+00, 1.932049606546640e-03}},
  {{ 4.097571772366432e-06, 2.476739673478372e+00, 1.942862424210627e-03}},
  {{ 4.622849445770262e-06, 6.542846935894391e-01, 1.948134275251087e-03}},
  {{ 4.098021786747177e-07, 4.478092318805028e+00, 1.951102785110223e-03}},
  {{ 9.865359954273804e-07, 5.114846914000389e+00, 1.953406126291547e-03}},
  {{ 6.577318919877076e-07, 3.559551940631996e+00, 1.956643751733750e-03}},
  {{ 1.141609439284819e-05, 4.878643124436238e+00, 1.961915602774210e-03}},
  {{ 1.246701893241003e-05, 3.056131321602242e+00, 1.967187453814670e-03}},
  {{ 2.869127018959028e-06, 1.233485437260538e+00, 1.972459304855130e-03}},
  {{ 5.346245981429982e-07, 2.130300142857571e+00, 1.972728420438197e-03}},
  {{ 2.905389353328124e-07, 1.636237329753522e+00, 1.983272122519117e-03}},
  {{ 5.913037768933752e-07, 5.106224293368807e+00, 1.986368004130226e-03}},
  {{ 3.023788452701212e-07, 3.282662154041165e+00, 1.991639855170686e-03}},
};

static const struct Tass17Term IAPETUS_2_13[4] = {
  {{ 5.121289657996665e-06, 3.617695571228794e+00, 3.940372942969930e-01}},
  {{ 5.937514602713236e-04, 4.936792335851361e+00, 3.940425661480334e-01}},
  {{ 5.590721162724015e-06, 6.255890970618563e+00, 3.940478379990738e-01}},
  {{ 3.508789971395948e-07, 2.022944114109337e+00, 3.940670185493895e-01}},
};

static const struct Tass17Term IAPETUS_2_14[1] = {
  {{ 3.500731595844181e-06, 1.909247431651662e+00, 3.148459949249374e-01}},
};

static const struct Tass17Term IAPETUS_2_15[1] = {
  {{ 2.223396437049981e-06, 4.822744530752217e+00, 4.732391373711294e-01}},
};

static const struct Tass17Term IAPETUS_2_16[3] = {
  {{ 1.903491726480552e-06, 1.967591900767555e+00, 5.524465214118891e-01}},
  {{ 1.948656845764241e-06, 1.451237650194714e-01, 5.524517932629298e-01}},
  {{ 5.000991669041441e-07, 4.605684755931306e+00, 5.524570651139702e-01}},
};

static const struct Tass17Term IAPETUS_2_17[2] = {
  {{ 3.067385455485728e-07, 2.393030894759276e+00,-7.880606727086713e-01}},
  {{ 3.140166875052082e-07, 4.215499030507360e+00,-7.880659445597117e-01}},
};

static const struct Tass17Term IAPETUS_2_18[1] = {
  {{ 7.066767025578607e-06, 5.707015564669242e+00,-6.296567174448153e-01}},
};

static const struct Tass17Term IAPETUS_2_19[4] = {
  {{ 3.279448099865014e-07, 2.451375363875169e+00,-5.504601462217194e-01}},
  {{ 3.126490699275733e-07, 1.368667098072893e+00,-5.504739275492424e-01}},
  {{ 1.048627591564010e-05, 4.956843545116611e-02,-5.504791994002831e-01}},
  {{ 3.126490699275733e-07, 5.013655080009026e+00,-5.504844712513235e-01}},
};

static const struct Tass17Term IAPETUS_2_20[2] = {
  {{ 1.831410067152233e-06, 3.077113311796339e+00,-4.712826281771871e-01}},
  {{ 7.115752344230280e-07, 6.753063833723361e-01,-4.713016813557509e-01}},
};

static const struct Tass17Term IAPETUS_2_21[1] = {
  {{ 1.721342888159264e-05, 9.024276128068640e-01, 7.880606727086713e-01}},
};

static const struct Tass17Term IAPETUS_2_22[1] = {
  {{ 1.093705824191350e-06, 2.826816249991383e-01, 7.088831546641388e-01}},
};

static const struct Tass17Term IAPETUS_2_23[1] = {
  {{ 2.918130882538270e-06, 1.562086464859070e+00,-8.653197879163885e-01}},
};

static const struct Tass17Term IAPETUS_2_24[1] = {
  {{ 5.831454312634436e-07, 4.589631341204242e+00,-7.861232166932925e-01}},
};

static const struct Tass17Term IAPETUS_2_25[1] = {
  {{ 5.754083403713949e-07, 3.163232139672567e+00, 1.182078779269309e+00}},
};

static const struct Tass17Term IAPETUS_2_26[1] = {
  {{ 4.595462189000190e-07, 5.053348902770820e+00, 1.023723743180244e+00}},
};

static const struct Tass17Term IAPETUS_2_27[1] = {
  {{ 8.343508926599710e-07, 3.074604494266973e+00,-1.180160376432494e+00}},
};

static const struct Tass17Term IAPETUS_2_28[2] = {
  {{ 3.005442215364578e-07, 3.363297035584122e+00,-1.390853717105622e+00}},
  {{ 3.076753876071480e-07, 5.185765171332203e+00,-1.390858988956663e+00}},
};

static const struct Tass17Term IAPETUS_2_29[1] = {
  {{ 6.251473936893123e-06, 3.940966283548398e-01,-1.232449761841766e+00}},
};

static const struct Tass17Term IAPETUS_2_30[1] = {
  {{ 1.645103872537621e-05, 6.221338739275052e+00, 1.390853717105622e+00}},
};

static const struct Tass17Term IAPETUS_2_31[1] = {
  {{ 3.207481705138808e-06, 4.620837650975520e+00,-2.137313691168855e+00}},
};

static const struct Tass17Term IAPETUS_2_32[1] = {
  {{ 9.029410063641440e-06, 1.994597716654372e+00, 2.295717646432711e+00}},
};

static const struct Tass17Term IAPETUS_2_33[1] = {
  {{ 1.980399556792357e-06, 1.376326363638862e+00,-3.169902489790702e+00}},
};

static const struct Tass17Term IAPETUS_2_34[1] = {
  {{ 5.659369633167750e-06, 5.239109003991030e+00, 3.328306445054557e+00}},
};

static const struct Tass17Term IAPETUS_2_35[1] = {
  {{ 4.105962555389770e-07, 5.815663679651547e+00,-4.427132796269878e+00}},
};

static const struct Tass17Term IAPETUS_2_36[1] = {
  {{ 3.999910250873396e-07, 7.997716879783442e-01, 4.585536751533733e+00}},
};

static const struct Tass17Term IAPETUS_2_37[1] = {
  {{ 4.231711644426220e-07, 1.822484926062486e-01, 6.667061728781588e+00}},
};

static const struct Tass17MultiTerm IAPETUS_2[38] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 24,IAPETUS_2_0},
  {{ 0, 0, 0, 0, 0, 0,-1, 0}, 17,IAPETUS_2_1},
  {{ 0, 0, 0, 0, 0, 0, 1, 0}, 15,IAPETUS_2_2},
  {{ 0, 0, 0, 0, 0, 0,-2, 0},  1,IAPETUS_2_3},
  {{ 0, 0, 0, 0, 0, 0, 2, 0},  4,IAPETUS_2_4},
  {{ 0, 0, 0, 0, 0, 0, 3, 0},  8,IAPETUS_2_5},
  {{ 0, 0, 0, 0, 0, 0, 4, 0},  1,IAPETUS_2_6},
  {{ 0, 0, 0, 0, 0,-1, 0, 0},  4,IAPETUS_2_7},
  {{ 0, 0, 0, 0, 0,-1, 1, 0},  3,IAPETUS_2_8},
  {{ 0, 0, 0, 0, 0,-1, 2, 0},  3,IAPETUS_2_9},
  {{ 0, 0, 0, 0, 0,-1, 3, 0},  4,IAPETUS_2_10},
  {{ 0, 0, 0, 0, 0,-1, 4, 0},  5,IAPETUS_2_11},
  {{ 0, 0, 0, 0, 0,-1, 5, 0}, 14,IAPETUS_2_12},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  4,IAPETUS_2_13},
  {{ 0, 0, 0, 0, 0, 1,-1, 0},  1,IAPETUS_2_14},
  {{ 0, 0, 0, 0, 0, 1, 1, 0},  1,IAPETUS_2_15},
  {{ 0, 0, 0, 0, 0, 1, 2, 0},  3,IAPETUS_2_16},
  {{ 0, 0, 0, 0, 0,-2, 0, 0},  2,IAPETUS_2_17},
  {{ 0, 0, 0, 0, 0,-2, 2, 0},  1,IAPETUS_2_18},
  {{ 0, 0, 0, 0, 0,-2, 3, 0},  4,IAPETUS_2_19},
  {{ 0, 0, 0, 0, 0,-2, 4, 0},  2,IAPETUS_2_20},
  {{ 0, 0, 0, 0, 0, 2, 0, 0},  1,IAPETUS_2_21},
  {{ 0, 0, 0, 0, 0, 2,-1, 0},  1,IAPETUS_2_22},
  {{ 0, 0, 0, 0, 0,-3, 4, 0},  1,IAPETUS_2_23},
  {{ 0, 0, 0, 0, 0,-3, 5, 0},  1,IAPETUS_2_24},
  {{ 0, 0, 0, 0, 0, 3, 0, 0},  1,IAPETUS_2_25},
  {{ 0, 0, 0, 0, 0, 3,-2, 0},  1,IAPETUS_2_26},
  {{ 0, 0, 0, 0, 0,-4, 5, 0},  1,IAPETUS_2_27},
  {{ 0, 0, 0, 0,-1, 0, 0, 0},  2,IAPETUS_2_28},
  {{ 0, 0, 0, 0,-1, 0, 2, 0},  1,IAPETUS_2_29},
  {{ 0, 0, 0, 0, 1, 0, 0, 0},  1,IAPETUS_2_30},
  {{ 0, 0, 0,-1, 0, 0, 2, 0},  1,IAPETUS_2_31},
  {{ 0, 0, 0, 1, 0, 0, 0, 0},  1,IAPETUS_2_32},
  {{ 0, 0,-1, 0, 0, 0, 2, 0},  1,IAPETUS_2_33},
  {{ 0, 0, 1, 0, 0, 0, 0, 0},  1,IAPETUS_2_34},
  {{ 0,-1, 0, 0, 0, 0, 2, 0},  1,IAPETUS_2_35},
  {{ 0, 1, 0, 0, 0, 0, 0, 0},  1,IAPETUS_2_36},
  {{ 1, 0, 0, 0, 0, 0, 0, 0},  1,IAPETUS_2_37},
};

static const struct Tass17Term IAPETUS_3_0[28] = {
  {{ 6.794549145709775e-02, 5.047886356662284e+00,-5.271851040460129e-06}},
  {{ 6.892434301339112e-04, 1.398038368144667e+00, 5.271851040460129e-06}},
  {{ 4.570393856724005e-05, 3.663028182224427e+00,-1.054370208092026e-05}},
  {{ 2.873148969672351e-05, 1.130417841604620e+00, 1.067825987245346e-05}},
  {{ 1.915644482319336e-05, 1.666983826523547e+00, 1.608466870444679e-05}},
  {{ 5.927925628073686e-06, 2.236765758751786e+00,-1.621922649597999e-05}},
  {{ 1.388791091408471e-05, 6.199545784892948e+00,-1.870277299460525e-05}},
  {{ 2.730634121085477e-04, 3.074026406224565e+00,-2.445240135601595e-05}},
  {{ 9.851993483313069e-06, 3.568018055505390e+00, 2.445240135601595e-05}},
  {{ 1.363049113672050e-05, 1.573081241951383e+00,-2.972425239647608e-05}},
  {{ 6.564159971602750e-06, 5.899199575040269e+00, 2.986599622756999e-05}},
  {{ 1.115402869499410e-05, 2.124370914205684e+00, 4.891917479115332e-05}},
  {{ 5.075828846612914e-06, 3.331278759394790e-01, 5.419102583161344e-05}},
  {{ 2.829811667283325e-05, 3.179813769687029e+00, 5.787093127593016e-04}},
  {{ 4.494417792607357e-05, 5.092981879591376e+00,-5.839811637997618e-04}},
  {{ 3.010291742672222e-05, 4.323389769715201e+00, 5.839811637997618e-04}},
  {{ 2.832841674512957e-05, 3.775418852980096e+00,-5.892530148402220e-04}},
  {{ 9.901282214730337e-06, 3.022565890567675e+00, 5.892530148402220e-04}},
  {{ 3.855202341799354e-06, 3.260409788853447e+00, 1.149259554604918e-03}},
  {{ 6.779699256950095e-07, 3.450394767068876e+00,-1.167962327599524e-03}},
  {{ 2.641112196820165e-04, 6.085250961969193e+00, 1.167962327599524e-03}},
  {{ 5.003929435494984e-06, 5.249025520673378e+00,-1.173234178639984e-03}},
  {{ 1.816919041955310e-04, 1.123018507358733e+00, 1.173234178639984e-03}},
  {{ 4.700210983954140e-06, 3.997089887831509e+00,-1.178506029680444e-03}},
  {{ 3.932389676076612e-06, 5.902900444967474e+00, 1.186665100594129e-03}},
  {{ 3.374620641006819e-05, 1.072212121537805e+00, 1.751943491399285e-03}},
  {{ 2.348785638862567e-05, 2.392069283444719e+00, 1.757215342439746e-03}},
  {{ 3.494901083668216e-06, 2.384257843599439e+00, 2.335924655199047e-03}},
};

static const struct Tass17Term IAPETUS_3_1[4] = {
  {{ 3.472415791165123e-07,-2.485713646106026e+00, 7.803942174838812e-02}},
  {{ 3.695428746936740e-07, 7.056695618928304e-01, 7.917751804453235e-02}},
  {{ 3.421209496536026e-07, 3.666045991093121e-01, 7.920738404075992e-02}},
  {{ 3.695428746936740e-07, 2.768173152147268e+00, 7.922643721932350e-02}},
};

static const struct Tass17Term IAPETUS_3_2[8] = {
  {{ 3.970411713643165e-07, 2.468316120918740e+00, 1.566520118252981e-01}},
  {{ 1.388077368193791e-06, 2.364532110656320e+00, 1.572307211204436e-01}},
  {{ 1.564612965731580e-06,-2.599830403698914e+00, 1.572359929714840e-01}},
  {{ 3.930946817313776e-07, 5.320498414089342e+00, 1.578199741176699e-01}},
  {{ 4.741596443169412e-07,-2.890040852186059e+00, 1.584039552638558e-01}},
  {{ 1.832371513844348e-06, 1.571448595037313e+00, 1.584092271148963e-01}},
  {{ 3.902069246730890e-07, 1.467528632730159e+00, 1.589879364100418e-01}},
  {{ 4.268928712637706e-07, 3.104364787604514e+00, 1.595719175562276e-01}},
};

static const struct Tass17Term IAPETUS_3_3[2] = {
  {{ 1.141957796039554e-06, 1.706497420624183e+00,-3.940371597392014e-01}},
  {{ 5.876712089500301e-07, 3.528959925655309e+00,-3.940424315902420e-01}},
};

static const struct Tass17Term IAPETUS_3_4[3] = {
  {{ 2.987819907657320e-05, 1.592450553501527e+00,-3.148405885161054e-01}},
  {{ 1.598538661690393e-05, 3.414911760531445e+00,-3.148458603671461e-01}},
  {{ 5.196347763201437e-07, 1.364682192599308e+00,-3.148650409174615e-01}},
};

static const struct Tass17Term IAPETUS_3_5[1] = {
  {{ 4.754237067161859e-07, 4.619994519724736e+00,-2.356440172930096e-01}},
};

static const struct Tass17Term IAPETUS_3_6[2] = {
  {{ 3.836235546728265e-07, 6.225559278035010e+00,-1.564313614012092e-01}},
  {{ 7.454540946216246e-07, 1.764836475886549e+00,-1.564366332522496e-01}},
};

static const struct Tass17Term IAPETUS_3_7[7] = {
  {{ 5.950685735865521e-07, 2.276553135246288e+00, 1.937456015378634e-03}},
  {{ 3.062968607914330e-07, 4.541169170095360e-01, 1.942727866419093e-03}},
  {{ 8.833068155471169e-07, 4.678333554199201e+00, 1.956509193942217e-03}},
  {{ 4.545649462411443e-07, 2.855871049168076e+00, 1.961781044982677e-03}},
  {{ 8.604417421849733e-07, 5.078832057446901e+00, 1.967322011606203e-03}},
  {{ 1.310815780246894e-06, 3.256355709659855e+00, 1.972593862646663e-03}},
  {{ 6.718820321393613e-07, 1.433809611894344e+00, 1.977865713687124e-03}},
};

static const struct Tass17Term IAPETUS_3_8[2] = {
  {{ 1.141957796039554e-06, 5.137033501494948e+00, 3.940479725568654e-01}},
  {{ 5.876712089500302e-07, 3.314570996463822e+00, 3.940532444079058e-01}},
};

static const struct Tass17Term IAPETUS_3_9[1] = {
  {{ 3.730666308347880e-07, 3.531492325926554e+00, 3.148353166650650e-01}},
};

static const struct Tass17Term IAPETUS_3_10[3] = {
  {{ 2.011086466288782e-05, 5.022984906438330e+00, 4.732445437799614e-01}},
  {{ 1.081178008232983e-05, 3.200523219219202e+00, 4.732498156310018e-01}},
  {{ 3.781314342005297e-07, 5.250621987578875e+00, 4.732689961813175e-01}},
};

static const struct Tass17Term IAPETUS_3_11[2] = {
  {{ 8.562203475402176e-07, 5.620822378456090e+00,-7.088586950767433e-01}},
  {{ 4.406257818911077e-07, 1.160099576307630e+00,-7.088639669277837e-01}},
};

static const struct Tass17Term IAPETUS_3_12[1] = {
  {{ 4.385541576256689e-07, 6.246560326377261e+00,-6.296811770322111e-01}},
};

static const struct Tass17Term IAPETUS_3_13[1] = {
  {{ 3.504051883832708e-07, 3.688750412526320e-01, 7.880851322960668e-01}},
};

static const struct Tass17Term IAPETUS_3_14[2] = {
  {{ 6.998372934208747e-07, 9.946129891738023e-01, 8.672626503405993e-01}},
  {{ 3.601483607531824e-07, 5.455335791322262e+00, 8.672679221916397e-01}},
};

static const struct Tass17Term IAPETUS_3_15[2] = {
  {{ 7.978736105198133e-07, 3.079032121013479e-01,-1.311651739473694e+00}},
  {{ 4.105995430914010e-07, 2.130365717132474e+00,-1.311657011324735e+00}},
};

static const struct Tass17Term IAPETUS_3_16[2] = {
  {{ 7.118995492853203e-07, 2.434684834895837e-02, 1.470055694737550e+00}},
  {{ 3.663558059942478e-07, 4.485069650497419e+00, 1.470060966588590e+00}},
};

static const struct Tass17Term IAPETUS_3_17[1] = {
  {{ 4.090821761277136e-07, 4.534644234722027e+00,-2.216515668800783e+00}},
};

static const struct Tass17Term IAPETUS_3_18[1] = {
  {{ 3.817968724121190e-07, 2.080791132907865e+00, 2.374919624064639e+00}},
};

static const struct Tass17MultiTerm IAPETUS_3[19] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 28,IAPETUS_3_0},
  {{ 0, 0, 0, 0, 0, 0, 1, 0},  4,IAPETUS_3_1},
  {{ 0, 0, 0, 0, 0, 0, 2, 0},  8,IAPETUS_3_2},
  {{ 0, 0, 0, 0, 0,-1, 0, 0},  2,IAPETUS_3_3},
  {{ 0, 0, 0, 0, 0,-1, 1, 0},  3,IAPETUS_3_4},
  {{ 0, 0, 0, 0, 0,-1, 2, 0},  1,IAPETUS_3_5},
  {{ 0, 0, 0, 0, 0,-1, 3, 0},  2,IAPETUS_3_6},
  {{ 0, 0, 0, 0, 0,-1, 5, 0},  7,IAPETUS_3_7},
  {{ 0, 0, 0, 0, 0, 1, 0, 0},  2,IAPETUS_3_8},
  {{ 0, 0, 0, 0, 0, 1,-1, 0},  1,IAPETUS_3_9},
  {{ 0, 0, 0, 0, 0, 1, 1, 0},  3,IAPETUS_3_10},
  {{ 0, 0, 0, 0, 0,-2, 1, 0},  2,IAPETUS_3_11},
  {{ 0, 0, 0, 0, 0,-2, 2, 0},  1,IAPETUS_3_12},
  {{ 0, 0, 0, 0, 0, 2, 0, 0},  1,IAPETUS_3_13},
  {{ 0, 0, 0, 0, 0, 2, 1, 0},  2,IAPETUS_3_14},
  {{ 0, 0, 0, 0,-1, 0, 1, 0},  2,IAPETUS_3_15},
  {{ 0, 0, 0, 0, 1, 0, 1, 0},  2,IAPETUS_3_16},
  {{ 0, 0, 0,-1, 0, 0, 1, 0},  1,IAPETUS_3_17},
  {{ 0, 0, 0, 1, 0, 0, 1, 0},  1,IAPETUS_3_18},
};

static const struct Tass17Term HYPERION_0_0[109] = {
  {{ 5.269198501828300e-03, 5.535906566693734e+00, 9.810539955099672e-03}},
  {{ 9.845757121969750e-04, 5.390367216813617e+00, 1.974675300542636e-01}},
  {{-9.447929974549504e-04, 2.695183608406955e+00, 9.873376502713178e-02}},
  {{-6.016015548174626e-04, 3.442462348892786e+00, 8.892322507203211e-02}},
  {{ 5.722146672387600e-04, 1.802365518040839e+00, 2.962012950813954e-01}},
  {{ 5.148150044366294e-04, 1.947904867920752e+00, 1.085443049822315e-01}},
  {{ 3.064232824160758e-04, 4.497549126447647e+00, 3.949350601085272e-01}},
  {{ 2.328679324996273e-04, 9.095474276756056e-01, 4.936688251356589e-01}},
  {{ 1.830222466848013e-04, 3.604731036082414e+00, 5.924025901627907e-01}},
  {{ 1.428121183641368e-04, 1.672933731036699e-02, 6.911363551899224e-01}},
  {{-1.309800881858535e-04, 4.597328162679121e+00, 1.072766731000612e-02}},
  {{ 1.186076675279319e-04, 1.912996635287834e-01, 8.893412600193221e-03}},
  {{ 1.030589558021878e-04, 2.711912945716443e+00, 7.898701202170543e-01}},
  {{ 9.162128151198170e-05, 2.549644258526895e+00, 2.863907551262957e-01}},
  {{ 8.420360344777332e-05, 5.244827866933704e+00, 3.851245201534275e-01}},
  {{-7.587557241443920e-05, 4.189741089378624e+00, 7.911268511693244e-02}},
  {{ 7.477636779983881e-05, 5.407096554123251e+00, 8.886038852441861e-01}},
  {{ 6.245968777723361e-05, 1.656826168161656e+00, 4.838582851805592e-01}},
  {{-5.925277975412419e-05, 1.055086777554782e+00, 3.060118350364951e-01}},
  {{-5.866195836719566e-05, 4.576031364902324e-02, 1.965504026993571e-01}},
  {{-5.838551971068579e-05, 3.750270385962327e+00, 4.047456000636268e-01}},
  {{ 5.517120904542451e-05, 1.819094855351211e+00, 9.873376502713178e-01}},
  {{ 5.291804011662172e-05, 4.352009776567740e+00, 5.825920502076911e-01}},
  {{ 4.785477163890776e-05, 7.640080777956939e-01, 6.813258152348228e-01}},
  {{ 4.648293960962418e-05, 6.137645957299674e+00, 1.876569900991639e-01}},
  {{-4.509683939781330e-05, 1.622686871902795e-01, 5.034793650907585e-01}},
  {{-4.394073687298339e-05, 1.200626127435425e+00, 1.183548449373311e-01}},
  {{ 4.393229670558661e-05, 3.459191686201771e+00, 7.800595802619547e-01}},
  {{ 4.272829406941570e-05, 1.848125831689869e+00, 4.927516977807525e-01}},
  {{ 4.154122350033837e-05, 4.514278463755098e+00, 1.086071415298450e+00}},
  {{-4.056750796467554e-05, 2.857452295597089e+00, 6.022131301178903e-01}},
  {{-3.857616137051121e-05, 5.552635904003895e+00, 7.009468951450221e-01}},
  {{ 3.815083219130052e-05, 6.154375294608579e+00, 8.787933452890865e-01}},
  {{-3.619913197295652e-05, 1.964634205231123e+00, 7.996806601721539e-01}},
  {{-3.615550506243641e-05, 6.017393906061373e-01, 1.778464501440642e-01}},
  {{ 3.211045925583688e-05, 2.566373595836530e+00, 9.775271103162182e-01}},
  {{-3.191646069822739e-05, 4.659817813637930e+00, 8.984144251992857e-01}},
  {{ 3.106998351274690e-05, 9.262767649808578e-01, 1.184805180325582e+00}},
  {{-2.986485373026291e-05, 4.041349085722073e+00, 2.943161986529901e-02}},
  {{-2.797299420355516e-05, 3.895809735842240e+00, 2.170886099644629e-01}},
  {{-2.722632461441940e-05, 1.071816114865886e+00, 9.971481902264174e-01}},
  {{-2.676900656082160e-05, 3.971018944640962e+00, 1.095544904247980e+00}},
  {{ 2.670876942507434e-05, 5.261557204243347e+00, 1.076260875343350e+00}},
  {{ 2.422130320267833e-05, 4.788627826207911e+00, 1.962107991019934e-02}},
  {{ 2.299675108056304e-05, 3.621460373393509e+00, 1.283538945352713e+00}},
  {{-2.297557612018265e-05, 3.766999723274154e+00, 1.095881955253549e+00}},
  {{ 2.232369387111974e-05, 1.673555505469107e+00, 1.174994640370482e+00}},
  {{-1.940722128323030e-05, 1.789980244999149e-01, 1.194615720280681e+00}},
  {{ 1.861668359762009e-05, 4.368739113874453e+00, 1.273728405397614e+00}},
  {{ 1.721156885182891e-05, 4.381040752907420e+00, 8.800609771712566e-02}},
  {{ 1.683393793859745e-05, 3.345867461927939e-02, 1.382272710379845e+00}},
  {{-1.635856648806699e-05, 2.874181632905266e+00, 1.293349485307813e+00}},
  {{ 1.543735278843180e-05, 7.807374151075219e-01, 1.372462170424745e+00}},
  {{-1.367379954537092e-05, 5.569365241310616e+00, 1.392083250334945e+00}},
  {{-1.338883184199104e-05, 2.503883944878085e+00, 8.984035242693857e-02}},
  {{-1.336950395903937e-05, 3.078080370694616e-01, 3.158223749915947e-01}},
  {{-1.320526200861202e-05, 3.296922999012952e+00, 2.765802151711960e-01}},
  {{-1.308495447734650e-05, 6.027455951817885e+00, 2.000408834733950e+00}},
  {{-1.302898520468604e-05, 1.009326463906489e+00, 1.094614323371379e-01}},
  {{ 1.289973076525889e-05, 4.062988177672174e+00, 9.924156005528160e-03}},
  {{-1.276260623147002e-05, 7.256396485357081e-01, 9.696923904671184e-03}},
  {{ 1.271138964892950e-05, 3.475921023512875e+00, 1.471195935451877e+00}},
  {{-1.249786948042500e-05, 4.914148009431628e+00, 2.962012605910000e-01}},
  {{ 1.223977806847720e-05, 2.728642283024626e+00, 1.481006475406977e+00}},
  {{-1.206158887296372e-05, 3.633762012421656e+00, 9.781663767222532e-02}},
  {{-1.131018256569493e-05, 1.981363542543681e+00, 1.490817015362076e+00}},
  {{-1.089636645498038e-05, 2.740943922055832e+00, 2.952841677264889e-01}},
  {{ 1.039179172259797e-05, 6.171104631918221e+00, 1.569929700479009e+00}},
  {{ 1.012836012521588e-05, 2.886483271935745e+00, 1.076271776273250e-01}},
  {{-9.589777592386616e-06, 5.344606903164943e+00, 9.171273549064512e-04}},
  {{-9.298418091702524e-06, 4.676547150949028e+00, 1.589550780389208e+00}},
  {{ 8.847111426033778e-06, 5.423825891429972e+00, 1.579740240434109e+00}},
  {{-8.783581973124320e-06, 4.617972413769575e+00, 5.894496611094870e-01}},
  {{ 8.461716341891088e-06, 2.583102933143981e+00, 1.668663465506141e+00}},
  {{-7.901292288839958e-06, 4.643088476327559e+00, 2.072780700093633e-01}},
  {{ 7.719706043802130e-06, 4.937019829864389e+00, 6.930214516183278e-02}},
  {{-7.609973567682646e-06, 1.088545452174794e+00, 1.688284545416340e+00}},
  {{ 7.430427565196998e-06, 3.313652336323319e+00, 9.677165703611184e-01}},
  {{ 7.356833812577206e-06, 6.184687279150488e-01, 8.689828053339868e-01}},
  {{ 7.165256187015007e-06, 6.008835944731589e+00, 1.066450335388250e+00}},
  {{ 6.955885415513037e-06, 4.206470426688558e+00, 7.702490403068549e-01}},
  {{ 6.885291075415051e-06, 5.278286541549328e+00, 1.767397230533273e+00}},
  {{ 6.614895521474710e-06, 1.511286818281744e+00, 6.715152752797231e-01}},
  {{ 6.331562819397564e-06, 1.835824192663037e+00, 1.678474005461240e+00}},
  {{-6.179612339504240e-06, 3.783729060580140e+00, 1.787018310443472e+00}},
  {{ 6.131629134251356e-06, 5.116017854362701e+00, 1.263917865442514e+00}},
  {{ 5.583256693630428e-06, 1.528016155588465e+00, 1.362651630469646e+00}},
  {{ 5.538856606051041e-06, 1.690284842782397e+00, 1.866130995560404e+00}},
  {{-5.031578459421872e-06, 4.533473869493745e-01, 1.281653848924308e-01}},
  {{-4.727461773298300e-06, 1.403512188080000e+00, 1.136827010127225e-02}},
  {{-4.490169397561938e-06, 8.800270869150743e-01, 9.899519638809918e-03}},
  {{ 4.445857119863126e-06, 3.908600739292801e+00, 9.721560271389426e-03}},
  {{ 3.929644136581539e-06, 3.385115638127897e+00, 8.252809808927092e-03}},
  {{ 3.847298333647876e-06, 5.128319493393325e+00, 7.819555776202598e-02}},
  {{-3.502084178980730e-06, 3.251162685363996e+00, 8.002981247183889e-02}},
  {{-2.961456426220440e-06, 5.853277499591428e+00, 2.952841654680000e-01}},
  {{ 2.811429184555117e-06, 3.658749758664493e+00, 1.164479466491257e-02}},
  {{ 2.625536042038141e-06, 1.756605204392253e+00, 9.965089238203824e-02}},
  {{ 2.190398167707681e-06, 2.620477234204319e-01, 1.192719722922376e-01}},
  {{-2.173672397129618e-06, 4.979927489736701e+00, 2.851449251039256e-02}},
  {{ 2.165574780684495e-06, 3.102770681707372e+00, 3.034874722020547e-02}},
  {{-2.086152830538867e-06, 2.139204531449687e+00, 1.174377175824247e-01}},
  {{ 1.758820758802031e-06, 2.435682350271240e+00, 9.835176321817914e-03}},
  {{-1.729303956187257e-06, 2.352945475936649e+00, 9.785903588381428e-03}},
  {{ 1.628729863936739e-06, 5.684298570350299e+00, 5.949160520673310e-02}},
  {{-1.424540056085036e-06, 4.915380737914367e+00, 8.880960902160362e-02}},
  {{ 1.386736928659767e-06, 1.969543959871211e+00, 8.903684112246060e-02}},
  {{ 1.373772890937228e-06, 4.749864788991031e-01, 1.086579210326600e-01}},
  {{-1.342309573473890e-06, 3.420823256942400e+00, 1.084306889318030e-01}},
};

static const struct Tass17MultiTerm HYPERION_0[1] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},109,HYPERION_0_0},
};

static const struct Tass17Term HYPERION_1_0[227] = {
  {{ 1.591300460227652e-01, 5.535906566693734e+00, 9.810539955099672e-03}},
  {{ 4.042489669732959e-03, 1.912996635287834e-01, 8.893412600193221e-03}},
  {{-3.674456394728999e-03, 4.597328162679121e+00, 1.072766731000612e-02}},
  {{ 2.477704948047725e-03, 5.390367216813617e+00, 1.974675300542636e-01}},
  {{ 1.876329764520020e-03, 2.695183608406955e+00, 9.873376502713178e-02}},
  {{-1.559041896665946e-03, 3.442462348892786e+00, 8.892322507203211e-02}},
  {{ 1.534084173919484e-03, 5.344606903164943e+00, 9.171273549064512e-04}},
  {{ 1.177430869304958e-03, 1.802365518040839e+00, 2.962012950813954e-01}},
  {{ 1.132234522571428e-03, 1.947904867920752e+00, 1.085443049822315e-01}},
  {{ 7.097516189849207e-04, 4.497549126447647e+00, 3.949350601085272e-01}},
  {{ 4.276809299095375e-04, 9.095474276756056e-01, 4.936688251356589e-01}},
  {{-3.898971540977004e-04, 7.256396485357081e-01, 9.696923904671184e-03}},
  {{ 3.851351224149008e-04, 4.062988177672174e+00, 9.924156005528160e-03}},
  {{-3.602762810400590e-04, 4.810266918158017e+00, 1.136160504284885e-04}},
  {{-3.108861291415183e-04, 4.189741089378624e+00, 7.911268511693244e-02}},
  {{-3.036800061325884e-04, 4.041349085722073e+00, 2.943161986529901e-02}},
  {{ 2.883086646474427e-04, 3.604731036082414e+00, 5.924025901627907e-01}},
  {{ 2.741386484621230e-04, 2.880428915709079e+00, 1.167962290513110e-03}},
  {{ 2.445211406069563e-04, 6.137645957299674e+00, 1.876569900991639e-01}},
  {{ 2.397418179235776e-04, 4.432376484564475e+00, 5.839811452565560e-04}},
  {{ 2.079619720142985e-04, 2.549644258526895e+00, 2.863907551262957e-01}},
  {{ 1.997665368454137e-04, 1.672933731036699e-02, 6.911363551899224e-01}},
  {{-1.967051700749114e-04, 3.182961090757090e+00, 2.463636671824243e-05}},
  {{ 1.584700380869004e-04, 4.788627826207911e+00, 1.962107991019934e-02}},
  {{ 1.532039999508838e-04, 5.244827866933704e+00, 3.851245201534275e-01}},
  {{ 1.443751571550919e-04, 3.385115638127897e+00, 8.252809808927092e-03}},
  {{ 1.391797303858132e-04, 2.711912945716443e+00, 7.898701202170543e-01}},
  {{ 1.354323755526581e-04, 3.908600739292801e+00, 9.721560271389426e-03}},
  {{-1.347079373955693e-04, 1.055086777554782e+00, 3.060118350364951e-01}},
  {{-1.344254972843197e-04, 8.800270869150743e-01, 9.899519638809918e-03}},
  {{-1.336552626452528e-04, 1.200626127435425e+00, 1.183548449373311e-01}},
  {{-1.248289604761352e-04, 1.403512188080000e+00, 1.136827010127225e-02}},
  {{-1.168617638452060e-04, 4.643088476327559e+00, 2.072780700093633e-01}},
  {{ 1.158407408725238e-04, 1.656826168161656e+00, 4.838582851805592e-01}},
  {{-1.110204610592649e-04, 3.750270385962327e+00, 4.047456000636268e-01}},
  {{ 9.876297087120832e-05, 5.407096554123251e+00, 8.886038852441861e-01}},
  {{ 9.360835142429720e-05, 4.352009776567740e+00, 5.825920502076911e-01}},
  {{-9.343384687595748e-05, 2.150790928565845e+00, 1.557730146172579e-03}},
  {{-8.787340150688395e-05, 1.622686871902795e-01, 5.034793650907585e-01}},
  {{ 8.558910102815169e-05, 1.627305827400927e+00, 8.897968371024604e-05}},
  {{ 7.626783708559587e-05, 7.640080777956939e-01, 6.813258152348228e-01}},
  {{-7.423412446912493e-05, 2.857452295597089e+00, 6.022131301178903e-01}},
  {{ 7.224202068094286e-05, 3.658749758664493e+00, 1.164479466491257e-02}},
  {{ 7.068861066099871e-05, 1.819094855351211e+00, 9.873376502713178e-01}},
  {{-6.947176438271949e-05, 4.576031364902324e-02, 1.965504026993571e-01}},
  {{-6.514978275214879e-05, 6.017393906061373e-01, 1.778464501440642e-01}},
  {{-6.245521353308536e-05, 5.552635904003895e+00, 7.009468951450221e-01}},
  {{ 6.236351497568400e-05, 3.459191686201771e+00, 7.800595802619547e-01}},
  {{ 5.296941292250232e-05, 2.435682350271240e+00, 9.835176321817914e-03}},
  {{-5.247540827620734e-05, 2.352945475936649e+00, 9.785903588381428e-03}},
  {{-5.211914123734037e-05, 1.964634205231123e+00, 7.996806601721539e-01}},
  {{ 5.087526477014214e-05, 4.514278463755098e+00, 1.086071415298450e+00}},
  {{ 5.058590687048317e-05, 6.154375294608579e+00, 8.787933452890865e-01}},
  {{-4.813389965573155e-05, 3.895809735842240e+00, 2.170886099644629e-01}},
  {{-4.301007833478336e-05, 4.659817813637930e+00, 8.984144251992857e-01}},
  {{ 4.189085863844754e-05, 1.756605204392253e+00, 9.965089238203824e-02}},
  {{ 4.104292740965665e-05, 2.566373595836530e+00, 9.775271103162182e-01}},
  {{ 4.098739653872672e-05, 4.381040752907420e+00, 8.800609771712566e-02}},
  {{-4.092616660610000e-05, 9.897290504821536e-01, 1.751943435769670e-03}},
  {{ 3.764144636960646e-05, 4.406028499150309e+00, 1.834254709812902e-03}},
  {{ 3.665390355309391e-05, 9.262767649808578e-01, 1.184805180325582e+00}},
  {{-3.538628961664771e-05, 1.071816114865886e+00, 9.971481902264174e-01}},
  {{-3.531748699358457e-05, 1.009326463906489e+00, 1.094614323371379e-01}},
  {{ 3.326628366799721e-05, 5.261557204243347e+00, 1.076260875343350e+00}},
  {{-2.947898798731622e-05, 4.858034027149458e+00, 8.139193758498604e-03}},
  {{-2.902403206479552e-05, 3.766999723274154e+00, 1.095881955253549e+00}},
  {{ 2.841090381393687e-05, 2.886483271935745e+00, 1.076271776273250e-01}},
  {{ 2.693554901487583e-05, 1.673555505469107e+00, 1.174994640370482e+00}},
  {{ 2.676761780821079e-05, 4.937019829864389e+00, 6.930214516183278e-02}},
  {{ 2.669007886238697e-05, 5.436127530462639e+00, 3.940179327536207e-01}},
  {{ 2.640617243698899e-05, 3.621460373393509e+00, 1.283538945352713e+00}},
  {{ 2.465038854019734e-05, 6.213779106238012e+00, 1.148188615170074e-02}},
  {{ 2.383157781622512e-05, 6.778725395442773e-01, 1.671346196601068e-03}},
  {{-2.373722745643357e-05, 1.789980244999149e-01, 1.194615720280681e+00}},
  {{-2.358765186013457e-05, 2.503883944878085e+00, 8.984035242693857e-02}},
  {{-2.328181620162680e-05, 5.210323605092877e+00, 2.952860072513110e-03}},
  {{-2.308684467527070e-05, 4.979927489736701e+00, 2.851449251039256e-02}},
  {{ 2.176062809432465e-05, 4.368739113874453e+00, 1.273728405397614e+00}},
  {{ 2.173788459895790e-05, 3.102770681707372e+00, 3.034874722020547e-02}},
  {{-1.934646504415605e-05, 2.874181632905266e+00, 1.293349485307813e+00}},
  {{ 1.897373895483440e-05, 3.345867461927939e-02, 1.382272710379845e+00}},
  {{ 1.754329413716687e-05, 7.807374151075219e-01, 1.372462170424745e+00}},
  {{ 1.642710507186450e-05, 1.129878067543419e+00, 7.976285245286770e-03}},
  {{-1.600410709697660e-05, 6.149453095252681e+00, 2.035764949513110e-03}},
  {{ 1.572430747504168e-05, 4.451788812799354e+00, 1.983846574091700e-01}},
  {{-1.571827863857085e-05, 5.569365241310616e+00, 1.392083250334945e+00}},
  {{ 1.553609418389716e-05, 5.128319493393325e+00, 7.819555776202598e-02}},
  {{-1.539945531353985e-05, 3.078080370694616e-01, 3.158223749915947e-01}},
  {{ 1.460196299297440e-05, 4.914148009431628e+00, 2.962012605910000e-01}},
  {{ 1.410585893877412e-05, 3.475921023512875e+00, 1.471195935451877e+00}},
  {{ 1.357189445488529e-05, 2.728642283024626e+00, 1.481006475406977e+00}},
  {{ 1.339823501057820e-05, 3.971018944640962e+00, 1.095544904247980e+00}},
  {{-1.337650472587841e-05, 3.251162685363996e+00, 8.002981247183889e-02}},
  {{ 1.288836483864210e-05, 4.542301028988324e+00, 1.281578356513110e-03}},
  {{-1.273330553828309e-05, 1.981363542543681e+00, 1.490817015362076e+00}},
  {{ 1.163556995976533e-05, 1.511286818281744e+00, 6.715152752797231e-01}},
  {{ 1.139577901854967e-05, 5.099288517053791e+00, 5.727815102525914e-01}},
  {{ 1.139366110710306e-05, 4.206470426688558e+00, 7.702490403068549e-01}},
  {{ 1.135306132914049e-05, 2.740943922055832e+00, 2.952841677264889e-01}},
  {{-1.132855895484880e-05, 3.124409773657545e+00, 1.084128336043461e-02}},
  {{ 1.131587409293219e-05, 6.171104631918221e+00, 1.569929700479009e+00}},
  {{ 1.088679752730659e-05, 6.070246551700696e+00, 1.061405125957763e-02}},
  {{ 1.085425461534599e-05, 6.184687279150488e-01, 8.689828053339868e-01}},
  {{-1.056787968358008e-05, 3.296922999012952e+00, 2.765802151711960e-01}},
  {{ 1.037117124427068e-05, 2.404104908646983e+00, 4.740477452254596e-01}},
  {{-1.027660284731680e-05, 4.676547150949028e+00, 1.589550780389208e+00}},
  {{ 1.027273355964679e-05, 5.199067553284680e+00, 1.885741174540704e-01}},
  {{-1.024878235180840e-05, 4.617972413769575e+00, 5.894496611094870e-01}},
  {{ 1.006767559393374e-05, 3.313652336323319e+00, 9.677165703611184e-01}},
  {{-1.000802392783223e-05, 4.533473869493745e-01, 1.281653848924308e-01}},
  {{ 9.633989740567134e-06, 5.423825891429972e+00, 1.579740240434109e+00}},
  {{ 9.178075378506732e-06, 6.008835944731589e+00, 1.066450335388250e+00}},
  {{ 9.065206170156284e-06, 2.583102933143981e+00, 1.668663465506141e+00}},
  {{ 8.851132878153258e-06, 8.637871140265755e-01, 2.971184224363018e-01}},
  {{ 8.798422861268053e-06, 5.684298570350299e+00, 5.949160520673310e-02}},
  {{-8.257707350863237e-06, 1.088545452174794e+00, 1.688284545416340e+00}},
  {{ 8.253549910000110e-06, 2.420834245957355e+00, 1.165184100415382e+00}},
  {{ 8.204317973862647e-06, 1.757809810726964e+00, 8.163830125216846e-03}},
  {{ 7.905521775023910e-06, 1.611065854511902e+00, 2.873078824812022e-01}},
  {{-7.853735166683912e-06, 5.581666880342553e+00, 2.063609426544568e-01}},
  {{ 7.724851326708509e-06, 5.001566581686802e+00, 9.007028650621710e-03}},
  {{ 7.341396041786271e-06, 5.116017854362701e+00, 1.263917865442514e+00}},
  {{ 7.254032016403240e-06, 6.027455951817885e+00, 2.000408834733950e+00}},
  {{ 7.228880718926971e-06, 5.278286541549328e+00, 1.767397230533273e+00}},
  {{-7.215481275780168e-06, 1.664218052550344e+00, 8.779796549764733e-03}},
  {{-6.984482201975051e-06, 2.342090592094629e+00, 1.045114274636580e-02}},
  {{-6.882066002549454e-06, 3.030818015480898e+00, 1.145724978498250e-02}},
  {{ 6.764635964795921e-06, 1.835824192663037e+00, 1.678474005461240e+00}},
  {{ 6.650438793970224e-06, 2.620477234204319e-01, 1.192719722922376e-01}},
  {{-6.615254376434763e-06, 3.783729060580140e+00, 1.787018310443472e+00}},
  {{ 6.543758079294575e-06, 3.912539073151877e+00, 9.082249651543854e-01}},
  {{ 6.469301087553129e-06, 1.528016155588465e+00, 1.362651630469646e+00}},
  {{ 6.431124552019571e-06, 3.245373743805582e-01, 1.006958730181517e+00}},
  {{-6.344479273710200e-06, 3.488222662541158e+00, 2.854736277713893e-01}},
  {{ 6.326716714869053e-06, 1.217355464745067e+00, 8.094912001272536e-01}},
  {{-6.296309881986443e-06, 2.139204531449687e+00, 1.174377175824247e-01}},
  {{ 6.126061421202252e-06, 3.019720982785904e+00, 1.105692495208649e+00}},
  {{ 6.110229923833445e-06, 3.633762012421656e+00, 9.781663767222532e-02}},
  {{ 6.015743140846809e-06, 3.089369332580482e+00, 6.406027912661278e-04}},
  {{ 5.753474696726435e-06, 4.306249462919441e+00, 3.860416475083339e-01}},
  {{ 5.748353093941175e-06, 1.690284842782397e+00, 1.866130995560404e+00}},
  {{ 5.708428891129938e-06, 5.714904591191261e+00, 1.204426260235781e+00}},
  {{ 5.649105509135010e-06, 4.223199763993809e+00, 1.461385395496778e+00}},
  {{ 5.648928770703093e-06, 4.805357163517846e+00, 7.107574351001218e-01}},
  {{-5.558326355997969e-06, 3.778096755966772e+00, 1.646709829882825e-03}},
  {{ 5.443666713560605e-06, 4.649337840653702e-01, 1.228539745617870e-02}},
  {{ 5.432340482357107e-06, 5.992106607419760e+00, 3.753139801983278e-01}},
  {{ 5.352373966360052e-06, 3.558970722433383e+00, 3.958521874634336e-01}},
  {{-5.276359293355092e-06, 1.957273618059048e-01, 1.885752075470604e+00}},
  {{ 5.250152718378343e-06, 1.848125831689869e+00, 4.927516977807525e-01}},
  {{ 5.229887464078713e-06, 2.126902892417018e+00, 1.303160025262913e+00}},
  {{ 4.890118052517932e-06, 6.351980652268782e-01, 1.560119160523909e+00}},
  {{ 4.723632226316384e-06, 4.822086500829672e+00, 1.401893790290044e+00}},
  {{ 4.693369092482988e-06, 4.531007801068383e+00, 1.777207770488372e+00}},
  {{ 4.561946954009516e-06, 4.385468451187749e+00, 1.964864760587536e+00}},
  {{-4.420710568728507e-06, 4.044201739498272e+00, 2.667696752160964e-01}},
  {{-4.380976601809811e-06, 1.349018131091464e+00, 1.680359101889646e-01}},
  {{ 4.337634584565960e-06, 2.110173555110300e+00, 6.120236700729901e-01}},
  {{ 4.289012976116846e-06, 7.182477641466621e-01, 4.847754125354657e-01}},
  {{ 4.260384871171156e-06, 3.330381673632227e+00, 1.658852925551041e+00}},
  {{ 4.220548662286636e-06, 1.234084802055436e+00, 1.500627555317176e+00}},
  {{-4.199363160735397e-06, 2.890910970218556e+00, 1.984485840497735e+00}},
  {{-3.958927857591537e-06, 2.637836897642000e+00, 1.806334976402677e-02}},
  {{ 3.941866401660965e-06, 6.224633990080018e+00, 1.081664699371637e-02}},
  {{-3.779778844595939e-06, 2.970022335278144e+00, 1.063868762629588e-02}},
  {{ 3.732023312137390e-06, 3.929268410460785e+00, 1.599361320344308e+00}},
  {{ 3.704937010630530e-06, 2.931053342080318e+00, 1.192598652513110e-03}},
  {{ 3.635875031439811e-06, 7.974667524135091e-01, 2.063598525614668e+00}},
  {{-3.613554557427136e-06, 4.915380737914367e+00, 8.880960902160362e-02}},
  {{ 3.510914428131760e-06, 1.969543959871211e+00, 8.903684112246060e-02}},
  {{-3.500526370624341e-06, 6.183406270948697e+00, 3.842073927985210e-01}},
  {{ 3.460035905216260e-06, 5.853277499591428e+00, 2.952841654680000e-01}},
  {{ 3.455891490801950e-06, 3.413431372552740e+00, 5.835091775625976e-01}},
  {{-3.450615143245680e-06, 8.053972782328979e-01, 1.118669826513110e-03}},
  {{ 3.406532781873487e-06, 6.254154330840199e+00, 4.945859524905654e-01}},
  {{ 2.958781539525876e-06, 4.749864788991031e-01, 1.086579210326600e-01}},
  {{-2.877066321058344e-06, 3.420823256942400e+00, 1.084306889318030e-01}},
  {{-2.684534114394913e-06, 1.818605490929709e+00, 8.982392283903467e-03}},
  {{ 2.527885180883367e-06, 4.847179143307436e+00, 8.804432916482976e-03}},
  {{-2.228949169524294e-06, 2.568430696700499e+00, 2.954523591572750e-02}},
  {{ 2.203418048212828e-06, 5.514267474743575e+00, 2.931800381487053e-02}},
  {{ 2.084495541017897e-06, 2.546791604750330e+00, 4.905269977549836e-02}},
  {{-2.078328653497296e-06, 5.875598233879090e+00, 6.838501780692632e-02}},
  {{ 2.053262782839373e-06, 1.212212524551202e+00, 2.474857501079031e-03}},
  {{ 1.926895597615764e-06, 3.998441425849760e+00, 7.021927251673923e-02}},
  {{ 1.866581488955838e-06, 3.825993270575803e-01, 1.778682520038644e-02}},
  {{ 1.793161742446505e-06, 1.291671420326874e+00, 8.736549492585954e-02}},
  {{ 1.728658611291661e-06, 5.989253953642903e+00, 1.379759248475305e-01}},
  {{ 1.676857763913550e-06, 6.562334475941619e-01, 2.117881005637192e-02}},
  {{ 1.535772080970505e-06, 4.776710899143922e-02, 8.025577708070115e-03}},
  {{-1.524956909904084e-06, 1.497103946256642e+00, 1.075230367672436e-02}},
  {{-1.522980567370601e-06, 5.662659478400199e+00, 7.899906906650395e-02}},
  {{-1.495199523539624e-06, 5.488139457702296e+00, 1.784962247029556e-03}},
  {{ 1.487447331838316e-06, 1.414367071922007e+00, 1.070303094328788e-02}},
  {{ 1.476058477371991e-06, 2.716822700357048e+00, 7.922630116736093e-02}},
  {{ 1.450002439077977e-06, 8.691722030730533e-01, 1.056475879679429e-02}},
  {{-1.242111402697583e-06, 4.740860717216437e+00, 1.159550220212923e-02}},
  {{ 1.240886283959093e-06, 1.815156521491816e+00, 8.883424538832187e-02}},
  {{-1.224261037265257e-06, 2.720171354649790e+00, 1.256192201981903e-02}},
  {{-1.221915897644608e-06, 5.069768176293690e+00, 8.901220475574236e-02}},
  {{-1.201215374450881e-06, 4.098695796487174e+00, 1.101020351284040e-01}},
  {{-1.186953182299199e-06, 2.911471018178655e+00, 2.145533462001224e-02}},
  {{-1.129771140532139e-06, 5.319619156922048e+00, 8.708897036221921e-02}},
  {{-1.097793441792323e-06, 5.275200702223381e+00, 1.239901350660719e-02}},
  {{-1.056701003062998e-06, 1.483920036565450e-01, 4.968106525163343e-02}},
  {{ 1.028753109593422e-06, 3.374260754285869e+00, 8.918048966911464e-03}},
  {{-1.020891709037016e-06, 3.575210695322014e+00, 1.086332846659417e-01}},
  {{-1.012670463985482e-06, 3.291523879951270e+00, 8.868776233474979e-03}},
  {{ 1.000328987513614e-06, 3.205990405202193e-01, 1.084553252985212e-01}},
  {{ 9.414415383367402e-07, 4.110755286663576e+00, 1.794973371359828e-02}},
  {{-9.256226899654040e-07, 2.164192277692745e+00, 3.126587457511192e-02}},
  {{-8.873327693660700e-07, 3.396916671853406e-01, 5.857447785182665e-02}},
  {{-8.599515368071076e-07, 3.230728199748524e+00, 8.050214074788358e-03}},
  {{-8.578646315557892e-07, 1.890558157156235e+00, 2.787388971912643e-02}},
  {{ 8.458120789255400e-07, 3.315709437186336e+00, 1.973469596062783e-02}},
  {{-8.217941676205146e-07, 5.593253277458703e+00, 9.048095521820468e-02}},
  {{ 7.998618448176001e-07, 2.446537234113254e+00, 9.169937163833544e-03}},
  {{ 7.948809078699551e-07, 6.192140014287844e+00, 3.098935001147160e-02}},
  {{ 7.924651953382082e-07, 4.745720166335665e+00, 6.040873256163955e-02}},
  {{-7.833211544506204e-07, 1.391925790964367e+00, 1.272482575375243e-01}},
  {{ 7.730107483777507e-07, 5.668654913122977e+00, 2.952059954900926e-02}},
  {{-7.622738839419112e-07, 2.414043258321102e+00, 2.934264018158877e-02}},
  {{ 7.622712310485859e-07, 7.074805989149507e-02, 1.103785596920444e-01}},
  {{ 7.575143203880983e-07, 6.080299246534646e+00, 1.069865748360589e-01}},
  {{-7.315354488166669e-07, 6.261546215229414e+00, 1.950746385977086e-02}},
  {{ 7.005448425844999e-07, 1.616450943558919e+00, 7.542188416946162e-04}},
  {{-6.870417571431780e-07, 6.010893045593364e+00, 1.184684609877596e-01}},
};

static const struct Tass17MultiTerm HYPERION_1[1] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},227,HYPERION_1_0},
};

static const struct Tass17Term HYPERION_2_0[195] = {
  {{ 1.030661479148230e-01, 3.619080362792600e+00,-8.924811235147779e-04}},
  {{ 2.448184191185018e-02, 2.680501958777963e+00, 2.464623139167320e-05}},
  {{-2.500610695618523e-03, 2.871801622306748e+00, 8.918058831584894e-03}},
  {{-1.653120911968409e-03, 4.366359103278453e+00,-1.070302107861445e-02}},
  {{-1.121964769453605e-03, 3.107866401998293e-02, 9.784128390361700e-02}},
  {{ 7.518101576911162e-04, 9.238967543856604e-01,-9.962624615064656e-02}},
  {{ 3.778282825702326e-04, 4.511898453158731e+00,-1.983600111777784e-01}},
  {{-3.775434250722031e-04, 2.726262272426791e+00, 1.965750489307488e-01}},
  {{-3.597821116452316e-04, 1.833444182060827e+00, 3.940425789850124e-01}},
  {{-2.927952161795262e-04, 5.421445880833606e+00, 2.953088139578806e-01}},
  {{ 2.580134073493171e-04, 1.766180138998234e-01,-8.981570619554689e-02}},
  {{ 2.216814079711899e-04, 1.816714844751185e+00,-2.970937762049101e-01}},
  {{ 1.927636081642950e-04, 2.546497551037094e+00, 2.060411181513110e-03}},
  {{-1.702244907149874e-04, 1.671175494871279e+00,-1.094367861057462e-01}},
  {{-1.630491456609473e-04, 5.566985230713519e+00, 1.076518238587167e-01}},
  {{ 1.502092233208532e-04, 7.783574045058210e-01, 8.803074394851733e-02}},
  {{ 1.403880753848180e-04, 5.404716543523964e+00,-3.958275412320419e-01}},
  {{ 1.282680047161120e-04, 3.764619712672674e+00,-1.885494712226787e-01}},
  {{ 1.080722389283692e-04, 1.933223218292120e+00, 9.835186186491344e-03}},
  {{-9.843461962138636e-05, 5.259177193644051e+00,-2.081705511328780e-01}},
  {{ 9.302812855413870e-05, 2.709532935117156e+00,-4.945613062591737e-01}},
  {{ 8.608901766955960e-05, 1.069436104265866e+00,-2.872832362498105e-01}},
  {{ 8.563197359093362e-05, 4.557658766807236e+00,-1.809608478421229e-03}},
  {{ 8.231557260635240e-05, 3.485627041196898e+00, 1.143316058513110e-03}},
  {{-7.641879996400636e-05, 3.810380026321384e+00, 8.000931476678443e-03}},
  {{-6.976234551437248e-05, 2.563993585237243e+00,-3.069043161600098e-01}},
  {{ 6.410167941575658e-05, 5.091998751814161e+00,-1.006097173943266e-03}},
  {{ 6.394074317345045e-05, 4.657437803037907e+00,-3.860170012769422e-01}},
  {{ 6.387046194265387e-05, 1.434932671034752e-02,-5.932950712863055e-01}},
  {{-6.379700394998453e-05, 2.146161973771036e+00,-7.788650730862894e-04}},
  {{-5.338815029861825e-05, 6.151995284010021e+00,-4.056380811871416e-01}},
  {{ 5.173543661985320e-05, 3.342022422925534e+00, 2.755133995131110e-04}},
  {{ 4.908475624063901e-05, 1.962254194631099e+00,-4.847507663040740e-01}},
  {{ 4.811302148563021e-05, 4.528627790467636e+00, 4.927763440121442e-01}},
  {{ 4.486679433374308e-05, 3.602351025483124e+00,-6.920288363134373e-01}},
  {{-4.178177074434045e-05, 3.456811675602482e+00,-5.043718462142733e-01}},
  {{ 4.042027917523730e-05, 2.250321955375842e+00, 2.953088117000000e-01}},
  {{-3.880672848400252e-05, 2.580722922546885e+00, 3.842320390299127e-01}},
  {{ 3.819239194924841e-05, 5.550255893404608e+00,-5.834845313312059e-01}},
  {{-3.738690787954837e-05, 2.124522881820873e+00, 1.872859878668457e-02}},
  {{-3.553990230264197e-05, 6.168724621318925e+00, 2.854982740027810e-01}},
  {{ 3.415645857185234e-05, 9.406260916955893e-01, 5.915101090392759e-01}},
  {{ 3.339668674029588e-05, 1.086165441574771e+00, 4.038531189401121e-01}},
  {{-3.301162776329309e-05, 7.616280671963972e-01,-6.031056112414052e-01}},
  {{ 3.269372364602020e-05, 3.813563431717510e+00, 2.644392326769670e-03}},
  {{ 3.180628253542403e-05, 9.071674170763108e-01,-7.907626013405691e-01}},
  {{ 3.177451875567787e-05, 5.375685567184869e+00, 9.875841125852346e-02}},
  {{ 3.067833613233151e-05, 5.304937507293081e+00,-1.162014843352090e-02}},
  {{ 3.003432659990370e-05, 2.855072284997068e+00,-6.822182963583376e-01}},
  {{ 2.732958451060629e-05, 5.769871291358446e+00, 6.652490226578010e-04}},
  {{ 2.729633446248457e-05, 3.573320049143738e+00,-1.974428838228719e-01}},
  {{-2.629885713884026e-05, 4.349629765968445e+00,-7.018393762685369e-01}},
  {{ 2.559468055750869e-05, 5.712524580593572e+00,-8.000516624044722e-02}},
  {{ 2.473774782254921e-05, 4.674167140347548e+00, 3.051193539129803e-01}},
  {{-2.417336169407593e-05, 3.473541012912848e+00, 1.867645089756491e-01}},
  {{ 2.384156418867662e-05, 1.598886765909899e-01,-7.809520613854695e-01}},
  {{ 2.259673699575893e-05, 4.495169115849089e+00,-8.894963663677009e-01}},
  {{-2.109852115260203e-05, 1.654446157561637e+00,-8.005731412956687e-01}},
  {{ 2.024562165367701e-05, 1.525636144991658e+00, 7.822020399341766e-02}},
  {{-2.019380769983444e-05, 1.991774535391674e+00,-9.814608072250240e-04}},
  {{ 2.008621003222866e-05, 5.246386190193526e+00,-8.035014398045319e-04}},
  {{ 1.902408812662748e-05, 3.747890375363762e+00,-8.796858264126013e-01}},
  {{-1.811127409136190e-05, 5.113637843764284e+00,-2.051356103371412e-02}},
  {{ 1.706204149397460e-05, 3.017340972186618e+00,-1.787389312675790e-01}},
  {{-1.698064243423227e-05, 5.242447856334416e+00,-8.993069063228005e-01}},
  {{ 1.679462249970688e-05, 4.220819753398175e+00, 1.769539690205495e-01}},
  {{-1.632299947318510e-05, 4.886425133804248e+00,-3.084677457434440e-04}},
  {{ 1.603779600347512e-05, 1.799985507441551e+00,-9.882301313948326e-01}},
  {{ 1.519891681318009e-05, 1.052706766956224e+00,-9.784195914397330e-01}},
  {{ 1.379512640799549e-05, 3.221573637798089e-01,-2.774726962947108e-01}},
  {{-1.366561535006994e-05, 2.547264247926875e+00,-9.980406713499322e-01}},
  {{ 1.283396500574140e-05, 1.556768500554912e+00, 3.084677457434440e-04}},
  {{ 1.279090832954947e-05, 5.438175218143973e+00, 9.864451691478030e-01}},
  {{ 1.243806749353366e-05, 1.231704791455413e+00, 2.161961288409481e-01}},
  {{ 1.235958412008419e-05, 3.427780699263810e+00,-9.785893723707998e-03}},
  {{ 1.225443721836814e-05, 9.863864053438862e-01, 7.880605117386331e-01}},
  {{ 1.225216491402911e-05, 2.742991609736432e+00, 8.877114041206713e-01}},
  {{-1.221520345313342e-05, 5.275906530953691e+00, 4.829658040570445e-01}},
  {{ 1.213225148001182e-05, 4.640708465730461e+00,-1.077153356466865e+00}},
  {{ 1.198327722769453e-05, 3.910159062552588e+00,-3.762064613218426e-01}},
  {{ 1.186021515881310e-05, 4.752692921877314e+00, 1.727297203769670e-03}},
  {{ 1.166603449799361e-05, 1.377244141335036e+00, 2.853913874178424e-02}},
  {{ 1.135648805958188e-05, 5.387987206211405e+00,-1.086963896421964e+00}},
  {{ 1.108533869996140e-05, 6.221340900016817e+00, 1.390853715947980e+00}},
  {{-1.098010740280340e-05, 6.135265946699647e+00,-1.096774436377064e+00}},
  {{ 1.086995985899863e-05, 5.860916584250122e+00,-3.032410098881379e-02}},
  {{ 1.069580937462716e-05, 1.214975454145048e+00,-4.749402263489743e-01}},
  {{ 1.067386083025773e-05, 3.781349049983041e+00, 5.025868839672437e-01}},
  {{-1.043068210990957e-05, 5.521224917064781e+00,-8.889857884064044e-02}},
  {{ 9.693581297810074e-06, 1.850173519369733e+00, 1.085178934174935e+00}},
  {{ 9.678835539512692e-06, 1.945524857325114e+00,-1.175887121493997e+00}},
  {{ 9.562800946711422e-06, 4.802977152918551e+00,-5.736739913761062e-01}},
  {{ 9.468784690060410e-06, 4.780800132962380e-02, 7.889776390935395e-01}},
  {{ 8.801022040147010e-06, 3.915534848785861e+00,-2.941408494094870e-01}},
  {{-8.797753682670658e-06, 3.440082338294301e+00,-1.195508201404196e+00}},
  {{ 8.599214007703230e-06, 1.787683868411798e+00, 1.974921762856553e-01}},
  {{ 8.485093197565512e-06, 2.107793544511012e+00,-6.724077564032379e-01}},
  {{-8.365240462865090e-06, 1.279431670356606e+00, 1.476430036256560e-03}},
  {{ 8.240909734627314e-06, 4.361192720355115e-01,-9.171174902330204e-04}},
  {{-8.205019885615929e-06, 5.188561463701097e-01,-8.678447567965355e-04}},
  {{ 7.999837192197997e-06, 6.268503657550692e+00,-9.870911879574012e-02}},
  {{ 7.994800633732257e-06, 2.692803597806052e+00,-1.185697661449096e+00}},
  {{ 7.964077512935541e-06, 4.819706490227462e+00, 1.174623638138164e-01}},
  {{ 7.702743188457386e-06, 5.533526556092043e+00,-1.274620886521128e+00}},
  {{ 7.473079103968928e-06, 5.695795243283790e+00,-7.711415214303697e-01}},
  {{ 7.388692911769633e-06, 6.328180546253961e-01, 2.756877340476813e-01}},
  {{ 7.270826776415052e-06, 1.468289434226756e+00,-2.450211269687357e-03}},
  {{ 7.236960740797404e-06, 4.628406826699250e+00, 1.085689512136231e-01}},
  {{ 7.144251731442444e-06, 4.545357127775086e+00, 1.183912699202067e+00}},
  {{-7.043838984557596e-06, 7.448987298889554e-01,-1.294241966431328e+00}},
  {{-6.917942470086600e-06, 3.896696083322066e+00,-2.060411181513110e-03}},
  {{ 6.782083637837408e-06, 3.311272325723300e+00,-3.167148561151095e-01}},
  {{ 6.750671838792509e-06, 6.160887173164911e-01,-4.154486211422413e-01}},
  {{ 6.736625416531154e-06, 4.252643881934955e-01,-2.518783322486501e-04}},
  {{ 6.716695065274547e-06, 1.115196417914451e+00,-9.073283355045334e-02}},
  {{ 6.651880335792781e-06, 6.006455934130108e+00,-2.179810910879777e-01}},
  {{ 6.570135525528793e-06, 4.204090416089263e+00,-5.141823861693731e-01}},
  {{ 6.551076448609189e-06, 3.000611634877713e+00,-8.698752864575016e-01}},
  {{ 6.445693456193227e-06, 3.926888399861492e+00, 3.149298938680800e-01}},
  {{ 6.221700343295405e-06, 1.508906807682454e+00,-6.129161511965049e-01}},
  {{-6.210141308502555e-06, 5.467206194483361e+00, 4.918592166572376e-01}},
  {{ 6.128034935351225e-06, 2.838342947686692e+00,-1.373354651548260e+00}},
  {{ 5.752222076851271e-06, 5.096908506454503e+00,-7.116499162236366e-01}},
  {{ 5.720920731236437e-06, 3.054280264694361e-01,-9.686090514846332e-01}},
  {{ 5.708438487292833e-06, 4.383088440588454e+00, 6.804333341113080e-01}},
  {{-5.618758010321190e-06, 4.332900428655887e+00,-1.392975731458459e+00}},
  {{ 5.586433308293342e-06, 6.280805296580291e+00,-1.284431426476228e+00}},
  {{ 5.455429436394752e-06, 9.573554290008455e-01, 1.282646464229199e+00}},
  {{ 5.234563142340540e-06, 2.401724898047694e+00,-8.103836812507684e-01}},
  {{-5.207388474970705e-06, 4.296952902336882e+00, 7.788650730862894e-04}},
  {{ 5.204438871550596e-06, 2.418454235357329e+00,-1.192473260608459e-01}},
  {{ 5.046334690770730e-06, 2.893440714256877e+00,-1.058940502818596e-02}},
  {{-5.035619025711394e-06, 5.839277492300028e+00,-1.081663712904294e-02}},
  {{ 4.976902542853610e-06, 3.893429725242215e+00,-1.067342816511765e+00}},
  {{ 4.951970227904307e-06, 1.978983531940741e+00, 2.063855888858485e-01}},
  {{-4.950572722250481e-06, 4.320598789629788e+00,-2.072534237779716e-01}},
  {{ 4.864691847876214e-06, 1.431593392813457e-01,-1.472088416575392e+00}},
  {{-4.835154150013290e-06, 2.223783275489257e-01, 1.067346965038102e-01}},
  {{ 4.711996026340074e-06, 5.989726596820463e+00,-9.091174462779002e-01}},
  {{ 4.647700174472706e-06, 8.781364407369228e-01,-2.961766488500037e-01}},
  {{-4.611548111165785e-06, 1.398883233285187e+00, 9.031674882013382e-03}},
  {{-4.463536929701217e-06, 1.637716820250533e+00,-1.491709496485591e+00}},
  {{ 4.448389546383381e-06, 4.344720011328309e+00, 8.804442781156406e-03}},
  {{ 4.423835447429758e-06, 3.652539037413501e+00, 1.381380229256330e+00}},
  {{ 4.297419749912965e-06, 1.198246116836868e+00,-1.166076581538897e+00}},
  {{ 4.206030088019200e-06, 3.294542988413657e+00,-1.007851211305032e+00}},
  {{-4.153052701221075e-06, 7.210106937409027e-01, 7.360328685412315e-03}},
  {{-4.097668847161432e-06, 2.888530959617075e+00, 7.000544140215073e-01}},
  {{ 3.894866749084315e-06, 3.635809700103133e+00, 6.902438740664076e-01}},
  {{-3.876996622725754e-06, 2.272914885477417e+00, 6.840966403831800e-02}},
  {{ 3.843534248051829e-06, 3.585621688174942e+00,-1.383165191503360e+00}},
  {{ 3.842326107820616e-06, 3.731161038055586e+00,-1.570822181602524e+00}},
  {{ 3.724216491694377e-06, 5.993593800083099e-01,-1.106584976332164e+00}},
  {{ 3.699201962974299e-06, 4.786247815611109e+00,-1.264810346566029e+00}},
  {{ 3.608611557983843e-06, 2.609753898886272e+00,-1.103539134606527e-01}},
  {{-3.538397263248576e-06, 5.225718519024773e+00,-1.590443261512723e+00}},
  {{ 3.531502882337352e-06, 6.453733863926071e-02, 1.480113994283462e+00}},
  {{ 3.397352921611207e-06, 4.482867476818612e+00, 2.962259413127871e-01}},
  {{-3.113267226101596e-06, 4.965245840107813e+00,-7.019462628534756e-02}},
  {{-2.954159247722890e-06, 2.957566593162261e+00,-1.143316058513110e-03}},
  {{ 2.885109484385860e-06, 3.947295643644445e+00, 6.086273772565560e-04}},
  {{ 2.832059006428450e-06, 2.402892932765738e+00, 1.192608522513110e-03}},
  {{-2.683799905115401e-06, 1.716935808520376e+00, 8.711361659361089e-02}},
  {{ 2.540204369555767e-06, 4.072427749742135e+00, 1.272729037689160e-01}},
  {{-2.530442498805404e-06, 9.696570680345386e-01, 9.692415654871056e-02}},
  {{ 2.471798836623408e-06, 1.185944477806245e+00, 1.964572614159102e-02}},
  {{-2.393809826972641e-06, 3.063101285835501e+00, 1.781147143177812e-02}},
  {{ 2.322013871583706e-06, 3.165732975843386e+00,-1.290578660159456e-01}},
  {{-2.252919673323646e-06, 9.946448142775208e-01, 1.075231354139779e-02}},
  {{-2.040203000028772e-06, 4.773946176578943e+00,-7.908803888554077e-02}},
  {{ 1.978250600690922e-06, 5.022592550872615e+00, 1.047578897775747e-02}},
  {{-1.817067803131338e-06, 2.941207823248317e+00,-2.563827320115845e-03}},
  {{-1.760241959810971e-06, 5.496237170821872e+00,-2.726735833327680e-03}},
  {{ 1.708248552006820e-06, 2.739053275877476e+00,-1.079200076232469e-02}},
  {{-1.707373996050395e-06, 5.993664930679350e+00,-1.061404139490420e-02}},
  {{ 1.638819397704571e-06, 4.499107449707681e+00, 9.007038515295140e-03}},
  {{-1.594635824936384e-06, 1.244495794905822e+00, 8.829079147874648e-03}},
  {{-1.549153538011889e-06, 4.175059439749656e+00,-1.959643367880767e-02}},
  {{-1.541304650138550e-06, 2.495897990714715e+00,-6.086273772565560e-04}},
  {{ 1.506189225615625e-06, 1.113991811579792e+00, 7.542287063680470e-04}},
  {{ 1.387762233686380e-06, 4.153420347799528e+00,-8.896981903681528e-05}},
  {{-1.337929891212176e-06, 2.339647246647041e-01,-9.145290932441870e-03}},
  {{ 1.284796105929756e-06, 3.679176774285410e-01,-8.092229359535366e-02}},
  {{ 1.274183971113668e-06, 2.489202295249182e+00,-8.868766368801548e-03}},
  {{-1.083761451668346e-06, 2.464214549006287e+00, 7.730307663851121e-02}},
  {{-1.082213439029503e-06, 4.386657373204058e-01, 2.945626609669069e-02}},
  {{ 1.074660210279234e-06, 6.052216247778919e+00,-2.143068838862057e-02}},
  {{-1.060143973086585e-06, 5.235531306351518e+00,-1.382622818201617e-04}},
  {{-1.050457836219505e-06, 5.297110302121204e-01,-1.533083914780906e-03}},
  {{ 1.016870649805557e-06, 1.862475158400216e+00,-1.005433735055530e-01}},
  {{ 9.272030248231280e-07, 2.193929082762471e+00, 7.246712634983826e-03}},
  {{ 8.522968078631408e-07, 2.315822545349735e+00, 2.762201138687778e-02}},
  {{-7.513552848257232e-07, 5.163096810852366e-01,-3.124122834372025e-02}},
  {{-7.452690477984808e-07, 4.217967099621830e+00,-6.038408633024787e-02}},
  {{-6.901926058355343e-07, 6.122964307670780e+00, 8.894787130342378e-02}},
};

static const struct Tass17MultiTerm HYPERION_2[1] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0},195,HYPERION_2_0},
};

static const struct Tass17Term HYPERION_3_0[58] = {
  {{ 5.948511882002843e-03, 4.694475827075523e+00,-1.136160504284885e-04}},
  {{ 1.535891024624297e-03, 3.859634729686247e-02,-2.463636671824243e-05}},
  {{ 1.572522677808419e-04, 6.085851736842245e+00, 1.167962290513110e-03}},
  {{-1.497345442720214e-04, 2.543684898509682e+00,-1.671346196601068e-03}},
  {{-4.911846033201391e-05, 1.605106494495041e+00,-7.542188416946162e-04}},
  {{-2.465654101797190e-05, 1.954530936499135e+00,-5.839811452565560e-04}},
  {{-2.213337603547740e-05, 4.211325867661734e+00, 1.751943435769670e-03}},
  {{ 2.189031408512060e-05, 5.633054231090162e+00,-1.030743405334940e-03}},
  {{ 1.598165380887390e-05, 4.295054520089123e+00, 5.839811452565560e-04}},
  {{ 1.496627946379793e-05, 4.016603287531243e+00,-1.784962247029556e-03}},
  {{ 1.404188064697451e-05, 1.796406158023830e+00, 8.139193758498604e-03}},
  {{-1.314294835383983e-05, 3.755897423060887e+00, 8.035113044779626e-04}},
  {{-9.122789187630200e-06, 3.498933647377842e+00,-1.167962290513110e-03}},
  {{ 8.389523097875214e-06, 3.290963638995533e+00,-1.148188615170074e-02}},
  {{ 7.467530692297956e-06, 6.665280904804041e-01, 1.629085132118349e-04}},
  {{-4.916203322979170e-06, 1.106474128302829e+00, 9.862014897670330e-02}},
  {{ 4.509531977488889e-06, 3.078024883516604e+00,-8.678348921231047e-04}},
  {{-4.145588215282021e-06, 9.163790711087549e-01,-1.760325880311314e-03}},
  {{ 3.982965022125390e-06, 1.480712538988318e+00, 1.281578356513110e-03}},
  {{ 3.839542395242535e-06, 1.999292218668653e+00,-9.884738107756028e-02}},
  {{ 3.566024262476873e-06, 5.441754567561372e+00,-9.924156005528160e-03}},
  {{ 3.148398194156444e-06, 5.587293917441577e+00,-1.975811461046921e-01}},
  {{ 3.030592492776347e-06, 5.238868506916578e+00, 9.706241883053072e-02}},
  {{-2.682452719829683e-06, 4.160135842068596e+00,-9.171273549064512e-04}},
  {{-2.169784851229061e-06, 3.801657736709643e+00, 1.973539140038351e-01}},
  {{-2.004460730134680e-06, 6.131686597281963e+00,-1.004051112237328e-01}},
  {{ 1.834573960516513e-06, 1.650866808143945e+00, 1.957961838576625e-01}},
  {{ 1.780425037193046e-06, 2.892110309034762e+00,-2.963149111318239e-01}},
  {{-1.562250771899869e-06, 3.269324547045398e+00, 8.025577708070115e-03}},
  {{-1.398643272650284e-06, 2.139446479501969e+00, 4.929246278334639e-05}},
  {{-1.272802367749216e-06, 6.260985974273701e+00,-8.431985254048623e-04}},
  {{ 1.255563077270305e-06, 2.282979034039322e+00, 9.171273549064512e-04}},
  {{ 1.221711462471724e-06, 1.049127417537954e+00, 1.794973371359828e-02}},
  {{-1.189056338328650e-06, 5.384407856796491e+00,-9.059457126863318e-02}},
  {{ 1.144957852083430e-06, 6.152650159259904e+00, 1.192598652513110e-03}},
  {{-1.121833101144503e-06, 5.986147247402415e+00, 8.725187887543105e-02}},
  {{ 1.116583047991100e-06, 4.491589766430666e+00, 1.068729587856304e-01}},
  {{-8.876506818660424e-07, 4.763882028017109e+00,-1.159550220212923e-02}},
  {{ 8.308837128254335e-07, 2.352385234980906e+00,-1.056475879679429e-02}},
  {{ 8.288384247459950e-07, 5.957800305884270e-01,-1.102156511788325e-01}},
  {{ 7.816180007614254e-07, 6.167394216097088e+00,-2.272321008569770e-04}},
  {{-7.628434629767395e-07, 1.853752868788740e+00, 8.880960902160362e-02}},
  {{ 7.436509468905391e-07, 3.199918346103792e+00, 1.950746385977086e-02}},
  {{-7.276927363562636e-07, 6.189033308047209e+00,-1.973469596062783e-02}},
  {{ 7.144299265706248e-07, 9.771747513115055e-01,-9.417637216246936e-04}},
  {{ 6.925549696724206e-07, 8.578277540091867e-01, 9.056321113405056e-03}},
  {{ 6.584378044173228e-07, 3.482263302524318e+00,-2.588473551507519e-03}},
  {{-6.290271131270627e-07, 4.885775490604302e+00, 8.779796549764733e-03}},
  {{-5.558619276909469e-07, 4.038242379481365e+00,-2.129242610680041e-02}},
  {{ 5.420684261778156e-07, 5.916741046460914e+00, 9.873376502713178e-02}},
  {{-4.533375266393237e-07, 9.714766439641385e-02,-1.084128336043461e-02}},
  {{ 4.270467112203478e-07, 1.691003306229041e-01, 8.050214074788358e-03}},
  {{-4.241248416918695e-07, 5.263738296470040e-01,-9.873376502713178e-02}},
  {{ 4.217192739334928e-07, 4.300290102902023e+00, 9.797954618543716e-02}},
  {{-4.111285058745586e-07, 3.968836178539811e+00,-9.810539955099672e-03}},
  {{ 3.911603306295834e-07, 4.503176163546744e+00,-9.007028650621710e-03}},
  {{ 3.588448340600037e-07, 1.252013478182742e+00,-8.903684112246060e-02}},
  {{-3.517973623230076e-07, 5.383203250461813e+00, 8.924909881882087e-04}},
};

static const struct Tass17MultiTerm HYPERION_3[1] = {
  {{ 0, 0, 0, 0, 0, 0, 0, 0}, 58,HYPERION_3_0},
};

const int nr_of_tass17bodies = 8;
const struct Tass17Body tass17bodies[8] = {
  { "MIMAS",
    8.457558957423141e-08,
    6.667061728781588e+00,
    {-5.196910356411064e-03,
      1.822484926062486e-01,
      0.000000000000000e+00,
      0.000000000000000e+00,
      0.000000000000000e+00,
      0.000000000000000e+00},
    {
      {2,MIMAS_0},
      {3,MIMAS_1},
      {4,MIMAS_2},
      {2,MIMAS_3},
    }
  },
  { "ENCELADUS",
    8.457559689847700e-08,
    4.585536751533733e+00,
    {-3.147075653259473e-03,
      7.997716657090215e-01,
      0.000000000000000e+00,
      0.000000000000000e+00,
     -4.817894243161427e-06,
     -3.858288993746540e-07},
    {
      {2,ENCELADUS_0},
      {10,ENCELADUS_1},
      {9,ENCELADUS_2},
      {2,ENCELADUS_3},
    }
  },
  { "TETHYS",
    8.457567386225863e-08,
    3.328306445054557e+00,
    {-2.047958691903563e-03,
      5.239109365414447e+00,
      0.000000000000000e+00,
      0.000000000000000e+00,
     -1.603681402396474e-05,
     -1.284293647967145e-06},
    {
      {3,TETHYS_0},
      {13,TETHYS_1},
      {12,TETHYS_2},
      {3,TETHYS_3},
    }
  },
  { "DIONE",
    8.457575023401118e-08,
    2.295717646432711e+00,
    {-1.245046723085128e-03,
      1.994592585279060e+00,
      0.000000000000000e+00,
      0.000000000000000e+00,
     -5.892691888978987e-05,
     -4.719091057203836e-06},
    {
      {9,DIONE_0},
      {17,DIONE_1},
      {17,DIONE_2},
      {3,DIONE_3},
    }
  },
  { "RHEA",
    8.457594957866317e-08,
    1.390853715957114e+00,
    {-6.263338154589970e-04,
      6.221340947932125e+00,
      0.000000000000000e+00,
      0.000000000000000e+00,
     -4.193141432895800e-04,
     -3.357667186781713e-05},
    {
      {10,RHEA_0},
      {17,RHEA_1},
      {18,RHEA_2},
      {1,RHEA_3},
    }
  },
  { "TITAN",
    8.459559800923616e-08,
    3.940425676910059e-01,
    {-1.348089930929860e-04,
      4.936792168079816e+00,
     -1.249553518183080e-06,
      2.792373338061224e-06,
     -5.584488311754492e-03,
     -4.471842487301890e-04},
    {
      {6,TITAN_0},
      {9,TITAN_1},
      {10,TITAN_2},
      {1,TITAN_3},
    }
  },
  { "IAPETUS",
    8.457584639645043e-08,
    7.920197763192791e-02,
    {-4.931880677088688e-04,
      1.661250302251527e-01,
      3.904890046320212e-04,
     -9.380651872794318e-04,
     -1.315950341063651e-01,
     -1.054097030879299e-02},
    {
      {26,IAPETUS_0},
      {29,IAPETUS_1},
      {38,IAPETUS_2},
      {19,IAPETUS_3},
    }
  },
  { "HYPERION",
    8.457558674940690e-08,
    2.953088138695055e-01,
    {-1.574686065780747e-03,
      2.250358656361423e+00,
      0.000000000000000e+00,
      0.000000000000000e+00,
     -4.939409467982673e-03,
     -3.958228521883369e-04},
    {
      {1,HYPERION_0},
      {1,HYPERION_1},
      {1,HYPERION_2},
      {1,HYPERION_3},
    }
  },
};

static void CalcLon(double t,double lon[7])
{
	int i;
	for (i=0;i<7;i++,lon++)
	{
		const struct Tass17MultiTerm *const tmt_begin =	tass17bodies[i].series[1].multi_terms;
		const struct Tass17Term *const tt_begin = tmt_begin->terms;
		const struct Tass17Term *tt = tt_begin + tmt_begin->nr_of_terms;
		*lon = 0;
		while (--tt >= tt_begin)
		{
			*lon += tt->s[0]*sin(tt->s[1]+tt->s[2]*t);
		}
	}
}

static void CalcTass17Elem(double t,const double lon[7],int body,double elem[6])
{
	const struct Tass17MultiTerm *tmt_begin,*tmt;
	int i;
	for (i=0;i<6;i++) elem[i] = tass17bodies[body].s0[i];

	tmt_begin = tass17bodies[body].series[0].multi_terms;
	tmt = tmt_begin + tass17bodies[body].series[0].nr_of_multi_terms;
	while (--tmt >= tmt_begin)
	{
		const struct Tass17Term *const tt_begin = tmt->terms;
		const struct Tass17Term *tt = tt_begin + tmt->nr_of_terms;
		double arg = 0;
		for (i=0;i<7;i++) arg += tmt->i[i]*lon[i];
		while (--tt >= tt_begin) elem[0] += tt->s[0]*cos(tt->s[1]+tt->s[2]*t+arg);
	}
	elem[0] = tass17bodies[body].aam * (1.0 + elem[0]);

	tmt_begin = tass17bodies[body].series[1].multi_terms;
	tmt = tmt_begin + tass17bodies[body].series[1].nr_of_multi_terms;
	if (body != 7)
	{ /* first multiterm already calculated: lon[body];*/
		tmt_begin++;
		elem[1] += lon[body];
	}
	while (--tmt >= tmt_begin)
	{
		const struct Tass17Term *const tt_begin = tmt->terms;
		const struct Tass17Term *tt = tt_begin + tmt->nr_of_terms;
		double arg = 0;
		for (i=0;i<7;i++) arg += tmt->i[i]*lon[i];
		while (--tt >= tt_begin) elem[1] += tt->s[0]*sin(tt->s[1]+tt->s[2]*t+arg);
	}
	elem[1] += tass17bodies[body].aam * t;

	tmt_begin = tass17bodies[body].series[2].multi_terms;
	tmt = tmt_begin + tass17bodies[body].series[2].nr_of_multi_terms;
	while (--tmt >= tmt_begin)
	{
		const struct Tass17Term *const tt_begin = tmt->terms;
		const struct Tass17Term *tt = tt_begin + tmt->nr_of_terms;
		double arg = 0;
		for (i=0;i<7;i++) arg += tmt->i[i]*lon[i];
		while (--tt >= tt_begin)
		{
			const double x = tt->s[1] + tt->s[2]*t + arg;
			elem[2] += tt->s[0]*cos(x);
			elem[3] += tt->s[0]*sin(x);
		}
	}

	tmt_begin = tass17bodies[body].series[3].multi_terms;
	tmt = tmt_begin + tass17bodies[body].series[3].nr_of_multi_terms;
	while (--tmt >= tmt_begin)
	{
		const struct Tass17Term *const tt_begin = tmt->terms;
		const struct Tass17Term *tt = tt_begin + tmt->nr_of_terms;
		double arg = 0;
		for (i=0;i<7;i++) arg += tmt->i[i]*lon[i];
		while (--tt >= tt_begin)
		{
			const double x = tt->s[1] + tt->s[2]*t + arg;
			elem[4] += tt->s[0]*cos(x);
			elem[5] += tt->s[0]*sin(x);
		}
	}
}


static
const double TASS17toVSOP87[9] = {
  -9.833473364922412278e-01,-1.603871593615649693e-01, 8.546329577978411975e-02,
   1.817361158757799129e-01,-8.678312794665074866e-01, 4.624292968291581735e-01,
   0.000000000000000000e+00, 4.702603847778936010e-01, 8.825277165667645230e-01
};

/*
static
const double TASS17toJ2000[9] = {
  -9.833472564628459035e-01,-1.603876313013248428e-01, 8.546333092352678089e-02,
   1.667401119524148001e-01,-9.832783769705406668e-01, 7.322136606398094752e-02,
   7.229044385733251626e-02, 8.625219479949252372e-02, 9.936471459321866589e-01
};
*/

#define TASS17_DIM (8*6)
static double t_0 = -1e100;
static double t_1 = -1e100;
static double t_2 = -1e100;
static double tass17_elem_0[TASS17_DIM];
static double tass17_elem_1[TASS17_DIM];
static double tass17_elem_2[TASS17_DIM];
/* 1 day: */
#define DELTA_T 1.0

static double tass17_jd0 = -1e100;
static double tass17_elem[TASS17_DIM];

void CalcAllTass17Elem(const double t,double elem[TASS17_DIM], void *user)
{
	int body;
	double lon[8];
	CalcLon(t,lon);
	for (body=0;body<=7;body++) CalcTass17Elem(t,lon,body,elem+(body*6));
}

void GetTass17Coor(double jd,int body,double *xyz, double *xyzdot)
{
	double xyz6[6];
	GetTass17OsculatingCoor(jd,jd,body,xyz6);
	xyz[0]   =xyz6[0]; xyz[1]   =xyz6[1]; xyz[2]   =xyz6[2];
	xyzdot[0]=xyz6[3]; xyzdot[1]=xyz6[4]; xyzdot[2]=xyz6[5];
}

void GetTass17OsculatingCoor(const double jd0,const double jd, const int body,double *xyz)
{
	double x[6];
	if (jd0 != tass17_jd0)
	{
		const double t0 = jd0 - 2444240.0;
		tass17_jd0 = jd0;
		CalcInterpolatedElements(t0,tass17_elem,
					 TASS17_DIM,
					 &CalcAllTass17Elem,DELTA_T,
					 &t_0,tass17_elem_0,
					 &t_1,tass17_elem_1,
					 &t_2,tass17_elem_2,
					 0);
		/*
		printf("GetTass17Coor(%d): %f %f  %f %f  %f %f\n",
			body,
			tass17_elem[body*6+0],tass17_elem[body*6+1],tass17_elem[body*6+2],
			tass17_elem[body*6+3],tass17_elem[body*6+4],tass17_elem[body*6+5]);
		*/
	}
	EllipticToRectangularN(tass17bodies[body].mu,tass17_elem+(body*6),jd-jd0,x);
	xyz[0] = TASS17toVSOP87[0]*x[0]+TASS17toVSOP87[1]*x[1]+TASS17toVSOP87[2]*x[2];
	xyz[1] = TASS17toVSOP87[3]*x[0]+TASS17toVSOP87[4]*x[1]+TASS17toVSOP87[5]*x[2];
	xyz[2] = TASS17toVSOP87[6]*x[0]+TASS17toVSOP87[7]*x[1]+TASS17toVSOP87[8]*x[2];
	// GZ Updated to a 6-vector including speed...
	xyz[3] = TASS17toVSOP87[0]*x[3]+TASS17toVSOP87[1]*x[4]+TASS17toVSOP87[2]*x[5];
	xyz[4] = TASS17toVSOP87[3]*x[3]+TASS17toVSOP87[4]*x[4]+TASS17toVSOP87[5]*x[5];
	xyz[5] = TASS17toVSOP87[6]*x[3]+TASS17toVSOP87[7]*x[4]+TASS17toVSOP87[8]*x[5];
}



