/*
    httpd.c - Contest progress displayed via http
    Copyright (C) 2011 Ladislav Vaiz <ok1zia@nagano.cz>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

*/

#include "header.h"


struct httpd *httpd;

struct httpd *init_httpd(void){
    struct httpd *httpd;
    struct sockaddr_in sin;

    dbg("init_httpd\n");
    httpd = g_new0(struct httpd, 1);
    httpd->port = 9862; // TODO 

    httpd->sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);

#ifndef WIN32 
    {
        int on;
        on=1;
        if (setsockopt(net->tcpsock, SOL_SOCKET, SO_REUSEADDR, SO4 &on, sizeof(on))){
            trace(cfg->trace_sock, "Can't set SO_REUSEADDR\n");
            goto x;
        }
    }
#endif    
    
    if (fcntl(httpd->sock, F_SETFL, O_NONBLOCK)){
        trace(cfg->trace_sock, "Can't set O_NONBLOCK\n");
        goto x;
    }
        
    memset(&sin, 0, sizeof(struct sockaddr_in));
    sin.sin_family = AF_INET;
    sin.sin_port = htons(httpd->port);
    sin.sin_addr.s_addr = INADDR_ANY;
    if (bind(httpd->sock, (struct sockaddr *)&sin, sizeof(sin))) {
        trace(cfg->trace_sock, "Can't bind port %d, error %d\n", httpd->port, sock_errno);
        goto x;
    }
    
    if (listen(httpd->sock, 10)){
        trace(cfg->trace_sock, "Can't listen on socket %d, tcp port %d \n", httpd->sock, httpd->port);
        goto x;
    }
    
    set_handlers(httpd->sock, httpd_accept_handler, NULL, NULL, CBA0);
    httpd->conns = g_ptr_array_new();
    dbg("httpd active\n");

    return httpd;
x:;
    closesocket(httpd->sock);
    httpd->sock = -1;
    return httpd;
}

void free_httpd(struct httpd *httpd){
    closesocket(httpd->sock);
    g_free(httpd);
}

void free_httpconn(struct httpconn *conn){
    if (conn->sock >= 0) {
        set_handlers(conn->sock, NULL, NULL, NULL, CBA0);
        closesocket(conn->sock);
    }
    g_string_free(conn->request, TRUE);
    g_string_free(conn->response, TRUE);
    g_free(conn);
}

void httpd_accept_handler(cba_t cba){
    int sock;
    struct sockaddr_in sin;
    socklen_t socklen;
    
    struct httpconn *conn;
    

    socklen = sizeof(sin);
    sock = accept(httpd->sock, (struct sockaddr *)&sin, &socklen);
    if (!socklen || sock<0) return;

    trace(cfg->trace_sock, "Accepted socket %d %s:%d\n", sock, inet_ntoa(sin.sin_addr), ntohs(sin.sin_port));

    conn = g_new0(struct httpconn, 1);
    /*conn->sin.sin_family      = AF_INET;
    conn->sin.sin_addr.s_addr = sin.sin_addr.s_addr;
    conn->sin.sin_port        = sin.sin_port;*/
    conn->sock = sock;
    conn->request = g_string_sized_new(500);
    conn->response = g_string_sized_new(500);
    
    set_handlers(conn->sock, httpd_read_handler, NULL, NULL, (cba_t)conn);
    g_ptr_array_add(httpd->conns, conn);
}

void httpd_read_handler(cba_t cba){
    struct httpconn *conn;
    char s[1030], *c, lf;
    int ret, err;

    conn = (struct httpconn*)GETCBA(cba, conn);
    ret = read(conn->sock, s, 1024);
    err=sock_errno;
    if (ret <= 0){
        dbg("http read error\n");
        g_ptr_array_remove(httpd->conns, conn);
        free_httpconn(conn);
        return;
    }
    s[ret] = '\0';
    g_string_append(conn->request, s);

    lf = 0;
    for (c = conn->request->str; *c != '\0'; c++){
        if (*c == '\r') continue;
        if (*c != '\n') {
            lf = 0;
            continue;
        }
        lf++;
        if (lf < 2) continue;
        break;
    }
    if (lf < 2) return;
    dbg("http request: >>>%s<<<\n", conn->request->str);

    if (strncasecmp(conn->request->str, "GET ", 4) == 0){
        httpd_get(conn);
    }else{
        httpd_header(conn, 400);
        g_string_sprintf(conn->response, "<html><body>Bad request</body></html>");
    }
/*    for (i = 0; i < 1000; i++){
        g_string_append_c(conn->response, '0' + (i / 100));
    }*/
    set_handlers(conn->sock, NULL, httpd_write_handler, NULL, cba);
}

void httpd_write_handler(cba_t cba){
    struct httpconn *conn;
    int ret, err, len;

    conn = (struct httpconn*)GETCBA(cba, conn);

	if (conn->response_mem != NULL){
		len = conn->response_len;
		if (len > 1400) len = 1400;
		ret = write(conn->sock, conn->response_mem + conn->response_i, len);
		err = sock_errno;
		if (ret <= 0){
			dbg("http write error\n");
			g_ptr_array_remove(httpd->conns, conn);
			free_httpconn(conn);
			return;
		}
		conn->response_i += ret;
		conn->response_len -= ret;
		if (conn->response_len <= 0){
			dbg("http done\n");
			g_ptr_array_remove(httpd->conns, conn);
			free_httpconn(conn);
			return;
		}
	}else{
		len = conn->response->len;
		if (len > 1400) len = 1400;
		ret = write(conn->sock, conn->response->str, len);
		err = sock_errno;
		if (ret <= 0){
			dbg("http write error\n");
			g_ptr_array_remove(httpd->conns, conn);
			free_httpconn(conn);
			return;
		}

		g_string_erase(conn->response, 0, ret);
		if (conn->response->len <= 0){
			dbg("http done\n");
			g_ptr_array_remove(httpd->conns, conn);
			free_httpconn(conn);
			return;
		}
	}
}

void httpd_header(struct httpconn *conn, int status){
    g_string_sprintf(conn->response, "HTTP/1.1 %d Status%d\r\n", status, status);
    g_string_sprintfa(conn->response, "Server: Tucnak %s\r\n", PACKAGE_VERSION);
    g_string_sprintfa(conn->response, "Connection: close\r\n");
    g_string_sprintfa(conn->response, "Content-type: text/html; charset=iso-8859-2\r\n");
    g_string_sprintfa(conn->response, "\r\n");

}

static void httpd_get_style_css(struct httpconn *conn){
	g_string_append(conn->response, 
"body {\n"
"    font-size: 12pt;\n"
/*"    font-family: Tahoma, Helvetica, Arial, sans-serif}\n"*/
"    font-family: Courier, Courier-new}\n"

"a {\n"
"    font-weight: bold}\n"

"h1, h2, h3, h4 {\n"
"    font-family: Tahoma, Helvetica, Arial, sans-serif;\n"
"    color: #f8f8f8};\n"

"h2 {\n"
"    margin-top: 20;\n"
"    margin-bottom: 5;\n"
"    margin-left:10 }\n"
/*
input {                                                                        
    padding: 2px;
    margin: 2px;
    font-weight: normal; 
    font-size: 12px; 
    color: #000000; 
    font-family: sans-serif}                                                                               

li {
    margin-bottom: 20px
}

p.stress{
    font-style: italic}
      */
"table.qsos {\n"
"    border: black 2px solid;\n"
"    padding: 5px;\n"
"    background: #606060}\n"

"th {\n"
"    padding-left: 5px;\n"
"    padding-right: 5px}\n"

"td {\n"
"    padding-left: 5px;\n"
"    padding-right: 5px}\n"
);

}

static void httpd_get_mem(struct httpconn *conn, const unsigned char *mem, int len){
	conn->response_mem = (char *)mem;
	conn->response_len = len;
	conn->response_i = 0;
}


static void httpd_get_index(struct httpconn *conn){
    GString *gs2, *title;
    int i;
    GString *gs = conn->response;

    httpd_header(conn, 200);

    if (!ctest){
        html_header(conn->response, "No contest", HTML_ICON);
        g_string_sprintfa(conn->response, "<p>No contest opened</p>\n");
        html_footer(conn->response);
        return;
    }
    
    gs2 = g_string_sized_new(1024);
    title = g_string_sized_new(200);
    
    zg_string_eprintfa("h", title, "%S - %s", ctest->pcall, ctest->tname);
    html_header(gs, title->str, HTML_ICON);

    g_string_sprintfa(gs, "<a href=\"complete\">Complete report</a><br>\n");
    for (i = 0; i < ctest->bands->len; i++){
        struct band *b;
        b = (struct band*)g_ptr_array_index(ctest->bands, i);

        zg_string_eprintfa("h", gs, "<a name=\"%c\"></a><h2>Band %s</h2>\n", b->bandchar, b->bandname);
        g_string_sprintfa(gs, "<p>QSOs: %d<br>\n", b->stats->nqsos);
        g_string_sprintfa(gs, "Points: %d<br>\n", b->stats->ntotal);
        g_string_sprintfa(gs, "WWLs: %d<br>\n", g_hash_table_size(b->stats->wwls));
        g_string_sprintfa(gs, "DXCCs: %d<br>\n", g_hash_table_size(b->stats->dxcs));
        if (b->stats->nqsos){
            g_string_sprintfa(gs, "AVG: %5.2f pts/qso<br>\n", (double)b->stats->ntotal/(double)b->stats->nqsos);
        }
        if (b->stats->odxcall){
            zg_string_eprintfa("h", gs, "ODX: %S %S %d km op %S<br>\n", b->stats->odxcall, b->stats->odxwwl, b->stats->odxqrb_int, b->stats->odxoperator);
        }
        g_string_sprintfa(gs, "<a href=\"stats/%c\">Statistics</a>\n", b->bandchar);
        g_string_sprintfa(gs, "<a href=\"qsos/%c\">QSOs</a><br>\n", b->bandchar);
        g_string_sprintfa(gs, "</p>\n");


    }

    html_footer(conn->response);
    g_string_free(gs2, TRUE);
    g_string_free(title, TRUE);
}

void httpd_get_stats(struct httpconn *conn, const char *bandchar_str){
	int i;
	struct band *b;

	httpd_header(conn, 200);
	if (!ctest){
        html_header(conn->response, "No contest", HTML_ICON); 
        g_string_sprintfa(conn->response, "<p>No contest opened</p>\n");
        html_footer(conn->response);
        return;
    }
	b = find_band_by_bandchar(bandchar_str[0]);
	if (!b){
        html_header(conn->response, "Band not found", HTML_ICON); 
		g_string_sprintfa(conn->response, "<p>Band %c not found</p>", bandchar_str[0]);
        html_footer(conn->response);
		return;
	}

    html_header(conn->response, "Statistics", HTML_ICON); 
    g_string_sprintfa(conn->response, "<pre>");
	for (i = 0; i < b->statsfifo1->items->len ; i++){
		char *s = (char*)g_ptr_array_index(b->statsfifo1->items, i);
		g_string_sprintfa(conn->response, "%s\n", s);
	}
    g_string_sprintfa(conn->response, "</pre>");
    html_footer(conn->response);

}

void httpd_get_qsos(struct httpconn *conn, const char *bandchar_str){
	struct band *b;
    struct config_band *confb;
	
    httpd_header(conn, 200);
	if (!ctest){
        html_header(conn->response, "No contest", HTML_ICON); 
        g_string_sprintfa(conn->response, "<p>No contest opened</p>\n");
        html_footer(conn->response);
        return;
    }
	b = find_band_by_bandchar(bandchar_str[0]);
    confb = get_config_band_by_bandchar(bandchar_str[0]);
	if (!b || !confb){
        html_header(conn->response, "Band not found", HTML_ICON); 
		g_string_sprintfa(conn->response, "<p>Band %c not found</p>", bandchar_str[0]);
        html_footer(conn->response);
		return;
	}

    html_header(conn->response, "QSOs", HTML_ICON); 
    html_band_header(conn->response, b, confb, HTML_ICON, NULL, NULL);
    html_qsos(conn->response, b, confb);
    html_band_footer(conn->response);
    html_footer(conn->response);
}

void httpd_get_complete(struct httpconn *conn){
    httpd_header(conn, 200);
	if (!ctest){
        html_header(conn->response, "No contest", HTML_ICON); 
        g_string_sprintfa(conn->response, "<p>No contest opened</p>\n");
        html_footer(conn->response);
        return;
    }
    html_complete(conn->response, HTML_IMG_ROOT);
}

void httpd_get(struct httpconn *conn){
	char *c, *page, *tmp, *page0, *page1/*, *page2*/;
	
    tmp = NULL;
	page = conn->request->str + strlen("GET") + 1;
	while (*page == ' ') page++;
	c = strchr(page, ' ');
	if (c != NULL) *c = '\0';

	if (strcmp(page, "/") == 0){
		httpd_get_index(conn);
		return;
	}

	page0 = strtok_r(page, "/", &tmp);
	page1 = strtok_r(NULL, "/", &tmp);
	//page2 = strtok_r(NULL, "/", &tmp);


	if (strcmp(page0, "style.css") == 0){
		httpd_get_style_css(conn);
	}else if (strcmp(page0, "tucnak64.png") == 0){
		httpd_get_mem(conn, icon_tucnak64, sizeof(icon_tucnak64));
	}else if (strcmp(page0, "stats") == 0){
		httpd_get_stats(conn, page1);
	}else if (strcmp(page0, "qsos") == 0){
		httpd_get_qsos(conn, page1);
	}else if (strcmp(page0, "complete") == 0){
		httpd_get_complete(conn);
//    }else if (strncmp(page0, "map") == 0){
//        httpd_get_complete(conn, );
	}else{
		httpd_get_index(conn);
	}

}
