/*******************************************************************************
* admin.cpp:
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: admin.cpp,v 1.1 2001/10/06 21:23:36 bozo Exp $
*
* Authors: Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

#include <crypt.h>

#include "config.h"
#include "../core/core.h"
#include "../core/network.h"
#include "request.h"
#include "admin.h"
#include "telnet.h"
#include "nativeadmin.h"

#include "../core/network.cpp"


/*******************************************************************************
* C_C_CommandDesc class
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_CommandDesc::C_CommandDesc(const C_String& strName,
                             const C_String& strHelp,
                             const C_String& strLongHelp) :
                     m_strName(strName),
                     m_strHelp(strHelp),
                     m_strLongHelp(strLongHelp)
{
  m_bExtendedOptions = false;
}


//------------------------------------------------------------------------------
// Usage's auto-build from the current description
//------------------------------------------------------------------------------
void C_CommandDesc::BuildUsage()
{
  m_strUsage = "Usage: " + m_strName;

  for(unsigned int ui = 0; ui < m_vMandatoryArgs.Size(); ui++)
    m_strUsage += " <" + m_vMandatoryArgs[ui] + ">";

  for(unsigned int ui = 0; ui < m_vOptionalArgs.Size(); ui++)
    m_strUsage += " [" + m_vOptionalArgs[ui] + "]";

  for(unsigned int ui = 0; ui < m_vOptions.Size(); ui++)
    m_strUsage += " [--" + m_vOptions[ui] + " value]";

  for(unsigned int ui = 0; ui < m_vBooleans.Size(); ui++)
    m_strUsage += " [--" + m_vBooleans[ui] + "]";

  if(m_bExtendedOptions)
    m_strUsage += " [--... value] [--...]";
}


/*******************************************************************************
* C_AdminGroup class
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_AdminGroup::C_AdminGroup(const C_String& strName) : m_strName(strName)
{
}


/*******************************************************************************
* C_AdminUser class
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_AdminUser::C_AdminUser(const C_String& strLogin,
                         const C_String& strEncPasswd,
                         C_AdminGroup* pGroup) : m_strLogin(strLogin),
                                                 m_strEncPasswd(strEncPasswd)
{
  m_pGroup = pGroup;
}


/*******************************************************************************
* C_AdminSession class
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_AdminSession::C_AdminSession(C_Admin* pAdmin)
{
  ASSERT(pAdmin);

  m_pAdmin = pAdmin;
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_AdminSession::~C_AdminSession()
{
  // Does nothing but has to be virtual
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_AdminSession::Init()
{
  OnInit();
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_AdminSession::Close()
{
  OnClose();
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
int C_AdminSession::Authentificate(const C_String& strLogin,
                                   const C_String& strPasswd)
{
  return m_pAdmin->Authentificate(this, strLogin, strPasswd);
}


/*******************************************************************************
* C_Admin class
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_Admin::C_Admin(handle hLogger, C_RequestHub* pRequestHub)
{
  ASSERT(hLogger);
  ASSERT(pRequestHub);

  m_hLog = hLogger;
  m_pRequestHub = pRequestHub;
  m_bRequestsEnabled = true;

  m_pNativeAdmin = new C_NativeAdmin(m_hLog, this);

  m_pTelnet = new C_Telnet(m_hLog, this);
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_Admin::~C_Admin()
{
  delete m_pNativeAdmin;
  delete m_pTelnet;
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
int C_Admin::Init()
{
  C_Application* pApp = C_Application::GetApp();
  int iRc = 0;

  // Commands' descriptions
  // -- help [command]
  C_CommandDesc* pCmdDesc = new C_CommandDesc("help",
        "gives help on the specified command.",
        "Called with no argument, \"help\" gives the list of all the commands"
                " (available or not). Called with one argument it gives"
                " details about how to use the specified command.");
  C_String* pStr = new C_String("command");
  pCmdDesc->m_vOptionalArgs.Add(pStr);
  pCmdDesc->BuildUsage();
  m_cCmdDescriptions.Add(pCmdDesc->m_strName, pCmdDesc);
  // -- browse [input]
  pCmdDesc = new C_CommandDesc("browse",
        "gives the program list",
        "Called with one argument, \"browse\" gives all programs of inputs."
                " Called with one argument it only gives the programs of the"
                "specified input. Each program is given with its status.");
  pStr = new C_String("input");
  pCmdDesc->m_vOptionalArgs.Add(pStr);
  pCmdDesc->BuildUsage();
  m_cCmdDescriptions.Add(pCmdDesc->m_strName, pCmdDesc);
  // -- start <program> <channel> <input> [--loop]
  pCmdDesc = new C_CommandDesc("start",
        "launches a program.",
        "\"start\" launches the specified program of the specified input"
                " and broadcasts it through the specified channel.");
  pStr = new C_String("program");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pStr = new C_String("channel");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pStr = new C_String("input");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pStr = new C_String("loop");
  pCmdDesc->m_vBooleans.Add(pStr);
  pCmdDesc->m_bExtendedOptions = true;
  pCmdDesc->BuildUsage();
  m_cCmdDescriptions.Add(pCmdDesc->m_strName, pCmdDesc);
  // -- resume <input> <program>
  pCmdDesc = new C_CommandDesc("resume",
        "resumes streaming.",
        "\"resume\" resumes the broadcast of the specified program of the"
                " specified input.");
  pStr = new C_String("input");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pStr = new C_String("program");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pCmdDesc->BuildUsage();
  m_cCmdDescriptions.Add(pCmdDesc->m_strName, pCmdDesc);
  // -- suspend <input> <program>
  pCmdDesc = new C_CommandDesc("suspend",
        "suspends streaming.",
        "\"suspend\" suspends the broadcast of the specified program of the"
                " specified input.");
  pStr = new C_String("input");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pStr = new C_String("program");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pCmdDesc->BuildUsage();
  m_cCmdDescriptions.Add(pCmdDesc->m_strName, pCmdDesc);
  // -- stop <input> <program>
  pCmdDesc = new C_CommandDesc("stop",
        "stops a program.",
        "\"stop\" ends the broadcast of the specified program of the"
                " specified input.");
  pStr = new C_String("input");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pStr = new C_String("program");
  pCmdDesc->m_vMandatoryArgs.Add(pStr);
  pCmdDesc->BuildUsage();
  m_cCmdDescriptions.Add(pCmdDesc->m_strName, pCmdDesc);
  // -- shutdown
  pCmdDesc = new C_CommandDesc("shutdown",
        "stops the server.",
        "\"shutdown\" stops all the programs and then the VideoLAN Server"
                " stops.");
  pCmdDesc->BuildUsage();
  m_cCmdDescriptions.Add(pCmdDesc->m_strName, pCmdDesc);
  // -- logout
  pCmdDesc = new C_CommandDesc("logout",
        "terminates the administration session.",
        "\"logout\" closes the current administration session and the remote"
                " connection.");
  pCmdDesc->BuildUsage();
  m_cCmdDescriptions.Add(pCmdDesc->m_strName, pCmdDesc);

  // Groups
  C_Vector<C_Setting> vGroups = pApp->GetSettings("Groups");

  for(unsigned int ui = 0; ui < vGroups.Size(); ui++)
  {
    C_Setting cCurrent = vGroups[ui];
    C_String strGroupName = cCurrent.GetName();
    if(m_cGroups.Get(strGroupName))
    {
      Log(m_hLog, LOG_ERROR,
          "Admin group \"" + strGroupName + "\" already exists");
      iRc = GEN_ERR;
      break;
    }

    C_AdminGroup* pGroup = new C_AdminGroup(strGroupName);
    ASSERT(pGroup);

    C_String strCommands = cCurrent.GetValue();
    C_StringTokenizer cTokenizer(strCommands, '|');
    while(cTokenizer.HasMoreToken())
    {
      C_String strCmd = cTokenizer.NextToken();
      pCmdDesc = m_cCmdDescriptions.Get(strCmd);
      if(!pCmdDesc)
      {
        Log(m_hLog, LOG_ERROR, "Admin group \"" + strGroupName +
            "\" not valid (the command \"" + strCmd + "\" doesn't exist)");
        delete pGroup;
        iRc = GEN_ERR;
        break;
      }

      C_String* pstrCmd = new C_String(strCmd);
      pGroup->m_vCommands.Add(pstrCmd);
    }

    Log(m_hLog, LOG_NOTE, "New admin group \"" + strGroupName + "\" is ok");
    m_cGroups.Add(strGroupName, pGroup);
  }

  // Users
  C_Vector<C_Setting> vUsers = pApp->GetSettings("Users");

  for(unsigned int ui = 0; ui < vUsers.Size(); ui++)
  {
    C_Setting cCurrent = vUsers[ui];
    C_String strUserLogin = cCurrent.GetName();
    if(m_cUsers.Get(strUserLogin))
    {
      Log(m_hLog, LOG_ERROR,
          "Admin user \"" + strUserLogin + "\" already exists");
      iRc = GEN_ERR;
      break;
    }

    C_String strUserInfo = cCurrent.GetValue();
    C_StringTokenizer cTokenizer(strUserInfo, ':');
    if(cTokenizer.CountTokens() != 2)
    {
      Log(m_hLog, LOG_ERROR, "Admin user \"" + strUserLogin + "\" not valid");
      iRc = GEN_ERR;
      break;
    }

    C_String strEncPasswd = cTokenizer.NextToken();
    C_String strGroup = cTokenizer.NextToken();

    C_AdminGroup* pGroup = m_cGroups.Get(strGroup);
    if(!pGroup)
    {
      Log(m_hLog, LOG_ERROR, "Admin user \"" + strUserLogin +
          "\" not valid (the group \"" + strGroup + "\" doesn't exists");
      iRc = GEN_ERR;
      break;
    }

    C_AdminUser* pUser = new C_AdminUser(strUserLogin, strEncPasswd, pGroup);
    ASSERT(pUser);

    Log(m_hLog, LOG_NOTE, "New admin user \"" + strUserLogin + "\" is ok");
    m_cUsers.Add(strUserLogin, pUser);
  }

  if(!iRc)
    iRc = m_pNativeAdmin->Init();

  if(!iRc)
    iRc = m_pTelnet->Init();

  return iRc;
}


//------------------------------------------------------------------------------
// Execution
//------------------------------------------------------------------------------
int C_Admin::Run()
{
  int iRc = m_pNativeAdmin->Run();

  if(!iRc)
    iRc = m_pTelnet->Run();

  iRc |= FullStop();

  return iRc;
}


//------------------------------------------------------------------------------
// Stop the execution of the telnet daemon (main thread)
//------------------------------------------------------------------------------
int C_Admin::Stop()
{
  return m_pTelnet->Stop();
}

//------------------------------------------------------------------------------
// Stop the execution of the the remaining daemons
//------------------------------------------------------------------------------
int C_Admin::FullStop()
{
  return m_pNativeAdmin->Stop();
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
int C_Admin::Destroy()
{
  int iRc = m_pTelnet->Destroy();
  iRc |= m_pNativeAdmin->Destroy();
  return iRc;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
int C_Admin::Authentificate(C_AdminSession *pSession,
                            const C_String& strLogin,
                            const C_String& strPasswd)
{
  C_AdminUser* pUser = m_cUsers.Get(strLogin);
  if(!pUser)
  {
    Log(m_hLog, LOG_WARN,
        "Authentification failed for user \"" + strLogin + "\"");
    return GEN_ERR;
  }

  C_String strCrypt(crypt(strPasswd.GetString(),
                          pUser->m_strEncPasswd.GetString()));
  if(strCrypt == pUser->m_strEncPasswd.GetString())
  {
    Log(m_hLog, LOG_NOTE,
        "User \"" + strLogin + "\" successfully authentificated");
    pSession->m_strUser = strLogin;
    pSession->m_strGroup = strLogin;
    pSession->m_vCommands = pUser->m_pGroup->m_vCommands;
    return 0;
  }

  Log(m_hLog, LOG_WARN,
      "Authentification failed for user \"" + strLogin + "\"");
  return GEN_ERR;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Admin::ParseCmdLine(C_AdminSession* pSession,
                               const C_String& strCmdLine,
                               C_Request& cRequest)
{
  ASSERT(pSession);

  C_Answer cAnswer("Admin");

  // Hash the command line
  // #### to be replaced by a shell-like hash -- bozo
  C_Vector<C_String> vArgs;
  C_StringTokenizer cTokenizer(strCmdLine, ' ');
  while(cTokenizer.HasMoreToken())
  {
    C_String* pStr = new C_String(cTokenizer.NextToken());
    vArgs.Add(pStr);
  }

  // Empty command line ?
  if(vArgs.Size() == 0)
  {
    cAnswer.SetStatus(ADMIN_EMPTY_COMMAND);
    return cAnswer;
  }

  // Unknown command ?
  C_String strCmd = vArgs[0];
  C_CommandDesc* pCmdDesc = m_cCmdDescriptions.Get(strCmd);
  if(!pCmdDesc)
  {
    cAnswer.SetStatus(ADMIN_UNKNOWN_COMMAND);
    cAnswer.AddMessage(strCmd + ": unknown command.");
    return cAnswer;
  }

  // The user is not allowed to run this command ?
  if(pSession->m_vCommands.Find(strCmd) < 0)
  {
    cAnswer.SetStatus(ADMIN_COMMAND_DENIED);
    cAnswer.AddMessage(strCmd + ": permission denied.");
    return cAnswer;
  }

  // Command is OK, now parse the arguments
  C_Request cRq(strCmd);
  // Default boolean value is "0"
  for(unsigned int ui = 0; ui < pCmdDesc->m_vBooleans.Size(); ui++)
    cRq.SetArg(pCmdDesc->m_vBooleans[ui], "0");

  int iError = 0;
  unsigned int uiMandatory = 0;
  unsigned int uiOptional = 0;

  for(unsigned int ui = 1; ui < vArgs.Size(); ui++)
  {
    C_String strArg(vArgs[ui]);
    if(strArg.StartsWith("--"))
    {
      C_String strTmp(strArg.SubString(2, strArg.Length()));
      if(pCmdDesc->m_vOptions.Find(strTmp) >= 0)
      {
        // The argument is an option
        if(++ui < vArgs.Size())
        {
          cRq.SetArg(strTmp, vArgs[ui]);
        }
        else
        {
          // Missing value
          cAnswer.AddMessage(strCmd + ": missing value for option \"" +
                             strTmp + "\"");
          cAnswer.AddMessage(pCmdDesc->m_strUsage);
          iError = GEN_ERR;
          break;
        }
      }
      else if(pCmdDesc->m_vBooleans.Find(strTmp) >= 0)
      {
        // The argument is a boolean
        cRq.SetArg(strTmp, "1");
      }
      else if(pCmdDesc->m_bExtendedOptions)
      {
        // Extended option or boolean
        if(++ui < vArgs.Size())
        {
          strArg = vArgs[ui];
          if(strArg.StartsWith("--"))
          {
            // The argument is a boolean
            cRq.SetArg(strTmp, "1");
            ui--;
          }
          else
          {
            // The argument is an option
            cRq.SetArg(strTmp, vArgs[ui]);
          }
        }
        else
        {
          // The argument is a boolean
          cRq.SetArg(strTmp, "1");
        }
      }
      else
      {
        // Bad argument
        cAnswer.AddMessage(strCmd + ": unknown option \"" + strTmp + "\"");
        cAnswer.AddMessage(pCmdDesc->m_strUsage);
        iError = GEN_ERR;
        break;
      }
    }
    else
    {
      if(uiMandatory < pCmdDesc->m_vMandatoryArgs.Size())
      {
        // Mandatory argument
        cRq.SetArg(pCmdDesc->m_vMandatoryArgs[uiMandatory++], strArg);
      }
      else if(uiOptional < pCmdDesc->m_vOptionalArgs.Size())
      {
        // Optional argument
        cRq.SetArg(pCmdDesc->m_vOptionalArgs[uiOptional++], strArg);
      }
      else
      {
        // Standalone arguments overflow
        cAnswer.AddMessage(strCmd + ": too many arguments");
        cAnswer.AddMessage(pCmdDesc->m_strUsage);
        iError = GEN_ERR;
        break;
      }
    }
  }

  // Check if there are enough arguments
  if(uiMandatory < pCmdDesc->m_vMandatoryArgs.Size())
  {
    cAnswer.AddMessage(strCmd + ": too few arguments");
    cAnswer.AddMessage(pCmdDesc->m_strUsage);
    iError = GEN_ERR;
  }

  if(iError)
  {
    cAnswer.SetStatus(ADMIN_COMMAND_NOT_VALID);
  }
  else
  {
    cRequest = cRq;
    cAnswer.SetStatus(ADMIN_WELLFORMED_COMMAND);
  }

  return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Admin::ValidateRequest(C_AdminSession* pSession,
                                  C_Request& cRequest)
{
  ASSERT(pSession);

  C_Answer cAnswer("Admin");

  C_String strCmd = cRequest.GetCmd();

   // Empty command line ?
  if(strCmd.Length() == 0)
  {
    cAnswer.SetStatus(ADMIN_EMPTY_COMMAND);
    return cAnswer;
  }

  // Unknown command ?
  C_CommandDesc* pCmdDesc = m_cCmdDescriptions.Get(strCmd);
  if(!pCmdDesc)
  {
    cAnswer.SetStatus(ADMIN_UNKNOWN_COMMAND);
    cAnswer.AddMessage(strCmd + ": unknown command.");
    return cAnswer;
  }

  // The user is not allowed to run this command ?
  if(pSession->m_vCommands.Find(strCmd) < 0)
  {
    cAnswer.SetStatus(ADMIN_COMMAND_DENIED);
    cAnswer.AddMessage(strCmd + ": permission denied.");
    return cAnswer;
  }

  // Check arguments' validity
  //   - mandatory arguments shouldn't be empty;
  //   - booleans should be "0" or "1";
  //   - we don't check arguments which aren't in the description because we
  //     don't have access to the hashtable in the C_Request. It doesn't matter
  //     because they are ignored.
  int iError = 0;

  for(unsigned int ui = 0; ui < pCmdDesc->m_vMandatoryArgs.Size(); ui++)
  {
    C_String strArg = cRequest.GetArg(pCmdDesc->m_vMandatoryArgs[ui]);
    if(strArg.Length() == 0)
    {
      cAnswer.AddMessage(strCmd + ": \"" +
                         pCmdDesc->m_vMandatoryArgs[ui] + "\" is mandatory");
      cAnswer.AddMessage(pCmdDesc->m_strUsage);
      iError = GEN_ERR;
    }
  }

  for(unsigned int ui = 0; ui < pCmdDesc->m_vBooleans.Size(); ui++)
  {
    C_String strArg = cRequest.GetArg(pCmdDesc->m_vBooleans[ui]);
    if((strArg != "0") && (strArg != "1"))
    {
      cAnswer.AddMessage(strCmd + ": \"" +
                         strArg + " isn't a boolean value (\"0\" or \"1\")");
      cAnswer.AddMessage(pCmdDesc->m_strUsage);
      iError = GEN_ERR;
    }
  }


  if(iError)
    cAnswer.SetStatus(ADMIN_COMMAND_NOT_VALID);
  else
    cAnswer.SetStatus(ADMIN_WELLFORMED_COMMAND);

  return cAnswer;
}


//------------------------------------------------------------------------------
// Request treatment
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Admin::HandleRequest(const C_Request& cRequest)
{
  C_Answer cAnswer("Admin");

  if(m_bRequestsEnabled)
  {
    // Lock the mutex so that only one command can be processed at a time
    // No more needed, done in the manager
    m_cRequestLock.Lock();

    C_String strCmd = cRequest.GetCmd();
    if(strCmd == "help")
    {
      C_String strArg = cRequest.GetArg("command");
      if(strArg.Length() == 0)
      {
        cAnswer.SetStatus(NO_ERR);
        C_HashTableIterator<C_String, C_CommandDesc> cIterator =
                        m_cCmdDescriptions.CreateIterator();
        while(cIterator.HasNext())
        {
          C_HashTableNode<C_String, C_CommandDesc>* pNode =
                        cIterator.GetNext();
          C_CommandDesc* pCmdDesc = pNode->GetValue();
          cAnswer.AddMessage(pCmdDesc->m_strName + ": " + pCmdDesc->m_strHelp);
        }
      }
      else
      {
        C_CommandDesc* pCmdDesc = m_cCmdDescriptions.Get(strArg);
        if(pCmdDesc)
        {
          cAnswer.SetStatus(NO_ERR);
          cAnswer.AddMessage(pCmdDesc->m_strUsage);
          cAnswer.AddMessage(pCmdDesc->m_strLongHelp);
        }
        else
        {
          cAnswer.AddMessage(strArg + ": unknown command.");
        }
      }
    }
    else if(strCmd == "shutdown")
    {
      Log(m_hLog, LOG_NOTE, "Received 'shutdown' command: stopping vls");
      C_Application* pApp = C_Application::GetApp();
      pApp->Stop();

      cAnswer.SetStatus(NO_ERR);
      cAnswer.AddMessage("Shutdowning the server");
    }
    else
    {
      // Forward the request to the app
      cAnswer = m_pRequestHub->ForwardRequest(cRequest);
    }
  }
  else
  {
    cAnswer.SetStatus(GEN_ERR);
    cAnswer.AddMessage("Requests are disabled");
  }

  // Unlock the mutex to allow another request to be processed
  m_cRequestLock.UnLock();

  return cAnswer;
}

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Admin::HandleEvent(const C_Event& cEvent)
{
  m_pNativeAdmin->PropagateEvent(cEvent);

  // To do: send something to the users loggued by telnet
}


