/*******************************************************************************
* streamdescr.cpp: MPEG stream description
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: streamdescr.cpp,v 1.1 2001/10/06 21:23:36 bozo Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*          Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
* TODO: DTS support
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

#include "../core/core.h"
#include "mpeg.h"
#include "ts.h"
#include "streamdescr.h"


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_ProgramDescriptor::C_ProgramDescriptor(unsigned int iMpegVersion)
{
  ASSERT(iMpegVersion == 1 || iMpegVersion == 2);

  memset(m_apElementaryStreams, 0, sizeof(m_apElementaryStreams));  
  memset(m_apP1ElementaryStreams, 0, sizeof(m_apP1ElementaryStreams));  

  // A virer
  ASSERT(sizeof(m_apElementaryStreams) == 4*0xFF);

  // Init the PAT and the PMT. The PMT for the program will be carried on the
  // 0x50 pid as we're sure that there is nothing else here. The PCR_PID is set
  // to 0x0 since it is legal and we don't have any other at that time
  m_cPAT.BuildHeader(0x1, 0, true);
  m_cPAT.AddPgrm(0x10, 0x50);

  m_cPMT.BuildHeader(0x50, 0x10, 0, 0, true);

  m_iNextPid = 0x80;
  m_iMpegVersion = iMpegVersion;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_ElementDescriptor* C_ProgramDescriptor::AddES(u8 iId, u8 iSubId)
{
  C_ElementDescriptor* pDescr;
  if(iId == PES_ID_PRIVATE_1)
    pDescr = m_apP1ElementaryStreams[iSubId];
  else
    pDescr = m_apElementaryStreams[iId];

  if(pDescr)
  {
    // pgrm already added -> just add the descriptors if any (to do)
  }
  else
  {
    u16 iPid = GetNewPid();
    printf("New Pid assigned: %x\n", iPid);
    pDescr = new C_ElementDescriptor(iId, iPid);
    if(iId == PES_ID_PRIVATE_1)
      m_apP1ElementaryStreams[iSubId] = pDescr;
    else
      m_apElementaryStreams[iId] = pDescr;

    // Update the PMT. The data we filter are not the correct ones -> to 
    // change
    u8 iESType = TS_TYPE_NULL;
    // MPEG video ES
    if(iId >= PES_ID_MIN_VIDEO && iId <= PES_ID_MAX_VIDEO)
    {
      printf("Video: 0x%x , %d\n", pDescr->GetPid(), pDescr->GetPid());
      if(m_iMpegVersion == 1)
        iESType = TS_TYPE_MPEG1_VIDEO;
      else
        iESType = TS_TYPE_MPEG2_VIDEO;
    }
    // MPEG audio ES
    else if(iId >= PES_ID_MIN_AUDIO && iId <= PES_ID_MAX_AUDIO)
    {
      printf("Audio: 0x%x , %d\n", pDescr->GetPid(), pDescr->GetPid());
      if(m_iMpegVersion == 1)
        iESType = TS_TYPE_MPEG1_AUDIO;
      else
        iESType = TS_TYPE_MPEG2_AUDIO;
    }
    // MPEG Private 1 ES
    else if(iId == PES_ID_PRIVATE_1)
    {
      printf("Private 1: SubId: 0x%x , Pid: 0x%x , %d\n",
             iSubId, pDescr->GetPid(), pDescr->GetPid());
      // subtitles
      if((iSubId >= PES_PRIV_ID_MIN_SPU) &&
         (iSubId <= PES_PRIV_ID_MAX_SPU))
        iESType = TS_TYPE_SPU;
      // ac3 audio stream
      else if((iSubId >= PES_PRIV_ID_MIN_AC3) &&
              (iSubId <= PES_PRIV_ID_MAX_AC3))
        iESType = TS_TYPE_AC3;
      // lpcm audio stream
      else if((iSubId >= PES_PRIV_ID_MIN_LPCM) &&
              (iSubId <= PES_PRIV_ID_MAX_LPCM))
        iESType = TS_TYPE_LPCM;
      // other
      else
        iESType = TS_TYPE_NULL;
    }
    // Padding
    else if(iId == PES_ID_PADDING)
      iESType = TS_TYPE_MPEG2_PRIVATE;

    if(iESType != TS_TYPE_NULL)
    {
      printf("PMT Add, PID : 0x%x , Type : 0x%x\n", pDescr->GetPid(), iESType);
      int iRc = m_cPMT.AddEs(iESType, pDescr->GetPid());
      if(iRc)
      {
        // Pas assez de place dans la table -> koi faire ????
      }
    }

    // Check if this pgrm must become the PCR_PID
    if(iId >= PES_ID_MIN_VIDEO && iId <= PES_ID_MAX_VIDEO)
    {
      u16 iPcrPid = m_cPMT.GetPcrPid();
      printf("updating PCR_PID to value %d (current pid = %d)\n",
             pDescr->GetPid(), iPcrPid);
      ASSERT((iPcrPid & 0xe000) == 0);
      
      if(iPcrPid == 0)
      {
        printf("New pid: %d\n", pDescr->GetPid());
        m_cPMT.UpdatePcrPid(pDescr->GetPid());
      }
    }
  }
  
  return pDescr;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
C_ElementDescriptor* C_ProgramDescriptor::GetDescriptor(u8 iId, u8 iSubId)
{
  C_ElementDescriptor* pDescr;
  if(iId != PES_ID_PRIVATE_1)
    pDescr = m_apElementaryStreams[iId];
  else
    pDescr = m_apP1ElementaryStreams[iSubId];
  if(!pDescr)
  {
    // Add the descriptor
    pDescr = AddES(iId, iSubId);
  }
  
  return pDescr;
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_ProgramDescriptor::WritePAT(C_TsPacket* pPacket)
{
  m_cPAT.Write(pPacket);
}


//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
// 
//------------------------------------------------------------------------------
void C_ProgramDescriptor::WritePMT(C_TsPacket* pPacket)
{
  m_cPMT.Write(pPacket);
}


