/*****************************************************************************
 * callback.c
 * Provide all callbacks called by the snmplib when snmp_read and
 * snmp_timeout
 *****************************************************************************
 * Copyright (C) 1998, 1999, 2000, 2001 VideoLAN
 * $Id: callback.c,v 1.15 2001/04/30 14:58:54 gunther Exp $
 *
 * Authors: Damien Lucas <nitrox@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <arpa/inet.h>                                            /* types.h */
#include <stdlib.h>                                                /* malloc */
#include <semaphore.h>                                           /* sem_post */
#include <netdb.h>                                              /* if_snmp.h */
#include <ucd-snmp/asn1.h>                                      /* if_snmp.h */
#include <ucd-snmp/snmp.h>                                      /* if_snmp.h */
#include <ucd-snmp/snmp_impl.h>                                 /* if_snmp.h */
#include <ucd-snmp/default_store.h>                             /* if_snmp.h */
#include <ucd-snmp/snmp_api.h>                                  /* if_snmp.h */
#include <ucd-snmp/snmp_client.h>                               /* if_snmp.h */
#include <ucd-snmp/mib.h>                                       /* if_snmp.h */

#include "../types.h"
#include "../logger.h"

#include "snmp_switch.h"                               /* struct SNMP_swicth */
#include "if_snmp.h"                                   /* SNMP_get_vlan, ... */
#include "snmp.h"                                         /* current_request */
#include "../vlanserver.h"                                             /* vs */
#include "callback.h"

/* Local static function */
static void end_init (void);

/*****************************************************************************
 *  Callback_Vlans()
 *****************************************************************************
 *  Treat the pdu in order to get the VLAN of a port
 *  And put it in the struct zwitch
 *  And send the GETNEXT packet if the walk is not at the end
 *****************************************************************************/
inline static unsigned int Callback_Vlans(struct snmp_session *session,
                                          struct SNMP_switch *zwitch,
                                          struct variable_list *pdu_var)
{
  char buffer[SPRINT_MAX_LEN];
  unsigned long long int vlan;
  unsigned int port;
   
  session->sessid++;
  zwitch->walk_vlans.session_num++;
  sprint_variable(buffer,pdu_var->name,pdu_var->name_length,pdu_var);

  /* Test if the walk is at the end */
  if(!memcmp(buffer, zwitch->walk_vlans.stop,zwitch->walk_vlans.stop_size))
  {
    /* Need to operate on received pdu
     * ifMIB.ifMIBObjects.ifStackTable.ifStackEntry.ifStackStatus.P.V
     *                                                           ^59  */
            
    switch(zwitch->type)
    {
      case VS_ST_3C1000:
      case VS_ST_3CDESK:
      case VS_ST_3C1100:
      case VS_ST_3C3300:
        if(sscanf(buffer+59,"%llu.%u = active(1)",&vlan,&port)==2)
        {
          if (port<Snmp_PORT(zwitch->nb_ports,zwitch->unit)+1 \
                                                      && port>100*zwitch->unit) 
          {
            zwitch->ports[PORT_Snmp(port,zwitch->unit)].vlan=vlan;
          }
        } 
        break;
    }
            
    /* Need to send the next request */
    return 1;
  }
  else
  {
    /* We reached end of the walk 
     * Let's see if the init was correctly done */
    for(port=1;port<zwitch->nb_ports+1;port++)
    {
      if(zwitch->ports[port].vlan==0)
      {
        if(zwitch->ports[port].flags&PF_BBPORT)
        {
          /* Consider here init failed */
          VS_log(LOGERROR,SNMP,"Switch %s init: Walk on VLANs failed",\
                                                          zwitch->s2.peername);
          SNMP_get_vlans(vs->snmp,zwitch);
          return 2;
        }
      }
    }
    /* When this part is reached init is correctly done */
    current_requests--;
    VS_log(LOGDEBUG,SNMP,"Walk on vlans succeeded. (CR: %u)",current_requests);
    zwitch->walk_vlans.session_num=0;
    
    if(current_requests==0)
    {
      /* We reach end of initialization */
      end_init();
    }
    return 2;
  }
} 


/*****************************************************************************
 *  Callback_Vlanids()
 ******************************************************************************
 *  Treat the pdu in order to get the VLANid of a VLAN
 *  And put it in the struct zwitch
 *  And send the GETNEXT packet if the walk is not at the end
 ******************************************************************************/

inline static unsigned int  Callback_Vlanid(struct snmp_session *session,
                                            struct SNMP_switch *zwitch,
                                            struct variable_list *pdu_var)

{
  char buffer[SPRINT_MAX_LEN];
  char sarass[64];
  unsigned long long int vlan;
  unsigned int i;
  int res;
  session->sessid++;
  zwitch->walk_table.session_num++;
  sprint_variable(buffer,pdu_var->name,pdu_var->name_length,pdu_var);

  /* Test if the walk is at the end */
  if(!memcmp(buffer,zwitch->walk_table.stop,zwitch->walk_table.stop_size))
  {
    /* Need to operate on received pdu
     * interfaces.ifTable.ifEntry.ifDescr.xxyy = RMON VLAN 1
     *                                   ^35                 */
    
    switch(zwitch->type)
    {
      case VS_ST_3C1000:
      case VS_ST_3CDESK:
        res=sscanf(buffer+35,"%Lu = RMON:VLAN %u:%s",&vlan,&i,sarass);
        if(res>=2)
        {
          zwitch->vlanid[i]=vlan;
        }
        break;

      case VS_ST_3C1100:
        if(sscanf(buffer+35,"%Lu = RMON VLAN %u",&vlan,&i)==2)
        {
          zwitch->vlanid[i]=vlan;
        }
        break;
                           
      case VS_ST_3C3300:
        if(sscanf(buffer+35,"%Lu = RMON VLAN %u",&vlan,&i)==2)
        {
          zwitch->vlanid[i]=vlan;
        }
        break;
    }
            
    /* Need to send the next request */
    return 1;
  }
  else
  {
    /* We reached end of the walk 
     * Let's see if the init was correctly done */
    for(i=1;i<MAX_VLAN_NB+1;i++)
    {
      if(zwitch->vlanid[i]==0)
      {
        /* Consider here init failed */
        VS_log(LOGERROR,SNMP,"Switch %s init: Walk on VLANids failed",\
                                                          zwitch->s2.peername);
        SNMP_get_vlanids(vs->snmp,zwitch);
        return 2;
      }
    }
    /* When this part is reached init is correctly done */
    current_requests--;
    VS_log(LOGDEBUG,SNMP,"Walk on Vlanids succeeded.(CR: %u)",\
                                                             current_requests);
    zwitch->walk_table.session_num=0;
    if(current_requests==0)
    {
      /* We reach end of initialization */
      end_init();
    }
    return 2;
  }
}

/*****************************************************************************
 *  Callback_Macs()
 *****************************************************************************
 *  Treat the pdu in order to get one the MAC of a port
 *  And put it in the struct zwitch
 *  And send the GETNEXT packet if the walk is not at the end
 *****************************************************************************/
inline static unsigned int Callback_Macs(struct snmp_session *session,
                                         struct SNMP_switch *zwitch,
                                         struct variable_list *pdu_var)
{
  char buffer[SPRINT_MAX_LEN];
  VS_MachineId mac,i2;
  unsigned int i3,i4,i5,i6;
  unsigned int port;
  
  session->sessid++;
  zwitch->walk_macs.session_num++;
        
  sprint_variable(buffer,pdu_var->name,pdu_var->name_length,pdu_var);

  /* Test if the walk is at the end */
  if(!memcmp(buffer, zwitch->walk_macs.stop,zwitch->walk_macs.stop_size))
  {
    /* Need to operate on received pdu
     * enterprises.43.10.22.2.1.3.unit.port.i1.i2.i3...
     *                                ^ 29             */
                
    if(sscanf(buffer+29,"%u.%Lu.%Lu.%u.%u.%u.%u",&port,&mac,\
                                                       &i2,&i3,&i4,&i5,&i6)!=7)
    {
      VS_log(LOGERROR,SNMP,"Bad snmp GETRESPONSE");
      (*zwitch->walk_macs.machine)->next=NULL;
      for(i3=zwitch->walk_macs.last_port+1;i3<=zwitch->nb_ports;i3++)
      {
        zwitch->machines[i3]=NULL;
      }
      zwitch->walk_macs.session_num=0;
      return 2;
    }
   
    /* Check for the port number */
    if(port>zwitch->nb_ports)
    {
      VS_log(LOGERROR,SNMP,"Found MAC on an out of range port, skipping");
      return 1;
    }


    mac<<=8; mac|=i2;
    mac<<=8; mac|=i3;
    mac<<=8; mac|=i4;
    mac<<=8; mac|=i5;
    mac<<=8; mac|=i6;
               
    /* Chesck for the mac */
    if(mac>0xffffffffffff)
    {
      VS_log(LOGERROR,SNMP,"Found an illegal mac, skipping");
      return 1;
    }

    *zwitch->walk_macs.machine=malloc(sizeof(struct SNMP_machines_elt));
    
    if(*zwitch->walk_macs.machine==NULL)
    {
      VS_log(LOGERROR,SNMP,"Unable to allocate memory");
      for(i3=zwitch->walk_macs.last_port+1;i3<=zwitch->nb_ports;i3++)
      {
        zwitch->machines[i3]=NULL;
      }
      zwitch->walk_macs.session_num=0;
      return 2;
    }
                    
    if(port==zwitch->walk_macs.last_port)
    {
      (*zwitch->walk_macs.machine)->machine=mac;
      zwitch->walk_macs.machine=&(*zwitch->walk_macs.machine)->next;
    }
    else
    {
      (*zwitch->walk_macs.machine)=NULL;
      
      for(i3=zwitch->walk_macs.last_port+1;i3<port;i3++)
      {
        zwitch->machines[i3]=NULL;
      }
       
      zwitch->walk_macs.machine=zwitch->machines+i3;
      *zwitch->walk_macs.machine=malloc(sizeof(struct SNMP_machines_elt));
                        
      zwitch->walk_macs.last_port=port;
      (*zwitch->walk_macs.machine)->machine=mac;
      zwitch->walk_macs.machine=&(*zwitch->walk_macs.machine)->next;

    }
                     
    /* Need to send the next request */
    return 1;
  }
  else
  {
    /* complete the MAC table */
    *zwitch->walk_macs.machine=NULL;
    for(i3=zwitch->walk_macs.last_port+1;i3<=zwitch->nb_ports;i3++)
    {
      zwitch->machines[i3]=NULL;
    }
    
    /* We reached end of the walk
     * BUT we can't check if all the macs have correctly been learned
     * so we'll just check for their size */
    
    /* When this part is reached init is correctly done */
    zwitch->walk_macs.session_num=0;
    current_requests--;
    VS_log(LOGDEBUG,SNMP,"Walk on macs succeeded. (CR: %u)",current_requests);
    if(current_requests==0)
    {
      /* We reach end of initialization */
      end_init();
    }
    return 2;
  }
}


/*****************************************************************************
 *  Callback_Portid()
 *****************************************************************************
 *  Treat the pdu in order to get the PORTid of a PORT (on Unit X)
 *  And put it in the struct zwitch
 *  And send the GETNEXT packet if the walk is not at the end
 *****************************************************************************/

inline static unsigned int Callback_Portid(struct snmp_session *session,
                                   struct SNMP_switch *zwitch,
                                   struct variable_list *pdu_var)

{
  char buffer[SPRINT_MAX_LEN];
  unsigned long long int port;
  unsigned int i,unit;
  
  zwitch->walk_ports.session_num++;
  session->sessid++;
  sprint_variable(buffer,pdu_var->name,pdu_var->name_length,pdu_var);

  /* Test if the walk is at the end */
  if(!memcmp(buffer,zwitch->walk_ports.stop,zwitch->walk_ports.stop_size))
  {
    /* Need to operate on received pdu
     * interfaces.ifTable.ifEntry.ifDescr.xxyy = RMON VLAN 1
     *                                   ^35                 */

    if (sscanf(buffer+35,"%Lu = RMON:V3 Port %u",&port,&i) == 2)
    {
      zwitch->portid[i][1]=port;	   
    }
    
    else if (sscanf(buffer+35,"%Lu = RMON:V2 Port %u on Unit %u",\
	                                                  &port,&i,&unit) == 3)
    {
      zwitch->portid[i][unit]=port;
    }

    else if (sscanf(buffer+35,"%Lu = RMON:V2 Port %u",&port,&i) == 2)
    {
      zwitch->portid[i][1]=port;
    }
    
    else if (sscanf(buffer+35,"%Lu = RMON:10/100 Port %u on Unit %u",\
		                                          &port,&i,&unit) == 3)
    {
      zwitch->portid[i][unit]=port;
    }	    
	    
	    /* Need to send the next request */
    return 1;
  }
  else
  {
    /* We reached end of the walk 
     * Let's see if the init was correctly done 
     * 
     * (We only test the nb_ports ports of the unit n
     * we are not sure that the others ports do exist)    */ 
	  
    for(i=1;i<=zwitch->nb_ports;i++)
    {
      if((zwitch->portid[i][zwitch->unit]==0) &&
         !(zwitch->ports[i].flags&PF_ATMPORT) &&
         !(zwitch->ports[i].flags&PF_UNUSEDPORT))
      {
        /* Consider here init failed */
        VS_log(LOGERROR,SNMP,"Switch %s init: Walk on PORTids failed",\
                                                          zwitch->s2.peername);
        SNMP_get_portids(vs->snmp,zwitch);
        return 2;
      }
    }
    /* When this part is reached init is correctly done */
    current_requests--;
    VS_log(LOGDEBUG,SNMP,"Walk on Portids successed.(CR: %u)",\
                                                             current_requests);
    zwitch->walk_ports.session_num=0;
    if(current_requests==0)
    {
      /* We reach end of initialization */
      end_init();
    }
    return 2;
  }
}

/*****************************************************************************
 * SNMP_SW_callback: Called after any reply of a switch.
 *****************************************************************************
 * This function is called each time a snmp_read occured
 * In fact, it is called for each answer and each walk step
 * It calls SNMP_get_next cause we can't send anything
 *  (we are inside snmp_read !)
 * Return value of the sub functions are:
 *   2 means nothing to be done
 *   1 means everything ok, send next packet
 *   0 means errored packet, need to be reasked
 *****************************************************************************/

int SNMP_SW_callback(int op,struct snmp_session *session,int reqid,\
                     struct snmp_pdu *pdu, struct SNMP_switch *zwitch)
{
  unsigned int rc=2;
  
  if(op==RECEIVED_MESSAGE)
  {
    switch(pdu->command)
    {
      case SNMP_MSG_RESPONSE:
        /* Response to a set or the next response to a walk */ 
        if(session->sessid == zwitch->walk_macs.session_num)
        {
          /* Pdu received is part of the walk on Macs */
          rc=Callback_Macs(session,zwitch,pdu->variables);
        }
        else if(session->sessid == zwitch->walk_table.session_num)
        {
          /* Pdu received is part of the walk on vlanids */
          rc=Callback_Vlanid(session,zwitch,pdu->variables);
        }

        else if(session->sessid == zwitch->walk_ports.session_num)
        {
          /* Pdu received is part of the walk on portids */
          rc=Callback_Portid(session,zwitch,pdu->variables);
	}
	
	else if(session->sessid == zwitch->walk_vlans.session_num)
        {
          /* Pdu received is part of the walk on vlans */
          rc=Callback_Vlans(session,zwitch,pdu->variables);
        }
        session->s_snmp_errno=SNMPERR_SUCCESS;
        break;

      case SNMP_MSG_TRAP:
        /* Packet was a snmp-trap */
        VS_log(LOGINFO,SNMP,"TRAP !!!!!"); 
        /* Be carefull not to free it */
        break;
    }

  }
  else if(op==TIMED_OUT)
  {
    /* pdu have been resent n times (default is 5 in surrent lib */
    /* we have to consider the switch is unusable */
    VS_log(LOGERROR,SNMP,"Timeout about switch %s!",zwitch->s2.peername);
    current_requests--;
    zwitch->badinit=1;
    if(current_requests==0)
    {
      /* We reach end of initialization */
      end_init();
    }
    
        
  }

  switch(rc)
  {
    case 0:
      return 0;
      break;
      
    case 1:
      /* Need to send next packet */
      SNMP_get_next(vs->snmp,zwitch,pdu,session);
      return 1;
      break;
      
    case 2:
      return 1;
      break;
      
      
  }
  return 1;
}


/*****************************************************************************
 * end_init: Called when current_request reach 0 means end of init
 *****************************************************************************/
static void end_init(void)
{
  sem_post(vs->snmp->sem_end_init);
}

