/*******************************************************************************
* converter.cpp: Threaded Mpeg converters
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: converter.cpp,v 1.1 2001/10/06 21:23:36 bozo Exp $
*
* Authors: Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../../core/defs.h"

#include "../../core/core.h"

#include "../../mpeg/mpeg.h"
#include "../../mpeg/ts.h"
#include "../../mpeg/streamdescr.h"
#include "../../mpeg/ps2ts.h"

#include "../../server/buffer.h"
#include "../../server/program.h"
#include "../../server/broadcast.h"
#include "../../server/request.h"
#include "../../server/input.h"

#include "reader.h"
#include "converter.h"


#include "../../mpeg/ps2ts.cpp"


#define STATUS_RUNNING          0
#define STATUS_PAUSE_REQUEST    1
#define STATUS_PAUSED           2


/*******************************************************************************
* C_MpegConverter
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_MpegConverter::C_MpegConverter(handle hLog, C_Broadcast* pBroadcast,
                                 C_MpegReader* pReader,
                                 C_NetList* pTsProvider, C_SyncFifo* pBuffer,
                                 C_EventHandler* pEventHandler)
{
  ASSERT(hLog);
  ASSERT(pBroadcast);
  ASSERT(pReader);
  ASSERT(pTsProvider);
  ASSERT(pBuffer);
  ASSERT(pEventHandler);

  m_hLog = hLog;
  m_pBroadcast = pBroadcast;
  m_pReader = pReader;
  m_pTsProvider = pTsProvider;
  m_pBuffer = pBuffer;
  m_pEventHandler = pEventHandler;
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_MpegConverter::~C_MpegConverter()
{
  delete m_pReader;
};


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
void C_MpegConverter::InitWork()
{
  m_iShortPauseStatus = m_iLongPauseStatus = STATUS_RUNNING;

  m_cResumeCond.Protect();

  m_bStop = false;
}


//------------------------------------------------------------------------------
// Stop request
//------------------------------------------------------------------------------
void C_MpegConverter::StopWork()
{
  m_bStop = true;

  DelayedInterruption();
}


//------------------------------------------------------------------------------
// Resume streaming
//------------------------------------------------------------------------------
void C_MpegConverter::Resume()
{
  m_cResumeCond.Signal();

  m_cResumeCond.Release();
}


//------------------------------------------------------------------------------
// Suspend streaming
//------------------------------------------------------------------------------
void C_MpegConverter::Suspend()
{
  m_iLongPauseStatus = STATUS_PAUSE_REQUEST;

  m_cResumeCond.Protect();
}


//------------------------------------------------------------------------------
// Short pause (use it to access for a very short time to the reader
//------------------------------------------------------------------------------
void C_MpegConverter::ShortPause()
{
  m_iShortPauseStatus = STATUS_PAUSE_REQUEST;

  m_cResumeCond.Protect();
}


/*******************************************************************************
* C_Ts2TsConverter
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_Ts2TsConverter::C_Ts2TsConverter(handle hLog, C_Broadcast* pBroadcast,
                                   C_MpegReader* pReader,
                                   C_NetList* pTsProvider, C_SyncFifo* pBuffer,
                                   C_EventHandler* pEventHandler) :
                            C_MpegConverter(hLog, pBroadcast,
                                            pReader,
                                            pTsProvider, pBuffer,
                                            pEventHandler)
{
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
void C_Ts2TsConverter::InitWork()
{
  // Inherited method
  C_MpegConverter::InitWork();

  int iRc = 0;

  // Fill the buffer
  C_TsPacket* pPacket = m_pTsProvider->GetPacket();
  ASSERT(pPacket);
  iRc = SyncFillPacket(pPacket);
  m_pBuffer->Push(pPacket);

  if(!iRc)
  {
    for(unsigned int ui = 1; ui < m_pBuffer->Capacity(); ui++)
    {
      C_TsPacket* pPacket = m_pTsProvider->GetPacket();
      ASSERT(pPacket);
      iRc = FillPacket(pPacket);
      m_pBuffer->Push(pPacket);
      if(iRc)
        break;
    }
  }

  if(iRc == MPEG_ENDOFSTREAM)
    iRc = NO_ERR;
  else if(iRc)
    throw E_Exception(GEN_ERR, "TS to TS converter failed to start");
}


//------------------------------------------------------------------------------
// Main job
//------------------------------------------------------------------------------
void C_Ts2TsConverter::DoWork()
{
  C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();

  LogDbg(m_hLog, "Starting to read program \"" + strPgrmName + "\"");

  int iRc = 0;
  bool bDiscontinuity = false;

  // Stream the file
  while(!m_bStop && !iRc)
  {
    // Suspend/Resume handling
    if(m_iShortPauseStatus == STATUS_PAUSE_REQUEST)
    {
      m_cResumeCond.Wait();
      m_iShortPauseStatus = STATUS_RUNNING;
    }

    if(m_iLongPauseStatus == STATUS_PAUSE_REQUEST)
    {
      m_cResumeCond.Wait();
      m_iLongPauseStatus = STATUS_RUNNING;
      bDiscontinuity = true;
    }

    // Get a packet to fill
    C_TsPacket* pPacket = m_pTsProvider->GetPacket();
    ASSERT(pPacket);
    // Fill it with the data from the file
    iRc = FillPacket(pPacket);
    // Check stream discontinuity
    if(!iRc && (m_pReader->HasDiscontinuity() || bDiscontinuity))
    {
      while((((*pPacket)[0] != 0x47) || !pPacket->HasPCR()) && !iRc)
        iRc = SyncFillPacket(pPacket);
      if(!iRc)
      {
        ASSERT(pPacket->SetDiscontinuityFlag());
        pPacket->SetDiscontinuityFlag();
        m_pReader->ResetDiscontinuity();
        bDiscontinuity = false;
      }
    }

    if(!iRc)
    {
      // Stores the packet in the buffer
      m_pBuffer->Push(pPacket);
    }
    else if(iRc == MPEG_ENDOFSTREAM)
    {
      C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();
      LogDbg(m_hLog, "End of program \"" + strPgrmName + "\" reached");
      m_pTsProvider->ReleasePacket(pPacket);
    }
    else
    {
      C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();
      Log(m_hLog, LOG_ERROR, "Read error for program \"" + strPgrmName + "\"");
      m_pTsProvider->ReleasePacket(pPacket);
    }
  }

  if(!m_bStop)
  {
    LogDbg(m_hLog, "Stopping converter by callback for program " + strPgrmName);
    m_bStop = true;
    C_Event cEvent(m_pBroadcast->GetInput()->GetName());
    cEvent.SetCode(EOF_EVENT);
    cEvent.SetBroadcast(m_pBroadcast);
    m_pEventHandler->HandleEvent(cEvent);
  }

  LogDbg(m_hLog, "Converter stopped for program " + strPgrmName);
}


//------------------------------------------------------------------------------
// Get the next 188 bytes to fill the TS packet
//------------------------------------------------------------------------------
int C_Ts2TsConverter::FillPacket(C_TsPacket* pPacket)
{
  int iRc = m_pReader->Read((*pPacket), TS_PACKET_LEN);

  if(iRc == TS_PACKET_LEN)
  {
    iRc = NO_ERR;
  }
  else if(iRc >= 0)
  {
    // Mark the packet as bad
    pPacket->SetErrorFlag();
    iRc = NO_ERR;
  }
  else if(iRc == MPEG_ENDOFSTREAM)
  {
    // Pad the end of the packet with 0 and mark it as bad if not some
    // decoders could crash
    (*pPacket)[0] = 0x47;
    for(int i = 1; i < TS_PACKET_LEN; i++)
      (*pPacket)[i] = 0;
    pPacket->SetErrorFlag();
  }
  else
  {
    // Mark the packet as bad
    pPacket->SetErrorFlag();
    iRc = MPEG_STREAMERROR;
  }

  return iRc;
}


//------------------------------------------------------------------------------
// Synchronize in the TS stream and then fill the TS packet
//------------------------------------------------------------------------------
int C_Ts2TsConverter::SyncFillPacket(C_TsPacket* pPacket)
{
  (*pPacket)[0] = 0;
  int iRc = 1;
  while(((*pPacket)[0] != 0x47) && (iRc == 1))
    iRc = m_pReader->Read((*pPacket), 1);

  if(iRc == 1)
  {
    iRc = NO_ERR;
  }
  else if(iRc >= 0)
  {
    // Mark the packet as bad
    pPacket->SetErrorFlag();
    iRc = NO_ERR;
  }
  else if(iRc == MPEG_ENDOFSTREAM)
  {
    // Pad the end of the packet with 0 and mark it as bad if not some
    // decoders could crash
    (*pPacket)[0] = 0x47;
    for(int i = 1; i < TS_PACKET_LEN; i++)
      (*pPacket)[i] = 0;
    pPacket->SetErrorFlag();
  }
  else
  {
    // Mark the packet as bad
    pPacket->SetErrorFlag();
    iRc = MPEG_STREAMERROR;
  }

  if(!iRc)
  {
    iRc = m_pReader->Read((byte*)(*pPacket) + 1, TS_PACKET_LEN - 1);

    if(iRc == TS_PACKET_LEN - 1)
    {
      iRc = NO_ERR;
    }
    else if(iRc >= 0)
    {
      // Mark the packet as bad
      pPacket->SetErrorFlag();
      iRc = NO_ERR;
    }
    else if(iRc == MPEG_ENDOFSTREAM)
    { 
      // Pad the end of the packet with 0 and mark it as bad if not some
      // decoders could crash
      for(int i = 1; i < TS_PACKET_LEN; i++)
      (*pPacket)[i] = 0;
      pPacket->SetErrorFlag();
    } 
    else
    {
      // Mark the packet as bad
      pPacket->SetErrorFlag();
      iRc = MPEG_STREAMERROR;
    }
  }

  return iRc;
}


/*******************************************************************************
* C_Ps2TsConverter
********************************************************************************
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_Ps2TsConverter::C_Ps2TsConverter(handle hLog, C_Broadcast* pBroadcast,
                                   C_MpegReader* pReader,
                                   C_NetList* pTsProvider, C_SyncFifo* pBuffer,
                                   C_EventHandler* pEventHandler,
                                   unsigned int iMpegVersion, bool bPreParse) :
                            C_MpegConverter(hLog, pBroadcast,
                                            pReader,
                                            pTsProvider, pBuffer,
                                            pEventHandler),
                            m_cConverter(pReader, m_pTsProvider, 2,
                                         iMpegVersion)
{
  m_bPreParse = bPreParse;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_ProgramDescriptor* C_Ps2TsConverter::GetPgrmDescriptor()
{
  return m_cConverter.GetPgrmDescriptor();
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
void C_Ps2TsConverter::InitWork()
{
  C_MpegConverter::InitWork();

  m_cConverter.Synch();

  int iRc = 0;

  if(m_bPreParse)
  {
    for(unsigned int i = 0; i < 100000; i++)
    {
      // preparsing to make the PAT and the PMT
      C_TsPacket* pPacket = m_cConverter.GetPacket();
      ASSERT(pPacket);
      m_pTsProvider->ReleasePacket(pPacket);
    }

    // Back to the beginning
    m_pReader->Seek(0, FILE_SEEK_BEGIN);
    m_cConverter.Synch();
  }

  for(unsigned int ui = 1; ui < m_pBuffer->Capacity(); ui++)
  {
    C_TsPacket* pPacket = m_cConverter.GetPacket();
    ASSERT(pPacket);
    m_pBuffer->Push(pPacket);
    if(m_cConverter.GetStatus())
    {
      iRc = MPEG_STREAMERROR;
      break;
    }
  }

  if(iRc == MPEG_ENDOFSTREAM)
    iRc = NO_ERR;
  else if(iRc)
    throw E_Exception(GEN_ERR, "PS to TS converter failed to start");
}


//------------------------------------------------------------------------------
// Main job
//------------------------------------------------------------------------------
void C_Ps2TsConverter::DoWork()
{
  C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();

  LogDbg(m_hLog, "Starting to read program \"" + strPgrmName + "\"");

  int iRc = 0;
  bool bDiscontinuity = false;

  // Stream the file
  while(!m_bStop && !iRc)
  {
    // Suspend/Resume handling
    if(m_iShortPauseStatus == STATUS_PAUSE_REQUEST)
    {
      m_cResumeCond.Wait();
      m_iShortPauseStatus = STATUS_RUNNING;
    }

    if(m_iLongPauseStatus == STATUS_PAUSE_REQUEST)
    {
      m_cResumeCond.Wait();
      m_iLongPauseStatus = STATUS_RUNNING;
      bDiscontinuity = true;
    }

    // Get a packet to fill
    C_TsPacket* pPacket = m_cConverter.GetPacket();

    switch(m_cConverter.GetStatus())
    {
    case 0:
      if(!pPacket)
        iRc = MPEG_STREAMERROR;
      break;
    case -99:
      iRc = MPEG_ENDOFSTREAM;
      break;
    default:
      iRc = MPEG_STREAMERROR;
      break;
    }

    // Check stream discontinuity
    if(    !iRc
        && (m_pReader->HasDiscontinuity() || bDiscontinuity)
        && pPacket->HasPCR())
    {
      ASSERT(pPacket);
      ASSERT(pPacket->SetDiscontinuityFlag());
      pPacket->SetDiscontinuityFlag();
      m_pReader->ResetDiscontinuity();
      bDiscontinuity = false;
    }

    if(!iRc)
    {
      // Stores the packet in the buffer
      m_pBuffer->Push(pPacket);
    }
    else if(iRc == MPEG_ENDOFSTREAM)
    {
      C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();
      LogDbg(m_hLog, "End of program \"" + strPgrmName + "\" reached");
      if(pPacket)
        m_pTsProvider->ReleasePacket(pPacket);
    }
    else
    {
      C_String strPgrmName = m_pBroadcast->GetProgram()->GetName();
      Log(m_hLog, LOG_ERROR, "Read error for program \"" + strPgrmName + "\"");
      if(pPacket)
        m_pTsProvider->ReleasePacket(pPacket);
    }
  }

  if(!m_bStop)
  {
    LogDbg(m_hLog, "Stopping converter by callback for program " + strPgrmName);
    m_bStop = true;
    C_Event cEvent(m_pBroadcast->GetInput()->GetName());
    cEvent.SetCode(EOF_EVENT);
    cEvent.SetBroadcast(m_pBroadcast);
    m_pEventHandler->HandleEvent(cEvent);
  }

  LogDbg(m_hLog, "Converter stopped for program " + strPgrmName);
}


