/*
 * Copyright (C) 2020 Collabora, Ltd.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef __BI_PACK_HELPERS_H
#define __BI_PACK_HELPERS_H

/* Helpers used by the autogenerated packing routines */

#include "compiler.h"

static inline void
bi_set_staging_register(bi_clause *clause, unsigned idx)
{
        assert(idx & BIR_INDEX_REGISTER);
        unsigned reg = idx & ~BIR_INDEX_REGISTER;
        assert(reg <= 63);
        clause->staging_register = reg;
}

static inline void
bi_read_staging_register(bi_clause *clause, bi_instruction *ins)
{
        bi_set_staging_register(clause, ins->src[0]);
}

static inline void
bi_write_staging_register(bi_clause *clause, bi_instruction *ins)
{
        bi_set_staging_register(clause, ins->dest);
}

static inline enum bifrost_packed_src
bi_get_src_reg_slot(bi_registers *regs, unsigned src)
{
        unsigned reg = src & ~BIR_INDEX_REGISTER;

        if (regs->slot[0] == reg && regs->enabled[0])
                return BIFROST_SRC_PORT0;
        else if (regs->slot[1] == reg && regs->enabled[1])
                return BIFROST_SRC_PORT1;
        else if (regs->slot[2] == reg && regs->slot23.slot2 == BIFROST_OP_READ)
                return BIFROST_SRC_PORT2;
        else
                unreachable("Tried to access register with no port");
}

/* Sources are usually strictly required, but in a few special cases they can
 * be made optional with the value passed arbitrary. Check that here */

static bool
bi_src_nullable(bi_instruction *ins, unsigned s)
{
        /* Z/S flags inferred */
        if (ins->type == BI_ZS_EMIT && s < 2)
                return true;

        return false;
}

static inline enum bifrost_packed_src
bi_get_src(bi_instruction *ins, bi_registers *regs, unsigned s)
{
        unsigned src = ins->src[s];

        if (src & BIR_INDEX_REGISTER)
                return bi_get_src_reg_slot(regs, src);
        else if (src & BIR_INDEX_PASS)
                return src & ~BIR_INDEX_PASS;
        else if (!src && bi_src_nullable(ins, s))
                return BIFROST_SRC_STAGE;
        else {
#ifndef NDEBUG
                bi_print_instruction(ins, stderr);
#endif
                unreachable("Unknown src in above instruction");
        }
}

#endif
