/*
Copyright 1985, 1986, 1987, 1991, 1998  The Open Group

Portions Copyright 2000 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/
/****************/
/* Header files */
/****************/

/* C headers */
#include <stdio.h>
#include <limits.h>
#include <iconv.h>

/* Xlib header */
#include <X11/Xlocale.h>

/* Xlib internal headers */
#include "Xlibint.h"
#include "XlcGeneric.h"

/* UNICODE locale header */
#include "XlcPublicUnicode.h"


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

/***********/
/* Defines */
/***********/

#define CS_ENTRY	CARD8
#define PATH_MAX	1024

#define MB_ICONV_NAME	"mb_encoding_name"
#define WC_ICONV_NAME	"wc_encoding_name"
#define WC_VALID_LENGTH_ENTRY	"wc_valid_length"
#define WC_TO_CS_CONV_TABLE_ENTRY	"wc_conversion_table"
#define CS_TO_WC_CONV_TABLE_ENTRY	"cs_conversion_table"

#define LUT_CODESET_UNKNOWN	0xff

#define KO_UTF8		1

/*************************/
/* structure and typedef */
/*************************/

typedef int (*UnicodeTableConverterProc)(
	void		*table,
	XPointer	*from,
	int		*from_left,
	XPointer	*to,
	int		*to_left,
	CS_ENTRY	*cnN
);

typedef struct _UnicodeMbstrInfo{
	char	*iconv_name;
#ifdef ICONVOPEN
	iconv_t	iconv_towc;
#endif
	UnicodeTableConverterProc	cs_conv_func;
} UnicodeMbstrInfo;
static UnicodeMbstrInfo	*UnicodeMbstrInfo_create(XLCd lcd);
static void	UnicodeMbstrInfo_destroy(UnicodeMbstrInfo *info);

typedef struct _UnicodeWcharInfo{
	int	valid_len;
	char	*iconv_name;
	void	*table;
	UnicodeTableConverterProc	cs_conv_func;
} UnicodeWcharInfo;
static UnicodeWcharInfo	*UnicodeWcharInfo_create(XLCd lcd);
static void	UnicodeWcharInfo_destroy(UnicodeWcharInfo *info);

typedef struct _UnicodeCs{
	CodeSet	codeset;
	XlcCharSet	charset;
	void	*table;
	UnicodeTableConverterProc	wc_conv_func;
	UnicodeTableConverterProc	mb_conv_func;
} UnicodeCs;

typedef struct _UnicodeCsInfo{
	UnicodeCs	*cs;
	int	cs_num;
	int	max_length;
} UnicodeCsInfo;
static UnicodeCsInfo	*UnicodeCsInfo_create(XLCd lcd);
static void	UnicodeCsInfo_destroy(UnicodeCsInfo *info);


/*************/
/* Functions */
/*************/

/* registration functions */
static XlcConv	open_ctstombs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_ctstowcs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_mbstocts(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_wcstocts(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_wcstocs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_cstowcs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_wcstombs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_mbstowcs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_cstombs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);
static XlcConv	open_mbstocs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type);

/* converter functions */
static int uc_wcstocts(XlcConv conv, XPointer *from, int *from_left,
		       XPointer *to, int *to_left,
		       XPointer *args, int num_args);
static int uc_mbstocts(XlcConv conv, XPointer *from, int *from_left,
		       XPointer *to, int *to_left,
		       XPointer *args, int num_args);
static int uc_ctstombs(XlcConv conv, XPointer *from, int *from_left,
		       XPointer *to, int *to_left,
		       XPointer *args, int num_args);
static int uc_ctstowcs(XlcConv conv, XPointer *from, int *from_left,
		       XPointer *to, int *to_left,
		       XPointer *args, int num_args);
static int	uc_wcstocs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args);
static int	uc_cstowcs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args);
static int	uc_mbstocs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args);
static int	uc_cstombs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args);
static int	uc_wcstombs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args);
static int	uc_mbstowcs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args);
static void	close_converter(XlcConv conv);

/* local converters */
static int	convert_wc16_to_cs(void *table, XPointer *from_wc, int *from_left, XPointer *to_cs, int *to_left, CS_ENTRY *csN);
static int	convert_mb16_to_cs(void *table, XPointer *from_mb, int *from_left, XPointer *to_cs, int *to_left, CS_ENTRY *csN);
static int	convert_cs8_to_wc16(void *table, XPointer *from_cs, int *from_left, XPointer *to_wc, int *to_left, CS_ENTRY *csN);
static int	convert_wcstombs(XPointer *from, int *from_left, XPointer *to, int *to_left);
static int	convert_mbstowcs(XPointer *from, int *from_left, XPointer *to, int *to_left);
static int	convert_cstombs(void *table, XPointer *from, int *from_left, XPointer *to, int *to_left, CS_ENTRY *csN);
static int	convert_mbstocs(void *table, XPointer *from, int *from_left, XPointer *to, int *to_left, CS_ENTRY *csN);
static int	convert_cs8_to_mb16(void *table, XPointer *from_cs, int *from_left, XPointer *to_mb, int *to_left, CS_ENTRY *csN);
static int	convert_cs16_to_wc16(void *table, XPointer *from_cs, int *from_left, XPointer *to_wc, int *to_left, CS_ENTRY *csN);
static int	convert_cs16_to_mb16(void *table, XPointer *from_cs, int *from_left, XPointer *to_mb, int *to_left, CS_ENTRY *csN);

/* miscellaneous functions */
static XlcConv	create_conv(XLCd lcd, XlcConvMethods methods);
static int	make_filename(char *str, char *filename);
static void	*read_table(char *str);

#ifdef MB_LEN_MAX
#undef MB_LEN_MAX
#endif
#define MB_LEN_MAX	8


/*************/
/* Variables */
/*************/
static UnicodeMbstrInfo	*uc_mb_info;
static UnicodeWcharInfo	*uc_wc_info;
static UnicodeCsInfo	*uc_cs_info;

enum { WCSTOCS, CSTOWCS, WCSTOMBS, MBSTOWCS, CSTOMBS, MBSTOCS,
       WCSTOCTS, MBSTOCTS, CTSTOMBS, CTSTOWCS };

enum { TO_MBS, TO_WCS};
static XlcConvMethodsRec conv_methods[] = {
	{close_converter, uc_wcstocs, NULL},
	{close_converter, uc_cstowcs, NULL},
	{close_converter, uc_wcstombs, NULL},
	{close_converter, uc_mbstowcs, NULL},
	{close_converter, uc_cstombs, NULL},
	{close_converter, uc_mbstocs, NULL},
	{close_converter, uc_wcstocts, NULL},
	{close_converter, uc_mbstocts, NULL},
	{close_converter, uc_ctstombs, NULL},
	{close_converter, uc_ctstowcs, NULL},
};



/*****************/
/* Program Codes */
/*****************/


/* loader function */

XLCd
_XlcUnicodeLoader(char *name)
{
	XLCd	lcd = (XLCd)NULL;
	uc_mb_info = (UnicodeMbstrInfo *)NULL;
	uc_wc_info = (UnicodeWcharInfo *)NULL;
	uc_cs_info = (UnicodeCsInfo *)NULL;

	_XlcInitPublicMethodsUnicode();
	lcd = _XlcCreateLC(name, _XlcGenericMethods);
	if (lcd == (XLCd)NULL)
		goto err_return;

	/* For ko.euc and ko.UTF-8 data exchanging in cut&paste */
	/* Following four converters were originally made for ko.UTF-8
	 * locale.  But to support interoperability between ko.UTF-8 
	 * and other UTF-8 locales, these convertes are necessary.
	 * I rewrote ctstombs/ctstowcs converter to work on other
	 * UTF-8 locale such as en_US.UTF-8 and ja_JP.UTF-8.  mbstocts
	 * and wcstocts converters work well without modification
	 * even on non ko.UTF-8 locale
	 */
	_XlcSetConverter(lcd, XlcNCompoundText,
			     lcd, XlcNMultiByte, open_ctstombs);
	_XlcSetConverter(lcd, XlcNMultiByte,
			     lcd, XlcNCompoundText, open_mbstocts);
	_XlcSetConverter(lcd, XlcNCompoundText,
			     lcd, XlcNWideChar, open_ctstowcs);
	_XlcSetConverter(lcd, XlcNWideChar,
			     lcd, XlcNCompoundText, open_wcstocts);

	_XlcSetConverter(lcd, XlcNWideChar, lcd, XlcNCharSet, open_wcstocs);
	_XlcSetConverter(lcd, XlcNCharSet, lcd, XlcNWideChar, open_cstowcs);
	_XlcSetConverter(lcd, XlcNWideChar, lcd, XlcNMultiByte, open_wcstombs);
	_XlcSetConverter(lcd, XlcNMultiByte, lcd, XlcNWideChar, open_mbstowcs);
	_XlcSetConverter(lcd, XlcNCharSet, lcd, XlcNMultiByte, open_cstombs);
	_XlcSetConverter(lcd, XlcNMultiByte, lcd, XlcNCharSet, open_mbstocs);

	return lcd;

err_return:
	if(uc_cs_info)
		UnicodeCsInfo_destroy(uc_cs_info);
	if(uc_mb_info)
		UnicodeMbstrInfo_destroy(uc_mb_info);
	if(uc_wc_info)
		UnicodeWcharInfo_destroy(uc_wc_info);
	if(lcd)
		_XlcDestroyLC(lcd);

	return (XLCd)NULL;
}


/* register functions */

static XlcConv
open_ctstombs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[CTSTOMBS]);
}

static XlcConv
open_ctstowcs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[CTSTOWCS]);
}

static XlcConv
open_mbstocts(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[MBSTOCTS]);
}

static XlcConv
open_wcstocts(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[WCSTOCTS]);
}

static XlcConv
open_wcstocs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[WCSTOCS]);
}

static XlcConv
open_cstowcs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[CSTOWCS]);
}

static XlcConv
open_wcstombs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[WCSTOMBS]);
}

static XlcConv
open_mbstowcs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[MBSTOWCS]);
}

static XlcConv
open_cstombs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[CSTOMBS]);
}

static XlcConv
open_mbstocs(XLCd from_lcd, char *from_type, XLCd to_lcd, char *to_type)
{
    return create_conv(from_lcd, &conv_methods[MBSTOCS]);
}


/* converter functions */

/*
   ctstostr:
   	converts Compound Text string to multibyte string in case that
	to_type is XlcNMultibyte, or to wide char string in case that
	to_type is XlcNWideChar.
*/
static
ctstostr(XLCd lcd, char *to_type, XPointer *from, int *from_left,
	 XPointer *to, int *to_left, XPointer *args, int num_args)
{
    XlcCharSet charset;
    XPointer tmp_args[1];
    XlcConv ctos_conv = _XlcOpenConverter(NULL, XlcNCompoundText,
					  NULL, XlcNCharSet);
    XlcConv stostr_conv = _XlcOpenConverter(lcd, XlcNCharSet,
					    lcd, to_type);
    char buf[BUFSIZ], *cs;
    int cs_left, ret, length, unconv_num = 0;
    int from_left_save = *from_left;
    if (ctos_conv == 0 || stostr_conv == 0) return (*from_left);

    cs = buf;
    cs_left = BUFSIZ;
    tmp_args[0] = (XPointer) &charset;
    ret = _XlcConvert(ctos_conv, (XPointer *)from, from_left,
		      &cs, &cs_left, tmp_args, 1);
    if (ret < 0)
	goto err;
    length = cs_left = cs - buf;
    cs = buf;
    tmp_args[0] = (XPointer) charset;
    ret = _XlcConvert(stostr_conv, &cs, &cs_left, to, to_left,
		      tmp_args, 1);
    if (ret < 0) {
	unconv_num += length / charset->char_size;
    }
err:
    _XlcCloseConverter(ctos_conv);
    _XlcCloseConverter(stostr_conv);
    if (cs_left > 0) {
	*from_left = cs_left;
	unconv_num = cs_left;
    }
    return unconv_num;
}

/*
   check_ascii:
   	checks if the Compound Text string designates ascii charset
*/
static
check_ascii(XPointer *from, int *from_left)
{
    if (!strncmp((char*)*from, "\033(B", 3) ||
	!strncmp((char*)*from, "\033-A", 3)) {
	*from += 3;
	*from_left -= 3;
	return True;
    }
    return False;
}

#ifdef KO_UTF8

/*
   check_koreanCT:
   	checks if the Compound Text string designates EUC charset
*/
static
check_koreanCT(XPointer *from, int *from_left)
{
    if (!strncmp((char*)*from, "\033$(C", 4) ||
	!strncmp((char*)*from, "\033$)C", 4)) {
	*from += 4;
	*from_left -= 4;
	return True;
    }
    return False;
}
#endif

/*
   ascii_to_uc:
   	converts ascii character string to UTF-8 or UCS-4 encoding
*/
static int
ascii_to_uc(int type, XPointer *from, int *from_left,
	    XPointer *to, int *to_left)
{
    register char *src, *dst;
    int length, i;

    length = *from_left;
    src = *((char **) from);

    if (type == TO_MBS) {
	dst = *((char **) to);
	for (i = 0; i < length; i++) {
	    if (*src == 0x1b || *src == NULL) break;
	    *dst++ = *src++ & 0x7f;
	}
	*from_left -= i;
	*((char **)from) += i;
	*((char **) to) += i;
	*to_left -= i;
    } else {
	wchar_t *bufptr;
	char *outbuf;
	int wc_conv_len;
	outbuf = Xmalloc(length + 1);
	if (outbuf == NULL) return(*from_left);
	dst = outbuf;
	for (i = 0; i < length; i++) {
	    if (*src == 0x1b || *src == NULL) break;
	    *dst++ = *src++ & 0x7f;
	}
	bufptr = *((wchar_t **)to);
	wc_conv_len = mbstowcs(NULL, outbuf, dst - outbuf);
	if (wc_conv_len > *to_left) {
	    /* overflow */
	    *to_left = 0;
	    return(*from_left);
	}
	if (wc_conv_len != mbstowcs(bufptr, outbuf, dst - outbuf))
	    return(*from_left);
	*to_left -= wc_conv_len;
	Xfree(outbuf);
	*from_left -= i;
	*((char **)from) += i;
	*((char **) to) += i;
	*to_left -= i;
    }
    return (*from_left);
}

#ifdef KO_UTF8
/*
   koreanCT_to_uc:
   	converts EUC-based Compound Text to UTF-8 or UCS-4 encoding
 */
static int
koreanCT_to_uc(int type, XPointer *from, int *from_left,
	    XPointer *to, int *to_left)
{
    iconv_t i_conv;
    size_t ret;
    int from_left_save, length;
    char *src;
    char *buf, *dst;
    int i;

    from_left_save = length = *from_left;

    if ((i_conv = iconv_open("ko_KR-UTF-8", "ko_KR-euc")) == (iconv_t)-1)
	return(*from_left);

    buf = Xmalloc(length + 1);
    if (buf == NULL) goto err;
    dst = buf;

    /* copy from buffer and turn MSB on */
    src = *((char **) from);
    for (i = 0; i < length; i++) {
	if (*src == 0x1b || *src == NULL) break;
	*dst++ = *src++ | 0x80;
    }
    *dst = 0;
    length = dst - buf;
    *from_left -= length;
    *((char **)from) += length;

    dst = buf;
    if (type == TO_MBS) {
	ret = iconv(i_conv, (const char **)&dst, (size_t*)&length,
		    to, (size_t*)to_left);
    } else {
	wchar_t *bufptr;
	char *outbuf = NULL, *out;
	size_t buf_len, outbytes;
	int mb_conv_len, wc_conv_len;
	buf_len = *from_left * MB_CUR_MAX /2;
   	outbuf = Xmalloc(buf_len);
	if (outbuf == NULL) goto err;

	out = outbuf;
	outbytes = buf_len;
	ret = iconv(i_conv, (const char **)&dst, (size_t*)&length,
		    (char**)&out, &outbytes);

	bufptr = *((wchar_t **)to);
	mb_conv_len = buf_len - outbytes;
	wc_conv_len = mbstowcs(NULL, outbuf, mb_conv_len);
	if (wc_conv_len > *to_left) {
	    /* overflow */
	    *from_left = from_left_save;
	    *to_left = 0;
	    goto err;
	}
	if (wc_conv_len != mbstowcs(bufptr, outbuf, mb_conv_len))
	    goto err;
	*to_left -= wc_conv_len;
	bufptr += wc_conv_len;
	Xfree(outbuf);
    }
err:
    Xfree(buf);
    iconv_close(i_conv);
    return (*from_left);
}

static int
johup92_to_euc(char *johup, char *euc)
{
    iconv_t i_conv;
    size_t inbytesleft = 2;
    size_t outbytesleft = 2;
    size_t sz;
    if ((i_conv = iconv_open("ko_KR-euc", "ko_KR-johap92")) == (iconv_t)-1)
	return -1;

    sz = iconv(i_conv, (const char**)&johup, &inbytesleft,
	       (char**)&euc, &outbytesleft);

    iconv_close(i_conv);
    return sz;
}
#endif

static int
convertToCT(XPointer from, int from_len, XPointer *to, int *to_left,
	    XlcCharSet charset)
{
    char *ct_sequence = charset->ct_sequence;
    int ct_sequence_len = strlen(ct_sequence);
    char *ctptr;
    char *current_encoding = NULL;
    char *csptr;
    const char *ko_euc_encoding = "\033$)C";
    int ext_segment_len, char_len = 0;
    char *M, *L;
    int to_left1 = *to_left;

    ctptr = *((char **) to);
    csptr = (char*)from;

    if (!strncmp(charset->ct_sequence, "\033%/", 3)) {
	/* extended segment */
	while (from_len > 0) {
	    char euc_buf[4];
#ifdef KO_UTF8
	    if (johup92_to_euc(csptr, euc_buf) > 0) {
		if (current_encoding == ct_sequence) {
		    ext_segment_len = ct_sequence_len + char_len - 6;
		    *M = (ext_segment_len / 128) | 0x80;
		    *L = (ext_segment_len % 128) | 0x80;
		}
		if (current_encoding != ko_euc_encoding) {
		    current_encoding = (char*)ko_euc_encoding;
		    if (to_left1 < 4) goto overflow;
		    memcpy(ctptr, ko_euc_encoding, 4);
		    ctptr += 4;
		    to_left1 -= 4;
		}
		if (to_left1 < 2) goto overflow;
		memcpy(ctptr, euc_buf, 2);
		ctptr += 2;
		to_left1 -= 2;
	    } else
#endif
	      {
		if (current_encoding != ct_sequence) {
		    char_len = 0;
		    current_encoding = ct_sequence; 
		    if (to_left1 < ct_sequence_len) goto overflow;
		    memcpy(ctptr, ct_sequence, ct_sequence_len);
		    M = &ctptr[4]; L = &ctptr[5];
		    ctptr += ct_sequence_len;
		    to_left1 -= ct_sequence_len;
		}
		if (to_left1 < 2) goto overflow;
		memcpy(ctptr, csptr, 2);
		ctptr += 2;
		char_len += 2;
		to_left1 -= 2;
	    }
	    csptr += 2;
	    from_len -= 2;
	}
	if (current_encoding == ct_sequence) {
	    ext_segment_len = ct_sequence_len + char_len - 6;
	    *M = (ext_segment_len / 128) | 0x80;
	    *L = (ext_segment_len % 128) | 0x80;
	}

    } else {
      if (to_left1 < ct_sequence_len + from_len) goto overflow;
      memcpy(ctptr, ct_sequence, ct_sequence_len);
      ctptr += ct_sequence_len;
      memcpy(ctptr, from, from_len);
      ctptr += from_len;
      to_left1 -= (ct_sequence_len + from_len);
    }
    *to_left -= ctptr - *((char **) to);
    *to = (XPointer) ctptr;
    return 0;
 overflow:
    return -1;
}

/*
   uc_mbstocts:
   	converter function from UTF-8 encoding to Compound Text.
	At Solaris 2.6, it is used only for ko.UTF-8 locale.
*/
static int
uc_mbstocts(XlcConv conv, XPointer *from, int *from_left, XPointer *to,
	    int *to_left, XPointer *args, int num_args)
{
    XlcCharSet charset = NULL;
    XPointer tmp_args[2];
    const int tmp_num = 1;
    int tmp_to_left;
    XPointer tmp_to, tmp_to_begin, tmp_to_save;
    int ret;
    int unconv_num = 0;
    int from_left_save;

    tmp_to = (XPointer)Xmalloc(*to_left * sizeof(char));
    if (tmp_to == NULL) return *from_left;
    tmp_to_save = tmp_to;

    tmp_to_left = *to_left;

    while(*from_left > 0 && *to_left > 0) {
	tmp_args[0] = (XPointer) &charset;
	tmp_to_begin = tmp_to;
	from_left_save = *from_left;
	ret = uc_mbstocs(conv, from, from_left, &tmp_to, &tmp_to_left,
			 tmp_args, tmp_num);
	if (ret == -1) break;
	if (convertToCT(tmp_to_begin, tmp_to - tmp_to_begin,
			to, to_left, charset) < 0) {
	  /* overflow */
	  *from_left = from_left_save;
	  unconv_num = from_left_save;
	  break;
	}
    }
    Xfree(tmp_to_save);

    return unconv_num;
}

/*
   uc_wcstocts:
   	converter function from UCS-4 encoding to Compound Text.
	At Solaris 2.6, it is used only for ko.UTF-8 locale.
*/
static int
uc_wcstocts(XlcConv conv, XPointer *from, int *from_left, XPointer *to,
	    int *to_left, XPointer *args, int num_args)
{
    wchar_t *src;
    char *buf, *dst;
    int buflen, mb_len;
    int wc_length;
    int ret;

    src = *((wchar_t **)from);
    wc_length = *from_left;
    mb_len = wcstombs(NULL, src, wc_length);
    buf = Xmalloc(mb_len);
    if (buf == NULL) return(*from_left);
    if (mb_len != wcstombs(buf, src, mb_len)) goto err;

    dst = buf;
    ret = uc_mbstocts(conv, (char**)&dst, (int*)&mb_len, to, to_left, args,
		      num_args);
    if (ret != 0) goto err;

    *((wchar_t **)from) += wc_length;
    *from_left -= wc_length;
err:
    Xfree(buf);
    return (*from_left);
}

/*
   uc_ctstombs:
   	converter function from Compound Text to UTF-8 encoding.
	At Solaris 2.6, it is used only for ko.UTF-8 locale.
*/
static int
uc_ctstombs(XlcConv conv, XPointer *from, int *from_left, XPointer *to,
	    int *to_left, XPointer *args, int num_args)
{
    XLCd lcd = (XLCd)conv->state;
    int unconv_num = 0;
    int		left, tmp_left;

    while (*from_left && **from && *to_left) {
#ifdef KO_UTF8
      if (check_koreanCT(from, from_left)) {
	unconv_num += koreanCT_to_uc(TO_MBS, from, from_left, to, to_left);
      } else
#endif
	{
	  int start=0;
	  if(**from == 0x1b)
	    start = 1;
	  for(left = start; left < *from_left; left++){
	    if(from[0][left] == 0x1b || from[0][left] == NULL)
	      break;
	  }
	  tmp_left = left;
	  unconv_num += ctstostr(lcd, XlcNMultiByte, from, &tmp_left, to,
				 to_left, args, num_args);
	  *from_left -= (left - tmp_left);
	}
    }
    return unconv_num;
}

/*
   uc_ctstowcs:
   	converter function from Compound Text to UCS-4 encoding.
	At Solaris 2.6, it is used only for ko.UTF-8 locale.
*/
static int
uc_ctstowcs(XlcConv conv, XPointer *from, int *from_left, XPointer *to,
	    int *to_left, XPointer *args, int num_args)
{
    XLCd lcd = (XLCd)conv->state;
    int unconv_num = 0;
    int		left, tmp_left;

    while (*from_left && **from && *to_left) {
#ifdef kO_UTF8
      if (check_koreanCT(from, from_left)) {
	unconv_num += koreanCT_to_uc(TO_WCS, from, from_left, to, to_left);
      } else
#endif
	{
	  int start=0;
	  if(**from == 0x1b)
	    start = 1;
	  for(left = start; left < *from_left; left++){
	    if(from[0][left] == 0x1b || from[0][left] == NULL)
	      break;
	  }
	  tmp_left = left;
	  unconv_num += ctstostr(lcd, XlcNWideChar, from, from_left, to,
				 to_left, args, num_args);
	  *from_left -= (left - tmp_left);
	}
    }
    return unconv_num;
}

static int
uc_wcstocs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args)
{
	int	status;
	CS_ENTRY	csN;

	if (uc_wc_info == NULL)
	    uc_wc_info = UnicodeWcharInfo_create((XLCd)conv->state);
	if (uc_cs_info == NULL)
	    uc_cs_info = UnicodeCsInfo_create((XLCd)conv->state);
	if (uc_wc_info == (UnicodeWcharInfo *)NULL ||
	    uc_cs_info == (UnicodeCsInfo *)NULL)
	    return -1;

	if (uc_wc_info->table == (void *)NULL)
	    return -1;
	if(uc_wc_info->cs_conv_func == NULL)
		return -1;

	status = uc_wc_info->cs_conv_func(uc_wc_info->table, from, from_left, to, to_left, &csN);
	if((status >= 0) && (num_args > 0)){
		if(csN != LUT_CODESET_UNKNOWN)
			*((XlcCharSet *)args[0]) = uc_cs_info->cs[csN].charset;
	}
	return status;
}

static int
uc_mbstocs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args)
{
	int	status;
	CS_ENTRY	csN;

	if (uc_wc_info == NULL)
	    uc_wc_info = UnicodeWcharInfo_create((XLCd)conv->state);
	if (uc_mb_info == NULL)
	    uc_mb_info = UnicodeMbstrInfo_create((XLCd)conv->state);
	if (uc_cs_info == NULL)
	    uc_cs_info = UnicodeCsInfo_create((XLCd)conv->state);
	if (uc_wc_info == (UnicodeWcharInfo *)NULL ||
	    uc_mb_info == (UnicodeMbstrInfo *)NULL ||
	    uc_cs_info == (UnicodeCsInfo *)NULL)
	    return -1;

#ifdef UC_SHORT_CONVERSION
	if(uc_wc_info->cs_conv_func != NULL)
		status = uc_mb_info->cs_conv_func(uc_wc_info->table, from, from_left, to, to_left, &csN);
	else
		return -1;
#else
	status = convert_mbstocs(uc_wc_info->table, from, from_left, to, to_left, &csN);
#endif

	if((status >= 0) && (num_args > 0)){
	    if(csN != LUT_CODESET_UNKNOWN)
		*((XlcCharSet *)args[0]) = uc_cs_info->cs[csN].charset;
	}
	if (status > 0) return -1; /* ignore unconv num */
	return status;
}

static int
uc_cstowcs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args)
{
	int	i, j, k;
	void	*table = (void *)NULL;
	CS_ENTRY	dummy;

	if (num_args < 1)
		return -1;

	if (uc_cs_info == NULL)
	    uc_cs_info = UnicodeCsInfo_create((XLCd)conv->state);
	if (uc_cs_info == (UnicodeCsInfo *)NULL)
	    return -1;

	for(i = 0, k = 0; k < uc_cs_info->cs_num; i++){
	    for(j = 0; j < uc_cs_info->cs[i].codeset->num_charsets; j++, k++){
		if(!strcmp(((XlcCharSet)(args[0]))->name, 
		    uc_cs_info->cs[i].codeset->charset_list[j]->name)){
			table = uc_cs_info->cs[i].table;
			goto finish_loop;
		}
	    }
	}
finish_loop:
	if(table == (void *)NULL)
		return -1;

	/* Skip ST (0x02) character used in Extended Segment */
	if(**from == 0x02){
		*from += 1; *from_left -= 1;
	}

	if(uc_cs_info->cs[i].wc_conv_func)
		return uc_cs_info->cs[i].wc_conv_func(table, from, from_left, to, to_left, &dummy);
	else
		return -1;
}

static int
uc_cstombs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args)
{
	CS_ENTRY	i, k, dummy;
	int		j;
	void	*table = (void *)NULL;

	if (uc_cs_info == NULL)
	    uc_cs_info = UnicodeCsInfo_create((XLCd)conv->state);
	if (uc_cs_info == (UnicodeCsInfo *)NULL)
	    return -1;

	for(i = 0, k = 0; k < uc_cs_info->cs_num; i++){
	    for(j = 0; j < uc_cs_info->cs[i].codeset->num_charsets; j++, k++){
		if(!strcmp(((XlcCharSet)(args[0]))->name, 
		    uc_cs_info->cs[i].codeset->charset_list[j]->name)){
			table = uc_cs_info->cs[i].table;
			goto finish_loop;
		}
	    }
	}
finish_loop:
	if(table == (void *)NULL)
		return -1;

	/* Skip ST (0x02) character used in Extended Segment */
	if(**from == 0x02){
		*from += 1; *from_left -= 1;
	}

/*
#ifdef UC_SHORT_CONVERSION
	if(uc_cs_info->cs[i].mb_conv_func)
		return uc_cs_info->cs[i].mb_conv_func(table, from, from_left, to, to_left, &dummy);
	else
		return -1;
#else
*/
	return convert_cstombs(table, from, from_left, to, to_left, &i);
/*
#endif
*/
}

static int
uc_wcstombs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args)
{
	return convert_wcstombs(from, from_left, to, to_left);
}

static int
uc_mbstowcs(XlcConv conv, XPointer *from, int *from_left, XPointer *to, int *to_left, XPointer *args, int num_args)
{
	return convert_mbstowcs(from, from_left, to, to_left);
}

static void
close_converter(XlcConv conv)
{
	Xfree(conv);
}


/* local converters */

static int
convert_wc16_to_cs(void *table, XPointer *from_wc, int *from_left, XPointer *to_cs, int *to_left, CS_ENTRY *csN)
{
	CARD32	offset;
	int	csn;
	int	cs_length, entry_length;
	int	ret_num;

	CARD8	*to = *((CARD8 **)to_cs);

	wchar_t	*from = *((wchar_t **)from_wc);
	ret_num = 0;

	if(*from_left <= 0)
		return ret_num;

	while(*from_left > 0){
	    offset = *((CARD16*)table + ((*from >> 8) & 0xff));
	    *csN = (int)(*((CARD8 *)table + offset * 256 + (*from & 0xff)));
	    if(*csN != LUT_CODESET_UNKNOWN)
		break;
	    from++;
	    (*from_left)--;
	    ret_num++;
	}
	if(*csN == LUT_CODESET_UNKNOWN) {
		*from_wc = (XPointer)from;
		return ret_num;
	}

	cs_length = uc_cs_info->cs[*csN].codeset->length;
	entry_length = uc_cs_info->max_length;
	if((entry_length == 2) && (cs_length == 2)){
loop_point22:
		if(*to_left > 1){
			CARD16	*tmp16;
			tmp16 = (CARD16 *)((CARD8 *)table + offset * 256 + 256) + (*from & 0xff);
			*to++ = *tmp16 >> 8;
			*to++ = *tmp16 & 0xff;
			from++;
			(*from_left)--;
			(*to_left) -= 2;
			while(*from_left > 0){
				offset = *((CARD16 *)table + ((*from >> 8) & 0xff));
				csn = (int)(*((CARD8 *)table + offset * 256 + (*from & 0xff)));
				if(csn == *csN)
					goto loop_point22;
				if(csn == LUT_CODESET_UNKNOWN){
					from++;
					(*from_left)--;
					ret_num++;
					break;
				}
				else
					break;
			}
		}
	}
	else if((entry_length == 2) && (cs_length == 1)){
loop_point21:
		if(*to_left > 0){
			CARD16	*tmp16;
			tmp16 = (CARD16 *)((CARD8 *)table + offset * 256 + 256) + (*from++ & 0xff);
			*to++ = *tmp16 & 0xff;
			(*from_left)--;
			(*to_left)--;
			while(*from_left > 0){
				offset = *((CARD16 *)table + (*from >> 8));
				csn = *((CARD8 *)table + offset * 256 + (*from & 0xff));
				if(csn == *csN)
					goto loop_point21;
				if(csn == LUT_CODESET_UNKNOWN){
					from++;
					(*from_left)--;
					ret_num++;
					break;
				}
				else
					break;
			}
		}
	}
	else if((entry_length == 1) && (cs_length == 1)){
loop_point11:
		if(*to_left > 0){
			*to++ = *((CARD8 *)table + offset * 256 + 256 + (*from++ & 0xff));
			(*from_left)--;
			(*to_left)--;
			while(*from_left > 0){
				offset = *((CARD16 *)table + (*from >> 8));
				csn = *((CARD8 *)table + offset * 256 + (*from & 0xff));
				if(csn == *csN)
					goto loop_point11;
				if(csn == LUT_CODESET_UNKNOWN){
					from++;
					(*from_left)--;
					ret_num++;
					break;
				}
				else
					break;
			}
		}
	}
	else{
		return -1;
	}

	*from_wc = (XPointer)from;
	*to_cs = (XPointer)to;

	return ret_num;
}

static int
convert_mb16_to_cs(void *table, XPointer *from_mb, int *from_left, XPointer *to_cs, int *to_left, CS_ENTRY *csN)
{
	CARD32	offset;
	int	csn;
	int	cs_length, entry_length;
	int	ret_num;

	char	*from = *((char **)from_mb);
	CARD8	*to = *((CARD8 **)to_cs);
	wchar_t	wc;
	int	len;

	ret_num = 0;

	while(*from_left > 0){
		len = mbtowc(&wc, from, *from_left);
		if(len > 0){
			offset = *((CARD16 *)table + ((wc >> 8) & 0xff));
			*csN = (int)(*((CARD8 *)table + offset * 256 + (wc & 0xff)));
			if(*csN != LUT_CODESET_UNKNOWN)
				break;
			from += len;
			(*from_left) -= len;
			ret_num += len;
		}
		else
			return ret_num;
	}
	if(*csN == LUT_CODESET_UNKNOWN) {
		*from_mb = (XPointer)from;
		return ret_num;
	}

	cs_length = uc_cs_info->cs[*csN].codeset->length;
	entry_length = uc_cs_info->max_length;
	if((entry_length == 2) && (cs_length == 2)){
loop_point22:
		if(*to_left > 1){
			CARD16	*tmp16;
			tmp16 = (CARD16 *)((CARD8 *)table + offset * 256 + 256) + (wc & 0xff);
			*to++ = *tmp16 >> 8;
			*to++ = *tmp16 & 0xff;
			from += len;
			(*from_left) -= len;
			*to_left -= 2;
			while(*from_left > 0){
				len = mbtowc(&wc, from, *from_left);
				if(len > 0){
					offset = *((CARD16 *)table + ((wc >> 8) & 0xff));
					csn = (int)(*((CARD8 *)table + offset * 256 + (wc & 0xff)));
					if(csn == *csN)
						goto loop_point22;
					if(csn == LUT_CODESET_UNKNOWN){
						from += len;
						(*from_left) -= len;
						ret_num += len;
						break;
					}
					else
						break;
				}
				else
					break;
			}
		}
	}
	if((entry_length == 2) && (cs_length == 1)){
loop_point21:
		if(*to_left > 0){
			CARD16	*tmp16;
			tmp16 = (CARD16 *)((CARD8 *)table + offset * 256 + 256) + (wc & 0xff);
			*to++ = *tmp16 & 0xff;
			from += len;
			(*from_left) -= len;
			(*to_left)--;
			while(*from_left > 0){
				len = mbtowc(&wc, from, *from_left);
				if(len > 0){
					offset = *((CARD16 *)table + (wc >> 8));
					csn = *((CARD8 *)table + offset * 256 + (wc & 0xff));
					if(csn == *csN)
						goto loop_point21;
					if(csn == LUT_CODESET_UNKNOWN){
						from += len;
						(*from_left) -= len;
						ret_num += len;
						break;
					}
					else
						break;
				}
				else
					break;
			}
		}
	}
	if((entry_length == 1) && (cs_length == 1)){
loop_point11:
		if(*to_left > 0){
			*to++ = *((CARD8 *)table + offset * 256 + 256 + (wc & 0xff));
			from += len;
			(*from_left) -= len;
			(*to_left)--;
			while(*from_left > 0){
				len = mbtowc(&wc, from, *from_left);
				if(len > 0){
					offset = *((CARD16 *)table + (wc >> 8));
					csn = *((CARD8 *)table + offset * 256 + (wc & 0xff));
					if(csn == *csN)
						goto loop_point11;
					if(csn == LUT_CODESET_UNKNOWN){
						from += len;
						(*from_left) -= len;
						ret_num += len;
						break;
					}
					else
						break;
				}
				else
					break;
			}
		}
	}
	else{
		return -1;
	}

	*from_mb = (XPointer)from;
	*to_cs = (XPointer)to;

	return 0;
}

static int
convert_cs8_to_wc16(void *table, XPointer *from_cs, int *from_left, XPointer *to_wc, int *to_left, CS_ENTRY *csN)
{
	CARD8	*from = *((CARD8 **)from_cs);
	wchar_t	*to = *((wchar_t **)to_wc);

	while((*from_left > 0) && (*to_left > 0)){
		*((wchar_t *)to++) = (wchar_t)*((CARD16 *)table + *((CARD8 *)from++));
		(*from_left)--;
		(*to_left)--;
	}
	
	*from_cs = (XPointer)from;
	*to_wc = (XPointer)to;

	return 0;
}

static int
convert_wcstombs(XPointer *from, int *from_left, XPointer *to, int *to_left)
{
#ifdef USE_ICONV
	iconv_t	i_conv;
	size_t	ret, inbytes, outbytes;

	i_conv = iconv_open(uc_mb_info->iconv_name, uc_wc_info->iconv_name);
	if(i_conv == (iconv_t)-1)
		return -1;

	inbytes = *from_left * sizeof(wchar_t), outbytes = *to_left;
	ret = iconv(i_conv, (const char **)from, &inbytes, (char **)to, &outbytes);
	*from_left = inbytes / sizeof(wchar_t), *to_left = outbytes;

	iconv_close(i_conv);

	return (int)ret;
#else
	size_t	num;
	wchar_t	*buf;
	
	buf = (wchar_t *)malloc((*from_left + 1) * sizeof(wchar_t));
	if(buf == (wchar_t *)NULL)
		return -1;

	wcsncpy(buf, (wchar_t *)(*from), *from_left);
	buf[*from_left] = (wchar_t)0;

	num = wcstombs(*to, buf, *to_left);
	free(buf);

	if(num == (size_t)-1)
		return -1;
	else{
		int	c_num = 0;
		char	*ptr = (char *)*to;
		int	len = 0;
		while(len < num){
			int	c_len = mblen(ptr, num - len);
			if(c_len > 0){
				c_num++;
				ptr += c_len;
				len += c_len;
			}
			else
				return -1;
		}
		*from += c_num * sizeof(wchar_t);
		*from_left -= c_num;
		*to += num;
		*to_left -= num;
		return 0;
	}
#endif
}

static int
convert_mbstowcs(XPointer *from, int *from_left, XPointer *to, int *to_left)
{
#ifdef USE_ICONV
	iconv_t	i_conv;
	size_t	ret, inbytes, outbytes;

#ifdef ICONVOPEN
	inbytes = *from_left, outbytes = *to_left * sizeof(wchar_t);
	ret = iconv(uc_mb_info->iconv_towc, (const char **)from, &inbytes, (char **)to, &outbytes);
	*from_left = inbytes, *to_left = outbytes / sizeof(wchar_t);
#else
	i_conv = iconv_open(uc_wc_info->iconv_name, uc_mb_info->iconv_name);
	if(i_conv == (iconv_t)-1)
		return -1;

	inbytes = *from_left, outbytes = *to_left * sizeof(wchar_t);
	ret = iconv(i_conv, (const char **)from, &inbytes, (char **)to, &outbytes);
	*from_left = inbytes, *to_left = outbytes / sizeof(wchar_t);

	iconv_close(i_conv);
#endif
	return (int)ret;
#else
	size_t	num;
	char	*buf;
	
	buf = (char *)malloc(*from_left + 1);
	if(buf == (char *)NULL)
		return -1;

	memcpy(buf, (char *)(*from), *from_left);
	buf[*from_left] = (char)0;

	num = mbstowcs((wchar_t *)(*to), buf, *to_left);
	free(buf);

	if(num == (size_t)-1)
		return -1;
	else{
		int	i;
		char	*ptr = (char *)(*from);
		int	len = 0;
		for(i = 0; i < num; i++){
			int	c_len = mblen(ptr, *from_left - len);
			if(c_len > 0){
				ptr += c_len;
				len += c_len;
			}
			else
				return -1;
		}
		*from += len; 
		*from_left -= len;
		*to += num * sizeof(wchar_t);
		*to_left -= num;
		return 0;
	}
#endif
}

static int
convert_cstombs(void *table, XPointer *from, int *from_left, XPointer *to, int *to_left, CS_ENTRY *csN)
{
	wchar_t	*wc_buf = (wchar_t *)NULL;
	int	wc_buf_size;
	XPointer	from1, from2, to1, to2;
	int		from_left1, from_left2, to_left1, to_left2;
	int		from_conv_size1, from_conv_size2, to_conv_size1;
	int		ret_num, ret;

	ret_num = 0;

	wc_buf_size = (*from_left)/(uc_cs_info->cs[*csN].codeset->length) * sizeof(wchar_t);
	wc_buf = (wchar_t *)Xmalloc(wc_buf_size);
	if(wc_buf == (wchar_t *)NULL){
		ret_num = -1;
		goto to_return;
	}

	from1 = *from, from_left1 = *from_left;
	to1 = (XPointer)(&(wc_buf[0])), to_left1 = wc_buf_size / sizeof(wchar_t); 
	if(uc_cs_info->cs[*csN].wc_conv_func)
		ret = uc_cs_info->cs[*csN].wc_conv_func(table, &from1, &from_left1, &to1, &to_left1, csN);
	else{
		ret_num = -1;
		goto to_return;
	}

	ret_num += ret;
	from_conv_size1 = *from_left - from_left1;
	to_conv_size1 = wc_buf_size / sizeof(wchar_t) - to_left1;

	from2 = (XPointer)(&(wc_buf[0])), from_left2 = to_conv_size1;
	to2 = *to, to_left2 = *to_left;
	ret = convert_wcstombs(&from2, &from_left2, &to2, &to_left2);
	if(ret > 0)
		ret_num += ret;

	if(from_left2 == 0){
		*from = from1;
		*from_left = from_left1;
		*to = to2;
		*to_left = to_left2;
		goto to_return;
	}
	else if(from_conv_size1/uc_cs_info->cs[*csN].codeset->length == to_conv_size1){
		from_conv_size2 = (to_conv_size1 - from_left2) * uc_cs_info->cs[*csN].codeset->length;
		*from += from_conv_size2;
		*from_left -= from_conv_size2;
		*to = to2;
		*to_left = to_left2;
		goto to_return;
	}
	else{
		if(uc_cs_info->cs[*csN].mb_conv_func)
			ret_num = uc_cs_info->cs[*csN].mb_conv_func(table, from, from_left, to, to_left, csN);
		else
			ret_num = -1;
		goto to_return;
	}

to_return:
	if(wc_buf)
		Xfree(wc_buf);
	return ret_num;
}

static int
convert_mbstocs(void *table, XPointer *from, int *from_left, XPointer *to, int *to_left, CS_ENTRY *csN)
{
	wchar_t	*wc_buf = (wchar_t *)NULL;
	int	wc_buf_size;
	XPointer	from1, from2, to1, to2;
	int		from_left1, from_left2, to_left1, to_left2;
	int		from_conv_size1, from_conv_size2, to_conv_size1;
	int		ret_num, ret1, ret2;

	ret_num = 0;

	wc_buf_size = (*to_left) * sizeof(wchar_t);
	wc_buf = (wchar_t *)Xmalloc(wc_buf_size);
	if(wc_buf == (wchar_t *)NULL){
		ret_num = -1;
		goto to_return;
	}

	from1 = *from, from_left1 = *from_left;
	to1 = (XPointer)(&(wc_buf[0])), to_left1 = wc_buf_size / sizeof(wchar_t); 
	ret1 = convert_mbstowcs(&from1, &from_left1, &to1, &to_left1);
	if (ret1 >= 0) {
	    ret_num += ret1;
	}else {
	    ret_num = -1;
	    *csN = LUT_CODESET_UNKNOWN;
	    goto to_return;
	}

	from_conv_size1 = *from_left - from_left1;
	to_conv_size1 = wc_buf_size / sizeof(wchar_t) - to_left1;

	from2 = (XPointer)(&(wc_buf[0])), from_left2 = to_conv_size1;
	to2 = *to, to_left2 = *to_left;
	if(uc_wc_info->cs_conv_func)
		ret2 = uc_wc_info->cs_conv_func(uc_wc_info->table, &from2, &from_left2, &to2, &to_left2, csN);
	else{
		ret_num = -1;
		goto to_return;
	}

	if(ret2 > 0)
		ret_num += ret2;

	if(from_left2 == 0){
		*from = from1;
		*from_left = from_left1;
		*to = to2;
		*to_left = to_left2;
		goto to_return;
	}
	else{
		ret_num = ret2;
		from1 = *from, from_left1 = *from_left;
		to1 = (XPointer)(&(wc_buf[0])), to_left1 = to_conv_size1 - from_left2; 
		ret1 = convert_mbstowcs(&from1, &from_left1, &to1, &to_left1);
		if(ret1 > 0)
			ret_num += ret1;
		if(to_left1 == 0){
			*from = from1;
			*from_left = from_left1;
			*to = to2;
			*to_left = to_left2;
			goto to_return;
		}
	}
	if(uc_wc_info->cs_conv_func != NULL)
		ret_num = uc_mb_info->cs_conv_func(uc_wc_info->table, from, from_left, to, to_left, csN);
	else
		ret_num = -1;
	goto to_return;

to_return:
	if(wc_buf)
		Xfree(wc_buf);
	return ret_num; 
}

static int
convert_cs8_to_mb16(void *table, XPointer *from_cs, int *from_left, XPointer *to_mb, int *to_left, CS_ENTRY *csN)
{
	CARD8	*from = *((CARD8 **)from_cs);
	CARD8	*to = *((CARD8 **)to_mb);
	wchar_t	wc;
	char	mb_buf[MB_LEN_MAX];
	int	len;
	int	ret_num;
	ret_num = 0;

	while((*from_left > 0) && (*to_left > 0)){
		wc = (wchar_t)*((CARD16 *)table + *((CARD8 *)from));
		len = wctomb(mb_buf, wc);
		if(*to_left >= len){
			from++;
			(*from_left)--;
			if(len > 0){
				strncpy((char *)to, mb_buf, len);
				to += len;
				(*to_left) -= len;
			}
			else{
				ret_num++;
			}
		}
		else
			break;
	}

	*from_cs = (XPointer)from;
	*to_mb = (XPointer)to;

	return ret_num;
}

static int
convert_cs16_to_wc16(void *table, XPointer *from_cs, int *from_left, XPointer *to_wc, int *to_left, CS_ENTRY *csN)
{
	CARD8	*from = *((CARD8 **)from_cs);
	wchar_t	*to = *((wchar_t **)to_wc);
	CARD32 offset;

	while((*from_left > 1) && (*to_left > 0)){
		offset = *((CARD16 *)table + *((CARD8 *)from++));
		*((wchar_t *)to++) = (wchar_t)*((CARD16 *)((CARD8 *)table + offset * 256 + *((CARD8 *)from++) * sizeof(CARD16)));
		(*from_left) = (*from_left) - 2;
		(*to_left)--;
	}

	*from_cs = (XPointer)from;
	*to_wc = (XPointer)to;

	return 0;
}

static int
convert_cs16_to_mb16(void *table, XPointer *from_cs, int *from_left, XPointer *to_mb, int *to_left, CS_ENTRY *csN)
{
	CARD8	*from = *((CARD8 **)from_cs);
	CARD8	*to = *((CARD8 **)to_mb);
	CARD32 offset;
	wchar_t	wc;
	char	mb_buf[MB_LEN_MAX];
	int	len;
	int	ret_num;
 
	ret_num = 0;
        while((*from_left > 1) && (*to_left > 0)){
                offset = *((CARD16 *)table + *((CARD8 *)from));
                wc = (wchar_t)*((CARD16 *)((CARD8 *)table + offset * 256 + *((CARD8 *)from+1) * sizeof(CARD16)));
		len = wctomb(mb_buf, wc);
		if(*to_left >= len){
                	(*from_left) -= 2;
			from += 2;
			if(len > 0){
				strncpy((char *)to, mb_buf, len);
				to += len;
				(*to_left) -= len;
			}
			else{
				ret_num++;
			}
		}
		else{
			break;
		}
        }

	*from_cs = (XPointer)from;
	*to_mb = (XPointer)to;

        return 0;
}


/* constructors and destructors */

static UnicodeMbstrInfo *
UnicodeMbstrInfo_create(XLCd lcd)
{
	UnicodeMbstrInfo	*info;
	char	**str_list;
	int	num;

	info = (UnicodeMbstrInfo *)Xmalloc(sizeof(UnicodeMbstrInfo));
	if(info == (UnicodeMbstrInfo *)NULL)
		return (UnicodeMbstrInfo *)NULL;

	info->iconv_name = (char *)NULL;
	_XlcGetResource(lcd, "XLC_XLOCALE", MB_ICONV_NAME, &str_list, &num);
	if(num > 0){
		info->iconv_name = (char *)Xmalloc(strlen(str_list[0]) + 1);
		if(info->iconv_name)
			strcpy(info->iconv_name, str_list[0]);
	}

#ifdef USE_ICONV
#ifdef ICONVOPEN
	info->iconv_towc = iconv_open(uc_wc_info->iconv_name, info->iconv_name);
#endif
#endif

	info->cs_conv_func = NULL;
	if(uc_wc_info->valid_len == 2)
		info->cs_conv_func = convert_mb16_to_cs;

	return info;
}

static void
UnicodeMbstrInfo_destroy(UnicodeMbstrInfo *info)
{
	if(info){
		if(info->iconv_name)
			Xfree(info->iconv_name);
	}
	Xfree(info);
}

static UnicodeWcharInfo *
UnicodeWcharInfo_create(XLCd lcd)
{
	UnicodeWcharInfo	*info;
	char	resource_name[256];
	char	filename[256];
	char	**str_list;
	int	num;

	info = (UnicodeWcharInfo *)Xmalloc(sizeof(UnicodeWcharInfo));
	if(info == (UnicodeWcharInfo *)NULL)
		return (UnicodeWcharInfo *)NULL;

	info->valid_len = 0;
	_XlcGetResource(lcd, "XLC_XLOCALE", WC_VALID_LENGTH_ENTRY, &str_list, &num);
	if(num > 0)
		info->valid_len = atoi(str_list[0]);

	info->iconv_name = (char *)NULL;
	_XlcGetResource(lcd, "XLC_XLOCALE", WC_ICONV_NAME, &str_list, &num);
	if(num > 0){
		info->iconv_name = (char *)Xmalloc(strlen(str_list[0]) + 1);
		if(info->iconv_name)
			strcpy(info->iconv_name, str_list[0]);
	}

	info->table = (void *)NULL;
	_XlcGetResource(lcd, "XLC_XLOCALE", WC_TO_CS_CONV_TABLE_ENTRY, &str_list, &num);
	if(num > 0){
		info->table = read_table(str_list[0]);
	}

	info->cs_conv_func = NULL;
	if(info->valid_len == 2)
		info->cs_conv_func = convert_wc16_to_cs;

	return info;
}

static void
UnicodeWcharInfo_destroy(UnicodeWcharInfo *info)
{
	if(info){
		if(info->iconv_name)
			Xfree(info->iconv_name);
		if(info->table)
			Xfree(info->table);
	}
	Xfree(info);
}

static UnicodeCsInfo *
UnicodeCsInfo_create(XLCd lcd)
{
	UnicodeCsInfo	*info;
	int	i, j;
	int	codeset_num;
	CodeSet	*codeset_list;
	int	count;
	char	resource_name[256];
	char	**str_list;
	int	num;
	char	filename[256];

	info = (UnicodeCsInfo *)Xmalloc(sizeof(UnicodeCsInfo));
	if(info == (UnicodeCsInfo *)NULL)
		return (UnicodeCsInfo *)NULL;

	info->cs_num = 0;
	codeset_num = XLC_GENERIC(lcd, codeset_num);
	codeset_list = XLC_GENERIC(lcd, codeset_list);
	for(i = 0; i < codeset_num; i++)
		info->cs_num += codeset_list[i]->num_charsets;

	info->cs = (UnicodeCs *)Xmalloc(sizeof(UnicodeCs) * info->cs_num);
	if(info->cs == (UnicodeCs *)NULL){
		Xfree(info);
		return (UnicodeCsInfo *)NULL;
	}

	count = 0;
	info->max_length = 0;
	for(i = 0; i < codeset_num; i++){
		sprintf(resource_name, "cs%d.%s", i, CS_TO_WC_CONV_TABLE_ENTRY);
		_XlcGetResource(lcd, "XLC_XLOCALE", resource_name, &str_list, &num);
/* for multiple charset definitions in a csN
		for(j = 0; j < codeset_list[i]->num_charsets; j++){
*/
		for(j = 0; j < 1; j++){
			info->cs[count].codeset = codeset_list[i];
			info->cs[count].charset = codeset_list[i]->charset_list[j];

			if(info->max_length < info->cs[count].codeset->length)
				info->max_length = info->cs[count].codeset->length;

			info->cs[count].table = (void *)NULL;
			if(j < num)
				info->cs[count].table = read_table(str_list[j]);

			info->cs[count].wc_conv_func = NULL;
			info->cs[count].mb_conv_func = NULL;
			if(info->cs[count].table){
			  if (uc_wc_info == NULL) {
			    uc_wc_info = UnicodeWcharInfo_create((XLCd)lcd);
			    if (!uc_wc_info) {
			      return NULL; /* error */
			    }
			  }
			  if(uc_wc_info->valid_len == 2){
			    switch(codeset_list[i]->length){
			    case 2:
			      info->cs[count].wc_conv_func = convert_cs16_to_wc16;
			      info->cs[count].mb_conv_func = convert_cs16_to_mb16;
			      break;
			    case 1:
			      info->cs[count].wc_conv_func = convert_cs8_to_wc16;
			      info->cs[count].mb_conv_func = convert_cs8_to_mb16;
			      break;
			    default:
			      break;
			    }
			  }
			}
			count++;
		}
	}

	return info;
}

static void
UnicodeCsInfo_destroy(UnicodeCsInfo *info)
{
	int	i;

	if(info){
		if(info->cs){
			for(i = 0; i < info->cs_num; i++){
				if(info->cs[i].table)
					Xfree(info->cs[i].table);
			}
		}
		Xfree(info->cs);
	}
	Xfree(info);
}


/* miscellaneous functions */

static XlcConv
create_conv(XLCd lcd, XlcConvMethods methods)
{
	XlcConv	conv;

	conv = (XlcConv)Xmalloc(sizeof(XlcConvRec));
	if(conv == (XlcConv)NULL)
		return (XlcConv)NULL;

	conv->methods = methods;
	conv->state = (XPointer) lcd;

	return conv;
}

static int
make_filename(char *str, char *filename)
{
  if(str == (char *)NULL)
    return -1;
  if(str[0] == '/'){
    if(strlen(str) >= PATH_MAX)
      return -1;
    strcpy(filename, str);
  }else{
    char *lc_name = strdup(setlocale(LC_CTYPE, (char *)NULL));
    char lc_dir[BUFSIZE];
    if (_XlcLocaleDirName(lc_dir, lc_name) == NULL) return 0;
    if (lc_dir == (char *)NULL)
      return -1;
    if (strlen(lc_dir) + 1 + strlen(str) >= PATH_MAX)
      return -1;
    strcpy(filename, lc_dir); strcat(filename, "/");
    strcat(filename, str);
    Xfree(lc_name);
  }
  return 0;
}

static void *
read_table(char *str)
{
	char	filename[PATH_MAX];
        CARD32  size;
        void    *table;
	char    *src;
	int     fildes;
	int     offset = sizeof(CARD32);

	/* try locale specific path 1st */
	if(make_filename(str, filename))
		return (void *)NULL;
	if ((fildes = open(filename, O_RDONLY)) == -1) {
	    /* try default path */
	    strcpy(filename, "/usr/openwin/lib/locale/common/");
	    strcat(filename, str);
	    if ((fildes = open(filename, O_RDONLY)) == -1) {
		return (void*)NULL;
	    }
	}

	if (read(fildes, &size, offset) != offset) {
	    close(fildes);
	    return (void*)NULL;
	}
	if ((table = (void *)Xmalloc(size * 256)) == (void *)NULL) {
	    close(fildes);
	    return (void *)NULL;
	}
#if defined(MAP_FAILED)
	if ((src = mmap(0, size * 256 + offset, PROT_READ,
			MAP_SHARED, fildes, 0)) == MAP_FAILED) {
#else
	if ((src = mmap(0, size * 256 + offset, PROT_READ,
			MAP_SHARED, fildes, 0)) == -1) {
#endif
	    close(fildes);
	    return (void *)NULL;
        }
	src += offset;
	memcpy(table, src, size * 256);
	munmap(src, size * 256 + offset);
	close(fildes);
        return table;
}
